// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "basecircle.h"

#include "framebase.h"
#include "util.h"

BaseCircle::BaseCircle(const BaseCircle& a) : BaseMarker(a)
{
  radii = new double[a.annuli];
  for (int i=0; i<a.annuli; i++)
    radii[i] = a.radii[i];
}

BaseCircle::BaseCircle(FrameBase* p, const Vector& ctr, double r,
		       const char* clr, int w, const char* f, 
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag)
  : BaseMarker(p, ctr, 0, clr, w, f, t, prop, c, tag)
{
  radii = NULL;
}

BaseCircle::~BaseCircle()
{
  if (radii)
    delete [] radii;
}

void BaseCircle::updateCoords(const Matrix& m)
{
  Matrix mm = m;
  for (int i=0; i<annuli; i++)
    radii[i] *= mm[0][0];

  Marker::updateCoords(m);
}

#if __GNUC__ >= 3
void BaseCircle::ps(int mode)
{
  Marker::ps(mode);

  Vector c = center*parent->refToCanvas;

  for (int i=0; i<annuli; i++) {
    double r = radii[i];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    ostringstream str;
    str << "newpath " << c.TkCanvasPs(parent->canvas) << r
	<< " 0 360 arc stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  if (!(properties & INCLUDE)) {
    double r = radii[annuli-1];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    Matrix m = Rotate(degToRad(-45))*Translate(c);
    Vector ll = Vector(r,0)*m;
    Vector ur = Vector(-r,0)*m;

    psColor(mode, "red");

    ostringstream str;
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}
#else
void BaseCircle::ps(int mode)
{
  Marker::ps(mode);

  Vector c = center*parent->refToCanvas;

  for (int i=0; i<annuli; i++) {
    double r = radii[i];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    char buf[256];
    ostrstream str(buf,256);
    str << "newpath " << c.TkCanvasPs(parent->canvas) << r
	<< " 0 360 arc stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  if (!(properties & INCLUDE)) {
    double r = radii[annuli-1];
    if (!(properties & FIXED))
      r *= parent->getZoom();

    Matrix m = Rotate(degToRad(-45))*Translate(c);
    Vector ll = Vector(r,0)*m;
    Vector ur = Vector(-r,0)*m;

    psColor(mode, "red");

    char buf[256];
    ostrstream str(buf,256);
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }
}
#endif

int BaseCircle::isIn(const Vector& v)
{
  // v is in canvas coords

  double r = radii[annuli-1];

  // zero radius

  if (!r)
    return 0;

  if (properties & FIXED)
    r /= parent->getZoom();

  Vector p = v * parent->canvasToRef * Translate(-center);

  if ((p[0]*p[0] + p[1]*p[1])/(r*r) <= 1)
    return 1;
  else
    return 0;
}

// private

void BaseCircle::render(Drawable drawable, const Matrix& matrix, double zoom, 
			RenderMode mode)
{
  setGC(mode);

  Vector c = center * matrix;
  for (int i=0; i<annuli; i++) {
    Vector r = Vector(radii[i],radii[i]);

    if (!(properties & FIXED))
      r *= zoom;

    Vector ll = (c-r).round();
    Vector size = (r*2).round();

    // Verify size is positive
    // XDrawArc is sensative to bad data, and may hang the XServer

    if (size[0]>0 && size[1]>0)
      XDRAWARC(display, drawable, gc, (int)ll[0], (int)ll[1],
	       (int)size[0], (int)size[1], 0, 360*64);
  }

  if (!(properties & INCLUDE)) {
    double r = radii[annuli-1];
    if (!(properties & FIXED))
      r *= zoom;

    Matrix m = Rotate(degToRad(-45))*Translate(c);
    Vector ll = (Vector( r,0) * m).round();
    Vector ur = (Vector(-r,0) * m).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }
}
