// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "basemarker.h"
#include "framebase.h"
#include "util.h"

// Base Markers Public

BaseMarker::BaseMarker(const BaseMarker& a) : Marker(a)
{
  segments = a.segments;
  annuli = a.annuli;
  vertices = NULL;
}

BaseMarker::BaseMarker(FrameBase* p, const Vector& ctr, int s, 
		       const char* clr, int w, const char* f, 
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag)
  : segments(s), Marker(p, ctr, clr, w, f, t, prop, c, tag)
{
  annuli = 0;
  vertices = NULL;
}

void BaseMarker::updateBBox()
{
  // bound marker

  newVertices();
  calcPoints(parent->getZoom());

  Matrix& m = parent->refToCanvas;
  bbox = BBox(vertices[0][0] * m, vertices[0][0] * m); // init the bbox
  for (int i=0; i<annuli; i++)
    for (int j=0; j<segments; j++)
      bbox.bound(vertices[i][j] * m);

  deleteVertices();

  // generate handles

  updateHandles();

  // bound handles

  {
    for (int i=0; i<numHandle; i++)
      bbox.bound(handle[i]);
  }

  // make room for handles

  bbox.expand(3);

  // calculate overall bbox

  calcAllBBox();
}

#if __GNUC__ >= 3
void BaseMarker::ps(int mode)
{
  Marker::ps(mode);

  newVertices();
  calcPoints(parent->getZoom());

  for (int i=0; i<annuli; i++) {
    for (int j=0; j<segments; j++) {
      Vector v =  vertices[i][j] * parent->refToCanvas;

      ostringstream str;
      if (j==0)
	str << "newpath " << endl
	    << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
      else
	str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;
      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    }

    ostringstream str;
    str << "closepath stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  if (!(properties & INCLUDE)) {
    Vector ll = vertices[annuli][0] * parent->refToCanvas;
    Vector ur = vertices[annuli][1] * parent->refToCanvas;

    psColor(mode, "red");

    ostringstream str;
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  deleteVertices();
}
#else
void BaseMarker::ps(int mode)
{
  Marker::ps(mode);

  newVertices();
  calcPoints(parent->getZoom());

  for (int i=0; i<annuli; i++) {
    for (int j=0; j<segments; j++) {
      Vector v =  vertices[i][j] * parent->refToCanvas;

    char buf[256];
    ostrstream str(buf,256);
      if (j==0)
	str << "newpath " << endl
	    << v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
      else
	str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;
      Tcl_AppendResult(parent->interp, buf, NULL);
    }

    char buf[256];
    ostrstream str(buf,256);
    str << "closepath stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  if (!(properties & INCLUDE)) {
    Vector ll = vertices[annuli][0] * parent->refToCanvas;
    Vector ur = vertices[annuli][1] * parent->refToCanvas;

    psColor(mode, "red");

    char buf[256];
    ostrstream str(buf,256);
    str << "newpath " 
	<< ll.TkCanvasPs(parent->canvas) << "moveto"
	<< ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  deleteVertices();
}
#endif

// Base Markers Private

void BaseMarker::render(Drawable drawable, const Matrix& matrix, double zoom,
			RenderMode mode)
{
  setGC(mode);

  newVertices();
  calcPoints(zoom);

  int count = segments+1;
  for (int i=0; i<annuli; i++) {
    XPoint* p = new XPoint[count];
    for (int j=0; j<segments; j++) {
      Vector v = (vertices[i][j] * matrix).round();
      p[j].x = (short)v[0];
      p[j].y = (short)v[1];
    }
    p[segments] = p[0];
    XDRAWLINES(display, drawable, gc, p, count, CoordModeOrigin);
    delete [] p;
  }

  if (!(properties & INCLUDE)) {
    Vector ll = (vertices[annuli][0] * matrix).round();
    Vector ur = (vertices[annuli][1] * matrix).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }

  deleteVertices();
}

void BaseMarker::newVertices() 
{
  if (vertices)
    deleteVertices();

  vertices = new Vector*[annuli+1];
  for (int i=0; i<annuli; i++)
    vertices[i]= new Vector[segments];
  vertices[annuli] = new Vector[2];
}

void BaseMarker::deleteVertices()
{
  for (int i=0; i<annuli+1; i++)
    delete [] vertices[i];
  delete [] vertices;
  vertices = NULL;
}
