// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "boxannulus.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

BoxAnnulus::BoxAnnulus(const BoxAnnulus& a) : BaseBox(a) {}

BoxAnnulus::BoxAnnulus(FrameBase* p, 
		       const Vector& ctr, const Vector& s, double a,
		       const char* clr, int w, const char* f,
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag)
  : BaseBox(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = 1;
  size = new Vector[1];
  size[0] = s;

  strcpy(type,"box annulus");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

BoxAnnulus::BoxAnnulus(FrameBase* p, const Vector& ctr,
		       const Vector& inner, const Vector& outer, int num,
		       double a,
		       const char* clr, int w, const char* f,
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag)
  : BaseBox(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = num+1;
  size = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    size[i] = ((outer-inner)/num)*i+inner;

  strcpy(type,"box annulus");
  numHandle = 4 + annuli;
  handle = new Vector[numHandle];

  updateBBox();
}

BoxAnnulus::BoxAnnulus(FrameBase* p, const Vector& ctr, double a, int an,
		       Vector* s,
		       const char* clr, int w, const char* f, 
		       const char* t, unsigned short prop, const char* c,
		       const List<Tag>& tag)
  : BaseBox(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = an;
  size = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    size[i] = s[i];
  sortSize();

  strcpy(type, "box annulus");
  numHandle = 4 + annuli;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

void BoxAnnulus::editBegin(int h)
{
  if (h<5) {
    switch (h) {
    case 1:
      return;
    case 2:
      size[annuli-1] = Vector(-size[annuli-1][0],size[annuli-1][1]);
      return;
    case 3:
      size[annuli-1] = -size[annuli-1];
      return;
    case 4:
      size[annuli-1] = Vector(size[annuli-1][0],-size[annuli-1][1]);
      return;
    }
  }

  doCallBack(&editBeginCB);
}

void BoxAnnulus::edit(const Vector& v, int h)
{
  // This sizes about the opposite node

  if (h<5) {
    Vector o = size[annuli-1];
    Vector n = center+size[annuli-1]/2 - 
      (v * Translate(-center) * flip * Rotate(-angle) * Translate(center));

    // don't go thru opposite node
    if (n[0]!=0 && n[1]!=0) {
      size[annuli-1] = n;
      center = v+((size[annuli-1]/2) * (Rotate(angle) * flip));

      for (int i=0; i<annuli-1; i++) {
	size[i][0] *= fabs(n[0]/o[0]);
	size[i][1] *= fabs(n[1]/o[1]);
      }
    }
  }
  else {
    // we must have some length
    double l = (v * Translate(-center) * flip * Rotate(-angle) * 2).length();
    if (l>0)
      size[h-5] *= l/size[h-5][0];
  }

  updateBBox();
  doCallBack(&editCB);
}

void BoxAnnulus::editEnd()
{
  for (int i=1; i<annuli; i++)
    size[i] = size[i].abs();
  sortSize();

  updateBBox();
  doCallBack(&editEndCB);
}

void BoxAnnulus::setSize(const Vector& inner, const Vector& outer,
			  int num)
{
  annuli = num+1;
  if (size)
    delete [] size;
  size = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    size[i] = ((outer-inner)/num)*i+inner;
  sortSize();

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

void BoxAnnulus::setSize(const Vector* s, int num)
{
  annuli = num;
  if (size)
    delete [] size;
  size = new Vector[annuli];

  for (int i=0; i<annuli; i++)
    size[i] = s[i];
  sortSize();

  numHandle = 4 + annuli;
  delete [] handle;
  handle = new Vector[numHandle];

  updateBBox();
  doCallBack(&editCB);
}

// private

void BoxAnnulus::updateHandles()
{
  // handles are in canvas coords
  // we can't garantee that the radii have been sorted yet

  Vector max;
  for(int i=0; i<annuli; i++)
    if (max[0]<size[i][0])
      max = size[i];

  Vector s = max;
  if (properties & FIXED)
    s /= parent->getZoom();

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  handle[0] = Vector(-s[0],-s[1])/2 * m;
  handle[1] = Vector( s[0],-s[1])/2 * m;
  handle[2] = Vector( s[0], s[1])/2 * m;
  handle[3] = Vector(-s[0], s[1])/2 * m;

  // the rest are radii

  {
    for (int i=0; i<annuli; i++) {
      Vector ss = Vector((size[i][0])/2.,0) * m;
      if (properties & FIXED)
	ss /= parent->getZoom();
      handle[i+4] = ss;
    }
  }
}

void BoxAnnulus::sortSize()
{
  for (int i=0; i<annuli; i++)
    for (int j=i+1; j<annuli; j++)
      if (size[i][0]>size[j][0]) {
	Vector d = size[i];
	size[i] = size[j];
	size[j] = d;
      }
}

int BoxAnnulus::addSize(const Vector& v)
{
  double l = (v * Translate(-center) * flip * Rotate(-angle) * 2).length();
  if (l>0) {

    // new size array
    Vector* old = size;
    size = new Vector[annuli+1];

    // copy old values
    for (int i=0; i<annuli; i++)
      size[i] = old[i];

    // delete old
    if (old)
      delete [] old;

    // new size on end
    size[annuli] = size[0] * (l/size[0][0]);
    annuli++;

    numHandle = 4 + annuli;
    delete [] handle;
    handle = new Vector[numHandle];

    updateBBox();
  }

  // return handle number
  return annuli+4;
}

void BoxAnnulus::deleteSize(int h)
{
  if (h>4) {
    int hh = h-4-1;

    if (annuli>2 && hh<annuli) {
      // new size array
      Vector* old = size;
      size = new Vector[annuli-1];

      // copy up to size in question
      {
	for (int i=0; i<hh; i++)
	  size[i] = old[i];
      }

      // copy remainder
      {
	for (int i=hh; i<annuli-1; i++)
	  size[i] = old[i+1];
      }

      if (old)
	delete [] old;
      annuli--;

      numHandle = 4 + annuli;
      delete [] handle;
      handle = new Vector[numHandle];

      updateBBox();
      doCallBack(&editCB);
    }
  }
}

// list

void BoxAnnulus::list(ostream& str, CoordSystem sys, SkyFrame sky,
		      SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector v = ptr->mapFromRef(center,sys);
      str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',';
      for (int i=0; i<annuli; i++) {
	Vector r = ptr->mapLenFromRef(size[i],sys);
	str << r[0] << ',' << r[1] << ',' ;
      }
      str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',';
	    for (int i=0; i<annuli; i++) {
	      Vector r = ptr->mapLenFromRef(size[i],sys,ARCSEC);
	      str << r[0] << "\"" << ',' << r[1] << "\"" << ',' ;
	    }
	    str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;

	    str << "box(" << ra << ',' << dec << ',' ;
	    for (int i=0; i<annuli; i++) {
	      Vector r = ptr->mapLenFromRef(size[i],sys,ARCSEC);
	      str << r[0] << "\""<< ',' << r[1] << "\""<< ',' ;
	    }
	    str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',';
	for (int i=0; i<annuli; i++) {
	  Vector r = ptr->mapLenFromRef(size[i],sys);
	  str << r[0] << ',' << r[1] << ',' ;
	}
	str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void BoxAnnulus::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
			  SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      for (int i=0; i<annuli; i++) {
	listProsCoordSystem(str,sys,sky);
	str << "; ";

	Vector r = ptr->mapLenFromRef(size[i],IMAGE);
	str << "box " << setprecision(8) << v << r << radToDeg(angle);

	if (i!=0) {
	  Vector r1 = ptr->mapLenFromRef(size[i-1],IMAGE);
	  str << " & !box " << setprecision(8) << v << r1 << radToDeg(angle);
	}

	str << delim;
      }
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  for (int i=0; i<annuli; i++) {
	    listProsCoordSystem(str,sys,sky);
	    str << "; ";

	    Vector r = ptr->mapLenFromRef(size[i],sys,ARCSEC);
	    str << "box " << setprecision(8) << v[0] << "d " << v[1] << "d "
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);

	    if (i!=0) {
	      Vector r1 = ptr->mapLenFromRef(size[i-1],sys,ARCSEC);
	      str << " & !box " << setprecision(8) 
		  << v[0] << "d " << v[1] << "d " 
		  << r1[0] << "\" " << r1[1] << "\" " << radToDeg(angle);
	    }

	    str << delim;
	  }
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;

	  for (int i=0; i<annuli; i++) {
	    listProsCoordSystem(str,sys,sky);
	    str << "; ";

	    Vector r = ptr->mapLenFromRef(size[i],sys,ARCSEC);
	    str << "box " << ra << ' ' << dec << ' ' 
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);

	    if (i!=0) {
	      Vector r1 = ptr->mapLenFromRef(size[i-1],sys,ARCSEC);
	      str << " & !box " << ra << ' ' << dec << ' ' 
		  << r1[0] << "\" " << r1[1] << "\" " << radToDeg(angle);
	    }

	    str << delim;
	  }
	}
	break;
      }
    }
    else 
      str << "";
    break;
  }
}

void BoxAnnulus::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // all coords are in image coords

  //  if (!(properties&INCLUDE))
  //    str << '-';

  for (int i=0; i<annuli; i++) {
    Vector v = ptr->mapFromRef(center,IMAGE);
    str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',' 
	<< size[i][0] << ',' << size[i][1] << ',' << radToDeg(angle) << ')';

    if (i!=0)
      str << " & !box(" << setprecision(8) << v[0] << ',' << v[1] 
	  << ',' << size[i-1][0] << ',' << size[i-1][1] << ',' 
	  << radToDeg(angle) << ')';

    str << delim;
  }
}
