// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#if __GNUC__ >= 3
#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;
#else
#include <iostream.h>
#include <strstream.h>
#include <iomanip.h>
#endif

#include "util.h"

int sexSign;     // used by parser and lex to indicate sign of dms or hms

void swap2(char* src, char* dest) {
  *(dest  ) = *(src+1);
  *(dest+1) = *(src  );
}

void swap4(char* src, char* dest) {
  *(dest  ) = *(src+3);
  *(dest+1) = *(src+2);
  *(dest+2) = *(src+1);
  *(dest+3) = *(src  );
}

void swap8(char* src, char* dest) {
  *(dest  ) = *(src+7);
  *(dest+1) = *(src+6);
  *(dest+2) = *(src+5);
  *(dest+3) = *(src+4);
  *(dest+4) = *(src+3);
  *(dest+5) = *(src+2);
  *(dest+6) = *(src+1);
  *(dest+7) = *(src  );
}

int lsb()
{
  return (*(short *)"\001\000" & 0x0001);
}

char* dupstr(const char* str)
{
  char* copy;
  if (str) {
    copy=new char[strlen(str)+1];
    strcpy(copy,str);
  }
  else
    copy=NULL;

  return copy;
}

char* toLower(const char* str)
{
  char* r = dupstr(str);
  char* ptr = r;
  while (*ptr) {
    *ptr = (char)(tolower(((int)(*ptr))));
    ptr++;
  }
  return r;
}

char* toUpper(const char* str)
{
  char* r = dupstr(str);
  char* ptr =r;
  while (*ptr) {
    *ptr = (char)(toupper(((int)(*ptr))));
    ptr++;
  }
  return r;
}

double degToRad(double d)
{
  return M_PI * d / 180.;
}

double radToDeg(double r)
{
  double d = 180. * r / M_PI;

  if (d > 0) 
    while(d >= 360)
      d -= 360.;
  else
    while(d < 0)
      d += 360.;

  return d;
}

double hmsToDegree(int sign, double hour, double min, double sec)
{
  return dmsToDegree(sign,hour,min,sec)/24.*360.;
}

double dmsToDegree(int sign, double degree, double min, double sec)
{
  double r = sign * (fabs(degree) + (min/60.) + (sec/60./60.));
  return r;
}

double parseDMS(const char* d)
{
  char* dms = dupstr(d); // its going to get clobbered
  char* str = dms;

  int sign = (d[0]!='-') ? 1 : -1;
  int degree = atoi(strtok(str,":"));
  int minute = atoi(strtok(NULL,":"));
  float sec = atof(strtok(NULL,":"));

  delete [] dms;

  return dmsToDegree(sign,degree,minute,sec);
}

unsigned char decodeMask(unsigned char mask, int* s)
{
  *s=0;
  for (int i=0; i<8; i++, (*s)++) {
    if (mask & 0x80)
      break;
    mask <<= 1;
  }
  return mask;
}

unsigned short decodeMask(unsigned short mask, int* s)
{
  // first calc ffs for mask

  unsigned short m1 = mask;
  int m1s;
  for (m1s=0; m1s<16; m1s++) {
    if (m1 & 0x1)
      break;
    m1 >>= 1;
  }

  // then shift mask to mask a char

  for (int i=0; i<16; i++) {
    if (mask & 0x8000)
      break;
    mask <<= 1;
  }
  mask >>= 8;

  // now calc ffs for new mask

  unsigned short m2 = mask;
  int m2s;
  for (m2s=0; m2s<16; m2s++) {
    if (m2 & 0x1)
      break;
    m2 >>= 1;
  }

  *s = m1s-m2s;
  return mask;
}

int decodeMask(unsigned long mask)
{
  switch (mask) {
  case 0xff:
    return 0;
  case 0xff00:
    return 8;
  case 0xff0000:
    return 16;
  case 0xff000000:
    return 24;
  }
}

double RGB2Gray(double red, double green, double blue)
{
  return 0.30*red + 0.59*green + 0.11*blue;
}

unsigned char RGB2Gray(unsigned char red, unsigned char green, 
		       unsigned char blue)
{
  return (unsigned char)(0.30*red + 0.59*green + 0.11*blue);
}

void RGB2CMYK(unsigned char red, unsigned char green, unsigned char blue,
	      unsigned char* cyan, unsigned char* magenta, 
	      unsigned char* yellow, unsigned char* black)
{
  // convert To CMY

  *cyan = 255-red;
  *magenta = 255-green;
  *yellow = 255-blue;
  *black =0;

  // determine black

  *black = 255;
  if (*cyan < *black)
    *black = *cyan;
  if (*magenta < *black)
    *black = *magenta;
  if (*yellow < *black)
    *black = *yellow;

  // substract out black

  *cyan -= *black;
  *magenta -= *black;
  *yellow -= *black;
}

ostream& psColorGray(const char* colorName, ostream& str)
{
  str << dec; // we want decimal
  if (!strcmp(colorName,"black"))
    str << "0.0";
  else if (!strcmp(colorName,"white"))
    str << "1.0";
  else if (!strcmp(colorName,"red"))
    str << RGB2Gray(1.,0.,0.);
  else if (!strcmp(colorName,"green"))
    str << RGB2Gray(0.,1.,0.);
  else if (!strcmp(colorName,"blue"))
    str << RGB2Gray(0.,0.,1.);
  else if (!strcmp(colorName,"cyan"))
    str << RGB2Gray(0.,1.,1.);
  else if (!strcmp(colorName,"magenta"))
    str << RGB2Gray(1.,0.,1.);
  else if (!strcmp(colorName,"yellow"))
    str << RGB2Gray(1.,1.,0.);
  else {
    cerr << "psColorGray Internal Error: Unable to Parser color: "
	 << colorName << endl;
    str << "0.0";
  }

  return str;
}

ostream& psColorRGB(const char* colorName, ostream& str)
{
  str << dec; // we want decimal
  if (!strcmp(colorName,"black"))
    str << "0 0 0";
  else if (!strcmp(colorName,"white"))
    str << "1 1 1";
  else if (!strcmp(colorName,"red"))
    str << "1 0 0";
  else if (!strcmp(colorName,"green"))
    str << "0 1 0";
  else if (!strcmp(colorName,"blue"))
    str << "0 0 1";
  else if (!strcmp(colorName,"cyan"))
    str << "0 1 1";
  else if (!strcmp(colorName,"magenta"))
    str << "1 0 1";
  else if (!strcmp(colorName,"yellow"))
    str << "1 1 0";
  else {
    cerr << "psColorRGB Internal Error: Unable to Parser color: "
	 << colorName << endl;
    str << "0 0 0";
  }

  return str;
}

ostream& psColorCMYK(const char* colorName, ostream& str)
{
  str << dec; // we want decimal
  if (!strcmp(colorName,"black"))
    str << "0 0 0 1";
  else if (!strcmp(colorName,"white"))
    str << "0 0 0 0";
  else if (!strcmp(colorName,"red"))
    str << "0 1 1 0";
  else if (!strcmp(colorName,"green"))
    str << "1 0 1 0";
  else if (!strcmp(colorName,"blue"))
    str << "1 1 0 0";
  else if (!strcmp(colorName,"cyan"))
    str << "1 0 0 0";
  else if (!strcmp(colorName,"magenta"))
    str << "0 1 0 0";
  else if (!strcmp(colorName,"yellow"))
    str << "0 0 1 0";
  else {
    cerr << "psColorCMYK Internal Error: Unable to Parser color: "
	 << colorName << endl;
    str << "0 0 0 1";
  }

  return str;
}

char* psStr = NULL; // psQuote returned string
char* psQuote(const char* str)
{
  // we must must quote '(', ')', and '\'
  if (psStr)
    delete [] psStr;

  psStr = new char[strlen(str)*2+1]; // worst case size
  
  char* out = psStr;
  const char* in = str;
  while (in && *in) {
    if (*in == '(' || *in == ')' || *in == '\\')
      *out++ = '\\';
    *out++ = *in++;
  }

  *out++ = '\0'; // terminating char
  return psStr;
}

int fCompare(const void* a, const void* b)
{
  float* aa = (float*)a;
  float* bb = (float*)b;

  if (*aa < *bb)
    return -1;
  if (*aa > *bb)
    return 1;
  return 0;
}

int dCompare(const void* a, const void* b)
{
  double* aa = (double*)a;
  double* bb = (double*)b;

  if (*aa < *bb)
    return -1;
  if (*aa > *bb)
    return 1;
  return 0;
}

