#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc OpenURL {} {
    global http

    set fn {}
    if {[EntryDialog [msgcat::mc {URL}] [msgcat::mc {Enter World Wide Web Location (URL)}] 80 fn {}]} {
	StartLoad
	LoadURL $fn
	FinishLoad
    }
}

proc LoadURL {url} {
    if {[string length $url] == 0} {
	return
    }

    ParseURL $url r
    switch -- $r(scheme) {
	ftp {LoadURLFTP $r(authority) $r(path)}
	file {LoadURLFile $r(path)}
	http -
	default {LoadURLHTTP $url}
    }
}

proc LoadURLFTP {host path} {
    global loadParam
    global ds9
    global debug

    set ftp [ftp::Open $host "ftp" "-ds9@" -mode passive]
    if {$ftp > -1} {
	set fn "$ds9(tmpdir)/[file tail $path]"
	set ftp::VERBOSE $debug(tcl,ftp)
	set "ftp::ftp${ftp}(Output)" FTPLog
	ftp::Type $ftp binary
	if [ftp::Get $ftp $path $fn] {
	    # alloc it because we are going to delete it after load
	    set loadParam(load,type) allocgz
	    set loadParam(load,layer) {}
	    set loadParam(file,type) fits
	    set loadParam(file,mode) {}
	    set loadParam(file,name) $fn
	    set loadParam(file,fn) $fn
	    ConvertFile
	    ProcessLoad
	}

	ftp::Close $ftp

	if [file exists $fn] {
	    catch {file delete -force $fn}
	}
    }
}

proc LoadURLFile {path} {
    global loadParam
    global ds9

    # alloc it because we can't assume it will last
    set loadParam(load,type) allocgz
    set loadParam(load,layer) {}
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    set loadParam(file,name) $path
    set loadParam(file,fn) $path
    ConvertFile
    ProcessLoad
}

proc LoadURLHTTP {url} {
    global ds9
    global ihttp

    ParseURL $url r
    set fn "$ds9(tmpdir)/[file tail $r(path)]"

    set code 200
    set meta {}
    set mime "application/fits"
    set encoding {}

    set ch [open $fn w]
    set token [http::geturl $url \
		   -protocol 1.0 \
		   -timeout $ihttp(timeout) \
		   -channel $ch \
		   -binary 1 \
		   -headers "[ProxyHTTP]"]

    # reset errorInfo (may be set in http::geturl)
    global errorInfo
    set errorInfo {}

    catch {close $ch}

    upvar #0 $token t

    # Code
    set code [http::ncode $token]

    # Meta
    set meta $t(meta)

    # Mime-type
    # we want to strip and extra info after ';'
    regexp -nocase {([^;])*} $t(type) mime

    # Content-Encoding
    foreach {name value} $meta {
	if {[regexp -nocase ^content-encoding $name]} {
	    switch -- [string tolower $value] {
		gzip -
		x-gzip {set encoding gzip}
		compress -
		bzip2 {set encoding bzip2}
		Z {set encoding compress}
		pack -
		z {set encoding pack}
		default {}
	    }
	}
    }

    HTTPLog $token
    # Result?
    switch -- $code {
	200 {}
	default {
	    Error "HTTP [msgcat::mc {Error}] $code"
	    return
	}
    }

    http::cleanup $token

    global debug
    if {$debug(tcl,hv)} {
	puts stderr "Load HTTP: fn $fn : code $code : meta $meta : mime $mime : encoding $encoding"
    }

    # NOTE: error notices may come as text/html
    switch -- [string tolower $mime] {
	"image/fits" -
	"application/fits" {}

	"application/fits-image" -
	"application/fits-table" -
	"application/fits-group" {}

	"image/x-fits" -
	"binary/x-fits" -
	"application/x-fits" {}

	"image/x-gfits" -
	"binary/x-gfits" -
	"image/gz-fits" -
	"display/gz-fits" {set encoding gzip}

	"image/bz2-fits" -
	"display/bz2-fits" {set encoding bzip2}

	"image/x-cfits" -
	"binary/x-cfits" {set encoding compress}

	"image/x-zfits" -
	"binary/x-zfits" {set encoding pack}

	default {
	    Error "[msgcat::mc {File not Found or Unable to load FITS data MIME type}] $mime"
	    return
	}
    }

    # alloc it because we are going to delete it after load
    StartLoad
    global loadParam
    set loadParam(load,type) allocgz
    set loadParam(load,layer) {}
    set loadParam(file,type) fits
    set loadParam(file,mode) {}
    set loadParam(file,name) $fn
    set loadParam(file,fn) $fn

    # may have to convert the file, based on content-encoding
    switch -- "$encoding" {
	bzip2 {
	    catch {set ch [open "| bunzip2 < $fn " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
	compress {
	    catch {set ch [open "| uncompress < $fn " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
	pack {
	    catch {set ch [open "| pcat $fn " r]}
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	}
    }

    ProcessLoad
    FinishLoad

    if {[file exists $fn]} {
	catch {file delete -force $fn}
    }
}

proc MultiLoad {{layer {}}} {
    global ds9
    global current

    global debug
    if {$debug(tcl,layout)} {
	puts stderr "MultiLoad"
    }

    if {$layer != {}} {
	return
    }

    if {$current(frame) != {}} {
	if {![$current(frame) has fits]} {
	    return
	}
	switch -- [$current(frame) get type] {
	    base {CreateFrame}
	    rgb {}
	    3d {Create3DFrame}
	}
    } else {
	CreateFrame
	return
    }

    # go into tile mode if more than one
    set cnt [llength $ds9(frames)]
    if {$cnt > 1 && $current(display) != "tile"} {
	set current(display) tile
	DisplayMode
    }
}

proc MultiLoadBase {} {
    global ds9
    global current

    global debug
    if {$debug(tcl,layout)} {
	puts stderr "MultiLoadBase"
    }

    if {$current(frame) != {}} {
	if {![$current(frame) has fits]} {
	    return
	}
	CreateFrame
    } else {
	CreateFrame
	return
    }

    # go into tile mode if more than one
    set cnt [llength $ds9(frames)]
    if {$cnt > 1 && $current(display) != "tile"} {
	set current(display) tile
	DisplayMode
    }
}

proc MultiLoadRGB {} {
    global ds9
    global current

    global debug
    if {$debug(tcl,layout)} {
	puts stderr "MultiLoadRGB"
    }

    if {$current(frame) != {}} {
	if {![$current(frame) has fits]} {
	    return
	}
	CreateRGBFrame
    } else {
	CreateRGBFrame
	return
    }

    # go into tile mode if more than one
    set cnt [llength $ds9(frames)]
    if {$cnt > 1 && $current(display) != "tile"} {
	set current(display) tile
	DisplayMode
    }
}

# used by backup
proc ProcessLoad {{err 1}} {
    global current
    global loadParam

    if [catch {
	switch -- $loadParam(load,type) {
	    alloc -
	    allocgz {$current(frame) load $loadParam(file,type) \
			 $loadParam(file,mode) \
			 \{$loadParam(file,name)\} \
			 $loadParam(load,type) \
			 \{$loadParam(file,fn)\} \
			 $loadParam(load,layer)}
	    channel {
		fconfigure $loadParam(channel,name) -translation binary \
		    -encoding binary
		$current(frame) load $loadParam(file,type) \
		    $loadParam(file,mode) \
		    \{$loadParam(file,name)\} \
		    $loadParam(load,type) \
		    $loadParam(channel,name) \
		    $loadParam(load,layer)

		# clean up
		catch {close $loadParam(channel,name)}
	    }
	    mmap -
	    mmapincr {$current(frame) load $loadParam(file,type) \
			  $loadParam(file,mode) \
			  \{$loadParam(file,name)\} \
			  $loadParam(load,type) \
			  $loadParam(load,layer)}
	    smmap {$current(frame) load $loadParam(file,type) \
		       $loadParam(file,mode) \
		       \{$loadParam(file,header)\} \
		       \{$loadParam(file,name)\} \
		       $loadParam(load,type) \
		       $loadParam(load,layer)}
	    shared {$current(frame) load $loadParam(file,type) \
			$loadParam(file,mode) \
			\{$loadParam(file,name)\} \
			$loadParam(load,type) \
			$loadParam(shared,idtype) \
			$loadParam(shared,id) \
			$loadParam(load,layer)}
	    sshared {$current(frame) load $loadParam(file,type) \
			 $loadParam(file,mode) \
			 \{$loadParam(file,name)\} \
			 $loadParam(load,type) \
			 $loadParam(shared,idtype) \
			 $loadParam(shared,hdr) \
			 $loadParam(shared,id) \
			 $loadParam(load,layer)}
	    socket -
	    socketgz {$current(frame) load $loadParam(file,type) \
			  $loadParam(file,mode) \
			  \{$loadParam(file,name)\} \
			  $loadParam(load,type) \
			  $loadParam(socket,id) \
			  $loadParam(load,layer)}
	    var {$current(frame) load $loadParam(file,type) \
		     $loadParam(file,mode) \
		     \{$loadParam(file,name)\} \
		     $loadParam(load,type) \
		     $loadParam(var,name) \
		     $loadParam(load,layer)}
	    photo {$current(frame) load $loadParam(file,type) \
		       $loadParam(file,mode) \
		       $loadParam(var,name) \{$loadParam(file,name)\}
	    }
	}
    } rr] {
	if {$err} {
	    Error "[msgcat::mc {Unable to load}] $loadParam(file,type) $loadParam(file,mode) $loadParam(file,name)"
	}
    }

    # save loadParam
    if {$loadParam(load,layer) == {}} {
	switch -- [$current(frame) get type] {
	    base -
	    3d {ProcessLoadSaveParams $current(frame)}
	    rgb {
		switch -- $loadParam(file,mode) {
		    {rgb image} -
		    {rgb cube} {ProcessLoadSaveParams $current(frame)}
		    default {
			ProcessLoadSaveParams \
			    "$current(frame)[$current(frame) get rgb channel]"
		    }
		}
	    }
	}
    }

    unset loadParam
    return $rr
}

proc ProcessLoadSaveParams {varname} {
    global loadParam
    global current

    switch $loadParam(file,mode) {
	slice -
	{mosaic wcs} -
	{mosaic iraf} {
	    # special case
	    global $varname
	    if {[info exists $varname]} {
		set varname "$varname.[$current(frame) get fits count]"
	    }
	}
    }

    global $varname
    if [info exists $varname] {
	unset $varname
    }

    array set $varname [array get loadParam]

    # always save absolute path
    upvar #0 $varname var
    if {[file pathtype $var(file,name)] == {relative}} {
	set var(file,name) [file join [pwd] $var(file,name)]
    }
}

proc StartLoad {} {
    SetWatchCursor
}

proc FinishLoadPre {} {
    global loadParam
    global current
    global threed

    # generate grid so updatemenu is correct
    GridUpdate

    # generate contour so updatemenu is correct
    UpdateContourScale
    ContourUpdate

    # just in case, frame may have been deleted before FinishLoad during startup
    if {$current(frame) == {}} {
	return
    }

    # if header(s) were open, remove them
    DestroyHeader $current(frame)

    # Cube?
    if [$current(frame) has fits cube] {
	CubeDialog
    }
}

proc FinishLoadPost {} {
    UpdateDS9
    UnsetWatchCursor
}

proc FinishLoad {} {
    FinishLoadPre
    FinishLoadPost
}

proc ConvertFile {} {
    global ds9

    foreach t $ds9(fileProcs) {
	if [$t] {
	    return
	}
    }
}

# File Types

proc Stdin {} {
    global loadParam

    # find -, -[], -[foo] but not -abc
    if [regexp -- {^-(\[.*)?$} $loadParam(file,name)] {
	set loadParam(load,type) allocgz
	set loadParam(file,name) "stdin[string range $loadParam(file,name) 1 end]"
	set loadParam(file,fn) $loadParam(file,name)
	return 1
    } elseif {[string range $loadParam(file,name) 0 4] == "stdin" || 
	      [string range $loadParam(file,name) 0 4] == "STDIN"} {
	set loadParam(load,type) allocgz
	set loadParam(file,name) "stdin[string range $loadParam(file,name) 5 end]"
	set loadParam(file,fn) $loadParam(file,name)
	return 1
    }

    return 0
}

proc BZip2Fits {} {
    global loadParam

    if { [regexp {(.*)\.bz2($|\[)} $loadParam(file,name) matched root] } {
	if [catch {set ch [open "| bunzip2 < $root.bz2 " r]}] {
	    return 0
	}
	set loadParam(load,type) channel
	set loadParam(channel,name) $ch
	return 1
    }

    return 0
}

proc CompressFits {} {
    global loadParam

    if { [regexp {(.*)\.Z($|\[)} $loadParam(file,name) matched root] } {
	if [catch {set ch [open "| uncompress < $root.Z " r]}] {
	    return 0
	}
	set loadParam(load,type) channel
	set loadParam(channel,name) $ch
	return 1
    }

    return 0
}

proc PackFits {} {
    global loadParam

    if { [regexp {(.*)\.z($|\[)} $loadParam(file,name) matched root] } {
	if [catch {set ch [open "| pcat $root.z " r]}] {
	    return 0
	}
	set loadParam(load,type) channel
	set loadParam(channel,name) $ch
	return 1
    }

    return 0
}

proc GzipFits {} {
    global loadParam

    if { [regexp {(.*)\.gz($|\[)} $loadParam(file,name) matched root] } {
	set loadParam(load,type) allocgz
	set loadParam(file,fn) $loadParam(file,name)
	return 1
    }

    return 0
}

proc FTZFits {} {
    global loadParam

    if { [regexp -nocase {(.*)\.(ftz)($|\[)} $loadParam(file,name) matched root ext] } {
	set loadParam(load,type) allocgz
	set loadParam(file,fn) $loadParam(file,name)
	return 1
    }

    return 0
}

proc ExternalFits {} {
    global loadParam
    global extFits

    foreach id [array names extFits] {
	if {[string match $id "$loadParam(file,name)"]} {
	    regsub -all {\$filename} $extFits($id) "$loadParam(file,name)" \
		result
	    set cmd "| $result"
	    if [catch {set ch [open "$cmd" r]} err] {
		Error "open $cmd failed: $err"
		return 0
	    }
	    set loadParam(load,type) channel
	    set loadParam(channel,name) $ch
	    return 1
	}
    }
    return 0
}

# Preserve

proc ProcessPreserveCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global ds9
    global scale
    global panzoom
    global marker

    switch -- [string tolower [lindex $var $i]] {
	scale {
	    incr i
	    set scale(preserve) [FromYesNo [lindex $var $i]]
	    PreserveScale
	}
	pan {
	    incr i
	    set panzoom(preserve) [FromYesNo [lindex $var $i]]
	    PreservePan
	}
	marker -
	regions {
	    incr i
	    set marker(preserve) [FromYesNo [lindex $var $i]]
	    MarkerPreserve
	}
    }
}

proc ProcessSendPreserveCmd {proc id param} {
    global scale
    global panzoom
    global marker

    switch -- [string tolower $param] {
	scale {$proc $id [ToYesNo $scale(preserve)]}
	pan {$proc $id [ToYesNo $panzoom(preserve)]}
	regions {$proc $id [ToYesNo $marker(preserve)]}
    }
}

# Update

proc ProcessUpdateCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global current
    global ds9

    if {$current(frame) == {}} {
	return
    }

    if {[lindex $var $i] != {} && [string range [lindex $var $i] 0 0] != {-}} {
	switch -- [string tolower [lindex $var $i]] {
	    on -
	    yes {set ds9(idletasks) 1}
	    no -
	    off {set ds9(idletasks) 0}

	    now {
		if {[string is integer [lindex $var [expr $i+1]]]} {
		    $current(frame) update now \
			[lindex $var [expr $i+1]] \
			[lindex $var [expr $i+2]] [lindex $var [expr $i+3]] \
			[lindex $var [expr $i+4]] [lindex $var [expr $i+5]]
		    
		    incr i 5
		} else {
		    $current(frame) update now
		}
	    }
	    {} {
		$current(frame) update
		incr i -1
	    }

	    default {
		$current(frame) update \
		    [lindex $var $i] \
		    [lindex $var [expr $i+1]] [lindex $var [expr $i+2]] \
		    [lindex $var [expr $i+3]] [lindex $var [expr $i+4]]
		incr i 4
	    }
	}
    } else {
	$current(frame) update
	incr i -1
    }
}
