/*
  operators.hh, copyright (c) 2006 by Vincent Fourmond: 
  The class definitions for the operators.
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

/* expression.hh should be included */

namespace SCalc {
  class Operator : public  Expression {
  protected:
    // The left and rigth members of the operator:
    Expression * left;
    Expression * right;
    virtual const char * name() = 0;

    /// This function should be redefined by the children to
    /// implement specific rules for simplification. It should
    /// return NULL if nothing has been found, or a
    /// simplified expression is case something was simplified
    virtual Expression * specific_rules(Expression * l,
					Expression * r) { return NULL;};
    
    /// To be redefined by children to 'remake' operation if
    /// necessary... times->op(a,b) -> "a * b"...
    virtual Operator * op(Expression * l, Expression * r) = 0;
  public:
    Operator(Session * s, Expression *l, Expression *r) : Expression(s) 
    { left = l; right = r;};
    virtual ~Operator() {delete left; delete right;};

    // Also pure virtual, as there is no implementation...
    virtual double evaluate(const double *, const double *) = 0;
    virtual void dump(::std::ostream & stream = ::std::cerr);
    virtual std::set<int> used_variables();

    virtual std::string pretty_print();

    /// One function for all, the subclasses should better reimplement the
    /// specific_rules.
    virtual Expression * simplify();
  };

  class Plus : public Operator {
  protected:
    virtual const char * name() { return "+";};
    virtual Operator * op(Expression * l, Expression * r) 
    { return new Plus(session(), l,r);};

    virtual Expression * specific_rules(Expression * l,
					Expression * r);
  public:
    Plus(Session * s, Expression *l, Expression *r) : Operator(s,l,r) {;};
    virtual ~Plus() {;} ;

    virtual double evaluate(const double * vals, const double * vars) 
    { return 
	left->evaluate(vals,vars) + 
	right->evaluate(vals,vars);
    };

    virtual Expression * derive(int id);
    virtual Expression * copy() 
    { 
      return new Plus(session(), 
		      left->copy(),
		      right->copy());
    };
  };

  class Minus : public Operator {
  protected:
    virtual const char * name() { return "-";};
    virtual Operator * op(Expression * l, Expression * r) 
    { return new Minus(session(), l,r);};

    virtual Expression * specific_rules(Expression * l,
					Expression * r);
  public:
    Minus(Session * s, Expression *l, Expression *r) : Operator(s,l,r) {;};
    virtual ~Minus() {;} ;

    virtual double evaluate(const double *vals, const double * vars) 
    { return 
	left->evaluate(vals,vars) - 
	right->evaluate(vals,vars);
    };

    virtual Expression * derive(int id);

    virtual Expression * copy() 
    { 
      return new Minus(session(), 
		       left->copy(),
		       right->copy());
    };
  };

  class Times : public Operator {
  protected:
    virtual const char * name() { return "*";};
    virtual Operator * op(Expression * l, Expression * r) 
    { return new Times(session(), l,r);};

    virtual Expression * specific_rules(Expression * l,
					Expression * r);
  public:
    Times(Session * s, Expression *l, Expression *r) : Operator(s,l,r) {;};
    virtual ~Times() {;} ;

    virtual double evaluate(const double *vals, const double * vars) 
    { return 
	left->evaluate(vals,vars) * 
	right->evaluate(vals,vars);
    };

    virtual Expression * derive(int id);

    virtual Expression * copy() 
    { 
      return new Times(session(), 
		       left->copy(),
		       right->copy());
    };
  };

  class Divides : public Operator {
  protected:
    virtual const char * name() { return "/";};
    virtual Operator * op(Expression * l, Expression * r) 
    { return new Divides(session(), l,r);};

    virtual Expression * specific_rules(Expression * l,
					Expression * r);
  public:
    Divides(Session * s, Expression *l, Expression *r) : Operator(s,l,r) {;};
    virtual ~Divides() {;} ;

    virtual double evaluate(const double *vals, const double * vars) 
    { return left->evaluate(vals,vars) / 
	right->evaluate(vals,vars);};

    virtual Expression * derive(int id);

    virtual Expression * copy() 
    { 
      return new Divides(session(), 
			 left->copy(),
			 right->copy());
    };
  };

  class Power : public Operator {
  protected:
    virtual const char * name() { return "^";};
    virtual Operator * op(Expression * l, Expression * r) 
    { return new Power(session(), l,r);};

    virtual Expression * specific_rules(Expression * l,
					Expression * r);
  public:
    Power(Session * s, Expression *l, Expression *r) : Operator(s,l,r) {;};
    virtual ~Power() {;} ;

    virtual double evaluate(const double *vals, const double * vars);

    virtual Expression * derive(int id);

    virtual Expression * copy() 
    { 
      return new Power(session(), 
		       left->copy(),
		       right->copy());
    };
  };

};
