/*
 * scamper_target.c
 *
 * $Id: scamper_target.c,v 1.11 2007/05/08 02:42:31 mjl Exp $
 *
 * Copyright (C) 2005-2007 The University of Waikato
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in.h>

#if defined(__APPLE__)
#include <stdint.h>
#endif

#include <stdlib.h>
#include <assert.h>

#if defined(DMALLOC)
#include <string.h>
#include <dmalloc.h>
#endif

#include "scamper.h"
#include "mjl_splaytree.h"
#include "scamper_addr.h"
#include "scamper_list.h"
#include "scamper_tlv.h"
#include "scamper_trace.h"
#include "scamper_target.h"
#include "utils.h"

static splaytree_t *tree[3];

static int target_addr4_cmp(const void *va, const void *vb)
{
  return addr4_cmp(((const scamper_target_t *)va)->addr->addr,
		   ((const scamper_target_t *)vb)->addr->addr);
}

static int target_addr6_cmp(const void *va, const void *vb)
{
  return addr6_cmp(((const scamper_target_t *)va)->addr->addr,
		   ((const scamper_target_t *)vb)->addr->addr);
}

static int target_link(scamper_target_t *target)
{
  /*
   * if splaytree_insert returns 1, it means the target was inserted
   * successfully.
   */
  if(splaytree_insert(tree[target->addr->type-1], target) != NULL)
    {
      return 0;
    }

  return -1;
}

static int target_unlink(scamper_target_t *target)
{
  splaytree_remove_item(tree[target->addr->type-1], target);
  return 0;
}

scamper_target_t *scamper_target_alloc(scamper_addr_t *addr)
{
  scamper_target_t *target;
  int linked = -1;

  if((target = malloc_zero(sizeof(scamper_target_t))) != NULL)
    {
      target->addr = scamper_addr_use(addr);

      if((linked = target_link(target)) == -1)
	{
	  goto err;
	}
    }

  return target;

 err:
  if(linked != -1) target_unlink(target);
  if(target->addr != NULL) scamper_addr_free(target->addr);
  return NULL;
}

void scamper_target_free(scamper_target_t *target)
{
  target_unlink(target);
  scamper_addr_free(target->addr);
  free(target);
  return;
}

scamper_target_t *scamper_target_find(scamper_addr_t *addr)
{
  scamper_target_t key;
  key.addr = addr;
  return splaytree_find(tree[addr->type-1], &key);
}

int scamper_target_count()
{
  int i;
  int count = 0;
  for(i=0; i<2; i++) count += splaytree_count(tree[i]);
  return count;
}

int scamper_targets_init()
{
  if((tree[0] = splaytree_alloc(target_addr4_cmp)) == NULL)
    {
      return -1;
    }

  if((tree[1] = splaytree_alloc(target_addr6_cmp)) == NULL)
    {
      return -1;
    }

  tree[2] = NULL;

  return 0;
}

void scamper_targets_cleanup()
{
  int i;
  for(i=0; i<3; i++)
    {
      if(tree[i] != NULL) splaytree_free(tree[i], NULL);
      tree[i] = NULL;
    }

  return;
}
