
############################################################
### TREE window

namespace eval ::tree {}
set tree(training) 0
set tree(locked) 0
set tree(base) 0
set tree(status) ""

proc ::tree::ConfigMenus {{lang ""}} {
  if {! [winfo exists .treeWin]} { return }
  if {$lang == ""} { set lang $::language }
  set m .treeWin.menu
  foreach menu {file sort opt help} tag {File Sort Opt Help} {
    configMenuName $m.$menu Tree$tag $lang
  }
  foreach idx {0 1 3 4 6 8} tag {Save Fill Best Graph Copy Close} {
    configMenuText $m.file.m $idx TreeFile$tag $lang
  }
  foreach idx {0 1 2 3} tag {Alpha ECO Freq Score} {
    configMenuText $m.sort.m $idx TreeSort$tag $lang
  }
  foreach idx {0 1 3} tag {Lock Training Autosave} {
    configMenuText $m.opt.m $idx TreeOpt$tag $lang
  }
  foreach idx {0 1} tag {Tree Index} {
    configMenuText $m.help.m $idx TreeHelp$tag $lang
  }
}

proc ::tree::copyToSelection {args} {
  set sel [join [.treeWin.f.tl get 0 end] "\n"]
  append sel "\n"
  return $sel
}

proc ::tree::make {} {
  global tree treeWin highcolor geometry helpMessage
  if {[winfo exists .treeWin]} {
    focus .
    destroy .treeWin
    set treeWin 0
    return
  }
  toplevel .treeWin
  set w .treeWin
  setWinLocation $w

  # Set the tree window title now:
  wm title $w "Scid: [tr WindowsTree]"
  set treeWin 1
  set tree(training) 0

  bind $w <Destroy> { set treeWin 0; set tree(locked) 0 }
  bind $w <F1> { helpWindow Tree }
  bind $w <Escape> { .treeWin.buttons.stop invoke }
  standardShortcuts $w

  frame $w.menu
  pack $w.menu -side top -fill x
  $w configure -menu $w.menu
  menubutton $w.menu.file -text TreeFile -menu $w.menu.file.m
  menubutton $w.menu.sort -text TreeSort -menu $w.menu.sort.m
  menubutton $w.menu.opt  -text TreeOpt  -menu $w.menu.opt.m
  menubutton $w.menu.help -text TreeHelp -menu $w.menu.help.m
  foreach i {file sort opt help} {
    menu $w.menu.$i.m -tearoff 0
    pack $w.menu.$i -side left
  }

  $w.menu.file.m add command -label TreeFileSave -command {
    busyCursor .
    update
    if {[catch {sc_tree write $tree(base)} result]} {
      tk_messageBox -type ok -icon warning -title "Scid: Error writing file" \
        -message $result
    }
    unbusyCursor .
  }
  set helpMessage($w.menu.file.m,0) TreeFileSave
  $w.menu.file.m add command -label TreeFileFill -command ::tree::prime
  set helpMessage($w.menu.file.m,1) TreeFileFill
  $w.menu.file.m add separator
  $w.menu.file.m add command -label TreeFileBest -command ::tree::best
  set helpMessage($w.menu.file.m,3) TreeFileBest
  $w.menu.file.m add command -label TreeFileGraph -command ::tree::graph
  set helpMessage($w.menu.file.m,4) TreeFileGraph
  $w.menu.file.m add separator
  $w.menu.file.m add command -label TreeFileCopy -command {
    clipboard clear
    clipboard append [::tree::copyToSelection]
    selection own .treeWin.f.tl
    selection get
  }
  set helpMessage($w.menu.file.m,6) TreeFileCopy
  $w.menu.file.m add separator
  $w.menu.file.m add command -label TreeFileClose \
    -command {.treeWin.buttons.close invoke}
  set helpMessage($w.menu.file.m,8) TreeFileClose

  foreach label {Alpha ECO Freq Score} value {alpha eco frequency score} {
    $w.menu.sort.m add radiobutton -label TreeSort$label \
      -variable tree(order) -value $value -command ::tree::refresh
  }

  $w.menu.opt.m add checkbutton -label TreeOptLock -variable tree(locked) \
    -command ::tree::toggleLock
  set helpMessage($w.menu.opt.m,0) TreeOptLock

  $w.menu.opt.m add checkbutton -label TreeOptTraining \
    -variable tree(training) -command ::tree::toggleTraining
  set helpMessage($w.menu.opt.m,1) TreeOptTraining

  $w.menu.opt.m add separator
  $w.menu.opt.m add checkbutton -label TreeOptAutosave \
    -variable tree(autoSave)
  set helpMessage($w.menu.opt.m,3) TreeOptAutosave

  $w.menu.help.m add command -label TreeHelpTree \
    -accelerator F1 -command {helpWindow Tree}
  $w.menu.help.m add command -label TreeHelpIndex -command {helpWindow Index}

  ::tree::ConfigMenus

  autoscrollframe $w.f listbox $w.f.tl \
    -width $::winWidth($w) -height $::winHeight($w) \
    -font font_Fixed -foreground black -background white \
    -selectmode browse -setgrid 1
  canvas $w.progress -width 250 -height 15 -bg white -relief solid -border 1
  $w.progress create rectangle 0 0 0 0 -fill blue -outline blue -tags bar
  selection handle $w.f.tl ::tree::copyToSelection
  bindMouseWheel $w $w.f.tl

  bind $w.f.tl <Destroy> {
    if {$tree(autoSave)} {
      busyCursor .
      catch {sc_tree write $tree(base)}
      unbusyCursor .
    }
  }

  bind $w <Configure> "recordWinSize $w"

  label $w.status -width 1 -anchor w -font font_Small \
    -relief sunken -textvar tree(status)
  pack $w.status -side bottom -fill x
  pack $w.progress -side bottom
  pack [frame $w.buttons -relief sunken] -side bottom -fill x
  pack $w.f -side top -expand 1 -fill both

  button $w.buttons.best -image b_list -command ::tree::best
  button $w.buttons.graph -image b_bargraph -command ::tree::graph
  checkbutton $w.buttons.lock -textvar ::tr(LockTree) \
    -variable tree(locked) -command ::tree::toggleLock
  checkbutton $w.buttons.training -textvar ::tr(Training) \
    -variable tree(training) -command ::tree::toggleTraining

  foreach {b t} {
    best TreeFileBest graph TreeFileGraph lock TreeOptLock
    training TreeOptTraining
  } {
    set helpMessage($w.buttons.$b) $t
  }

  dialogbutton $w.buttons.stop -textvar ::tr(Stop) -command { sc_progressBar }
  dialogbutton $w.buttons.close -textvar ::tr(Close) -command {
    set geometry(treeWin) [wm geometry .treeWin]
    focus .; destroy .treeWin
  }

  pack $w.buttons.best $w.buttons.graph $w.buttons.lock $w.buttons.training \
    -side left -padx 3 -pady 2
  packbuttons right $w.buttons.close $w.buttons.stop
  $w.buttons.stop configure -state disabled

  wm minsize $w 40 5

  bind $w.f.tl <Return> {
    tree::select [lindex [ .treeWin.f.tl curselection] 0]
  }
  bind $w.f.tl <ButtonRelease-1> {
    .treeWin.f.tl selection clear 0 end
    tree::select [ .treeWin.f.tl nearest %y ]
    .treeWin.f.tl selection clear 0 end
    break
  }

  wm protocol $w WM_DELETE_WINDOW { .treeWin.buttons.close invoke }
  ::tree::refresh
}

proc ::tree::toggleTraining {} {
  global tree
  if {$tree(training)} {
    ::tree::doTraining
  } else {
    ::tree::refresh
  }
}

proc ::tree::doTraining {{n 0}} {
  global tree
  if {$n != 1  &&  [winfo exists .analysisWin1]  &&  $::analysis(automove1)} {
    automove 1
    return
  }
  if {$n != 2  &&  [winfo exists .analysisWin2]  &&  $::analysis(automove2)} {
    automove 2
    return
  }
  if {[::tb::isopen]  &&  $::tbTraining} {
    ::tb::move
    return
  }
  if {! [winfo exists .treeWin]} { return }
  if {$tree(training) == 0} { return }
  set move [sc_tree move $tree(base) random]
  addSanMove $move -animate -notraining
  updateBoard -pgn
}

proc ::tree::toggleLock {} {
  global tree
  if {$tree(locked)} {
    set tree(base) [sc_base current]
  } else {
    set tree(base) 0
  }
  ::tree::refresh
}

proc ::tree::select { selection } {
  global tree
  if {! [winfo exists .treeWin]} { return }
  .treeWin.f.tl selection clear 0 end
  if {$selection == 0} {
    sc_move back
    updateBoard -pgn
    return
  }
  set move [sc_tree move $tree(base) $selection]
  if {$move == ""} { return }
  addSanMove $move -animate
  updateBoard -pgn
}

set tree(refresh) 0

proc ::tree::refresh {} {
  global tree treeWin glstart
  set w .treeWin

  if {![winfo exists $w]} { return }
  busyCursor .
  sc_progressBar $w.progress bar 251 16
  foreach button {best graph training lock close} {
    $w.buttons.$button configure -state disabled
  }
  $w.buttons.stop configure -state normal
  set tree(refresh) 1
  catch {grab $w.buttons.stop}

  update
  set base 0
  if {$tree(locked)} { set base $tree(base) }
  set moves [sc_tree search -hide $tree(training) -sort $tree(order) -base $base]
  catch {grab release $w.buttons.stop}
  set tree(refresh) 0
  foreach button {best graph training lock close} {
    $w.buttons.$button configure -state normal
  }
  $w.buttons.stop configure -state disabled -relief raised

  set moves [split $moves "\n"]
  set len [llength $moves]
  $w.f.tl delete 0 end
  for { set i 0 } { $i < $len } { incr i } {
    $w.f.tl insert end [lindex $moves $i]
  }
  catch {$w.f.tl itemconfigure 0 -foreground darkBlue}

  #set n [expr $len - 4]
  #if {$n > 0} {
  #  for {set i 1} {$i < $n} {incr i} {
  #    set f [string range [lindex $moves $i] 25 27]
  #    if {$f < 5} {
  #      catch {$w.f.tl itemconfigure $i -foreground gray50}
  #    }
  #  }
  #  while {$n < $len} {
  #    catch {$w.f.tl itemconfigure $n -foreground darkBlue}
  #    incr n
  #  }
  #}

  if {[winfo exists .treeBest]} { ::tree::best }

  unbusyCursor .
  $w.f.tl configure -cursor {}
  $w.f.tl selection clear 0 end

  ::tree::status
  set glstart 1
  ::windows::stats::Refresh
  if {[winfo exists .treeGraph]} ::tree::graph
  ::windows::gamelist::Refresh
  updateTitle
}

proc ::tree::status {{msg ""}} {
  global tree
  if {$msg != ""} {
    set tree(status) $msg
    return
  }
  set s "  $::tr(Database)"
  set base [sc_base current]
  if {$tree(locked)} { set base $tree(base) }
  set status "  $::tr(Database) $base: [file tail [sc_base filename $base]]"
  if {$tree(locked)} { append status " ($::tr(TreeLocked))" }
  append status "   $::tr(Filter)"
  append status ": [filterText $base]"
  set tree(status) $status
}

set tree(standardLines) {
  {}
  {1.c4}
  {1.c4 c5}
  {1.c4 c5 2.Nf3}
  {1.c4 e5}
  {1.c4 Nf6}
  {1.c4 Nf6 2.Nc3}
  {1.d4}
  {1.d4 d5}
  {1.d4 d5 2.c4}
  {1.d4 d5 2.c4 c6}
  {1.d4 d5 2.c4 c6 3.Nf3}
  {1.d4 d5 2.c4 c6 3.Nf3 Nf6}
  {1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3}
  {1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4}
  {1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 e6}
  {1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 e6 5.e3}
  {1.d4 d5 2.c4 e6}
  {1.d4 d5 2.c4 e6 3.Nc3}
  {1.d4 d5 2.c4 e6 3.Nc3 Nf6}
  {1.d4 d5 2.c4 e6 3.Nf3}
  {1.d4 d5 2.c4 dxc4}
  {1.d4 d5 2.c4 dxc4 3.Nf3}
  {1.d4 d5 2.c4 dxc4 3.Nf3 Nf6}
  {1.d4 d5 2.Nf3}
  {1.d4 d5 2.Nf3 Nf6}
  {1.d4 d5 2.Nf3 Nf6 3.c4}
  {1.d4 d6}
  {1.d4 d6 2.c4}
  {1.d4 Nf6}
  {1.d4 Nf6 2.c4}
  {1.d4 Nf6 2.c4 c5}
  {1.d4 Nf6 2.c4 d6}
  {1.d4 Nf6 2.c4 e6}
  {1.d4 Nf6 2.c4 e6 3.Nc3}
  {1.d4 Nf6 2.c4 e6 3.Nc3 Bb4}
  {1.d4 Nf6 2.c4 e6 3.Nf3}
  {1.d4 Nf6 2.c4 g6}
  {1.d4 Nf6 2.c4 g6 3.Nc3}
  {1.d4 Nf6 2.c4 g6 3.Nc3 Bg7}
  {1.d4 Nf6 2.c4 g6 3.Nc3 Bg7 4.e4}
  {1.d4 Nf6 2.c4 g6 3.Nc3 Bg7 4.e4 d6}
  {1.d4 Nf6 2.c4 g6 3.Nc3 Bg7 4.e4 d6 5.Nf3}
  {1.d4 Nf6 2.c4 g6 3.Nc3 Bg7 4.e4 d6 5.Nf3 O-O}
  {1.d4 Nf6 2.c4 g6 3.Nc3 Bg7 4.e4 d6 5.Nf3 O-O 6.Be2}
  {1.d4 Nf6 2.c4 g6 3.Nf3}
  {1.d4 Nf6 2.Bg5}
  {1.d4 Nf6 2.Bg5 Ne4}
  {1.d4 Nf6 2.Nf3}
  {1.d4 Nf6 2.Nf3 e6}
  {1.d4 Nf6 2.Nf3 g6}
  {1.e4}
  {1.e4 c5}
  {1.e4 c5 2.c3}
  {1.e4 c5 2.c3 d5}
  {1.e4 c5 2.c3 Nf6}
  {1.e4 c5 2.Nc3}
  {1.e4 c5 2.Nc3 Nc6}
  {1.e4 c5 2.Nf3}
  {1.e4 c5 2.Nf3 d6}
  {1.e4 c5 2.Nf3 d6 3.d4}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6}
  {1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6}
  {1.e4 c5 2.Nf3 d6 3.Bb5+}
  {1.e4 c5 2.Nf3 e6}
  {1.e4 c5 2.Nf3 Nc6}
  {1.e4 c5 2.Nf3 Nc6 3.d4}
  {1.e4 c5 2.Nf3 Nc6 3.Bb5}
  {1.e4 c6}
  {1.e4 c6 2.d4}
  {1.e4 c6 2.d4 d5}
  {1.e4 c6 2.d4 d5 3.e5}
  {1.e4 c6 2.d4 d5 3.Nc3}
  {1.e4 c6 2.d4 d5 3.Nd2}
  {1.e4 d5}
  {1.e4 d6}
  {1.e4 d6 2.d4}
  {1.e4 d6 2.d4 Nf6}
  {1.e4 d6 2.d4 Nf6 3.Nc3}
  {1.e4 e5}
  {1.e4 e5 2.Nf3}
  {1.e4 e5 2.Nf3 Nc6}
  {1.e4 e5 2.Nf3 Nc6 3.d4}
  {1.e4 e5 2.Nf3 Nc6 3.Bb5}
  {1.e4 e5 2.Nf3 Nc6 3.Bb5 a6}
  {1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4}
  {1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6}
  {1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O}
  {1.e4 e5 2.Nf3 Nc6 3.Bc4}
  {1.e4 e5 2.Nf3 Nf6}
  {1.e4 e6}
  {1.e4 e6 2.d4}
  {1.e4 e6 2.d4 d5}
  {1.e4 e6 2.d4 d5 3.Nc3}
  {1.e4 e6 2.d4 d5 3.Nc3 Bb4}
  {1.e4 e6 2.d4 d5 3.Nc3 Nf6}
  {1.e4 e6 2.d4 d5 3.Nd2}
  {1.e4 e6 2.d4 d5 3.Nd2 c5}
  {1.e4 e6 2.d4 d5 3.Nd2 Nf6}
  {1.e4 Nf6}
  {1.e4 Nf6 2.e5}
  {1.e4 Nf6 2.e5 Nd5}
  {1.Nf3}
  {1.Nf3 Nf6}
}

# ::tree::prime
#   Primes the tree for this database, filling it with a number of
#   common opening positions.
#
proc ::tree::prime {} {
  global tree
  if {! [winfo exists .treeWin]} { return }
  set base [sc_base current]
  if {$tree(locked)} { set base $tree(base) }
  if {! [sc_base inUse]} { return }
  set fname [sc_base filename $base]
  if {[string index $fname 0] == "\["  ||  [file extension $fname] == ".pgn"} {
    tk_messageBox -parent .treeWin -icon info -type ok -title "Scid" \
      -message "Sorry, only Scid-format database files can have a tree cache file."
    return
  }

  set ::interrupt 0
  progressWindow "Scid: [tr TreeFileFill]" "" $::tr(Cancel) {set ::interrupt 1}
  resetProgressWindow
  leftJustifyProgressWindow
  busyCursor .
  sc_game push
  set i 1
  set len [llength $tree(standardLines)]
  foreach line $tree(standardLines) {
    sc_game new
    set text [format "%3d/\%3d" $i $len]
    if {[llength $line] > 0}  {
      sc_move addSan $line
      changeProgressWindow "$text: $line"
    } else {
      changeProgressWindow "$text: start position"
    }
    sc_tree search -base $base
    updateProgressWindow $i $len
    incr i
    if {$::interrupt} {
      closeProgressWindow
      set ::interrupt 0
      sc_game pop
      unbusyCursor .
      ::tree::refresh
      return
    }
  }
  closeProgressWindow
  if {[catch {sc_tree write $base} result]} {
    #tk_messageBox -type ok -icon warning -title "Scid: Error writing file" \
        -message $result
  } else {
    #set a "$fname.stc: [sc_tree positions] positions, "
    #append a "$result bytes: "
    #set pergame [expr double($result) / double([sc_base numGames])]
    #append a [format "%.2f" $pergame]
    #append a " bytes per game"
    #tk_messageBox -type ok -parent .treeWin -title "Scid" -message $a
  }
  sc_game pop
  unbusyCursor .
  ::tree::refresh
}

set tree(bestMax) 50
trace variable tree(bestMax) w ::tree::best
set tree(bestRes) "1-0 0-1 1/2 *"
trace variable tree(bestRes) w ::tree::best

# ::tree::best
#   Updates the window of best (highest-rated) tree games.
#
proc ::tree::best {args} {
  global tree
  set w .treeBest
  if {! [winfo exists .treeWin]} { return }
  if {! [winfo exists $w]} {
    toplevel $w
    wm title $w "Scid: $::tr(TreeBestGames)"
    setWinLocation $w
    bind $w <Escape> "destroy $w"
    bind $w <F1> {helpWindow Tree Best}
    pack [frame $w.b] -side bottom -fill x
    pack [frame $w.opt] -side bottom -fill x
    set pane [::utils::pane::Create $w.pane blist bpgn 520 320 0.6]
    ::utils::pane::SetRange $w.pane 0.3 0.8
    pack $pane -side top -expand true -fill both
    scrollbar $pane.blist.ybar -command "$pane.blist.list yview" -takefocus 0
    listbox $pane.blist.list -background white \
      -yscrollcommand "$pane.blist.ybar set" -font font_Small
    pack $pane.blist.ybar -side right -fill y
    pack $pane.blist.list -side left -fill both -expand yes
    bind $pane.blist.list <<ListboxSelect>> ::tree::bestPgn
    bind $pane.blist.list <Double-Button-1> ::tree::bestBrowse

    scrollbar $pane.bpgn.ybar -command "$pane.bpgn.text yview" -takefocus 0
    text $pane.bpgn.text -width 50 -height 20 -background gray90 \
      -cursor top_left_arrow -yscrollcommand "$pane.bpgn.ybar set" -wrap word \
      -state disabled -font font_Small
    pack $pane.bpgn.ybar -side right -fill y
    pack $pane.bpgn.text -side left -fill both -expand yes
    set t $pane.bpgn.text
    bind $t <ButtonPress-1> "::pgn::ShowBoard $pane.bpgn.text 4 %x %y %X %Y"
    bind $t <ButtonRelease-1> ::pgn::HideBoard
    bind $t <ButtonPress-2> "::pgn::ShowBoard $pane.bpgn.text 4 %x %y %X %Y"
    bind $t <ButtonRelease-2> ::pgnHideBoard
    bind $t <ButtonPress-3> "::pgn::ShowBoard $pane.bpgn.text 4 %x %y %X %Y"
    bind $t <ButtonRelease-3> :::pgn::HideBoard

    label $w.opt.lmax -text $::tr(TreeBest:) -font font_Small
    set m [tk_optionMenu $w.opt.max tree(bestMax) 10 20 50 100 200 500]
    $m configure -font font_Small
    $w.opt.max configure -font font_Small
    label $w.opt.lres -text " $::tr(Result):" -font font_Small
    set m [tk_optionMenu $w.opt.res tree(bestRes) \
             "1-0 0-1 1/2 *" 1-0 0-1 "1-0 0-1" 1/2-1/2]
    $m configure -font font_Small
    $w.opt.res configure -font font_Small

    button $w.b.browse -text $::tr(BrowseGame) -command ::tree::bestBrowse
    button $w.b.load -text $::tr(LoadGame) -command ::tree::bestLoad
    button $w.b.merge -text $::tr(MergeGame) -command ::tree::bestMerge
    button $w.b.close -text $::tr(Close) -command "destroy $w"
    foreach i {browse load merge close} { $w.b.$i configure -font font_Small }
    pack $w.b.close $w.b.merge $w.b.load $w.b.browse \
      -side right -padx 1 -pady 2
    pack $w.opt.lmax $w.opt.max -side left -padx 0 -pady 2
    pack $w.opt.lres $w.opt.res -side left -padx 0 -pady 2
    bind $w <Configure> "recordWinSize $w"
    focus $w.pane.blist.list
  }
  $w.pane.blist.list delete 0 end
  set tree(bestList) {}
  set count 0
  if {! [sc_base inUse]} { return }
  foreach {idx line} [sc_tree best $tree(base) $tree(bestMax) $tree(bestRes)] {
    incr count
    $w.pane.blist.list insert end "[format %02d $count]:  $line"
    lappend tree(bestList) $idx
  }
  catch {$w.pane.blist.list selection set 0}
  ::tree::bestPgn
}

proc ::tree::bestLoad {} {
  global tree
  if {[catch {set sel [.treeBest.pane.blist.list curselection]}]} { return }
  if {[catch {set g [lindex $tree(bestList) $sel]}]} { return }
  if {$tree(locked)} { sc_base switch $tree(base) }
  ::game::Load $g
}

proc ::tree::bestMerge {} {
  global tree
  if {[catch {set sel [.treeBest.pane.blist.list curselection]}]} { return }
  if {[catch {set gnum [lindex $tree(bestList) $sel]}]} { return }
  set base [sc_base current]
  if {$tree(locked)} { set base $tree(base) }
  mergeGame $base $gnum
}

proc ::tree::bestBrowse {} {
  global tree
  if {[catch {set sel [.treeBest.pane.blist.list curselection]}]} { return }
  if {[catch {set gnum [lindex $tree(bestList) $sel]}]} { return }
  set base [sc_base current]
  if {$tree(locked)} { set base $tree(base) }
  ::gbrowser::new $base $gnum
}

proc ::tree::bestPgn {} {
  global tree
  set t .treeBest.pane.bpgn.text
  $t configure -state normal
  $t delete 1.0 end
  if {[catch {set sel [.treeBest.pane.blist.list curselection]}]} { return }
  if {[catch {set g [lindex $tree(bestList) $sel]}]} { return }
  set base [sc_base current]
  if {$tree(locked)} { set base $tree(base) }
  if {[catch {sc_game summary -base $base -game $g header} header]} { return }
  if {[catch {sc_game summary -base $base -game $g moves} moves]} { return }
  if {[catch {sc_filter value $base $g} ply]} { return }
  $t tag configure header -foreground darkBlue
  $t tag configure start -foreground darkRed
  $t insert end $header header
  $t insert end "\n\n"
  set m 0
  foreach move $moves {
    incr m
    if {$m < $ply} {
      $t insert end $move start
    } else {
      $t insert end $move
    }
    $t insert end " "
  }
  #catch {$t insert end [sc_game pgn -base $base -game $g \
  #                        -short 1 -indentC 1 -indentV 1 -symbol 1 -tags 0]}
  $t configure -state disabled
}

# ::tree::graph
#   Updates the tree graph window, creating it if necessary.
#
proc ::tree::graph {} {
  set w .treeGraph
  if {! [winfo exists .treeWin]} { return }
  if {! [winfo exists $w]} {
    toplevel $w
    setWinLocation $w
    bind $w <Escape> "destroy $w"
    bind $w <F1> {helpWindow Tree Graph}
    frame $w.menu -relief raised -borderwidth 2
    pack $w.menu -side top -fill x
    $w configure -menu $w.menu
    menubutton $w.menu.file -text GraphFile -menu $w.menu.file.m
    menu $w.menu.file.m
    $w.menu.file.m add command -label GraphFileColor \
      -command "saveGraph color $w.c"
    $w.menu.file.m add command -label GraphFileGrey \
      -command "saveGraph gray $w.c"
    $w.menu.file.m add separator
    $w.menu.file.m add command -label GraphFileClose -command "destroy $w"
    pack $w.menu.file -side left

    canvas $w.c -width 500 -height 300
    pack $w.c -side top -fill both -expand yes
    $w.c create text 25 10 -tag text -justify center -width 1 \
      -font font_Regular -anchor n
    bind $w <Configure> {
      .treeGraph.c itemconfigure text -width [expr {[winfo width .treeGraph.c] - 50}]
      .treeGraph.c coords text [expr {[winfo width .treeGraph.c] / 2}] 10
      ::utils::graph::configure tree -height [expr {[winfo height .treeGraph.c] - 100}]
      ::utils::graph::configure tree -width [expr {[winfo width .treeGraph.c] - 50}]
      ::utils::graph::redraw tree
    }
    bind $w.c <Button-1> ::tree::graph
    wm title $w "Scid: Tree Graph"
    # wm minsize $w 300 200
    standardShortcuts $w
    ::tree::configGraphMenus
  }

  $w.c itemconfigure text -width [expr {[winfo width $w.c] - 50}]
  $w.c coords text [expr {[winfo width $w.c] / 2}] 10
  set height [expr {[winfo height $w.c] - 100}]
  set width [expr {[winfo width $w.c] - 50}]
  ::utils::graph::create tree -width $width -height $height -xtop 25 -ytop 60 \
    -xmin 0.5 -xtick 1 -ytick 5 -font font_Small -canvas $w.c

  set data {}
  set xlabels {}
  set othersCount 0
  set numOthers 0
  set othersName "..."
  set count 0
  set othersScore 0.0
  set mean 50.0
  set totalGames 0
  set treeData [.treeWin.f.tl get 0 end]

  set numTreeLines [llength $treeData]
  set totalLineIndex [expr $numTreeLines - 2]

  for {set i 0} {$i < [llength $treeData]} {incr i} {
    # Extract info from each line of the tree window:
    # Note we convert "," decimal char back to "." where necessary.
    set line [lindex $treeData $i]
    set mNum [string trim [string range $line  0  1]]
    set freq [string trim [string range $line 17 23]]
    set fpct [string trim [string range $line 25 29]]
    regsub -all {,} $fpct . fpct
    set move [string trim [string range $line  4 9]]
    set score [string trim [string range $line 33 37]]
    regsub -all {,} $score . score
    if {$score > 99.9} { set score 99.9 }
    # Check if this line is "TOTAL:" line:
    if {$i == $totalLineIndex} {
      set mean $score
      set totalGames $freq
    }
    # Add info for this move to the graph if necessary:
    if {[string index $line 2] == ":"  &&  [string compare "<end>" $move]} {
      if {$fpct < 1.0  ||  $freq < 5  ||  $i > 5} {
        incr othersCount $freq
        incr numOthers
        set othersScore [expr {$othersScore + (double($freq) * $score)}]
        set m $move
        if {$numOthers > 1} { set m "..." }
      } else {
        incr count
        lappend data $count
        lappend data $score
        lappend xlabels [list $count "$move ([expr round($score)]%)\n$freq: [expr round($fpct)]%"]
      }
    }
  }

  # Add extra bar for other moves if necessary:
  if {$numOthers > 0  &&  $totalGames > 0} {
    incr count
    set fpct [expr {double($othersCount) * 100.0 / double($totalGames)}]
    set sc [expr {round($othersScore / double($othersCount))}]
    set othersName "$m ($sc%)\n$othersCount: [expr round($fpct)]%"
    lappend data $count
    lappend data [expr {$othersScore / double($othersCount)}]
    lappend xlabels [list $count $othersName]
  }

  # Plot fake bounds data so graph at least shows range 40-65:
  ::utils::graph::data tree bounds -points 0 -lines 0 -bars 0 -coords {1 41 1 64}

  # Replot the graph:
  ::utils::graph::data tree data -color red -points 0 -lines 0 -bars 1 \
    -barwidth 0.75 -outline black -coords $data
  ::utils::graph::configure tree -xlabels $xlabels -xmax [expr {$count + 0.5}] \
    -hline [list {gray80 1 each 5} {gray50 1 each 10} {black 2 at 50} \
              {black 1 at 55} [list red 2 at $mean]] \
    -brect [list [list 0.5 55 [expr {$count + 0.5}] 50 LightSkyBlue1]]

  ::utils::graph::redraw tree
  set moves ""
  catch {set moves [sc_game firstMoves 0 -1]}
  if {[string length $moves] == 0} { set moves $::tr(StartPos) }
  set title "$moves ([::utils::thousands $totalGames] $::tr(games))"
  $w.c itemconfigure text -text $title
}

proc ::tree::configGraphMenus {{lang ""}} {
  if {! [winfo exists .treeGraph]} { return }
  if {$lang == ""} { set lang $::language }
  set m .treeGraph.menu
  foreach menu {file} tag {File} {
    configMenuName $m.$menu Graph$tag $lang
  }
  foreach idx {0 1 3} tag {Color Grey Close} {
    configMenuText $m.file.m $idx GraphFile$tag $lang
  }
}
