/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_layer.h"
#include "sg_arrange_dialog.h"
#include "sg_dialogs.h"
#include "sg_stock.h"

static GtkWidget *rows_entry;
static GtkWidget *cols_entry;
static GtkWidget *hgap_entry;
static GtkWidget *vgap_entry;
static GtkWidget *left_entry;
static GtkWidget *right_entry;
static GtkWidget *top_entry;
static GtkWidget *bottom_entry;

gboolean arrange_busy=FALSE;

static gboolean
mw_destroy(GtkWidget *widget)
{
  /* This is needed to get out of gtk_main 
  gtk_main_quit ();*/
  arrange_busy=FALSE;
  return FALSE;
}

static void
update_plot(GtkWidget *widget, gpointer data)
{
  SGlayer *layer;
  gint nrows, ncols;
  gint i, j;
  gint nlayers;
  gdouble x, y, width, height;
  gdouble vgap, hgap, top, bottom, left, right;
  gchar text[255];

  ncols = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cols_entry));
  nrows = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(rows_entry));
  vgap = atof(gtk_entry_get_text(GTK_ENTRY(vgap_entry)));
  hgap = atof(gtk_entry_get_text(GTK_ENTRY(hgap_entry)));
  left = atof(gtk_entry_get_text(GTK_ENTRY(left_entry)));
  right = atof(gtk_entry_get_text(GTK_ENTRY(right_entry)));
  top = atof(gtk_entry_get_text(GTK_ENTRY(top_entry)));
  bottom = atof(gtk_entry_get_text(GTK_ENTRY(bottom_entry)));

  nlayers = ncols*nrows - active_plot->nlayers;
  if(nlayers > 0){
      sprintf(text,"You need to create %d layers. Go ahead?", nlayers);
      if(sg_accept_dialog(text, 0) != YES_CLICKED) return;
  }
  if(nlayers < 0){
      sprintf(text,"The number of layers is invalid");
      sg_message_dialog(text, 1);
      return;
  }

  nlayers = 0;

  width = 1. - left - right - (ncols - 1)*hgap;
  height = 1. - top - bottom - (nrows - 1)*vgap;
  width /= ncols;
  height /= nrows;

  y = top;
  for(i = 1; i <= nrows; i++){
    x = left;
    for(j = 1; j <= ncols; j++){
      nlayers ++;

      if(nlayers <= active_plot->nlayers){
          layer = (SGlayer *)g_list_nth_data(active_plot->layers, nlayers-1);
          gtk_plot_move(GTK_PLOT(layer->real_plot), x, y);
          gtk_plot_resize(GTK_PLOT(layer->real_plot), width, height);
      }else{
          layer = sg_layer_new(SG_LAYER_2D, width, height);
          sg_plot_add_layer(active_plot, layer, x, y);
      }

      x += width + hgap;
    }
    y += height + vgap;
  }

  arrange_rows = nrows;
  arrange_cols = ncols;
  arrange_hgap = hgap;
  arrange_vgap = vgap;
  arrange_top = top;
  arrange_bottom = bottom;
  arrange_left = left;
  arrange_right = right;

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(active_plot->real_canvas));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(active_plot->real_canvas));

}

static void
ok_clicked(GtkWidget *widget, gpointer data)
{
  update_plot(widget, data);
  arrange_busy=FALSE;
  gtk_widget_destroy(widget);
}

static void
init_dialog()
{
  gchar label[100];

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rows_entry), arrange_rows);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(cols_entry), arrange_cols);

  sprintf(label, "%*.*f", 1, 3, arrange_hgap);
  gtk_entry_set_text(GTK_ENTRY(hgap_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_vgap);
  gtk_entry_set_text(GTK_ENTRY(vgap_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_left);
  gtk_entry_set_text(GTK_ENTRY(left_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_right);
  gtk_entry_set_text(GTK_ENTRY(right_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_bottom);
  gtk_entry_set_text(GTK_ENTRY(bottom_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_top);
  gtk_entry_set_text(GTK_ENTRY(top_entry), label);

  sg_entry_set_numeric(GTK_ENTRY(hgap_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(vgap_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(left_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(right_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(top_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(bottom_entry), 3);
}


void
sg_arrange_dialog (GtkPlotCanvas *canvas)
{
  GtkWidget *window = NULL;
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *ok_button, *apply_button, *cancel_button;
  GtkWidget *action_area;
  GtkWidget *table;
  GtkWidget *label;
  GtkAdjustment *adj;
  GtkRequisition req;

  if (arrange_busy) return;
  else arrange_busy=TRUE;

  window=gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
  gtk_window_set_title (GTK_WINDOW(window),"Arrange layers");

  gtk_window_set_modal (GTK_WINDOW(window),FALSE);
  /* Create widgets */
  main_box = gtk_vbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 10);
  gtk_container_add (GTK_CONTAINER (window), main_box);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);

  table = gtk_table_new(9, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 0., 0.); 
  rows_entry = gtk_spin_button_new(adj, 0, 0);   
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 0., 0.); 
  cols_entry = gtk_spin_button_new(adj, 0, 0);   
  hgap_entry = gtk_entry_new();   
  vgap_entry = gtk_entry_new();   
  left_entry = gtk_entry_new();   
  right_entry = gtk_entry_new();   
  top_entry = gtk_entry_new();   
  bottom_entry = gtk_entry_new();   
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(rows_entry), TRUE);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(cols_entry), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(rows_entry), 0);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(cols_entry), 0);

  gtk_widget_size_request(cols_entry, &req);
  gtk_widget_set_usize(cols_entry, req.width, req.height);
  gtk_widget_set_usize(rows_entry, req.width, req.height);
  gtk_widget_set_usize(hgap_entry, req.width, req.height);
  gtk_widget_set_usize(vgap_entry, req.width, req.height);
  gtk_widget_set_usize(left_entry, req.width, req.height);
  gtk_widget_set_usize(right_entry, req.width, req.height);
  gtk_widget_set_usize(top_entry, req.width, req.height);
  gtk_widget_set_usize(bottom_entry, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new("Units in % of page size"),
                            0, 2, 0, 1);
  label = gtk_label_new("Number of rows"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 1, 2);
  label = gtk_label_new("Number of columns"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_hseparator_new(),
                            0, 2, 3, 4);
  label = gtk_label_new("Horizontal gap"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 4, 5);
  label = gtk_label_new("Vertical gap"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 5, 6);
  label = gtk_label_new("Left space"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 6, 7);
  label = gtk_label_new("Right space"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 7, 8);
  label = gtk_label_new("Top space"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 8, 9);
  label = gtk_label_new("Bottom space"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 9, 10);
  gtk_table_attach_defaults(GTK_TABLE(table), rows_entry, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), cols_entry, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), hgap_entry, 1, 2, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), vgap_entry, 1, 2, 5, 6);
  gtk_table_attach_defaults(GTK_TABLE(table), left_entry, 1, 2, 6, 7);
  gtk_table_attach_defaults(GTK_TABLE(table), right_entry, 1, 2, 7, 8);
  gtk_table_attach_defaults(GTK_TABLE(table), top_entry, 1, 2, 8, 9);
  gtk_table_attach_defaults(GTK_TABLE(table), bottom_entry, 1, 2, 9, 10);

  /* Action Area */
  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_end (GTK_BOX (main_box), action_area, FALSE, FALSE, 0);
  gtk_widget_show (action_area);

  ok_button = sg_stock_button(GNOME_STOCK_BUTTON_OK);
  GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (ok_button);
  gtk_widget_show (ok_button);

  apply_button = sg_stock_button(GNOME_STOCK_BUTTON_APPLY);
  GTK_WIDGET_SET_FLAGS (apply_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), apply_button, TRUE, TRUE, 0);
  gtk_widget_show (apply_button);

  cancel_button = sg_stock_button(GNOME_STOCK_BUTTON_CANCEL);
  GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (cancel_button);

  /* connect signals */
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (gtk_widget_destroy),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
                             GTK_SIGNAL_FUNC (ok_clicked),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (apply_button), "clicked",
                             GTK_SIGNAL_FUNC (update_plot),
                             GTK_OBJECT (window));

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (mw_destroy),NULL);

  /* Show widgets */

  init_dialog();

  gtk_widget_show_all (window);

  /* wait until dialog get destroyed */
  /*  gtk_main();*/

  return;
}
