/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_edit_dialog.h"
#include "sg_dataset.h"
#include "sg_dialogs.h"
#include "sg.h"

#define DATASET_NAME_LENGTH 100

static gint default_nargs[] = {2, 2, 2, 2, 2, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3};
static gint nargs[] = {2, 2, 2, 2, 2, 3, 4, 3, 3, 3, 3, 3, 3, 3, 3};

static gchar *column_labels[][10] = {
  {"X", "Y", "Z", "", "Xerr", "Yerr", "Zerr", "", "Lbls"},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "", "", "", "", "", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Amp", "", "", "", "Aerr", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "DX", "DY", "DZ", "Amp", ""},
  {"X", "Y", "Z", "Size", "", "", "", "Amp", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "Amp", "", "", "", "", ""},
  {"X", "Y", "Z", "", "", "", "", "", ""},};

static gchar *dialog_names_2d[] = {"Worksheet",
                                   "Functions",
                                   "Python variable/expression",
                                   NULL};

static gchar *dialog_names_3d[] = {"Worksheet",
                                   "Matrix",
                                   "Python variable/expression",
                                    NULL};



static GtkWidget *column_entry[9];
static GtkWidget *left_button[9];
static GtkWidget *right_button[9];
static GtkWidget *entries_box = NULL;
static GtkWidget *entries_table = NULL;
static gint toggle_selection = 0;
static SGdataStyle data_style;
static SGlayerType layer_type;
static gint num_entries;
static gint default_columns_mask[] = { 
                              SG_STYLE_LPOINTS_MASK,
                              SG_STYLE_VBARS_MASK,
                              SG_STYLE_HBARS_MASK,
                              SG_STYLE_VBOXES_MASK,
                              SG_STYLE_HBOXES_MASK,
                              SG_STYLE_BUBBLES_MASK,
                              SG_STYLE_FLUX_MASK,
                              SG_STYLE_COLORS_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK };
static gint columns_mask[] = {
                              SG_STYLE_LPOINTS_MASK,
                              SG_STYLE_VBARS_MASK,
                              SG_STYLE_HBARS_MASK,
                              SG_STYLE_VBOXES_MASK,
                              SG_STYLE_HBOXES_MASK,
                              SG_STYLE_BUBBLES_MASK,
                              SG_STYLE_FLUX_MASK,
                              SG_STYLE_COLORS_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK,
                              SG_STYLE_CONTOUR_MASK };


static SGdataset *the_dataset = NULL;
static GtkWidget *worksheet_combo;
static GtkWidget *columns_list;
static SGplot *plot;
static GtkWidget *toggle_combo;

static GtkWidget *entries[9];

static void real_select_columns				(GtkWidget *widget, 
							 gpointer data);
static void select_columns				(GtkWidget *widget, 
							 gint row, gint col,
							 gpointer data);
static void init_columns				();


static gint
get_column_from_name(gchar *name)
{
  SGworksheet *worksheet;
  gchar *worksheet_name = NULL;
  gint n = 0;

  worksheet_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  worksheet = sg_project_get_worksheet(worksheet_name);
  for(n = 0; n <= GTK_SHEET(worksheet->sheet)->maxcol; n++){
    if(strcmp(name, GTK_SHEET(worksheet->sheet)->column[n].name) == 0) return n;
  }

  return -1;
}

SGdataset *
sg_edit_columns_update()
{
  SGworksheet *worksheet;
  gchar *worksheet_name;
  gchar default_text[] = "";
  gint i, row = 0, narg = 0;
  gint dataset_columns[9];
  gchar *column_name[9];
  gboolean old_dataset = FALSE;

  if(the_dataset) old_dataset = TRUE;

  for(i = 0; i < 9; i++){
    dataset_columns[i] = -1;
    if(columns_mask[data_style] & (1 << i)){
       column_name[i] = gtk_entry_get_text(GTK_ENTRY(column_entry[i]));
       dataset_columns[i] = get_column_from_name(column_name[i]);
       if(dataset_columns[i] != -1) 
         narg++;
    }
  }

  if(narg < nargs[data_style]) return NULL;

  worksheet_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  worksheet = sg_project_get_worksheet(worksheet_name);

  if(!worksheet) return NULL;

  if(!old_dataset) the_dataset = sg_dataset_new(data_style);

  the_dataset->style = data_style;
  the_dataset->worksheet = worksheet;

  sg_dataset_set_worksheet(the_dataset, 
                           worksheet,
                           dataset_columns[0],
                           dataset_columns[1],
                           dataset_columns[2],
                           dataset_columns[3],
                           dataset_columns[4],
                           dataset_columns[5],
                           dataset_columns[6],
                           dataset_columns[7],
                           dataset_columns[8]);

  if(old_dataset) return (the_dataset);

  switch(toggle_selection){
    case 3:
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_IMPULSE;
      break;
    case 1:
      the_dataset->real_data->line.line_style = GTK_PLOT_LINE_SOLID;
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_NONE;
      break;
    case 2:
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_SQUARE;
      the_dataset->real_data->line.line_style = GTK_PLOT_LINE_SOLID;
      break;
    case 0:
    default:
      the_dataset->real_data->symbol.symbol_type = GTK_PLOT_SYMBOL_SQUARE;
      the_dataset->real_data->line.line_style = GTK_PLOT_LINE_NONE;
  }

/*
  for(i = 0; i < 9; i++){
    if(dataset_columns[i] != -1) 
        the_dataset->iterator_mask -= (1 << i); 
  }
*/

  return the_dataset;
}

static void
update_columns(GtkEntry *entry)
{
  gint i, nrows;
  SGworksheet *worksheet = NULL;
  GList *list;
  gchar name[DATASET_NAME_LENGTH];
  gchar *text;

  nrows = GTK_CLIST(columns_list)->rows;
  for(i = 0; i <= nrows; i++){
    gtk_clist_remove(GTK_CLIST(columns_list), 0);
  }

  text = gtk_entry_get_text(entry);
  list = worksheets;
  while(list){
    worksheet = (SGworksheet *)list->data;
    if(strcmp(text, worksheet->name) == 0) break;
    list = list->next;
  }

  for(i = 0; i <= GTK_SHEET(worksheet->sheet)->maxcol; i++){
     if(!GTK_SHEET(worksheet->sheet)->column[i].name)
        sprintf(name,"%d",i);
     else
        sprintf(name,"%s",GTK_SHEET(worksheet->sheet)->column[i].name);
     text=g_strdup(name);
     gtk_clist_append(GTK_CLIST(columns_list), &text);
  };

  for(i = 0; i < num_entries; i++)
    gtk_entry_set_text(GTK_ENTRY(column_entry[i]), "");

}

GtkWidget *
sg_edit_columns_dialog (SGdataset *dataset, SGlayerType type)
{
  GtkWidget *frame, *vbox;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *box, *sw;
  GList *list;
  GtkWidget *item;
  SGworksheet *worksheet;
  gint i;

  the_dataset = dataset;
  if(dataset)
    data_style = dataset->style;
  else
    data_style = (SGdataStyle)0;

  layer_type = type;

  for(i = 0; i < 12; i++){
     columns_mask[i] = default_columns_mask[i];
     nargs[i] = default_nargs[i];
  }

  if(layer_type == SG_LAYER_3D){
     columns_mask[SG_STYLE_LPOINTS] |= GTK_PLOT_DATA_Z | GTK_PLOT_DATA_DZ;
     columns_mask[SG_STYLE_LPOINTS] -= (GTK_PLOT_DATA_LABELS + 
                                      GTK_PLOT_DATA_DX + GTK_PLOT_DATA_DY);
     columns_mask[SG_STYLE_BUBBLES] |= GTK_PLOT_DATA_Z;
     columns_mask[SG_STYLE_COLORS] |= GTK_PLOT_DATA_Z;
     columns_mask[SG_STYLE_FLUX] |= GTK_PLOT_DATA_Z | GTK_PLOT_DATA_DZ;

     nargs[SG_STYLE_LPOINTS]++;
     nargs[SG_STYLE_BUBBLES]++;
     nargs[SG_STYLE_COLORS]++;
     nargs[SG_STYLE_FLUX] += 2;
  }

  /* Create widgets */

  vbox = gtk_vbox_new(FALSE, 5);

/*********************************************************************/

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new("Select Worksheet:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
  worksheet_combo = gtk_combo_new();
  gtk_box_pack_end (GTK_BOX (box), worksheet_combo, FALSE, FALSE, 0);
  gtk_widget_show(box);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry), FALSE);
  list = worksheets;
  while(list){
     worksheet = (SGworksheet *)list->data;
     item = gtk_list_item_new_with_label(worksheet->name);
     gtk_widget_show(item);
     gtk_container_add(GTK_CONTAINER(GTK_COMBO(worksheet_combo)->list), item);
     list = list->next;
  }

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new("Dataset Style:");
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);

  toggle_combo = sg_toggle_data_new(layer_type);
  gtk_box_pack_start (GTK_BOX (box), toggle_combo, FALSE, FALSE, 0);
  sg_toggle_data_init(toggle_combo, 0);

  if(dataset && dataset->worksheet){
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry), 
                       the_dataset->worksheet->name);
    gtk_widget_set_sensitive(GTK_WIDGET(worksheet_combo), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(toggle_combo), FALSE);
  }

/*********************************************************************/
  frame = gtk_frame_new("Columns");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  entries_table = NULL;
  entries_box = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(entries_box), 5);
  gtk_container_add (GTK_CONTAINER (frame), entries_box);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(sw, 180, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (entries_box), sw, FALSE, FALSE, 0);

  columns_list = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), columns_list);

  if(the_dataset && the_dataset->worksheet)
     init_columns();
  else
     select_columns(toggle_combo, 0, 0, NULL);

/********************************************************************/
  gtk_signal_connect(GTK_OBJECT(toggle_combo), "changed",
		     GTK_SIGNAL_FUNC(select_columns), NULL);

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(worksheet_combo)->entry), 
                     "changed", GTK_SIGNAL_FUNC(real_select_columns), NULL);

  return vbox;
}

static gint
set_column(GtkWidget *widget, gpointer data)
{
  GList *selection;
  gchar *text;
  gint row;

  selection = GTK_CLIST(columns_list)->selection;
  if(!selection) return TRUE;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(strlen(text) > 0) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_get_text(GTK_CLIST(columns_list), row, 0, &text);

  gtk_entry_set_text(GTK_ENTRY(data), text);
  gtk_clist_remove(GTK_CLIST(columns_list), row);
  return TRUE;
}

static gint
restore_column(GtkWidget *widget, gpointer data)
{
  gchar *text;
  gint col;

  text = gtk_entry_get_text(GTK_ENTRY(data));
  if(strlen(text) == 0) return TRUE;

  col = get_column_from_name(text);
  gtk_clist_insert(GTK_CLIST(columns_list), col, &text);

  gtk_entry_set_text(GTK_ENTRY(data), "");
  return TRUE;
}

static void
init_columns()
{
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint i, nrows = 0;
  gchar label[9][100];
  gint col_index[9];

  if(!the_dataset) return;

  col_index[0] = the_dataset->x;
  col_index[1] = the_dataset->y;
  col_index[2] = the_dataset->z;
  col_index[3] = the_dataset->a;
  col_index[4] = the_dataset->dx;
  col_index[5] = the_dataset->dy;
  col_index[6] = the_dataset->dz;
  col_index[7] = the_dataset->da;
  col_index[8] = the_dataset->l;

  update_columns(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(entries_box), entries_table);

  toggle_selection = 0;
 
  switch(layer_type){ 
    case SG_LAYER_3D:
      switch(data_style){
        case SG_STYLE_LPOINTS:
          toggle_selection = 0;
          break;
        case SG_STYLE_BUBBLES:
          toggle_selection = 1;
          break;
        case SG_STYLE_COLORS:
          toggle_selection = 2;
          break;
        case SG_STYLE_FLUX:
          toggle_selection = 3;
          break;
        case SG_STYLE_SURFACE:
          toggle_selection = 4;
          break;
        case SG_STYLE_CSURFACE:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_LAYER_POLAR:
      switch(data_style){
        case SG_STYLE_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_STYLE_BUBBLES:
          toggle_selection = 4;
          break;
        case SG_STYLE_COLORS:
          toggle_selection = 5;
          break;
      }
      break;
    case SG_LAYER_2D:
      switch(data_style){
        case SG_STYLE_LPOINTS:
          toggle_selection = 2;
          break;
        case SG_STYLE_VBARS:
          toggle_selection = 4;
          break;
        case SG_STYLE_HBARS:
          toggle_selection = 5;
          break;
        case SG_STYLE_VBOXES:
          toggle_selection = 6;
          break;
        case SG_STYLE_HBOXES:
          toggle_selection = 7;
          break;
        case SG_STYLE_BUBBLES:
          toggle_selection = 8;
          break;
        case SG_STYLE_COLORS:
          toggle_selection = 9;
          break;
        case SG_STYLE_FLUX:
          toggle_selection = 10;
          break;
        case SG_STYLE_CONTOUR:
          toggle_selection = 11;
          break;
      }
    break;
  }

  sg_toggle_data_init(toggle_combo, toggle_selection);

  entries_table = gtk_table_new(num_entries, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_style][i]);
    if(columns_mask[data_style] & (1 << i)){
      button_box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach_defaults(GTK_TABLE (entries_table), button_box, 
                                0, 1, nrows, nrows + 1);

      left_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(left_button[i]), 
                        gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(left_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), left_button[i], FALSE, FALSE, 0);
 
      right_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(right_button[i]), 
                        gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(right_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), right_button[i], FALSE, FALSE, 0);

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label, 
                                1, 2, nrows, nrows + 1);    
      column_entry[i] = gtk_entry_new();
      gtk_entry_set_editable(GTK_ENTRY(column_entry[i]), FALSE);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_entry[i], 
                                2, 3, nrows, nrows + 1);    

      if(col_index[i] != -1){
        gint row;
        gchar *text = GTK_SHEET(the_dataset->worksheet->sheet)->column[col_index[i]].name;
        gchar *ctext;

        gtk_entry_set_text(GTK_ENTRY(column_entry[i]), text);

        for(row = 0; row < GTK_CLIST(columns_list)->rows; row++){
          gtk_clist_get_text(GTK_CLIST(columns_list), row, 0, &ctext);
          if(strcmp(text, ctext) == 0)
            gtk_clist_remove(GTK_CLIST(columns_list), row);
        }
      }


      gtk_signal_connect(GTK_OBJECT(left_button[i]), "clicked",
                         (GtkSignalFunc) restore_column, column_entry[i]);
      gtk_signal_connect(GTK_OBJECT(right_button[i]), "clicked",
                         (GtkSignalFunc) set_column, column_entry[i]);

      nrows++;
    }
  }

  gtk_box_pack_start(GTK_BOX(entries_box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}


static void
select_columns(GtkWidget *widget, gint row, gint col, gpointer data)
{
  toggle_selection = sg_toggle_data_get_selection(widget);
  
  switch(toggle_selection){
    case 0:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_LPOINTS;
         break;
      }
    case 1:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_BUBBLES;
         break;
      }
    case 2:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_COLORS;
         break;
      }
    case 3:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_FLUX;
         break;
      }
      data_style = SG_STYLE_LPOINTS;
      break;
    case 4:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_SURFACE;
         break;
      }
      if(layer_type == SG_LAYER_POLAR){
         data_style = SG_STYLE_BUBBLES;
         break;
      }
      data_style = SG_STYLE_VBARS;
      break;
    case 5:
      if(layer_type == SG_LAYER_3D){
         data_style = SG_STYLE_CSURFACE;
         break;
      }
      if(layer_type == SG_LAYER_POLAR){
         data_style = SG_STYLE_COLORS;
         break;
      }
      data_style = SG_STYLE_HBARS;
      break;
    case 6:
      data_style = SG_STYLE_VBOXES;
      break;
    case 7:
      data_style = SG_STYLE_HBOXES;
      break;
    case 8:
      data_style = SG_STYLE_BUBBLES;
      break;
    case 9:
      data_style = SG_STYLE_COLORS;
      break;
    case 10:
      data_style = SG_STYLE_FLUX;
      break;
    case 11:
      data_style = SG_STYLE_CONTOUR;
      break;
  }    

  real_select_columns(widget, data);
}

static void
real_select_columns(GtkWidget *widget, gpointer data)
{
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint i, nrows = 0;
  gchar label[9][100];

  update_columns(GTK_ENTRY(GTK_COMBO(worksheet_combo)->entry));

  if(entries_table && GTK_IS_WIDGET(entries_table))
     gtk_container_remove(GTK_CONTAINER(entries_box), entries_table);

  entries_table = gtk_table_new(num_entries, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(entries_table), 5);

  for(i = 0; i < 9; i++){
    sprintf(label[i], "%s:", column_labels[data_style][i]);
    if(columns_mask[data_style] & (1 << i)){
      button_box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach_defaults(GTK_TABLE (entries_table), button_box, 
                                0, 1, nrows, nrows + 1);

      left_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(left_button[i]), 
                        gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(left_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), left_button[i], FALSE, FALSE, 0);
 
      right_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(right_button[i]), 
                        gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(right_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), right_button[i], FALSE, FALSE, 0);

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_label, 
                                1, 2, nrows, nrows + 1);    
      column_entry[i] = gtk_entry_new();
      gtk_entry_set_editable(GTK_ENTRY(column_entry[i]), FALSE);
      gtk_table_attach_defaults(GTK_TABLE(entries_table), column_entry[i], 
                                2, 3, nrows, nrows + 1);    

      gtk_signal_connect(GTK_OBJECT(left_button[i]), "clicked",
                         (GtkSignalFunc) restore_column, column_entry[i]);
      gtk_signal_connect(GTK_OBJECT(right_button[i]), "clicked",
                         (GtkSignalFunc) set_column, column_entry[i]);

      nrows++;
    }
  }

  gtk_box_pack_start(GTK_BOX(entries_box), entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(entries_table);
}

