//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_sph,vel_sph,jacob]=CL_co_car2sph(pos_car,vel_car)
// Cartesian coordinates to spherical coordinates
//
// Calling Sequence
// [pos_sph [,vel_sph,jacob]] = CL_co_car2sph(pos_car [,vel_car])
//
// Description
// <itemizedlist><listitem>
// This function converts cartesian coordinates into spherical coordinates.  
// <para> The input arguments can consist in position only, or position and velocity, in which case
// the derivatives of the spherical coordinates with respect to time are computed as well. </para> 
// <para/><inlinemediaobject><imageobject><imagedata fileref="spherical_coord.gif"/></imageobject></inlinemediaobject>
// </listitem>
// <listitem>
// Notes: 
// <para> - The transformation jacobian is computed if the corresponding output argument exists. </para>
// <para> - Be careful that the 3rd coordinate is the distance to the planet center and not an altitude. </para>
// </listitem>
// </itemizedlist>
//
// Parameters
// pos_car : [X;Y;Z] Position vector in cartesian coordinates [m] (3xN)
// vel_car : (optional, needed for vel_sph and jacob computation) [Vx;Vy;Vz] Velocity vector in cartesian coordinates [m/s] (3xN)
// pos_sph : [lon;lat;r] Position vector in spherical coordinates [rad,m](3xN)
// vel_sph : (optional) [d(lon)/dt;d(lat)/dt;d(r)/dt] Derivatives of spherical coordinates [rad/s,m/s] (3xN)
// jacob : (optional) Transformation jacobian (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Mecanique Spatiale, Cnes - Cepadues Editions, Tome I, section 3.2.3 (Les reperes de l'espace et du temps, Relations entre les coordonnees)
// 2 CNES - MSLIB FORTRAN 90, Volume T (mt_car_geoc)
//
// See also
// CL_co_sph2car
// CL_oe_car2cir
// CL_oe_car2cirEqua
// CL_co_car2ell
// CL_oe_car2kep
//
// Examples
// // Example 1
// pos_car = [3842403.1 ; -5057704.6 ; 577780.5];
// [pos_sph] = CL_co_car2sph(pos_car);
//
// // Example 2
// vel_car = [1000;1000;100];
// [pos_sph,vel_sph,jacob] = CL_co_car2sph(pos_car,vel_car);

// Declarations:


// Code:

[lhs,rhs] = argn();

select lhs
  case 1
    compute_velocity = %f
    compute_jacob = %f
  case 2
    compute_velocity = %t
    compute_jacob = %f
    if ~exists('vel_car','local') then CL__error('give velocity as input argument'); end
  case 3
    compute_velocity = %t
    compute_jacob = %t
    if ~exists('vel_car','local') then CL__error('give velocity as input argument'); end
  else
    CL__error('check number of ouptur arguments')
end

N = size(pos_car,2);

eps100 = 100*%eps

//GEOCENTRIC POSITION COMPUTATION
X = pos_car(1,:);
Y = pos_car(2,:);
Z = pos_car(3,:);

//distance
d = CL_colNorm(pos_car); //vector module
ii = find(d==0); // gestion des cas ou le vecteur est de norme nulle.
d(ii) = 1;

//latitude
lat = asin(Z./d);
//longitude
lon = [];
xy_nz = find(X ~= 0 | Y ~= 0);  //x != 0 ou y != 0
xy_z = find(X == 0 & Y == 0);  //x et y = 0
lon(1,xy_nz) = CL__sc2angle(X(xy_nz),Y(xy_nz));
lon(1,xy_z) = 0; //at poles indefined longitude: we put 0

pos_sph = [lon;lat;d];
pos_sph(:,ii) = 0;

//GEOCENTRIC velocity COMPUTATION
if compute_velocity
  VX = vel_car(1,:)
  VY = vel_car(2,:)
  VZ = vel_car(3,:)
  dist2_xy = X.^2 + Y.^2
  ii2 = find(dist2_xy ==0)
  dist2_xy(ii2) = 1;
  dist_xy = sqrt(dist2_xy)

  //distance
  sp = CL_dot(pos_car,vel_car) //scalar product
  vel_d = sp./d    //velocity distance

  //latitude
  z_vit_d_sur_d = Z .* (vel_d./d)
  vel_lat = (VZ - z_vit_d_sur_d) ./ dist_xy

  //longitude
  cp_z = X.*VY - Y.*VX //position-velocity CL_cross product z component
  vel_lon = cp_z ./ dist2_xy

  vel_sph=[vel_lon;vel_lat;vel_d];
  vel_sph(:,ii2) = 0;
end

//JACOBIAN COMPUTATION: partial derivatives of position and velocity vectors
if compute_jacob
  if(ii ~=[] | ii2 ~=[])
      CL__error('Impossible computation of jacobian matrix');
  end
  jacob = zeros(6,6,N)

  d2 = d.*d
  z_sur_d = Z./d
  z_sur_d2 = Z./d2
  z_vit_d_sur_d3 = z_sur_d2.*vel_d./d

  //latitude partial derivatives
  var1 = -z_sur_d2./dist_xy

  dlat_sur_dx = X .* var1
  jacob(2,1,1:N) = dlat_sur_dx

  dlat_sur_dy  = Y.*var1
  jacob(2,2,1:N) = dlat_sur_dy

  dlat_sur_dz  =  dist_xy / d2
  jacob(2,3,1:N) = dlat_sur_dz

  //longitud partial derivatives
  dlong_sur_dx  =  -Y ./ dist2_xy
  jacob(1,1,1:N) =  dlong_sur_dx

  dlong_sur_dy  = X ./ dist2_xy
  jacob(1,2,1:N) = dlong_sur_dy

  //d partial derivatives
  dd_sur_dx  = X ./ d
  jacob(3,1,1:N) = dd_sur_dx

  dd_sur_dy  = Y ./ d
  jacob(3,2,1:N) = dd_sur_dy

  jacob(3,3,1:N) = z_sur_d

  //velocity derivatives over d
  var4 = vel_d ./ d

  dvit_d_sur_dx = (VX - X.*var4) ./ d
  jacob(6,1,1:N) = dvit_d_sur_dx

  dvit_d_sur_dy = (VY - Y.*var4) ./ d
  jacob(6,2,1:N) = dvit_d_sur_dy

  dvit_d_sur_dz = (VZ - Z.*var4) / d
  jacob(6,3,1:N) = dvit_d_sur_dz

  jacob(6,4,1:N) = dd_sur_dx
  jacob(6,5,1:N) = dd_sur_dy
  jacob(6,6,1:N) = z_sur_d

  //velocity partial derivatives over latitude
  var2 = z_vit_d_sur_d3 - (vel_lat./dist_xy)
  jacob(5,1,1:N) = (X.*var2 - z_sur_d.*dvit_d_sur_dx) ./ dist_xy
  jacob(5,2,1:N) = (Y.*var2 - z_sur_d.*dvit_d_sur_dy) ./ dist_xy
  jacob(5,3,1:N) = (z_vit_d_sur_d3.*Z - z_sur_d.*dvit_d_sur_dz - vel_d./d) ./ dist_xy
  jacob(5,4,1:N) = dlat_sur_dx
  jacob(5,5,1:N) = dlat_sur_dy
  jacob(5,6,1:N) = dlat_sur_dz

  //velocity partial derivatives over longitude
  var3 = 2 * vel_lon
  jacob(4,1,1:N) = (VY - X.*var3) ./  dist2_xy
  jacob(4,2,1:N) =  -(VX + Y.*var3) ./ dist2_xy
  jacob(4,4,1:N) = dlong_sur_dx
  jacob(4,5,1:N) = dlong_sur_dy

end

endfunction
