//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [M,Inok] = CL_rot_defFrameVec(varargin)
// Frame transformation matrix based on 1 or 2 vectors. 
//
// Calling Sequence
// [M,Inok] = CL_rot_defFrameVec(u, v, n1, n2)
// [M,Inok] = CL_rot_defFrameVec(u, n1)
//
// Description
// <itemizedlist><listitem>
// <para>If u,n1 and v,n2 are given, the frame transformation matrix is such that: </para>
// <para>- axis n1: colinear to u (and same direction) </para>
// <para>- axis n2: in the (u,v) plane (in the half-plane delimited by u and containing v)</para>
// </listitem>
// <listitem>
// <para>If only (u,n1) are given, the frame transformation matrix is such that the n1 axis is 
// colinear to u (with same direction). The other axes are arbitrarily defined. </para>
// </listitem>
// <listitem>
// <para>Inok contains the indices for which the transformation matrix is not 
// uniquely defined. </para>
// </listitem></itemizedlist>
//
// Parameters
// n1: Number of axis colinear to vector u (1,2 or 3) (1xN)
// u: First vector (3xN)
// n2: (optional) Number of axis in the (u,v) plane (1,2 or 3) (1xN)
// v: (optional) Second vector (3xN)
// M: Frame transformation matrix (3x3xN)
// Inok: Indices for which the input arguments do not define a unique rotation. 
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_defQuat
// CL_rot_defRotVec
//
// Examples
// CL_rot_defFrameVec([1;0;0], [1;1;0], 1, 2) // => identity
//
// // Computes local orbital frame:
// kep = [7000.e3; 0.01; %pi/3 ; 0 ; 0; 0]; // orbital elements
// [r,v] = CL_oe_kep2car(kep); // position and velocity
//
// CL_rot_defFrameVec(r, v, 1, 2) // => "qsw" local frame
// CL_rot_defFrameVec(-r, v, 3, 1)  //=> "lvlh" local frame
//

// Declarations:

// Code:

// check number of input arguments 
[lhs,rhs] = argn();

if (rhs == 4)
  u = varargin(1); 
  v = varargin(2); 
  n1 = varargin(3); 
  n2 = varargin(4); 
elseif (rhs == 2)
  u = varargin(1); 
  n1 = varargin(2); 
else
  CL__error("Invalid number of input arguments (2 or 4 expected)");
end

// check argument values
if (find(n1 < 1 | n1 > 3 | n1 <> round(n1)) ~= [])
  CL__error("Wrong value for n1");
end

if (rhs > 2 & find(n2 < 1 | n2 > 3 | n2 <> round(n2)) ~= [])
  CL__error("Wrong value for n2");
end


Id = eye(3,3); // identity

if (rhs == 4)
  // (u,n1) and (v,n2) given
  u0 = Id(:,n1);
  v0 = Id(:,n2);

  [q,Inok] = CL_rot_defRotVec(u0, v0, u, v);
  M = CL_rot_quat2matrix(q); 

else
  // Only (u,n1) given

  u0 = Id(:,n1);

  [q,Inok] = CL_rot_defRotVec(u0, u0, u, u);
  M = CL_rot_quat2matrix(q);

end


endfunction
