//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [v] = CL_kp_E2v(ecc,E)
// Eccentric anomaly to true anomaly
//
// Calling Sequence
// v = CL_kp_E2v(ecc,E)
//
// Description
// <itemizedlist><listitem>
// Computes true anomaly from eccentric anomaly and eccentricity for elliptical and hyperbolic orbits.
// </listitem>
// </itemizedlist>
//
// Parameters
// ecc: Eccentricity (PxN or Px1)
// E: Eccentric anomaly [rad] (PxN or Px1)
// v: True anomaly [rad] (PxN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Chapter 3, equation 3.10a
// 2 Mecanique spatiale, CNES - Cepadues 1995, Tome I, section 4.6.6, page 179
//
// See also
// CL_kp_M2Ecir
// CL_kp_M2E
// CL_kp_v2E
//
// Examples
// // Reciprocity test
// E=[3.4794 5.7281];
// ecc=[0.37255 0.55];
// v = CL_kp_E2v(ecc,E);
// E2 = CL_kp_v2E(ecc,v);

// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;

// Code:

NE = size(E,2)
Ne = size(ecc,2)
N = max(NE,Ne)

if ~( NE==1 | Ne==1 | NE==Ne )
  CL__error('Wrong size of input arguments');
end


if (NE == 1); E = E * ones(1,N); end
if (Ne == 1); ecc = ecc * ones(1,N); end

epsilon = %CL_epsilon.parab

if (find(ecc < 0) ~= [])
  CL__error("Negative eccentricity");
end

if (find(ecc >= 1 - %CL_epsilon.parab & ecc <= 1 + %CL_epsilon.parab) ~= [])
  CL__error("Parabolic orbits not handled");
end

v = zeros(E);

I = find(ecc < 1); // elliptic
if (I ~= [])
  betaa = ecc(I)./(1+sqrt(1-ecc(I).^2));
  v(I) = E(I) + 2*atan(betaa.*sin(E(I)),1-betaa.*cos(E(I)));
end

I = find(ecc > 1);  // hyperbolic
if (I ~= [])
  w = sqrt((ecc(I)+1)./(ecc(I)-1)).*tanh(E(I)./2);
  v(I) = 2.0.*atan(w);
end

endfunction


