//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [sma,ecc] = CL_op_rava2ae(ra,va,mu)
// Apoapsis radius and velocity to semi major axis and eccentricity (elliptic orbits)
//
// Calling Sequence
// [sma,ecc] = CL_op_rava2ae(ra,va[,mu])
//
// Description
// <itemizedlist><listitem>
// Computes keplerian parameters semi major axis and eccentricity from
// apoapsis radius and velocity (only elliptic orbit).
// </listitem>
// </itemizedlist>
//
// Parameters
// ra: apoapsis radius [m] (1xN)
// va: velocity at apoapsis [m/s] (1xN)
// mu : (optional) geocentric gravitational constant [m^3/s^2] (default value is %CL_mu)
// sma: semi major axis [m] (1xN)
// ecc: eccentricity (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// ra = 7078.e3;
// va = 7.e3;
// [sma,ecc] = CL_op_rava2ae(ra,va)

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end

sma = (mu.*ra)./(2*mu-(va.^2).*ra)
ecc = (mu-sma.*va.^2)./(mu+sma.*va.^2)

endfunction
