//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [N]=CL_cw_Nmatrix(alt,delta_t,er,mu)
// Clohessy-Wiltshire N transformation Matrix
//
// Calling Sequence
// [N]=CL_cw_Nmatrix(alt,delta_t[,er,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the Clohessy-Wiltshire <emphasis role="bold">N</emphasis> transformation matrix.
// <para>The local orbital reference frame tied to the target is LVLH (See <link linkend="CL_fr_lvlhMat">CL_fr_lvlhMat</link>).</para>
// <para/><inlinemediaobject><imageobject><imagedata fileref="cw.gif"/></imageobject></inlinemediaobject>
// </listitem>
// </itemizedlist>
//
// Parameters
// alt:  Target altitude [m] (1xN)
// delta_t: Time step [s] (1xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default is %CL_mu)
// N: Clohessy-Wiltshire "N" transformation matrix (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_cw_Mmatrix
// CL_cw_twoImpulse
//
// Bibliography
// Mecanique spatiale, CNES - Cepadues 1995, Tome II, chapter 16 (Le rendez-vous orbital)
//
// Examples
// [N]=CL_cw_Nmatrix(350000,100)
//

// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end

S = size(alt,2)

N = hypermat([6 3 S])

omega = CL_kp_params('mm',alt+er,mu)
teta = omega.*delta_t
c = cos(teta)
s = sin(teta)

N(1,1,:) = (1.0./omega.^2).*( 4*(1-c) - (3/2).*teta.^2 )
N(3,1,:) = (2.0./omega.^2).*(s-teta)
N(4,1,:) = (4*s-3*teta)./omega
N(6,1,:) = (2.0./omega).*(c-1)
N(2,2,:) = (1-c)./omega.^2
N(5,2,:) = s./omega
N(1,3,:) = (2.0./omega.^2).*(teta-s)
N(3,3,:) = (1-c)./omega.^2
N(4,3,:) = (2.0./omega).*(1-c)
N(6,3,:) = s./omega

if S==1 then N=N(:,:,1); end

endfunction
