//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [dv,anv] = CL_man_apsidesLine(ai,ei,pomi,pomf,posman,mu)
// Delta V required to change the line of apsides
//
// Calling Sequence
// [dv,anv] = CL_man_apsidesLine(ai,ei,pomi,pomf [,posman,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the velocity increment needed to modify the line of apsides.
// <para> The maneuver consists in making the orbit rotate in the orbit plane. Thus, only the argument of periapsis is changed; the semi major axis or the eccentricity are not. </para>
// <para> The output argument <emphasis role="bold">dv</emphasis> is the velocity increment 
// expressed in spherical coordinates in the QSW local frame: </para>
// <para> <emphasis role="bold">dv</emphasis> = [lambda;phi;dv] (lambda: in-plane angle 
// (pi = towards planet and pi/2 ~ along velocity), phi: out-of-plane angle, positive towards 
// the angular momentum vector (the angular momentum vector is perpendicular to the orbit plane 
// and oriented according to the right hand rule), dv: norm of delta-V). </para>
// <para><emphasis role="bold">anv</emphasis> is the true anomaly at the position of maneuver.</para>
// <para><emphasis role="bold">posman</emphasis> can be used to define where the maneuver takes place: 0->close to periapsis, 1->close to apoapsis, default is 0)</para>
// </listitem>
// </itemizedlist>
//
// Parameters
// ai : Initial semi major axis [m] (1xN)
// ei: Initial eccentricity (1xN)
// pomi: Initial argument of periapsis [rad] (1xN)
// pomi: Final argument of periapsis [rad] (1xN)
// posman: (optional) Flag specifying the maneuver location (0->close to periapsis, 1->close to apoapsis, default is 0) (1xN)
// mu : (optional) Gravitational constant. [m^3/s^2] (default value is %CL_mu)
// dv : Delta-V in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad,rad,m/s] (3xN)
// anv: True anomaly at the position of maneuver [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_man_biElliptic
// CL_man_hohmann
// CL_man_sma
// CL_man_hohmannG
// CL_man_inclination
//
// Examples
// ai = 7200.e3;
// ei = 0.1;
// pomi = CL_deg2rad(65);
// pomf = CL_deg2rad(85);
// [dv,anv] = CL_man_apsidesLine(ai,ei,pomi,pomf,1)
// // Check results :
// anm = CL_kp_v2M(ei,anv);
// kep = [ai ; ei ; %pi/2 ; pomi ; 0 ; anm];
// kep1 = CL_man_applyDv(kep,dv)


// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('posman','local') then posman=0; end

Nai = size(ai,2);
Nei = size(ei,2);
Npomi = size(pomi,2);
Nposman = size(posman,2);
Npomf = size(pomf,2);

N = max(Nai,Nei,Npomi,Nposman,Npomf);
coherence = (Nai==N | Nai==1) &..
            (Nei==N | Nei==1) &..
            (Npomi==N | Npomi==1) &..
            (Nposman==N | Nposman==1) &..
            (Npomf==N | Npomf==1);

if ~coherence then CL__error('bad input argument sizes'); end
if N~=1
  if Nai==1 then ai=ai*ones(1,N); end
  if Nei==1 then ei=ei*ones(1,N); end
  if Npomi==1 then pomi=pomi*ones(1,N); end
  if Npomf==1 then pomf=pomf*ones(1,N); end
  if Nposman==1 then posman=posman*ones(1,N); end
end

dv = zeros(3,N);
anv = zeros(1,N);

if ~exists('mu','local') then mu=%CL_mu; end

//argument of periapsis correction (angle between -pi and pi)
dpom = modulo(pomf-pomi,2*%pi);

//true anomaly at maneuver execution point
i_p = find(posman==0) //maneuver next to periapsis
  anv(i_p) = pmodulo(dpom(i_p)*0.5,2*%pi)
i_a = find(posman==1)  //maneuver next to apoapsis
  anv(i_a) = pmodulo(%pi + dpom(i_a)*0.5,2*%pi)
if find(posman~=0 & posman~=1)~=[] then CL__error('bad value of posman argument'); end

//radius and velocity at maneuver execution point
rayon = ai.*(1-ei.*ei)./(1+ei.*cos(anv))
v1 = sqrt(mu.*(2.0./rayon-1.0./ai))

//calculation of velocity impulsion
deltav = abs( 2.0.*sqrt(mu./(ai.*(1-ei.*ei))).*ei.*sin(dpom.*0.5) )

//calculation of angle between initial velocity and impulsion
dangle = %pi/2 + asin(deltav./(2.0.*v1))

//impulsion in spheric coordinates
//attention please!: dpom is assumed to be in [-pi,pi]
dv(2,:) = 0
dv(3,:) = deltav
i1 = find(posman==0 & dpom>0) //maneuver at periapsis and positive argument of periapsis
  dv(1,i1) = %pi
i2 = find(posman==0 & ~(dpom>0))  //maneuver at periapsis and negative argument of periapsis
  dv(1,i2) = 0
i3 = find(posman==1 & dpom>0) //maneuver at apoapsis and positive argument of periapsis
  dv(1,i3) = 0
i4 = find(posman==1 & ~(dpom>0))  //maneuver at apoapsis and negative argument of periapsis
  dv(1,i4) = %pi

endfunction
