//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [output] = CL_man_consumption(type_output,par1,par2,par3,g0)
// Mass consumed as function of delta-V
//
// Calling Sequence
// [output] = CL_man_consumption(type_output,par1,par2,par3 [,g0])
// [dm] = CL_man_consumption('dm',dv,m,isp[,g0])
// [dv] = CL_man_consumption('dv',dm,m,isp[,g0])
// [m] = CL_man_consumption('m',dm,dv,isp[,g0])
// [isp] = CL_man_consumption('isp',dm,dv,m[,g0])
//
// Description
// <itemizedlist><listitem>
// The mass consumed by a maneuver can be computed by the following equation:
// <para><inlinemediaobject><imageobject><imagedata fileref="consumption.gif"/></imageobject></inlinemediaobject></para>
// <para> Given 3 of the 4 variables that appear in the equation (mass consumed, 
// norm of velocity increment, Isp or initial mass), the function computes the 4th one. </para> 
// <para>The output argument <emphasis role="bold">type_output</emphasis> defines the parameter 
// to be computed. It can be one of the following: </para>
// <para>- <emphasis role="bold">'dm'</emphasis>: mass consumed</para>
// <para>- <emphasis role="bold">'dv'</emphasis>: norm of delta-V</para>
// <para>- <emphasis role="bold">'isp'</emphasis>: propellant specific impulse. </para>
// <para>- <emphasis role="bold">'m'</emphasis>: initial mass</para>
// <para>Note: The input arguments are always in the same order: dm, dv, m, isp. </para>
// </listitem>
// </itemizedlist>
//
// Parameters
// type_output: String defining the parameter to be computed. It can be 'dm,'dv','m' or 'isp'. (1x1)
// par1: First input parameter: dm, dv, m or isp [kg m/s kg or s] (1xN)
// par2: Second input parameter: dv, m or isp [m/s kg or s] (1xN)
// par3: Third input parameter: m or isp [kg or s] (1xN)
// g0: (optional) Value of gravity acceleration. [m/s^2] (default value is %CL_g0) (1x1)
// output: Value of computed parameter: dm, dv, m or isp [kg m/s kg or s] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Chapter 6, equation 6.1
// 2 Mecanique spatiale, CNES - Cepadues 1995, Tome I, section 4.8.6
//
// See also
// CL_man_thrustDuration
//
// Examples
// // Compute mass consumed
// isp = 220; // seconds
// dv = 1; // m/s
// m = 180; // kg
// dm = CL_man_consumption('dm',dv,m,isp); // mass consumed in kg
//
// // Compute delta V
// isp = 220; // seconds
// dm = 1; // kg
// m = 180; // kg
// dv = CL_man_consumption('dv',dm,m,isp); // delta V in m/s
//
// // Compute initial mass
// isp = 220; // seconds
// dv = 1; // m/s
// dm = 1.8; // kg
// m = CL_man_consumption('m',dm,dv,isp); // initial mass in kg
//
// // Compute Isp
// dm = 1; // kg
// dv = 1; // m/s
// m = 180; // kg
// isp = CL_man_consumption('isp',dm,dv,m); // isp in sec



// Declarations:
if(~exists('%CL_g0')) then global %CL_g0; end;

// Code:

Npar1 = size(par1,2)
Npar2 = size(par2,2)
Npar3 = size(par3,2)
N = max(Npar1,Npar2,Npar3)
coherence = (Npar1==N|Npar1==1) & (Npar2==N|Npar2==1) & (Npar3==N|Npar3==1)
if ~coherence then CL__error('bad size of input arguments'); end
if N~=1
  if Npar1==1 then par1=par1*ones(1,N); end
  if Npar2==1 then par2=par2*ones(1,N); end
  if Npar3==1 then par3=par3*ones(1,N); end
end

if ~exists('g0','local') then g0=%CL_g0; end

select type_output
  case 'dm'
    dv = par1;
  m = par2;
    isp = par3;
    dm = m.*(1 - exp(-dv./g0./isp));
    output=dm;
  case 'dv'
    dm = par1;
  m = par2;
    isp = par3;
    dv = -g0.*isp.*log(1-dm./m);
    output=dv;
  case 'm'
    dm = par1;
    dv = par2;
    isp = par3;
    m = dm./(1 - exp(-dv./g0./isp));
    output=m;
  case 'isp'
    dm = par1;
    dv = par2;
    m = par3;
    isp = -dv./(g0.*log(1-dm./m));
    output=isp
  else
    CL__error('unknown type_output');
end

endfunction
