//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [varargout]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,type_par2)
// Visibility parameters (angles, distance, ...) for a spherical planet
//
// Calling Sequence
// [par2]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,type_par2)
// [res1,..resN]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,[type1,..,typeN])
// [result]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,'all')
//
// Description
// <itemizedlist><listitem>
// <p>Computes various visibility parameters: </p>
// <p>- satellite's semi view-angle (<b>sat</b>): 
// angle between the descending vertical and the direction of the target location </p>
// <p>- satellite's elevation (<b>elev</b>): 
// elevation of the satellite as seen from the target location </p>
// <p>- satellite's incidence (<b>incid</b> = pi/2 - elev): 
// incidence of the target-satellite direction from the target location </p>
// <p>- distance (<b>dist</b>): 
// distance between the satellite and the target location </p>
// <p>- centre angle (<b>cen</b>): 
// angle between the (planet centre -> satellite) direction and the (planet centre -> target location) direction. </p>
// <p>Given the distance from the planet centre to the satellite (<b>sat_radius</b>),
// the distance from the planet centre to the target (<b>target_radius</b>) 
// and one of the following parameters (<b>type_par1</b>):</p>
// <p>type_par1 = 'sat' , par1 = satellite's semi view angle in radians.</p>
// <p>type_par1 = 'elev' , par1 = elevation from the target location in radians.</p>
// <p>type_par1 = 'incid' , par1 = incidence (=pi/2-elev) from the target location in radians.</p>
// <p>type_par1 = 'dist' , par1 = distance between the target location and the satellite in meters.</p>
// <p>type_par1 = 'cen' , par1 = centre angle: angle from the planet centre between the target location and the satellite in radians.</p>
// <p>The function computes <b>par2</b> whose type is defined by <b>type_par2</b></p> 
// <p><b>type_par2</b> can also be an array of strings (any of the 5 parameters above), 
// or 'all' and in that case the result is a structure whose fields are the 5 parameters above</p>
// <p><inlinemediaobject><imageobject><imagedata fileref="visiParams.gif"/></imageobject></inlinemediaobject></p>
// <p></p></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p> - A spherical planet is assumed.</p>
// <p> - If the input parameter is 'sat' there usually exist 2 possible solutions.
// But only the one for which the elevation is positive is computed. </p>
// </listitem></itemizedlist>
//
// Parameters
// sat_radius: Distance from the satellite to the planet centre (satellite altitude + planet radius) [m] (1xN)
// target_radius: Distance from the target to the planet centre (target altitude + planet radius) [m] (1xN)
// type_par1: (string) Type of input parameter par1. It can be 'sat', 'elev', 'incid', 'dist', 'cen'
// par1: Satellite's semi view angle, elevation, indicence, distance or centre angle [rad,m] (1xN)
// type_par2: (string) Type of ouput parameter(s) to be computed. It can be 'sat', 'elev', 'incid', 'dist', 'cen' or a vector contaning any of them, or 'all'. If type_par2 == 'all', a structure is returned. 
// par2: Computed value(s): [rad,m] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// sat_r = %CL_eqRad + 700.e3; // 700 km altitude
// target_r = %CL_eqRad + 0; // Ground
// 
// // Distance to Ground incidence: 
// [incid]=CL_gm_visiParams(sat_r,target_r,'dist',800.e3,'incid')
//
// // Satellite view angle to Ground elevation:
// [elev]=CL_gm_visiParams(sat_r,target_r,'sat',CL_deg2rad(10),'elev') 
//
// // Centre angle to distance:
// [dist]=CL_gm_visiParams(sat_r,target_r,'cen',CL_deg2rad(7),'dist') 
//
// // Ground incidence to satellite view angle and distance:
// [sat,dist]=CL_gm_visiParams(sat_r,target_r,'incid',CL_deg2rad(15),['sat','dist'])
//
// // Satellite view angle to everything:
// [result]=CL_gm_visiParams(sat_r,target_r,'sat',CL_deg2rad(37),'all');


// Declarations:


// Code:

[lhs,rhs]=argn();

if (rhs ~= 5) 
   CL__error('This function requires 5 input arguments'); 
end

if (typeof(type_par2) <> 'string') 
   CL__error('Wrong type for argument type_par2'); 
end

if ~(size(type_par2,2) == lhs | (type_par2 == 'all' & lhs == 1) ) 
   CL__error('Wrong number of output parameters'); 
end

// calculation of angle at centre depending on input type_par1
if type_par1=='elev' | type_par1=='incid' // par1 = elevation or incidence

  if type_par1=='incid'
    angsit = %pi/2 - par1
  else
    angsit = par1
  end
  i1 = find( abs(angsit)>%pi/2 )
  if i1~=[] then CL__error('Elevation greater than pi/2'); end
  angcen = acos( (target_radius./sat_radius).*cos(angsit) ) - angsit

elseif type_par1=='sat'  //par1 is satellite angle

  angsat = par1
  i2 = find( abs(angsat)>asin(target_radius./sat_radius) )
  if i2~=[] then CL__error('Satellite angle greater than asin(target_radius/sat_radius) : no intersection with the Earth'); end
  angsit = acos( (sat_radius./target_radius) .* sin(angsat) )
  //2 solutions
  angcen = %pi/2 - angsat - angsit
  //warning('only solution for elev>=0 is computed');
  //angcen = %pi/2 - angsat + angsit

elseif type_par1=='dist' //par1 is distance

  dist = par1
  i3 = find( dist<sat_radius-target_radius | dist>sat_radius+target_radius )
  if i3~=[] then CL__error('Distance too low (<sat_radius-target_radius) or too high (>sat_radius+target_radius)'); end
  angcen = acos( (sat_radius.^2 + target_radius.^2 - dist.^2) ./ (2.*sat_radius.*target_radius))

elseif type_par1=='cen'  //par1 is angle at centre

  angcen = par1

else
  CL__error('Unknown type_par1 value');
end

// calculation of output parameters as function of centre angle
res = struct('sat', 0 , 'elev', 0 , 'incid', 0 , 'dist', 0, 'cen', 0);

res.elev = atan(cos(angcen) - target_radius./sat_radius, sin(angcen)); 
res.incid = %pi/2 - res.elev; 
res.sat = %pi/2 - angcen - res.elev;
res.dist = sqrt(sat_radius.^2 + target_radius.^2 - 2.*sat_radius.*target_radius.*cos(angcen));
res.cen = angcen;

// output
if (type_par2 == 'all') 
  varargout(1) = res;
else
  for k = 1:size(type_par2,2)
    varargout(k) = res(type_par2(k));
  end
end

endfunction
