//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [kep,jacob] = CL_oe_cir2kep(cir)
// Circular adapted to keplerian orbital elements
//
// Calling Sequence
// [kep[,jacob]] = CL_oe_cir2kep(cir)
//
// Description
// <itemizedlist><listitem>
// <p>Converts orbital elements adapted to near-circular orbits to 
// "classical" Keplerian orbital elements.</p>
// <p>The transformation jacobian is optionally computed.</p>
// <p>See <link linkend="Orbital Elements">Orbital elements</link> for more details. </p>
// <p></p></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p> - if eccentricity is 0, the argument of perigee is set to 0.</p>
// <p> - if eccentricity is 0 (actually less than threshold), the jacobian is set to %nan.</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// cir: Orbital elements adapted to near-circular orbits [sma;ex;ey;inc;raan;pom+anm] [m,rad] (6xN)
// kep: Keplerian orbital elements [sma;ecc;inc;pom;raan;anm] [m,rad] (6xN)
// jacob: (optional) Transformation jacobian (See <link linkend="Orbital Elements">Orbital elements</link> for more details) (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_oe_kep2cir
// CL_oe_cir2car
//
// Examples
// // Example 1
// cir = [42166.e3; 1e-4; 1e-4; 1; 2; %pi/2];
// [kep,jacob1] = CL_oe_cir2kep(cir);
// [cir2,jacob2] = CL_oe_kep2cir(kep); // cir2: same as cir
// jacob2 * jacob1 // => identity

// Declarations:
if (~exists('%CL_epsilon')) then global %CL_epsilon; end;

// Code:

// Handle [] cases
if (cir == [])
  kep = [];
  jacob = [];
  return;
end

ecc = sqrt(cir(2,:).^2 + cir(3,:).^2);

// check args
if (find(cir(1,:) <= 0 | ecc > 1-%CL_epsilon.parab) <> [])
  CL__error("Invalid orbital elements"); 
end

[lhs,rhs] = argn();
if (lhs >= 2)
  compute_jacob = %t;
else
  compute_jacob = %f;
end

kep = zeros(cir);

kep(1,:) = cir(1,:);                              // sma = sma
kep(2,:) = ecc;                                   // e = sqrt(ex^2+ey^2)
kep(3,:) = cir(4,:);                              // inc = inc
kep(4,:) = atan( cir(3,:), cir(2,:) );            // w = atan(ey,ex) (if ex and ey=0 --> w =0)
kep(5,:) = cir(5,:);                              // raan = raan
kep(6,:) = cir(6,:) - kep(4,:);                   // M = pso-w


// Jacobian computation
if (compute_jacob)
  // jacob = d(kep)/d(cir)
  // jacob(i,j) = d(kep_i)/d(cir_j)
  //
  // d(e)/d(ex) = cos(w)
  // d(e)/d(ey) = sin(w)
  //
  // d(w)/d(ex) = -sin(w)/e
  // d(w)/d(ey) = cos(w)/e
  //
  // d(M)/d(ex) = sin(w)/e
  // d(M)/d(ey) = -cos(w)/e
  // d(M)/d(pso) = 1

  N = size(kep,2);
  jacob = hypermat([6,6,N]);
  jacob(:,:,:) = %nan;
  
  I = find(kep(2,:) >= %CL_epsilon.cir);
  if (I <> [])
    jacob(:,:,I) = 0; 
  
    jacob(1,1,I) = 1; // da/da
  
    cos_w = cos(kep(4,I));
    sin_w = sin(kep(4,I));
  
    jacob(2,2,I) = cos_w; // de/dex
    jacob(2,3,I) = sin_w; // de/dey
  
    jacob(3,4,I) = 1; // di/di
  
    jacob(4,2,I) = -sin_w ./ ecc(I); // dw/dex
    jacob(4,3,I) = cos_w ./ ecc(I); // dw/dey
  
    jacob(5,5,I) = 1; // dgom/dgom
  
    jacob(6,2,I) = sin_w ./ ecc(I); // dM/dex
    jacob(6,3,I) = -cos_w ./ ecc(I); // dM/dey
    jacob(6,6,I) = 1; // dM/dpso
  
  end

  // conversion to matrix type
  if (N == 1); jacob = jacob(:,:,1); end
end

endfunction
