//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [q] = CL_rot_axAng2quat(u, alpha)
// Rotation axis and angle to quaternion
//
// Calling Sequence
// [q] = CL_rot_axAng2quat(u, alpha)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the quaternion <b>q</b> that corresponds to the 
// rotation of angle <b>alpha</b> around the rotation axis 
// <b>u</b>.</p>
// <p></p></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p> - The output quaternion has a norm equal to 1.  </p>
// <p> - For compatibility reasons, the argument order: (alpha, u) is also 
// possible but should be avoided.  </p>
// <p> - See <link linkend="Data types">Data types</link> or <link linkend="CL_rot_defQuat">CL_rot_defQuat</link> for more details on quaternions. </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// u : Rotation axis [ux;uy;uz]. ||u|| should be > 0 (3xN or 3x1)
// alpha : Rotation angle [rad] (1xN or 1x1)
// q : Quaternion (with norm equal to 1) (dim N)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
//
// Examples
// // 10 degree rotation around Z axis :
// alpha = CL_deg2rad(10);
// q1 = CL_rot_axAng2quat([0;0;1],alpha)
//
// // same : 
// q2 = CL_rot_eul2quat(0,0,alpha)
//
// // same :
// M = CL_rot_angles2matrix(3,alpha);
// q3 = CL_rot_matrix2quat(M)
//

// Declarations:


// Code:

// Code for compatibility (inversion of arguments) : 
if (size(alpha,1) == 3 & size(u,1) == 1) 
  tmp = alpha; 
  alpha = u; 
  u = tmp;
end

s = [size(u) ; size(alpha)];
 
smin = min(s, "r"); // min nb of rows, min nb of columns
smax = max(s, "r"); // max nb of rows, max nb of columns
N = smax(2); // max number of columns

// check rows (must be 3 for u, and 1 for alpha)
if (s(1,1) <> 3 | s(2,1) <> 1)
  CL__error("Invalid argument sizes (number of rows)"); 
end

// check columns (must be 1 or N)
I = find(s(:,2) <> 1 & s(:,2) <> N);
if (I <> [])
  CL__error("Invalid argument sizes (number of columns)");
end

if (s(1,2) < N); u = u * ones(1,N); end
if (s(2,2) < N); alpha = alpha * ones(1,N); end





[axis, axis_norm] = CL_unitVector(u); // axis = %nan if ||u|| == 0 

// if ||u|| == 0  or u = %nan AND alpha == 0 => quaternion = identity

I = find (isnan(sum(axis,'r'))); 

if (I <> [])
  J = I(find (abs(CL_rMod(alpha(I),-%pi,%pi)) <= %eps ));
  axis(:,J) = 0;
  J = I(find (abs(CL_rMod(alpha(I),-%pi,%pi)) > %eps )); // set alpha to %nan if u is %nan and alpha ~=0
  alpha(J) = %nan;
end

ri = zeros(4,size(u,2));
ri(1,:) = cos(alpha./2);
ri(2:4,:)= ( ones(size(u,1),1)*sin(alpha/2) ) .* axis;

q = CL_rot_defQuat(ri);

endfunction
