//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [dv, rph, tanoh] = CL_ip_escapeDv(sma, ecc, vinf, tanoe, mu)
// Delta-V needed to escape from a planet
//
// Calling Sequence
// [dv, rph, tanoh] = CL_ip_escapeDv(sma, ecc, vinf [,tanoe, mu])
//
// Description
// <itemizedlist><listitem>
// <p>Computes the delta-V needed for the transfer from an elliptical orbit around a given planet to an 
// hyperbolic escape orbit.</p>
// <p>The escape maneuver is tangential, i.e. the delta-V is parallel to the velocity vector on the elliptical orbit. </p>
// <p>The initial orbit is defined by its semi-major axis (<b>sma</b>) and eccentricity (<b>ecc</b>).</p>
// <p>The final orbit is defined by its velocity at infinity (<b>vinf</b>).</p>
// <p>The true anomaly of the maneuver on the initial orbit (<b>tanoe</b>) can optionally be specified. By default tanoe=0 (meaning 'at the periapsis').</p>
// <p>The planet is defined by its gravitational constant mu (default is %CL_mu)</p></listitem>
// </itemizedlist>
//
// Parameters
// sma: Semi-major axis of initial orbit [m]. (1xN)
// ecc: Eccentricity of initial orbit. (1xN)
// vinf: Target velocity on hyperbolic orbit at infinity [m/s]. (1xN)
// tanoe: (optional) True anomaly of maneuver [rad]. Default value is 0. (1xN)
// mu: (optional) Gravitational constant of the planet considered [m3/s2]. Default value is %CL_mu.
// dv: Norm of the delta-v. (1xN)
// rph: Periapsis radius of hyperbolic orbit [m]. (1xN)
// tanoh: True anomaly on the hyperbolic orbit at the time of the maneuver [rad]. (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Escape a LEO (circular) orbit :
// sma = [%CL_eqRad+350.e3 , %CL_eqRad+700.e3];
// ecc = zeros(sma);
// vinf = [5 , 6] * 1.e3;
// [dv, rph, tanoh] = CL_ip_escapeDv(sma, ecc, vinf)
//
// // Escape an orbit around Mars :
// global %CL_planetC;
// eqrad_mars = %CL_planetC.mars.eqRad;
// sma = [eqrad_mars+350.e3 , eqrad_mars+700.e3];
// ecc = [0.1 , 0.05];
// vinf = [5 , 6] * 1.e3;
// tanoe = [0, 0.1]; // radians
// mu_mars = %CL_planetC.mars.mu;
// [dv,rph,tanoh] = CL_ip_escapeDv(sma,ecc,vinf,tanoe,mu_mars)
//

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:
[arg_out ,arg_in] = argn();
if ~exists('tanoe','local') then tanoe = 0; end
if ~exists('mu','local') then mu = %CL_mu; end

  Nsma = size(sma,2);
  Necc = size(ecc,2);
  Nvinf = size(vinf,2);
  Ntanoe = size(tanoe,2);
  Nmax = max(Nsma,Necc,Nvinf,Ntanoe);
  coherence = ((Nsma==1) | (Nsma==Nmax)) & ...
			  ((Necc==1) | (Necc==Nmax)) & ...
			  ((Nvinf==1) | (Nvinf==Nmax)) & ...
			  ((Ntanoe==1) | (Ntanoe==Nmax))
			  	  
  if coherence
    if Nsma==1 then sma=sma*ones(1,Nmax); end
    if Necc==1 then ecc=ecc*ones(1,Nmax); end
	if Nvinf==1 then vinf=vinf*ones(1,Nmax); end
	if Ntanoe==1 then tanoe=tanoe*ones(1,Nmax); end
  else
    CL__error('bad size of input arguments');
  end
  
  ii = find(sma <= 0);
  if(ii ~= [])
	CL__error('sma must be strictly positive');
  end
  ii = find(ecc < 0 | ecc >= 1);
  if(ii ~= [])
	CL__error('ecc must be in [0,1[');
  end
  ii = find(vinf <= 0);
  if(ii ~= [])
	CL__error('vinf must be strictly positive');
  end
  
  
  // Vitesse initiale (orbite elliptique) :
  p_ell = sma.*(1-ecc.^2);
  r_ell = p_ell ./ (1 + ecc .* cos(tanoe));
  v_ell = sqrt(mu * (2 ./ r_ell - 1 ./ sma));
  
  
  // Vitesse finale (orbite hyperbolique) :
  r_hyp = r_ell;
  sma_hyp = mu ./ vinf.^2;
  v_hyp = sqrt(mu * (2 ./ r_hyp + 1 ./ sma_hyp));
  
  dv = v_hyp - v_ell;
  
  if(arg_out >= 2)
    // on utilise le fait que la pente est identique, avant et aprés la maneuvre :
    // C_ell ./ (r_ell .* v_ell) = C_hyp ./ (r_hyp .* v_hyp)
	C_ell = sqrt(mu * p_ell);
	C_hyp = C_ell .* v_hyp ./ v_ell;
	ecc_hyp = sqrt( 1 + C_hyp.^2 ./ (mu*sma_hyp));
    rph = sma_hyp .* (ecc_hyp - 1);
  end
  if(arg_out == 3)
    p_hyp = sma_hyp .* (ecc_hyp .^2 - 1);
	sin_gamma = ecc .* sqrt(mu ./ p_ell) .* sin(tanoe) ./ v_ell;
	
	sin_tanoh = sin_gamma ./ ecc_hyp ./ sqrt(mu ./ p_hyp) .* v_hyp;
	cos_tanoh = p_hyp ./ (r_hyp .* ecc_hyp) - 1 ./ ecc_hyp;
	
	tanoh = atan(sin_tanoh,cos_tanoh);
  
  end
  
  
  
  
  
  
endfunction
