/** @file scim_hangul_imengine.cpp
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_hangul_imengine.cpp 67 2004-08-28 17:55:42Z suzhe $
 */

#define Uses_SCIM_UTILITY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_LOOKUP_TABLE
#define Uses_SCIM_CONFIG_BASE

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#ifdef HAVE_GETTEXT
  #include <libintl.h>
  #define _(String) dgettext(GETTEXT_PACKAGE,String)
  #define N_(String) (String)
#else
  #define _(String) (String)
  #define N_(String) (String)
  #define bindtextdomain(Package,Directory)
  #define textdomain(domain)
  #define bind_textdomain_codeset(domain,codeset)
#endif

#include <scim.h>
#include "scim_hangul_imengine.h"

#define scim_module_init hangul_LTX_scim_module_init
#define scim_module_exit hangul_LTX_scim_module_exit
#define scim_imengine_module_init hangul_LTX_scim_imengine_module_init
#define scim_imengine_module_create_factory hangul_LTX_scim_imengine_module_create_factory

#define SCIM_CONFIG_IMENGINE_HANGUL_USE_CAPSLOCK                "/IMEngine/Hangul/UseCapslock"
#define SCIM_CONFIG_IMENGINE_HANGUL_USE_DVORAK                  "/IMEngine/Hangul/UseDvorak"
#define SCIM_CONFIG_IMENGINE_HANGUL_SHOW_CANDIDATE_COMMENT      "/IMEngine/Hangul/ShowCandidateComment"
#define SCIM_CONFIG_IMENGINE_HANGUL_TRIGGER_KEY                 "/IMEngine/Hangul/TriggerKey"
#define SCIM_CONFIG_IMENGINE_HANGUL_HANGUL_HANJA_KEY            "/IMEngine/Hangul/HangulHanjaKey"
#define SCIM_CONFIG_IMENGINE_HANGUL_MANUAL_MODE_KEY             "/IMEngine/Hangul/ManualModeKey"

#define SCIM_PROP_PREFIX                                        "/IMEngine/Hangul"
#define SCIM_PROP_INPUT_MODE                                    "/IMEngine/Hangul/InputMode"
#define SCIM_PROP_OUTPUT_MODE                                   "/IMEngine/Hangul/OutputMode"

#ifndef SCIM_HANGUL_ICON_FILE
    #define SCIM_HANGUL_ICON_FILE           (SCIM_ICONDIR "/scim-hangul.png")
#endif

struct CandidateItem {
    ucs4_t      ch;
    const char *comment;
};

#include "candidatetable.h"

/* Hangul keyboard map table:
 * 2set keyboard (dubulsik) */
static const ucs4_t keyboard_table_2 [] = {
  0x0021,    /* SCIM_KEY_exclam:        exclamation mark    */
  0x0022,    /* SCIM_KEY_quotedbl:    quotation mark         */
  0x0023,    /* SCIM_KEY_numbersign:    number sign         */
  0x0024,    /* SCIM_KEY_dollar:        dollar sign        */
  0x0025,    /* SCIM_KEY_percent:        percent sign        */
  0x0026,    /* SCIM_KEY_ampersand:    ampersand        */
  0x0027,    /* SCIM_KEY_apostrophe:    apostrophe        */
  0x0028,    /* SCIM_KEY_parenleft:    left parenthesis    */
  0x0029,    /* SCIM_KEY_parenright:    right parenthesis    */
  0x002a,    /* SCIM_KEY_asterisk:    asterisk        */
  0x002b,    /* SCIM_KEY_plus:        plus sign        */
  0x002c,    /* SCIM_KEY_comma:        comma            */
  0x002d,    /* SCIM_KEY_minus:        minus sign        */
  0x002e,    /* SCIM_KEY_period:        period            */
  0x002f,    /* SCIM_KEY_slash:        slash            */
  0x0030,    /* SCIM_KEY_0:        0            */
  0x0031,    /* SCIM_KEY_1:        1            */
  0x0032,    /* SCIM_KEY_2:        2            */
  0x0033,    /* SCIM_KEY_3:        3            */
  0x0034,    /* SCIM_KEY_4:        4            */
  0x0035,    /* SCIM_KEY_5:        5            */
  0x0036,    /* SCIM_KEY_6:        6            */
  0x0037,    /* SCIM_KEY_7:        7            */
  0x0038,    /* SCIM_KEY_8:        8            */
  0x0039,    /* SCIM_KEY_9:        9            */
  0x003a,    /* SCIM_KEY_colon:        colon            */
  0x003b,    /* SCIM_KEY_semicolon:    semicolon        */
  0x003c,    /* SCIM_KEY_less:        less-than sign        */
  0x003d,    /* SCIM_KEY_equal:        equals sign        */
  0x003e,    /* SCIM_KEY_greater:        greater-than sign    */
  0x003f,    /* SCIM_KEY_question:    question mark         */
  0x0040,    /* SCIM_KEY_at:        commercial at        */
  0x1106,    /* SCIM_KEY_A:        choseong mieum        */
  0x1172,    /* SCIM_KEY_B:        jungseong yu        */
  0x110e,    /* SCIM_KEY_C:        choseong chieuch    */
  0x110b,    /* SCIM_KEY_D:        choseong ieung        */
  0x1104,    /* SCIM_KEY_E:        choseong ssangtikeut    */
  0x1105,    /* SCIM_KEY_F:        choseong rieul        */
  0x1112,    /* SCIM_KEY_G:        choseong hieuh        */
  0x1169,    /* SCIM_KEY_H:        jungseong o        */
  0x1163,    /* SCIM_KEY_I:        jungseong ya        */
  0x1165,    /* SCIM_KEY_J:        jungseong eo        */
  0x1161,    /* SCIM_KEY_K:        jungseong a        */
  0x1175,    /* SCIM_KEY_L:        jungseong i        */
  0x1173,    /* SCIM_KEY_M:        jungseong eu        */
  0x116e,    /* SCIM_KEY_N:        jungseong u        */
  0x1164,    /* SCIM_KEY_O:        jungseong yae        */
  0x1168,    /* SCIM_KEY_P:        jungseong ye        */
  0x1108,    /* SCIM_KEY_Q:        choseong ssangpieup    */
  0x1101,    /* SCIM_KEY_R:        choseong ssangkiyeok    */
  0x1102,    /* SCIM_KEY_S:        choseong nieun        */
  0x110a,    /* SCIM_KEY_T:        choseong ssangsios    */
  0x1167,    /* SCIM_KEY_U:        jungseong yeo        */
  0x1111,    /* SCIM_KEY_V:        choseong phieuph    */
  0x110d,    /* SCIM_KEY_W:        choseong ssangcieuc    */
  0x1110,    /* SCIM_KEY_X:        choseong thieuth    */
  0x116d,    /* SCIM_KEY_Y:        jungseong yo        */
  0x110f,    /* SCIM_KEY_Z:        choseong khieukh    */
  0x005b,    /* SCIM_KEY_bracketleft:    left bracket        */
  0x005c,    /* SCIM_KEY_backslash:    backslash        */
  0x005d,    /* SCIM_KEY_bracketright:    right bracket        */
  0x005e,    /* SCIM_KEY_asciicircum:    circumflex accent    */
  0x005f,    /* SCIM_KEY_underscore:    underscore        */
  0x0060,    /* SCIM_KEY_quoteleft:    grave accent        */
  0x1106,    /* SCIM_KEY_a:        choseong  mieum        */
  0x1172,    /* SCIM_KEY_b:        jungseong yu        */
  0x110e,    /* SCIM_KEY_c:        choseong  chieuch    */
  0x110b,    /* SCIM_KEY_d:        choseong  ieung        */
  0x1103,    /* SCIM_KEY_e:        choseong  tikeut    */
  0x1105,    /* SCIM_KEY_f:        choseong  rieul        */
  0x1112,    /* SCIM_KEY_g:        choseong  hieuh        */
  0x1169,    /* SCIM_KEY_h:        jungseong o        */
  0x1163,    /* SCIM_KEY_i:        jungseong ya        */
  0x1165,    /* SCIM_KEY_j:        jungseong eo        */
  0x1161,    /* SCIM_KEY_k:        jungseong a        */
  0x1175,    /* SCIM_KEY_l:        jungseong i        */
  0x1173,    /* SCIM_KEY_m:        jungseong eu        */
  0x116e,    /* SCIM_KEY_n:        jungseong u        */
  0x1162,    /* SCIM_KEY_o:        jungseong ae        */
  0x1166,    /* SCIM_KEY_p:        jungseong e        */
  0x1107,    /* SCIM_KEY_q:        choseong  pieup        */
  0x1100,    /* SCIM_KEY_r:        choseong  kiyeok    */
  0x1102,    /* SCIM_KEY_s:        choseong  nieun        */
  0x1109,    /* SCIM_KEY_t:        choseong  sios        */
  0x1167,    /* SCIM_KEY_u:        jungseong yeo        */
  0x1111,    /* SCIM_KEY_v:        choseong  phieuph    */
  0x110c,    /* SCIM_KEY_w:        choseong  cieuc        */
  0x1110,    /* SCIM_KEY_x:        choseong  thieuth    */
  0x116d,    /* SCIM_KEY_y:        jungseong yo        */
  0x110f,    /* SCIM_KEY_z:        choseong  khieukh    */
  0x007b,    /* SCIM_KEY_braceleft:    left brace        */
  0x007c,    /* SCIM_KEY_bar:        vertical bar        */
  0x007d,    /* SCIM_KEY_braceright:    right brace        */
  0x007e,    /* SCIM_KEY_asciitilde:    tilde            */
};

/* Hangul keyboard map table:
 * 3set keyboard 2bul shifted (sebulsik dubulsik badchim) */
static const ucs4_t keyboard_table_32 [] = {
  0x0021,    /* SCIM_KEY_exclam:        exclamation mark        */
  0x0022,    /* SCIM_KEY_quotedbl:    quotation mark             */
  0x0023,    /* SCIM_KEY_numbersign:    number sign             */
  0x0024,    /* SCIM_KEY_dollar:        dollar sign            */
  0x0025,    /* SCIM_KEY_percent:        percent sign            */
  0x0026,    /* SCIM_KEY_ampersand:    ampersand             */
  0x0027,    /* SCIM_KEY_apostrophe:    apostrophe            */
  0x0028,    /* SCIM_KEY_parenleft:    left parenthesis        */
  0x0029,    /* SCIM_KEY_parenright:    right parenthesis        */
  0x002a,    /* SCIM_KEY_asterisk:    asterisk            */
  0x002b,    /* SCIM_KEY_plus:        plus sign            */
  0x002c,    /* SCIM_KEY_comma:        comma                */
  0x002d,    /* SCIM_KEY_minus:        minus sign            */
  0x002e,    /* SCIM_KEY_period:        period                */
  0x002f,    /* SCIM_KEY_slash:        slash                */
  0x0030,    /* SCIM_KEY_0:        0                */
  0x0031,    /* SCIM_KEY_1:        1                */
  0x0032,    /* SCIM_KEY_2:        2                */
  0x0033,    /* SCIM_KEY_3:        3                */
  0x0034,    /* SCIM_KEY_4:        4                */
  0x0035,    /* SCIM_KEY_5:        5                */
  0x0036,    /* SCIM_KEY_6:        6                */
  0x0037,    /* SCIM_KEY_7:        7                */
  0x0038,    /* SCIM_KEY_8:        8                */
  0x0039,    /* SCIM_KEY_9:        9                */
  0x003a,    /* SCIM_KEY_colon:        colon                */
  0x003b,    /* SCIM_KEY_semicolon:    semicolon            */
  0x003c,    /* SCIM_KEY_less:        less-than sign            */
  0x003d,    /* SCIM_KEY_equal:        equals sign            */
  0x003e,    /* SCIM_KEY_greater:        greater-than sign        */
  0x003f,    /* SCIM_KEY_question:    question mark             */
  0x0040,    /* SCIM_KEY_at:        commercial at            */
  0x11b7,    /* SCIM_KEY_A:        jongseong mieum            */
  0x1172,    /* SCIM_KEY_B:        jungseong yu            */
  0x11be,    /* SCIM_KEY_C:        jongseong chieuch        */
  0x11bc,    /* SCIM_KEY_D:        jongseong ieung            */
  0x11ae,    /* SCIM_KEY_E:        jongseong tikeut        */
  0x11af,    /* SCIM_KEY_F:        jongseong rieul            */
  0x11c2,    /* SCIM_KEY_G:        jongseong hieuh            */
  0x1169,    /* SCIM_KEY_H:        jungseong o            */
  0x1163,    /* SCIM_KEY_I:        jungseong ya            */
  0x1165,    /* SCIM_KEY_J:        jungseong eo            */
  0x1161,    /* SCIM_KEY_K:        jungseong a            */
  0x1175,    /* SCIM_KEY_L:        jungseong i            */
  0x1173,    /* SCIM_KEY_M:        jungseong eu            */
  0x116e,    /* SCIM_KEY_N:        jungseong u            */
  0x1164,    /* SCIM_KEY_O:        jungseong yae            */
  0x1168,    /* SCIM_KEY_P:        jungseong ye            */
  0x11b8,    /* SCIM_KEY_Q:        jongseong pieup            */
  0x11a8,    /* SCIM_KEY_R:        jongseong kiyeok        */
  0x11ab,    /* SCIM_KEY_S:        jongseong nieun            */
  0x11ba,    /* SCIM_KEY_T:        jongseong sios            */
  0x1167,    /* SCIM_KEY_U:        jungseong yeo            */
  0x11c1,    /* SCIM_KEY_V:        jongseong phieuph        */
  0x11bd,    /* SCIM_KEY_W:        jongseong cieuc            */
  0x11c0,    /* SCIM_KEY_X:        jongseong thieuth        */
  0x116d,    /* SCIM_KEY_Y:        jungseong yo            */
  0x11bf,    /* SCIM_KEY_Z:        jongseong khieukh        */
  0x005b,    /* SCIM_KEY_bracketleft:    left bracket            */
  0x005c,    /* SCIM_KEY_backslash:    backslash            */
  0x005d,    /* SCIM_KEY_bracketright:    right bracket            */
  0x005e,    /* SCIM_KEY_asciicircum:    circumflex accent        */
  0x005f,    /* SCIM_KEY_underscore:    underscore            */
  0x0060,    /* SCIM_KEY_quoteleft:    grave accent            */
  0x1106,    /* SCIM_KEY_a:        choseong  mieum            */
  0x1172,    /* SCIM_KEY_b:        jungseong yu            */
  0x110e,    /* SCIM_KEY_c:        choseong  chieuch        */
  0x110b,    /* SCIM_KEY_d:        choseong  ieung            */
  0x1103,    /* SCIM_KEY_e:        choseong  tikeut        */
  0x1105,    /* SCIM_KEY_f:        choseong  rieul            */
  0x1112,    /* SCIM_KEY_g:        choseong  hieuh            */
  0x1169,    /* SCIM_KEY_h:        jungseong o            */
  0x1163,    /* SCIM_KEY_i:        jungseong ya            */
  0x1165,    /* SCIM_KEY_j:        jungseong eo            */
  0x1161,    /* SCIM_KEY_k:        jungseong a            */
  0x1175,    /* SCIM_KEY_l:        jungseong i            */
  0x1173,    /* SCIM_KEY_m:        jungseong eu            */
  0x116e,    /* SCIM_KEY_n:        jungseong u            */
  0x1162,    /* SCIM_KEY_o:        jungseong ae            */
  0x1166,    /* SCIM_KEY_p:        jungseong e            */
  0x1107,    /* SCIM_KEY_q:        choseong  pieup            */
  0x1100,    /* SCIM_KEY_r:        choseong  kiyeok        */
  0x1102,    /* SCIM_KEY_s:        choseong  nieun            */
  0x1109,    /* SCIM_KEY_t:        choseong  sios            */
  0x1167,    /* SCIM_KEY_u:        jungseong yeo            */
  0x1111,    /* SCIM_KEY_v:        choseong  phieuph        */
  0x110c,    /* SCIM_KEY_w:        choseong  cieuc            */
  0x1110,    /* SCIM_KEY_x:        choseong  thieuth        */
  0x116d,    /* SCIM_KEY_y:        jungseong yo            */
  0x110f,    /* SCIM_KEY_z:        choseong  khieukh        */
  0x007b,    /* SCIM_KEY_braceleft:    left brace            */
  0x007c,    /* SCIM_KEY_bar:        vertical bar            */
  0x007d,    /* SCIM_KEY_braceright:    right brace            */
  0x007e,    /* SCIM_KEY_asciitilde:    tilde                */
};

/* Hangul keyboard map table:
 * 3set keyboard 390 (sebulsik 390) */
static const ucs4_t keyboard_table_390 [] = {
  0x11bd,    /* SCIM_KEY_exclam:        jongseong cieuc            */
  0x0022,    /* SCIM_KEY_quotedbl:    quotatioin mark            */
  0x0023,    /* SCIM_KEY_numbersign:    number sign             */
  0x0024,    /* SCIM_KEY_dollar:        dollar sign            */
  0x0025,    /* SCIM_KEY_percent:        percent sign            */
  0x0026,    /* SCIM_KEY_ampersand:    ampersand             */
  0x1110,    /* SCIM_KEY_apostrophe:    choseong thieuth        */
  0x0028,    /* SCIM_KEY_parenleft:    left parenthesis        */
  0x0029,    /* SCIM_KEY_parenright:    right parenthesis        */
  0x002a,    /* SCIM_KEY_asterisk:    asterisk            */
  0x002b,    /* SCIM_KEY_plus:        plus sign            */
  0x002c,    /* SCIM_KEY_comma:        comma                */
  0x002d,    /* SCIM_KEY_minus:        minus sign            */
  0x002e,    /* SCIM_KEY_period:        period                */
  0x1169,    /* SCIM_KEY_slash:        jungseong o            */
  0x110f,    /* SCIM_KEY_0:        choseong  khieukh        */
  0x11c2,    /* SCIM_KEY_1:        jongseong hieuh            */
  0x11bb,    /* SCIM_KEY_2:        jongseong ssangsios        */
  0x11b8,    /* SCIM_KEY_3:        jongseong pieup            */
  0x116d,    /* SCIM_KEY_4:        jungseong yo            */
  0x1172,    /* SCIM_KEY_5:        jungseong yu            */
  0x1163,    /* SCIM_KEY_6:        jungseong ya            */
  0x1168,    /* SCIM_KEY_7:        jungseong ye            */
  0x1174,    /* SCIM_KEY_8:        jungseong yi            */
  0x116e,    /* SCIM_KEY_9:        jungseong u            */
  0x003a,    /* SCIM_KEY_colon:        colon                */
  0x1107,    /* SCIM_KEY_semicolon:    choseong  pieup            */
  0x0032,    /* SCIM_KEY_less:        2                */
  0x003d,    /* SCIM_KEY_equal:        euals sign            */
  0x0033,    /* SCIM_KEY_greater:        3                */
  0x003f,    /* SCIM_KEY_question:    question mark             */
  0x0040,    /* SCIM_KEY_at:        commertial at            */
  0x11ae,    /* SCIM_KEY_A:        jongseong tikeut        */
  0x0021,    /* SCIM_KEY_B:        exclamation mark        */
  0x11b1,    /* SCIM_KEY_C:        jongseong rieul-mieum        */
  0x11b0,    /* SCIM_KEY_D:        jongseong rieul-kiyeok        */
  0x11bf,    /* SCIM_KEY_E:        jongseong khieukh        */
  0x11a9,    /* SCIM_KEY_F:        jongseong ssangkiyeok        */
  0x002f,    /* SCIM_KEY_G:        slash                */
  0x0027,    /* SCIM_KEY_H:        apostrophe            */
  0x0038,    /* SCIM_KEY_I:        8                */
  0x0034,    /* SCIM_KEY_J:        4                */
  0x0035,    /* SCIM_KEY_K:        5                */
  0x0036,    /* SCIM_KEY_L:        6                */
  0x0031,    /* SCIM_KEY_M:        1                */
  0x0030,    /* SCIM_KEY_N:        0                */
  0x0039,    /* SCIM_KEY_O:        9                */
  0x003e,    /* SCIM_KEY_P:        greater-than sign        */
  0x11c1,    /* SCIM_KEY_Q:        jongseong phieuph        */
  0x1164,    /* SCIM_KEY_R:        jungseong yae            */
  0x11ad,    /* SCIM_KEY_S:        jongseong nieun-hieuh        */
  0x003b,    /* SCIM_KEY_T:        semicolon            */
  0x0037,    /* SCIM_KEY_U:        7                */
  0x11b6,    /* SCIM_KEY_V:        jongseong rieul-hieuh        */
  0x11c0,    /* SCIM_KEY_W:        jongseong thikeuth        */
  0x11b9,    /* SCIM_KEY_X:        jongseong pieup-sios        */
  0x003c,    /* SCIM_KEY_Y:        less-than sign            */
  0x11be,    /* SCIM_KEY_Z:        jongseong chieuch        */
  0x005b,    /* SCIM_KEY_bracketleft:    left bracket            */
  0x005c,    /* SCIM_KEY_backslash:    backslash            */
  0x005d,    /* SCIM_KEY_bracketright:    right bracket            */
  0x005e,    /* SCIM_KEY_asciicircum:    circumflex accent        */
  0x005f,    /* SCIM_KEY_underscore:    underscore            */
  0x0060,    /* SCIM_KEY_quoteleft:    grave accent            */
  0x11bc,    /* SCIM_KEY_a:        jongseong ieung            */
  0x116e,    /* SCIM_KEY_b:        jungseong u            */
  0x1166,    /* SCIM_KEY_c:        jungseong e            */
  0x1175,    /* SCIM_KEY_d:        jungseong i            */
  0x1167,    /* SCIM_KEY_e:        jungseong yeo            */
  0x1161,    /* SCIM_KEY_f:        jungseong a            */
  0x1173,    /* SCIM_KEY_g:        jungseong eu            */
  0x1102,    /* SCIM_KEY_h:        choseong  nieun            */
  0x1106,    /* SCIM_KEY_i:        choseong  mieum            */
  0x110b,    /* SCIM_KEY_j:        choseong  ieung            */
  0x1100,    /* SCIM_KEY_k:        choseong  kiyeok        */
  0x110c,    /* SCIM_KEY_l:        choseong  cieuc            */
  0x1112,    /* SCIM_KEY_m:        choseong  hieuh            */
  0x1109,    /* SCIM_KEY_n:        choseong  sios            */
  0x110e,    /* SCIM_KEY_o:        choseong  chieuch        */
  0x1111,    /* SCIM_KEY_p:        choseong  phieuph        */
  0x11ba,    /* SCIM_KEY_q:        jongseong sios            */
  0x1162,    /* SCIM_KEY_r:        jungseong ae            */
  0x11ab,    /* SCIM_KEY_s:        jongseong nieun            */
  0x1165,    /* SCIM_KEY_t:        jungseong eo            */
  0x1103,    /* SCIM_KEY_u:        choseong  tikeut        */
  0x1169,    /* SCIM_KEY_v:        jungseong o            */
  0x11af,    /* SCIM_KEY_w:        jongseong rieul            */
  0x11a8,    /* SCIM_KEY_x:        jongseong kiyeok        */
  0x1105,    /* SCIM_KEY_y:        choseong  rieul            */
  0x11b7,    /* SCIM_KEY_z:        jongseong mieum            */
  0x007b,    /* SCIM_KEY_braceleft:    left brace            */
  0x007c,    /* SCIM_KEY_bar:        vertical line(bar)        */
  0x007d,    /* SCIM_KEY_braceright:    right brace            */
  0x007e,    /* SCIM_KEY_asciitilde:    tilde                */
};

/* Hangul keyboard map table:
 * 3set keyboard final (sebulsik choejong) */
static const ucs4_t keyboard_table_3final [] = {
  0x11a9,    /* SCIM_KEY_exclam:        jongseong ssangkiyeok        */
  0x00b7,    /* SCIM_KEY_quotedbl:    middle dot             */
  0x11bd,    /* SCIM_KEY_numbersign:    jognseong cieuc         */
  0x11b5,    /* SCIM_KEY_dollar:        jongseong rieul-phieuph        */
  0x11b4,    /* SCIM_KEY_percent:        jongseong rieul-thieuth        */
  0x201c,    /* SCIM_KEY_ampersand:    left double quotation mark     */
  0x1110,    /* SCIM_KEY_apostrophe:    choseong  thieuth        */
  0x0027,    /* SCIM_KEY_parenleft:    apostrophe            */
  0x007e,    /* SCIM_KEY_parenright:    Tilde                */
  0x201d,    /* SCIM_KEY_asterisk:    right double quotation mark    */
  0x002b,    /* SCIM_KEY_plus:        plus sign            */
  0x002c,    /* SCIM_KEY_comma:        comma                */
  0x0029,    /* SCIM_KEY_minus:        right parenthesis        */
  0x002e,    /* SCIM_KEY_period:        period                */
  0x1169,    /* SCIM_KEY_slash:        jungseong o            */
  0x110f,    /* SCIM_KEY_0:        choseong  khieukh        */
  0x11c2,    /* SCIM_KEY_1:        jongseong hieuh            */
  0x11bb,    /* SCIM_KEY_2:        jongseong ssangsios        */
  0x11b8,    /* SCIM_KEY_3:        jongseong pieup            */
  0x116d,    /* SCIM_KEY_4:        jungseong yo            */
  0x1172,    /* SCIM_KEY_5:        jungseong yu            */
  0x1163,    /* SCIM_KEY_6:        jungseong ya            */
  0x1168,    /* SCIM_KEY_7:        jungseong ye            */
  0x1174,    /* SCIM_KEY_8:        jungseong yi            */
  0x116e,    /* SCIM_KEY_9:        jungseong u            */
  0x0034,    /* SCIM_KEY_colon:        4                */
  0x1107,    /* SCIM_KEY_semicolon:    choseong  pieup            */
  0x002c,    /* SCIM_KEY_less:        comma                */
  0x003e,    /* SCIM_KEY_equal:        greater-than sign        */
  0x002e,    /* SCIM_KEY_greater:        period                */
  0x0021,    /* SCIM_KEY_question:    exclamation mark         */
  0x11b0,    /* SCIM_KEY_at:        jongseong rieul-kiyeok        */
  0x11ae,    /* SCIM_KEY_A:        jongseong tikeut        */
  0x003f,    /* SCIM_KEY_B:        question mark            */
  0x11bf,    /* SCIM_KEY_C:        jongseong khieukh        */
  0x11b2,    /* SCIM_KEY_D:        jongseong rieul-pieup        */
  0x11ac,    /* SCIM_KEY_E:        jongseong nieun-cieuc        */
  0x11b1,    /* SCIM_KEY_F:        jongseong rieul-mieum        */
  0x1164,    /* SCIM_KEY_G:        jungseong yae            */
  0x0030,    /* SCIM_KEY_H:        0                */
  0x0037,    /* SCIM_KEY_I:        7                */
  0x0031,    /* SCIM_KEY_J:        1                */
  0x0032,    /* SCIM_KEY_K:        2                */
  0x0033,    /* SCIM_KEY_L:        3                */
  0x0022,    /* SCIM_KEY_M:        double quotation mark        */
  0x002d,    /* SCIM_KEY_N:        minus sign            */
  0x0038,    /* SCIM_KEY_O:        8                */
  0x0039,    /* SCIM_KEY_P:        9                */
  0x11c1,    /* SCIM_KEY_Q:        jongseong phieuph        */
  0x11b6,    /* SCIM_KEY_R:        jongseong rieul-hieuh        */
  0x11ad,    /* SCIM_KEY_S:        jongseong nieun-hieuh        */
  0x11b3,    /* SCIM_KEY_T:        jongseong rieul-sios        */
  0x0036,    /* SCIM_KEY_U:        6                */
  0x11aa,    /* SCIM_KEY_V:        jongseong kiyeok-sios        */
  0x11c0,    /* SCIM_KEY_W:        jongseong thikeuth        */
  0x11b9,    /* SCIM_KEY_X:        jongseong pieup-sios        */
  0x0035,    /* SCIM_KEY_Y:        5                */
  0x11be,    /* SCIM_KEY_Z:        jongseong chieuch        */
  0x0028,    /* SCIM_KEY_bracketleft:    left parenthesis        */
  0x003a,    /* SCIM_KEY_backslash:    colon                */
  0x003c,    /* SCIM_KEY_bracketright:    less-than sign            */
  0x003d,    /* SCIM_KEY_asciicircum:    equals sign            */
  0x003b,    /* SCIM_KEY_underscore:    semicolon            */
  0x002a,    /* SCIM_KEY_quoteleft:    asterisk            */
  0x11bc,    /* SCIM_KEY_a:        jongseong ieung            */
  0x116e,    /* SCIM_KEY_b:        jungseong u            */
  0x1166,    /* SCIM_KEY_c:        jungseong e            */
  0x1175,    /* SCIM_KEY_d:        jungseong i            */
  0x1167,    /* SCIM_KEY_e:        jungseong yeo            */
  0x1161,    /* SCIM_KEY_f:        jungseong a            */
  0x1173,    /* SCIM_KEY_g:        jungseong eu            */
  0x1102,    /* SCIM_KEY_h:        choseong  nieun            */
  0x1106,    /* SCIM_KEY_i:        choseong  mieum            */
  0x110b,    /* SCIM_KEY_j:        choseong  ieung            */
  0x1100,    /* SCIM_KEY_k:        choseong  kiyeok        */
  0x110c,    /* SCIM_KEY_l:        choseong  cieuc            */
  0x1112,    /* SCIM_KEY_m:        choseong  hieuh            */
  0x1109,    /* SCIM_KEY_n:        choseong  sios            */
  0x110e,    /* SCIM_KEY_o:        choseong  chieuch        */
  0x1111,    /* SCIM_KEY_p:        choseong  phieuph        */
  0x11ba,    /* SCIM_KEY_q:        jongseong sios            */
  0x1162,    /* SCIM_KEY_r:        jungseong ae            */
  0x11ab,    /* SCIM_KEY_s:        jongseong nieun            */
  0x1165,    /* SCIM_KEY_t:        jungseong eo            */
  0x1103,    /* SCIM_KEY_u:        choseong  tikeut        */
  0x1169,    /* SCIM_KEY_v:        jungseong o            */
  0x11af,    /* SCIM_KEY_w:        jongseong rieul            */
  0x11a8,    /* SCIM_KEY_x:        jongseong kiyeok        */
  0x1105,    /* SCIM_KEY_y:        choseong  rieul            */
  0x11b7,    /* SCIM_KEY_z:        jongseong mieum            */
  0x0025,    /* SCIM_KEY_braceleft:    percent sign            */
  0x005c,    /* SCIM_KEY_bar:        backslash            */
  0x002f,    /* SCIM_KEY_braceright:    slash                */
  0x203b,    /* SCIM_KEY_asciitilde:    reference mark            */
};

/* Hangul keyboard map table:
 * 3set keyboard no-shift (sebulsik sunarae) */
static const ucs4_t keyboard_table_3sun [] = {
  0x0021,    /* SCIM_KEY_exclam:        exclamation mark        */
  0x0022,    /* SCIM_KEY_quotedbl:    quotatioin mark            */
  0x0023,    /* SCIM_KEY_numbersign:    number sign             */
  0x0024,    /* SCIM_KEY_dollar:        dollar sign            */
  0x0025,    /* SCIM_KEY_percent:        percent sign            */
  0x0026,    /* SCIM_KEY_ampersand:    ampersand             */
  0x1110,    /* SCIM_KEY_apostrophe:    choseong thieuth        */
  0x0028,    /* SCIM_KEY_parenleft:    left parenthesis        */
  0x0029,    /* SCIM_KEY_parenright:    right parenthesis        */
  0x002a,    /* SCIM_KEY_asterisk:    asterisk            */
  0x002b,    /* SCIM_KEY_plus:        plus sign            */
  0x002c,    /* SCIM_KEY_comma:        comma                */
  0x11bd,    /* SCIM_KEY_minus:        jongseong cieuc            */
  0x002e,    /* SCIM_KEY_period:        period                */
  0x11ae,    /* SCIM_KEY_slash:        jongseong tikeut        */
  0x1164,    /* SCIM_KEY_0:        choseong  yae            */
  0x11c2,    /* SCIM_KEY_1:        jongseong hieuh            */
  0x11bb,    /* SCIM_KEY_2:        jongseong ssangsios        */
  0x11b8,    /* SCIM_KEY_3:        jongseong pieup            */
  0x116d,    /* SCIM_KEY_4:        jungseong yo            */
  0x1172,    /* SCIM_KEY_5:        jungseong yu            */
  0x1163,    /* SCIM_KEY_6:        jungseong ya            */
  0x1168,    /* SCIM_KEY_7:        jungseong ye            */
  0x1174,    /* SCIM_KEY_8:        jungseong yi            */
  0x110f,    /* SCIM_KEY_9:        choseong khieukh        */
  0x003a,    /* SCIM_KEY_colon:        colon                */
  0x1107,    /* SCIM_KEY_semicolon:    choseong  pieup            */
  0x0032,    /* SCIM_KEY_less:        2                */
  0x11be,    /* SCIM_KEY_equal:        jongseong chieuch        */
  0x0033,    /* SCIM_KEY_greater:        3                */
  0x003f,    /* SCIM_KEY_question:    question mark             */
  0x0040,    /* SCIM_KEY_at:        commertial at            */
  0x11bc,    /* SCIM_KEY_A:        jongseong ieung            */
  0x0021,    /* SCIM_KEY_B:        exclamation mark        */
  0x005c,    /* SCIM_KEY_C:        backslash            */
  0x005d,    /* SCIM_KEY_D:        right bracket            */
  0x1167,    /* SCIM_KEY_E:        jungseong yeo            */
  0x1161,    /* SCIM_KEY_F:        jungseong a            */
  0x002f,    /* SCIM_KEY_G:        slash                */
  0x0027,    /* SCIM_KEY_H:        apostrophe            */
  0x0038,    /* SCIM_KEY_I:        8                */
  0x0034,    /* SCIM_KEY_J:        4                */
  0x0035,    /* SCIM_KEY_K:        5                */
  0x0036,    /* SCIM_KEY_L:        6                */
  0x0031,    /* SCIM_KEY_M:        1                */
  0x0030,    /* SCIM_KEY_N:        0                */
  0x0039,    /* SCIM_KEY_O:        9                */
  0x003e,    /* SCIM_KEY_P:        greater-than sign        */
  0x11ba,    /* SCIM_KEY_Q:        jongseong sios            */
  0x1162,    /* SCIM_KEY_R:        jungseong ae            */
  0x005b,    /* SCIM_KEY_S:        left bracket            */
  0x003b,    /* SCIM_KEY_T:        semicolon            */
  0x0037,    /* SCIM_KEY_U:        7                */
  0x1169,    /* SCIM_KEY_V:        jungseong o            */
  0x11af,    /* SCIM_KEY_W:        jongseong rieul            */
  0x003d,    /* SCIM_KEY_X:        equals sign            */
  0x003c,    /* SCIM_KEY_Y:        less-than sign            */
  0x002d,    /* SCIM_KEY_Z:        minus sign            */
  0x11c0,    /* SCIM_KEY_bracketleft:    jongseong thieuth        */
  0x11bf,    /* SCIM_KEY_backslash:    jongseong khieukh        */
  0x11c1,    /* SCIM_KEY_bracketright:    jongseong phieuph        */
  0x005e,    /* SCIM_KEY_asciicircum:    circumflex accent        */
  0x005f,    /* SCIM_KEY_underscore:    underscore            */
  0x0060,    /* SCIM_KEY_quoteleft:    grave accent            */
  0x11bc,    /* SCIM_KEY_a:        jongseong ieung            */
  0x116e,    /* SCIM_KEY_b:        jungseong u            */
  0x1166,    /* SCIM_KEY_c:        jungseong e            */
  0x1175,    /* SCIM_KEY_d:        jungseong i            */
  0x1167,    /* SCIM_KEY_e:        jungseong yeo            */
  0x1161,    /* SCIM_KEY_f:        jungseong a            */
  0x1173,    /* SCIM_KEY_g:        jungseong eu            */
  0x1102,    /* SCIM_KEY_h:        choseong  nieun            */
  0x1106,    /* SCIM_KEY_i:        choseong  mieum            */
  0x110b,    /* SCIM_KEY_j:        choseong  ieung            */
  0x1100,    /* SCIM_KEY_k:        choseong  kiyeok        */
  0x110c,    /* SCIM_KEY_l:        choseong  cieuc            */
  0x1112,    /* SCIM_KEY_m:        choseong  hieuh            */
  0x1109,    /* SCIM_KEY_n:        choseong  sios            */
  0x110e,    /* SCIM_KEY_o:        choseong  chieuch        */
  0x1111,    /* SCIM_KEY_p:        choseong  phieuph        */
  0x11ba,    /* SCIM_KEY_q:        jongseong sios            */
  0x1162,    /* SCIM_KEY_r:        jungseong ae            */
  0x11ab,    /* SCIM_KEY_s:        jongseong nieun            */
  0x1165,    /* SCIM_KEY_t:        jungseong eo            */
  0x1103,    /* SCIM_KEY_u:        choseong  tikeut        */
  0x1169,    /* SCIM_KEY_v:        jungseong o            */
  0x11af,    /* SCIM_KEY_w:        jongseong rieul            */
  0x11a8,    /* SCIM_KEY_x:        jongseong kiyeok        */
  0x1105,    /* SCIM_KEY_y:        choseong  rieul            */
  0x11b7,    /* SCIM_KEY_z:        jongseong mieum            */
  0x007b,    /* SCIM_KEY_braceleft:    left brace            */
  0x007c,    /* SCIM_KEY_bar:        vertical line(bar)        */
  0x007d,    /* SCIM_KEY_braceright:    right brace            */
  0x007e,    /* SCIM_KEY_asciitilde:    tilde                */
};

/* Hangul keyboard map table:
 * 3set keyboard yetguel (sebulsik yetguel)
 * see http://asadal.cs.pusan.ac.kr/hangeul/kbd/oldkbd.jpg */
static const ucs4_t keyboard_table_3yet [] = {
  0x11bd,    /* SCIM_KEY_exclam:        jongseong cieuc            */
  0x0022,    /* SCIM_KEY_quotedbl:    quotatioin mark            */
  0x0023,    /* SCIM_KEY_numbersign:    number sign             */
  0x0024,    /* SCIM_KEY_dollar:        dollar sign            */
  0x0025,    /* SCIM_KEY_percent:        percent sign            */
  0x0026,    /* SCIM_KEY_ampersand:    ampersand             */
  0x1110,    /* SCIM_KEY_apostrophe:    choseong thieuth        */
  0x0028,    /* SCIM_KEY_parenleft:    left parenthesis        */
  0x0029,    /* SCIM_KEY_parenright:    right parenthesis        */
  0x002a,    /* SCIM_KEY_asterisk:    asterisk            */
  0x002b,    /* SCIM_KEY_plus:        plus sign            */
  0x002c,    /* SCIM_KEY_comma:        comma                */
  0x002d,    /* SCIM_KEY_minus:        minus sign            */
  0x002e,    /* SCIM_KEY_period:        period                */
  0x1169,    /* SCIM_KEY_slash:        jungseong o            */
  0x110f,    /* SCIM_KEY_0:        choseong  khieukh        */
  0x11c2,    /* SCIM_KEY_1:        jongseong hieuh            */
  0x11bb,    /* SCIM_KEY_2:        jongseong ssangsios        */
  0x11b8,    /* SCIM_KEY_3:        jongseong pieup            */
  0x116d,    /* SCIM_KEY_4:        jungseong yo            */
  0x1172,    /* SCIM_KEY_5:        jungseong yu            */
  0x1163,    /* SCIM_KEY_6:        jungseong ya            */
  0x1168,    /* SCIM_KEY_7:        jungseong ye            */
  0x1174,    /* SCIM_KEY_8:        jungseong yi            */
  0x116e,    /* SCIM_KEY_9:        jungseong u            */
  0x003a,    /* SCIM_KEY_colon:        colon                */
  0x1107,    /* SCIM_KEY_semicolon:    choseong  pieup            */
  0x113c,    /* SCIM_KEY_less:        choseong chitueumsios        */
  0x003d,    /* SCIM_KEY_equal:        euals sign            */
  0x113e,    /* SCIM_KEY_greater:        choseong ceongchieumsios    */
  0x003f,    /* SCIM_KEY_question:    question mark             */
  0x11eb,    /* SCIM_KEY_at:        jongseong pansios        */
  0x11ae,    /* SCIM_KEY_A:        jongseong tikeut        */
  0x116e,    /* SCIM_KEY_B:        jungseong u            */
  0x11b1,    /* SCIM_KEY_C:        jongseong rieul-mieum        */
  0x11b0,    /* SCIM_KEY_D:        jongseong rieul-kiyeok        */
  0x11bf,    /* SCIM_KEY_E:        jongseong khieukh        */
  0x11a9,    /* SCIM_KEY_F:        jongseong ssangkiyeok        */
  0x119e,    /* SCIM_KEY_G:        jungseong araea            */
  0x1102,    /* SCIM_KEY_H:        choseong nieun            */
  0x1154,    /* SCIM_KEY_I:        choseong chitueumchieuch    */
  0x114c,    /* SCIM_KEY_J:        choseong yesieung        */
  0x114e,    /* SCIM_KEY_K:        choseong chitueumcieuc        */
  0x1150,    /* SCIM_KEY_L:        choseong ceongchieumcieuc    */
  0x1159,    /* SCIM_KEY_M:        choseong yeorinhieuh        */
  0x1140,    /* SCIM_KEY_N:        choseong pansios        */
  0x1155,    /* SCIM_KEY_O:        choseong ceongchieumchieuch    */
  0x1111,    /* SCIM_KEY_P:        choseong phieuph        */
  0x11c1,    /* SCIM_KEY_Q:        jongseong phieuph        */
  0x1164,    /* SCIM_KEY_R:        jungseong yae            */
  0x11ad,    /* SCIM_KEY_S:        jongseong nieun-hieuh        */
  0x1165,    /* SCIM_KEY_T:        jungseong eo            */
  0x302e,    /* SCIM_KEY_U:        single dot tone mark        */
  0x11b6,    /* SCIM_KEY_V:        jongseong rieul-hieuh        */
  0x11c0,    /* SCIM_KEY_W:        jongseong thikeuth        */
  0x11b9,    /* SCIM_KEY_X:        jongseong pieup-sios        */
  0x302f,    /* SCIM_KEY_Y:        double dot tone mark        */
  0x11be,    /* SCIM_KEY_Z:        jongseong chieuch        */
  0x005b,    /* SCIM_KEY_bracketleft:    left bracket            */
  0x005c,    /* SCIM_KEY_backslash:    backslash            */
  0x005d,    /* SCIM_KEY_bracketright:    right bracket            */
  0x005e,    /* SCIM_KEY_asciicircum:    circumflex accent        */
  0x005f,    /* SCIM_KEY_underscore:    underscore            */
  0x11f9,    /* SCIM_KEY_quoteleft:    jongseong yeorinhieuh        */
  0x11bc,    /* SCIM_KEY_a:        jongseong ieung            */
  0x116e,    /* SCIM_KEY_b:        jungseong u            */
  0x1166,    /* SCIM_KEY_c:        jungseong e            */
  0x1175,    /* SCIM_KEY_d:        jungseong i            */
  0x1167,    /* SCIM_KEY_e:        jungseong yeo            */
  0x1161,    /* SCIM_KEY_f:        jungseong a            */
  0x1173,    /* SCIM_KEY_g:        jungseong eu            */
  0x1102,    /* SCIM_KEY_h:        choseong  nieun            */
  0x1106,    /* SCIM_KEY_i:        choseong  mieum            */
  0x110b,    /* SCIM_KEY_j:        choseong  ieung            */
  0x1100,    /* SCIM_KEY_k:        choseong  kiyeok        */
  0x110c,    /* SCIM_KEY_l:        choseong  cieuc            */
  0x1112,    /* SCIM_KEY_m:        choseong  hieuh            */
  0x1109,    /* SCIM_KEY_n:        choseong  sios            */
  0x110e,    /* SCIM_KEY_o:        choseong  chieuch        */
  0x1111,    /* SCIM_KEY_p:        choseong  phieuph        */
  0x11ba,    /* SCIM_KEY_q:        jongseong sios            */
  0x1162,    /* SCIM_KEY_r:        jungseong ae            */
  0x11ab,    /* SCIM_KEY_s:        jongseong nieun            */
  0x1165,    /* SCIM_KEY_t:        jungseong eo            */
  0x1103,    /* SCIM_KEY_u:        choseong  tikeut        */
  0x1169,    /* SCIM_KEY_v:        jungseong o            */
  0x11af,    /* SCIM_KEY_w:        jongseong rieul            */
  0x11a8,    /* SCIM_KEY_x:        jongseong kiyeok        */
  0x1105,    /* SCIM_KEY_y:        choseong  rieul            */
  0x11b7,    /* SCIM_KEY_z:        jongseong mieum            */
  0x007b,    /* SCIM_KEY_braceleft:    left brace            */
  0x007c,    /* SCIM_KEY_bar:        vertical line(bar)        */
  0x007d,    /* SCIM_KEY_braceright:    right brace            */
  0x11f0,    /* SCIM_KEY_asciitilde:    jongseong yesieung        */
};

static const HangulCombination compose_table_yet[] = {
  { 0x11001100, 0x1101 }, /* CHOSEONG SSANGKIYEOK           = KIYEOK + KIYEOK */
  { 0x11021100, 0x1113 }, /* CHOSEONG NIEUN-KIYEOK          = NIEUN + KIYEOK */
  { 0x11021102, 0x1114 }, /* CHOSEONG SSANGNIEUN            = NIEUN + NIEUN */
  { 0x11021103, 0x1115 }, /* CHOSEONG NIEUN-TIKEUT          = NIEUN + TIKEUT */
  { 0x11021107, 0x1116 }, /* CHOSEONG NIEUN-PIEUP           = NIEUN + PIEUP */
  { 0x11031100, 0x1117 }, /* CHOSEONG TIKEUT-KIYEOK         = TIKEUT + KIYEOK */
  { 0x11031103, 0x1104 }, /* CHOSEONG SSANGTIKEUT           = TIKEUT + TIKEUT */
  { 0x11051102, 0x1118 }, /* CHOSEONG RIEUL-NIEUN           = RIEUL + NIEUN */
  { 0x11051105, 0x1119 }, /* CHOSEONG SSANGRIEUL            = RIEUL + RIEUL */
  { 0x1105110b, 0x111b }, /* CHOSEONG KAPYEOUNRIEUL         = RIEUL + IEUNG */
  { 0x11051112, 0x111a }, /* CHOSEONG RIEUL-HIEUH           = RIEUL + HIEUH */
  { 0x11061107, 0x111c }, /* CHOSEONG MIEUM-PIEUP           = MIEUM + PIEUP */
  { 0x1106110b, 0x111d }, /* CHOSEONG KAPYEOUNMIEUM         = MIEUM + IEUNG */
  { 0x11071100, 0x111e }, /* CHOSEONG PIEUP-KIYEOK          = PIEUP + KIYEOK */
  { 0x11071102, 0x111f }, /* CHOSEONG PIEUP-NIEUN           = PIEUP + NIEUN */
  { 0x11071103, 0x1120 }, /* CHOSEONG PIEUP-TIKEUT          = PIEUP + TIKEUT */
  { 0x11071107, 0x1108 }, /* CHOSEONG SSANGPIEUP            = PIEUP + PIEUP */
  { 0x11071109, 0x1121 }, /* CHOSEONG PIEUP-SIOS            = PIEUP + SIOS */
  { 0x1107110b, 0x112b }, /* CHOSEONG KAPYEOUNPIEUP         = PIEUP + IEUNG */
  { 0x1107110c, 0x1127 }, /* CHOSEONG PIEUP-CIEUC           = PIEUP + CIEUC */
  { 0x1107110e, 0x1128 }, /* CHOSEONG PIEUP-CHIEUCH         = PIEUP + CHIEUCH */
  { 0x11071110, 0x1129 }, /* CHOSEONG PIEUP-THIEUTH         = PIEUP + THIEUTH */
  { 0x11071111, 0x112a }, /* CHOSEONG PIEUP-PHIEUPH         = PIEUP + PHIEUPH */
  { 0x1108110b, 0x112c }, /* CHOSEONG KAPYEOUNSSANGPIEUP    = SSANGPIEUP + IEUNG */
  { 0x11091100, 0x112d }, /* CHOSEONG SIOS-KIYEOK           = SIOS + KIYEOK */
  { 0x11091102, 0x112e }, /* CHOSEONG SIOS-NIEUN            = SIOS + NIEUN */
  { 0x11091103, 0x112f }, /* CHOSEONG SIOS-TIKEUT           = SIOS + TIKEUT */
  { 0x11091105, 0x1130 }, /* CHOSEONG SIOS-RIEUL            = SIOS + RIEUL */
  { 0x11091106, 0x1131 }, /* CHOSEONG SIOS-MIEUM            = SIOS + MIEUM */
  { 0x11091107, 0x1132 }, /* CHOSEONG SIOS-PIEUP            = SIOS + PIEUP */
  { 0x11091109, 0x110a }, /* CHOSEONG SSANGSIOS             = SIOS + SIOS */
  { 0x1109110b, 0x1135 }, /* CHOSEONG SIOS-IEUNG            = SIOS + IEUNG */
  { 0x1109110c, 0x1136 }, /* CHOSEONG SIOS-CIEUC            = SIOS + CIEUC */
  { 0x1109110e, 0x1137 }, /* CHOSEONG SIOS-CHIEUCH          = SIOS + CHIEUCH */
  { 0x1109110f, 0x1138 }, /* CHOSEONG SIOS-KHIEUKH          = SIOS + KHIEUKH */
  { 0x11091110, 0x1139 }, /* CHOSEONG SIOS-THIEUTH          = SIOS + THIEUTH */
  { 0x11091111, 0x113a }, /* CHOSEONG SIOS-PHIEUPH          = SIOS + PHIEUPH */
  { 0x11091112, 0x113b }, /* CHOSEONG SIOS-HIEUH            = SIOS + HIEUH */
  { 0x110a1109, 0x1134 }, /* CHOSEONG SIOS-SSANGSIOS        = SSANGSIOS + SIOS */
  { 0x110b1100, 0x1141 }, /* CHOSEONG IEUNG-KIYEOK          = IEUNG + KIYEOK */
  { 0x110b1103, 0x1142 }, /* CHOSEONG IEUNG-TIKEUT          = IEUNG + TIKEUT */
  { 0x110b1106, 0x1143 }, /* CHOSEONG IEUNG-MIEUM           = IEUNG + MIEUM */
  { 0x110b1107, 0x1144 }, /* CHOSEONG IEUNG-PIEUP           = IEUNG + PIEUP */
  { 0x110b1109, 0x1145 }, /* CHOSEONG IEUNG-SIOS            = IEUNG + SIOS */
  { 0x110b110b, 0x1147 }, /* CHOSEONG SSANGIEUNG            = IEUNG + IEUNG */
  { 0x110b110c, 0x1148 }, /* CHOSEONG IEUNG-CIEUC           = IEUNG + CIEUC */
  { 0x110b110e, 0x1149 }, /* CHOSEONG IEUNG-CHIEUCH         = IEUNG + CHIEUCH */
  { 0x110b1110, 0x114a }, /* CHOSEONG IEUNG-THIEUTH         = IEUNG + THIEUTH */
  { 0x110b1111, 0x114b }, /* CHOSEONG IEUNG-PHIEUPH         = IEUNG + PHIEUPH */
  { 0x110b1140, 0x1146 }, /* CHOSEONG IEUNG-PANSIOS         = IEUNG + PANSIOS */
  { 0x110c110b, 0x114d }, /* CHOSEONG CIEUC-IEUNG           = CIEUC + IEUNG */
  { 0x110c110c, 0x110d }, /* CHOSEONG SSANGCIEUC            = CIEUC + CIEUC */
  { 0x110e110f, 0x1152 }, /* CHOSEONG CHIEUCH-KHIEUKH       = CHIEUCH + KHIEUKH */
  { 0x110e1112, 0x1153 }, /* CHOSEONG CHIEUCH-HIEUH         = CHIEUCH + HIEUH */
  { 0x11111107, 0x1156 }, /* CHOSEONG PHIEUPH-PIEUP         = PHIEUPH + PIEUP */
  { 0x1111110b, 0x1157 }, /* CHOSEONG KAPYEOUNPHIEUPH       = PHIEUPH + IEUNG */
  { 0x11121112, 0x1158 }, /* CHOSEONG SSANGHIEUH            = HIEUH + HIEUH */
  { 0x11211100, 0x1122 }, /* CHOSEONG PIEUP-SIOS-KIYEOK     = PIEUP-SIOS + KIYEOK */
  { 0x11211103, 0x1123 }, /* CHOSEONG PIEUP-SIOS-TIKEUT     = PIEUP-SIOS + TIKEUT */
  { 0x11211107, 0x1124 }, /* CHOSEONG PIEUP-SIOS-PIEUP      = PIEUP-SIOS + PIEUP */
  { 0x11211109, 0x1125 }, /* CHOSEONG PIEUP-SSANGSIOS       = PIEUP-SIOS + SIOS */
  { 0x1121110c, 0x1126 }, /* CHOSEONG PIEUP-SIOS-CIEUC      = PIEUP-SIOS + CIEUC */
  { 0x11321100, 0x1133 }, /* CHOSEONG SIOS-PIEUP-KIYEOK     = SIOS-PIEUP + KIYEOK */
  { 0x113c113c, 0x113d }, /* CHOSEONG CHITUEUMSSANGSIOS     = CHITUEUMSIOS + CHITUEUMSIOS */
  { 0x113e113e, 0x113f }, /* CHOSEONG CEONGCHIEUMSSANGSIOS  = CEONGCHIEUMSIOS + CEONGCHIEUMSIOS */
  { 0x114e114e, 0x114f }, /* CHOSEONG CHITUEUMSSANGCIEUC    = CHITUEUMCIEUC + CHITUEUMCIEUC */
  { 0x11501150, 0x1151 }, /* CHOSEONG CEONGCHIEUMSSANGCIEUC = CEONGCHIEUMCIEUC + CEONGCHIEUMCIEUC */
  { 0x11611169, 0x1176 }, /* JUNGSEONG A-O                  = A + O */
  { 0x1161116e, 0x1177 }, /* JUNGSEONG A-U                  = A + U */
  { 0x11611175, 0x1162 }, /* JUNGSEONG AE                   = A */
  { 0x11631169, 0x1178 }, /* JUNGSEONG YA-O                 = YA + O */
  { 0x1163116d, 0x1179 }, /* JUNGSEONG YA-YO                = YA + YO */
  { 0x11631175, 0x1164 }, /* JUNGSEONG YAE                  = YA */
  { 0x11651169, 0x117a }, /* JUNGSEONG EO-O                 = EO + O */
  { 0x1165116e, 0x117b }, /* JUNGSEONG EO-U                 = EO + U */
  { 0x11651173, 0x117c }, /* JUNGSEONG EO-EU                = EO + EU */
  { 0x11651175, 0x1166 }, /* JUNGSEONG E                    = EO */
  { 0x11671169, 0x117d }, /* JUNGSEONG YEO-O                = YEO + O */
  { 0x1167116e, 0x117e }, /* JUNGSEONG YEO-U                = YEO + U */
  { 0x11671175, 0x1168 }, /* JUNGSEONG YE                   = YEO */
  { 0x11691161, 0x116a }, /* JUNGSEONG WA                   = O + A */
  { 0x11691165, 0x117f }, /* JUNGSEONG O-EO                 = O + EO */
  { 0x11691166, 0x1180 }, /* JUNGSEONG O-E                  = O + E */
  { 0x11691168, 0x1181 }, /* JUNGSEONG O-YE                 = O + YE */
  { 0x11691169, 0x1182 }, /* JUNGSEONG O-O                  = O + O */
  { 0x1169116e, 0x1183 }, /* JUNGSEONG O-U                  = O + U */
  { 0x11691175, 0x116c }, /* JUNGSEONG OE                   = O */
  { 0x116a1175, 0x116b }, /* JUNGSEONG WAE                  = WA */
  { 0x116d1163, 0x1184 }, /* JUNGSEONG YO-YA                = YO + YA */
  { 0x116d1164, 0x1185 }, /* JUNGSEONG YO-YAE               = YO + YAE */
  { 0x116d1167, 0x1186 }, /* JUNGSEONG YO-YEO               = YO + YEO */
  { 0x116d1169, 0x1187 }, /* JUNGSEONG YO-O                 = YO + O */
  { 0x116d1175, 0x1188 }, /* JUNGSEONG YO-I                 = YO + I */
  { 0x116e1161, 0x1189 }, /* JUNGSEONG U-A                  = U + A */
  { 0x116e1162, 0x118a }, /* JUNGSEONG U-AE                 = U + AE */
  { 0x116e1165, 0x116f }, /* JUNGSEONG WEO                  = U + EO */
  { 0x116e1168, 0x118c }, /* JUNGSEONG U-YE                 = U + YE */
  { 0x116e116e, 0x118d }, /* JUNGSEONG U-U                  = U + U */
  { 0x116e1175, 0x1171 }, /* JUNGSEONG WI                   = U */
  { 0x116f1173, 0x118b }, /* JUNGSEONG U-EO-EU              = WEO + EU */
  { 0x116f1175, 0x1170 }, /* JUNGSEONG WE                   = WEO */
  { 0x11721161, 0x118e }, /* JUNGSEONG YU-A                 = YU + A */
  { 0x11721165, 0x118f }, /* JUNGSEONG YU-EO                = YU + EO */
  { 0x11721166, 0x1190 }, /* JUNGSEONG YU-E                 = YU + E */
  { 0x11721167, 0x1191 }, /* JUNGSEONG YU-YEO               = YU + YEO */
  { 0x11721168, 0x1192 }, /* JUNGSEONG YU-YE                = YU + YE */
  { 0x1172116e, 0x1193 }, /* JUNGSEONG YU-U                 = YU + U */
  { 0x11721175, 0x1194 }, /* JUNGSEONG YU-I                 = YU + I */
  { 0x1173116e, 0x1195 }, /* JUNGSEONG EU-U                 = EU + U */
  { 0x11731173, 0x1196 }, /* JUNGSEONG EU-EU                = EU + EU */
  { 0x11731175, 0x1174 }, /* JUNGSEONG YI                   = EU */
  { 0x1174116e, 0x1197 }, /* JUNGSEONG YI-U                 = YI + U */
  { 0x11751161, 0x1198 }, /* JUNGSEONG I-A                  = I + A */
  { 0x11751163, 0x1199 }, /* JUNGSEONG I-YA                 = I + YA */
  { 0x11751169, 0x119a }, /* JUNGSEONG I-O                  = I + O */
  { 0x1175116e, 0x119b }, /* JUNGSEONG I-U                  = I + U */
  { 0x11751173, 0x119c }, /* JUNGSEONG I-EU                 = I + EU */
  { 0x1175119e, 0x119d }, /* JUNGSEONG I-ARAEA              = I */
  { 0x119e1165, 0x119f }, /* JUNGSEONG ARAEA-EO             = ARAEA + EO */
  { 0x119e116e, 0x11a0 }, /* JUNGSEONG ARAEA-U              = ARAEA + U */
  { 0x119e1175, 0x11a1 }, /* JUNGSEONG ARAEA-I              = ARAEA + I */
  { 0x119e119e, 0x11a2 }, /* JUNGSEONG SSANGARAEA           = ARAEA + ARAEA */
  { 0x11a811a8, 0x11a9 }, /* JONGSEONG SSANGKIYEOK          = KIYEOK + KIYEOK */
  { 0x11a811af, 0x11c3 }, /* JONGSEONG KIYEOK-RIEUL         = KIYEOK + RIEUL */
  { 0x11a811ba, 0x11aa }, /* JONGSEONG KIYEOK-SIOS          = KIYEOK */
  { 0x11aa11a8, 0x11c4 }, /* JONGSEONG KIYEOK-SIOS-KIYEOK   = KIYEOK-SIOS + KIYEOK */
  { 0x11ab11a8, 0x11c5 }, /* JONGSEONG NIEUN-KIYEOK         = NIEUN + KIYEOK */
  { 0x11ab11ae, 0x11c6 }, /* JONGSEONG NIEUN-TIKEUT         = NIEUN + TIKEUT */
  { 0x11ab11ba, 0x11c7 }, /* JONGSEONG NIEUN-SIOS           = NIEUN + SIOS */
  { 0x11ab11bd, 0x11ac }, /* JONGSEONG NIEUN-CIEUC          = NIEUN */
  { 0x11ab11c0, 0x11c9 }, /* JONGSEONG NIEUN-THIEUTH        = NIEUN + THIEUTH */
  { 0x11ab11c2, 0x11ad }, /* JONGSEONG NIEUN-HIEUH          = NIEUN */
  { 0x11ab11eb, 0x11c8 }, /* JONGSEONG NIEUN-PANSIOS        = NIEUN */
  { 0x11ae11a8, 0x11ca }, /* JONGSEONG TIKEUT-KIYEOK        = TIKEUT + KIYEOK */
  { 0x11ae11af, 0x11cb }, /* JONGSEONG TIKEUT-RIEUL         = TIKEUT + RIEUL */
  { 0x11af11a8, 0x11b0 }, /* JONGSEONG RIEUL-KIYEOK         = RIEUL + KIYEOK */
  { 0x11af11ab, 0x11cd }, /* JONGSEONG RIEUL-NIEUN          = RIEUL + NIEUN */
  { 0x11af11ae, 0x11ce }, /* JONGSEONG RIEUL-TIKEUT         = RIEUL + TIKEUT */
  { 0x11af11af, 0x11d0 }, /* JONGSEONG SSANGRIEUL           = RIEUL + RIEUL */
  { 0x11af11b7, 0x11b1 }, /* JONGSEONG RIEUL-MIEUM          = RIEUL */
  { 0x11af11b8, 0x11b2 }, /* JONGSEONG RIEUL-PIEUP          = RIEUL */
  { 0x11af11ba, 0x11b3 }, /* JONGSEONG RIEUL-SIOS           = RIEUL */
  { 0x11af11bf, 0x11d8 }, /* JONGSEONG RIEUL-KHIEUKH        = RIEUL + KHIEUKH */
  { 0x11af11c0, 0x11b4 }, /* JONGSEONG RIEUL-THIEUTH        = RIEUL */
  { 0x11af11c1, 0x11b5 }, /* JONGSEONG RIEUL-PHIEUPH        = RIEUL */
  { 0x11af11c2, 0x11b6 }, /* JONGSEONG RIEUL-HIEUH          = RIEUL */
  { 0x11af11eb, 0x11d7 }, /* JONGSEONG RIEUL-PANSIOS        = RIEUL */
  { 0x11af11f9, 0x11d9 }, /* JONGSEONG RIEUL-YEORINHIEUH    = RIEUL */
  { 0x11b011ba, 0x11cc }, /* JONGSEONG RIEUL-KIYEOK-SIOS    = RIEUL-KIYEOK + SIOS */
  { 0x11b111a8, 0x11d1 }, /* JONGSEONG RIEUL-MIEUM-KIYEOK   = RIEUL-MIEUM + KIYEOK */
  { 0x11b111ba, 0x11d2 }, /* JONGSEONG RIEUL-MIEUM-SIOS     = RIEUL-MIEUM + SIOS */
  { 0x11b211ba, 0x11d3 }, /* JONGSEONG RIEUL-PIEUP-SIOS     = RIEUL-PIEUP + SIOS */
  { 0x11b211bc, 0x11d5 }, /* JONGSEONG RIEUL-KAPYEOUNPIEUP  = RIEUL-PIEUP + IEUNG */
  { 0x11b211c2, 0x11d4 }, /* JONGSEONG RIEUL-PIEUP-HIEUH    = RIEUL-PIEUP + HIEUH */
  { 0x11b311ba, 0x11d6 }, /* JONGSEONG RIEUL-SSANGSIOS      = RIEUL-SIOS + SIOS */
  { 0x11b711a8, 0x11da }, /* JONGSEONG MIEUM-KIYEOK         = MIEUM + KIYEOK */
  { 0x11b711af, 0x11db }, /* JONGSEONG MIEUM-RIEUL          = MIEUM + RIEUL */
  { 0x11b711b8, 0x11dc }, /* JONGSEONG MIEUM-PIEUP          = MIEUM + PIEUP */
  { 0x11b711ba, 0x11dd }, /* JONGSEONG MIEUM-SIOS           = MIEUM + SIOS */
  { 0x11b711bc, 0x11e2 }, /* JONGSEONG KAPYEOUNMIEUM        = MIEUM + IEUNG */
  { 0x11b711be, 0x11e0 }, /* JONGSEONG MIEUM-CHIEUCH        = MIEUM + CHIEUCH */
  { 0x11b711c2, 0x11e1 }, /* JONGSEONG MIEUM-HIEUH          = MIEUM + HIEUH */
  { 0x11b711eb, 0x11df }, /* JONGSEONG MIEUM-PANSIOS        = MIEUM */
  { 0x11b811af, 0x11e3 }, /* JONGSEONG PIEUP-RIEUL          = PIEUP + RIEUL */
  { 0x11b811ba, 0x11b9 }, /* JONGSEONG PIEUP-SIOS           = PIEUP */
  { 0x11b811bc, 0x11e6 }, /* JONGSEONG KAPYEOUNPIEUP        = PIEUP + IEUNG */
  { 0x11b811c1, 0x11e4 }, /* JONGSEONG PIEUP-PHIEUPH        = PIEUP + PHIEUPH */
  { 0x11b811c2, 0x11e5 }, /* JONGSEONG PIEUP-HIEUH          = PIEUP + HIEUH */
  { 0x11ba11a8, 0x11e7 }, /* JONGSEONG SIOS-KIYEOK          = SIOS + KIYEOK */
  { 0x11ba11ae, 0x11e8 }, /* JONGSEONG SIOS-TIKEUT          = SIOS + TIKEUT */
  { 0x11ba11af, 0x11e9 }, /* JONGSEONG SIOS-RIEUL           = SIOS + RIEUL */
  { 0x11ba11b8, 0x11ea }, /* JONGSEONG SIOS-PIEUP           = SIOS + PIEUP */
  { 0x11ba11ba, 0x11bb }, /* JONGSEONG SSANGSIOS            = SIOS + SIOS */
  { 0x11bc11a8, 0x11ec }, /* JONGSEONG IEUNG-KIYEOK         = IEUNG + KIYEOK */
  { 0x11bc11bc, 0x11ee }, /* JONGSEONG SSANGIEUNG           = IEUNG + IEUNG */
  { 0x11bc11bf, 0x11ef }, /* JONGSEONG IEUNG-KHIEUKH        = IEUNG + KHIEUKH */
  { 0x11c111b8, 0x11f3 }, /* JONGSEONG PHIEUPH-PIEUP        = PHIEUPH + PIEUP */
  { 0x11c111bc, 0x11f4 }, /* JONGSEONG KAPYEOUNPHIEUPH      = PHIEUPH + IEUNG */
  { 0x11c211ab, 0x11f5 }, /* JONGSEONG HIEUH-NIEUN          = HIEUH + NIEUN */
  { 0x11c211af, 0x11f6 }, /* JONGSEONG HIEUH-RIEUL          = HIEUH + RIEUL */
  { 0x11c211b7, 0x11f7 }, /* JONGSEONG HIEUH-MIEUM          = HIEUH + MIEUM */
  { 0x11c211b8, 0x11f8 }, /* JONGSEONG HIEUH-PIEUP          = HIEUH + PIEUP */
  { 0x11ce11c2, 0x11cf }, /* JONGSEONG RIEUL-TIKEUT-HIEUH   = RIEUL-TIKEUT + HIEUH */
  { 0x11dd11ba, 0x11de }, /* JONGSEONG MIEUM-SSANGSIOS      = MIEUM-SIOS + SIOS */
  { 0x11ec11a8, 0x11ed }, /* JONGSEONG IEUNG-SSANGKIYEOK    = IEUNG-KIYEOK + KIYEOK */
  { 0x11f011ba, 0x11f1 }, /* JONGSEONG YESIEUNG-SIOS        = YESIEUNG + SIOS */
  { 0x11f011eb, 0x11f2 }, /* JONGSEONG YESIEUNG-PANSIOS     = YESIEUNG + PANSIOS */
};

static const HangulCombination compose_table_default [] = {
  { 0x11001100, 0x1101 }, /* choseong  kiyeok + kiyeok	= ssangkiyeok	*/
  { 0x11031103, 0x1104 }, /* choseong  tikeut + tikeut	= ssangtikeut	*/
  { 0x11071107, 0x1108 }, /* choseong  pieup  + pieup	= ssangpieup	*/
  { 0x11091109, 0x110a }, /* choseong  sios   + sios	= ssangsios	*/
  { 0x110c110c, 0x110d }, /* choseong  cieuc  + cieuc	= ssangcieuc	*/
  { 0x11691161, 0x116a }, /* jungseong o      + a	= wa		*/
  { 0x11691162, 0x116b }, /* jungseong o      + ae	= wae		*/
  { 0x11691175, 0x116c }, /* jungseong o      + i	= oe		*/
  { 0x116e1165, 0x116f }, /* jungseong u      + eo	= weo		*/
  { 0x116e1166, 0x1170 }, /* jungseong u      + e	= we		*/
  { 0x116e1175, 0x1171 }, /* jungseong u      + i	= wi		*/
  { 0x11731175, 0x1174 }, /* jungseong eu     + i	= yi		*/
  { 0x11a811a8, 0x11a9 }, /* jongseong kiyeok + kiyeok	= ssangekiyeok	*/
  { 0x11a811ba, 0x11aa }, /* jongseong kiyeok + sios	= kiyeok-sois	*/
  { 0x11ab11bd, 0x11ac }, /* jongseong nieun  + cieuc	= nieun-cieuc	*/
  { 0x11ab11c2, 0x11ad }, /* jongseong nieun  + hieuh	= nieun-hieuh	*/
  { 0x11af11a8, 0x11b0 }, /* jongseong rieul  + kiyeok	= rieul-kiyeok	*/
  { 0x11af11b7, 0x11b1 }, /* jongseong rieul  + mieum	= rieul-mieum	*/
  { 0x11af11b8, 0x11b2 }, /* jongseong rieul  + pieup	= rieul-pieup	*/
  { 0x11af11ba, 0x11b3 }, /* jongseong rieul  + sios	= rieul-sios	*/
  { 0x11af11c0, 0x11b4 }, /* jongseong rieul  + thieuth = rieul-thieuth	*/
  { 0x11af11c1, 0x11b5 }, /* jongseong rieul  + phieuph = rieul-phieuph	*/
  { 0x11af11c2, 0x11b6 }, /* jongseong rieul  + hieuh	= rieul-hieuh	*/
  { 0x11b811ba, 0x11b9 }, /* jongseong pieup  + sios	= pieup-sios	*/
  { 0x11ba11ba, 0x11bb }, /* jongseong sios   + sios	= ssangsios	*/
};

static HangulFactoryData hangul_factory_data [] = {
    {
        "d75857a5-4148-4745-89e2-1da7ddaf70a9",
        N_("2bul"),
        HANGUL_COMPOSER_2,
        keyboard_table_2,
        sizeof (compose_table_default) / sizeof (compose_table_default [0]),
        compose_table_default,
        false
    },
    {
        "353a81d2-a065-4b3b-bf67-ee457f7b72a8",
        N_("3bul 2bul-shifted"),
        HANGUL_COMPOSER_3,
        keyboard_table_32,
        sizeof (compose_table_default) / sizeof (compose_table_default [0]),
        compose_table_default,
        false
    },
    {
        "c0d227c4-c70f-413a-8a35-a72155cca459",
        N_("3bul 390"),
        HANGUL_COMPOSER_3,
        keyboard_table_390,
        sizeof (compose_table_default) / sizeof (compose_table_default [0]),
        compose_table_default,
        false
    },
    {
        "a32d6d4f-2aef-40e7-95e0-8ff73c6d855a",
        N_("3bul Final"),
        HANGUL_COMPOSER_3,
        keyboard_table_3final,
        sizeof (compose_table_default) / sizeof (compose_table_default [0]),
        compose_table_default,
        false
    },
    {
        "56f62c53-ef31-457c-86bf-9ff692db5209",
        N_("3bul No-Shift"),
        HANGUL_COMPOSER_3,
        keyboard_table_3sun,
        sizeof (compose_table_default) / sizeof (compose_table_default [0]),
        compose_table_default,
        false
    },
    {
        "888b8e80-3419-4fcc-9110-92de45774053",
        N_("3bul Yetgeul"),
        HANGUL_COMPOSER_3,
        keyboard_table_3yet,
        sizeof (compose_table_yet) / sizeof (compose_table_yet [0]),
        compose_table_yet,
        true
    }
};

#define HANGUL_NUM_OF_FACTORY (sizeof(hangul_factory_data)/sizeof(hangul_factory_data[0]))

static ConfigPointer _scim_config (0);

// utility functions.
#define HCF	0x115f
#define HJF	0x1160

/* choseong to compatibility jamo */
static ucs4_t
im_hangul_choseong_to_cjamo (ucs4_t ch)
{
    static ucs4_t table[] = {
        0x3131,  /* 0x1100 */
        0x3132,  /* 0x1101 */
        0x3134,  /* 0x1102 */
        0x3137,  /* 0x1103 */
        0x3138,  /* 0x1104 */
        0x3139,  /* 0x1105 */
        0x3141,  /* 0x1106 */
        0x3142,  /* 0x1107 */
        0x3143,  /* 0x1108 */
        0x3145,  /* 0x1109 */
        0x3146,  /* 0x110a */
        0x3147,  /* 0x110b */
        0x3148,  /* 0x110c */
        0x3149,  /* 0x110d */
        0x314a,  /* 0x110e */
        0x314b,  /* 0x110f */
        0x314c,  /* 0x1110 */
        0x314d,  /* 0x1111 */
        0x314e   /* 0x1112 */
    };

    if (ch < 0x1100 || ch > 0x1112)
        return 0;
    return table [ch - 0x1100];
}

/* jungseong to compatibility jamo */
static ucs4_t
im_hangul_jungseong_to_cjamo (ucs4_t ch)
{
    static ucs4_t table[] = {
        0x314f,  /* 0x1161 */
        0x3150,  /* 0x1162 */
        0x3151,  /* 0x1163 */
        0x3152,  /* 0x1164 */
        0x3153,  /* 0x1165 */
        0x3154,  /* 0x1166 */
        0x3155,  /* 0x1167 */
        0x3156,  /* 0x1168 */
        0x3157,  /* 0x1169 */
        0x3158,  /* 0x116a */
        0x3159,  /* 0x116b */
        0x315a,  /* 0x116c */
        0x315b,  /* 0x116d */
        0x315c,  /* 0x116e */
        0x315d,  /* 0x116f */
        0x315e,  /* 0x1170 */
        0x315f,  /* 0x1171 */
        0x3160,  /* 0x1172 */
        0x3161,  /* 0x1173 */
        0x3162,  /* 0x1174 */
        0x3163   /* 0x1175 */
    };

    if (ch < 0x1161 || ch > 0x1175)
        return 0;
    return table [ch - 0x1161];
}

/* jongseong to compatibility jamo */
static ucs4_t
im_hangul_jongseong_to_cjamo (ucs4_t ch)
{
    static ucs4_t table[] = {
        0x3131,  /* 0x11a8 */
        0x3132,  /* 0x11a9 */
        0x3133,  /* 0x11aa */
        0x3134,  /* 0x11ab */
        0x3135,  /* 0x11ac */
        0x3136,  /* 0x11ad */
        0x3137,  /* 0x11ae */
        0x3139,  /* 0x11af */
        0x313a,  /* 0x11b0 */
        0x313b,  /* 0x11b1 */
        0x313c,  /* 0x11b2 */
        0x313d,  /* 0x11b3 */
        0x313e,  /* 0x11b4 */
        0x313f,  /* 0x11b5 */
        0x3140,  /* 0x11b6 */
        0x3141,  /* 0x11b7 */
        0x3142,  /* 0x11b8 */
        0x3144,  /* 0x11b9 */
        0x3145,  /* 0x11ba */
        0x3146,  /* 0x11bb */
        0x3147,  /* 0x11bc */
        0x3148,  /* 0x11bd */
        0x314a,  /* 0x11be */
        0x314b,  /* 0x11bf */
        0x314c,  /* 0x11c0 */
        0x314d,  /* 0x11c1 */
        0x314e   /* 0x11c2 */
    };

    if (ch < 0x11a8 || ch > 0x11c2)
        return 0;
    return table [ch - 0x11a8];
}

static ucs4_t
im_hangul_jamo_to_syllable (ucs4_t choseong,
                            ucs4_t jungseong,
                            ucs4_t jongseong)
{
    static const ucs4_t hangul_base = 0xac00;
    static const ucs4_t choseong_base = 0x1100;
    static const ucs4_t jungseong_base = 0x1161;
    static const ucs4_t jongseong_base = 0x11a7;

    /*  static const int nchoseong = 19; */
    static int njungseong = 21;
    static int njongseong = 28;

    /*  static const int njungjong = 21 * 28; */
    ucs4_t ch;

    /* we use 0x11a7 like a Jongseong filler */
    if (jongseong == 0)
        jongseong = 0x11a7;     /* Jongseong filler */

    if (!(choseong  >= 0x1100 && choseong  <= 0x1112))
        return 0;
    if (!(jungseong >= 0x1161 && jungseong <= 0x1175))
        return 0;
    if (!(jongseong >= 0x11a7 && jongseong <= 0x11c2))
        return 0;

    choseong  -= choseong_base;
    jungseong -= jungseong_base;
    jongseong -= jongseong_base;
 
    ch = ((choseong * njungseong) + jungseong) * njongseong + jongseong
         + hangul_base;
    return ch;
}

static int
im_hangul_dvorak_to_qwerty (int code)
{
    /* maybe safe if we use switch statement */
    static int table [] = {
        SCIM_KEY_exclam,         /* SCIM_KEY_exclam */
        SCIM_KEY_Q,              /* SCIM_KEY_quotedbl */
        SCIM_KEY_numbersign,     /* SCIM_KEY_numbersign */
        SCIM_KEY_dollar,         /* SCIM_KEY_dollar */
        SCIM_KEY_percent,        /* SCIM_KEY_percent */
        SCIM_KEY_ampersand,      /* SCIM_KEY_ampersand */
        SCIM_KEY_q,              /* SCIM_KEY_apostrophe */
        SCIM_KEY_parenleft,      /* SCIM_KEY_parenleft */
        SCIM_KEY_parenright,     /* SCIM_KEY_parenright */
        SCIM_KEY_asterisk,       /* SCIM_KEY_asterisk */
        SCIM_KEY_braceright,     /* SCIM_KEY_plus */
        SCIM_KEY_w,              /* SCIM_KEY_comma */
        SCIM_KEY_apostrophe,     /* SCIM_KEY_minus */
        SCIM_KEY_e,              /* SCIM_KEY_period */
        SCIM_KEY_bracketleft,    /* SCIM_KEY_slash */
        SCIM_KEY_0,              /* SCIM_KEY_0 */
        SCIM_KEY_1,              /* SCIM_KEY_1 */
        SCIM_KEY_2,              /* SCIM_KEY_2 */
        SCIM_KEY_3,              /* SCIM_KEY_3 */
        SCIM_KEY_4,              /* SCIM_KEY_4 */
        SCIM_KEY_5,              /* SCIM_KEY_5 */
        SCIM_KEY_6,              /* SCIM_KEY_6 */
        SCIM_KEY_7,              /* SCIM_KEY_7 */
        SCIM_KEY_8,              /* SCIM_KEY_8 */
        SCIM_KEY_9,              /* SCIM_KEY_9 */
        SCIM_KEY_Z,              /* SCIM_KEY_colon */
        SCIM_KEY_z,              /* SCIM_KEY_semicolon */
        SCIM_KEY_W,              /* SCIM_KEY_less */
        SCIM_KEY_bracketright,   /* SCIM_KEY_qual */
        SCIM_KEY_E,              /* SCIM_KEY_greater */
        SCIM_KEY_braceleft,      /* SCIM_KEY_question */
        SCIM_KEY_at,             /* SCIM_KEY_at */
        SCIM_KEY_A,              /* SCIM_KEY_A */
        SCIM_KEY_N,              /* SCIM_KEY_B */
        SCIM_KEY_I,              /* SCIM_KEY_C */
        SCIM_KEY_H,              /* SCIM_KEY_D */
        SCIM_KEY_D,              /* SCIM_KEY_E */
        SCIM_KEY_Y,              /* SCIM_KEY_F */
        SCIM_KEY_U,              /* SCIM_KEY_G */
        SCIM_KEY_J,              /* SCIM_KEY_H */
        SCIM_KEY_G,              /* SCIM_KEY_I */
        SCIM_KEY_C,              /* SCIM_KEY_J */
        SCIM_KEY_V,              /* SCIM_KEY_K */
        SCIM_KEY_P,              /* SCIM_KEY_L */
        SCIM_KEY_M,              /* SCIM_KEY_M */
        SCIM_KEY_L,              /* SCIM_KEY_N */
        SCIM_KEY_S,              /* SCIM_KEY_O */
        SCIM_KEY_R,              /* SCIM_KEY_P */
        SCIM_KEY_X,              /* SCIM_KEY_Q */
        SCIM_KEY_O,              /* SCIM_KEY_R */
        SCIM_KEY_colon,          /* SCIM_KEY_S */
        SCIM_KEY_K,              /* SCIM_KEY_T */
        SCIM_KEY_F,              /* SCIM_KEY_U */
        SCIM_KEY_greater,        /* SCIM_KEY_V */
        SCIM_KEY_less,           /* SCIM_KEY_W */
        SCIM_KEY_B,              /* SCIM_KEY_X */
        SCIM_KEY_T,              /* SCIM_KEY_Y */
        SCIM_KEY_question,       /* SCIM_KEY_Z */
        SCIM_KEY_minus,          /* SCIM_KEY_bracketleft */
        SCIM_KEY_backslash,      /* SCIM_KEY_backslash */
        SCIM_KEY_equal,          /* SCIM_KEY_bracketright */
        SCIM_KEY_asciicircum,    /* SCIM_KEY_asciicircum */
        SCIM_KEY_quotedbl,       /* SCIM_KEY_underscore */
        SCIM_KEY_grave,          /* SCIM_KEY_grave */
        SCIM_KEY_a,              /* SCIM_KEY_a */
        SCIM_KEY_n,              /* SCIM_KEY_b */
        SCIM_KEY_i,              /* SCIM_KEY_c */
        SCIM_KEY_h,              /* SCIM_KEY_d */
        SCIM_KEY_d,              /* SCIM_KEY_e */
        SCIM_KEY_y,              /* SCIM_KEY_f */
        SCIM_KEY_u,              /* SCIM_KEY_g */
        SCIM_KEY_j,              /* SCIM_KEY_h */
        SCIM_KEY_g,              /* SCIM_KEY_i */
        SCIM_KEY_c,              /* SCIM_KEY_j */
        SCIM_KEY_v,              /* SCIM_KEY_k */
        SCIM_KEY_p,              /* SCIM_KEY_l */
        SCIM_KEY_m,              /* SCIM_KEY_m */
        SCIM_KEY_l,              /* SCIM_KEY_n */
        SCIM_KEY_s,              /* SCIM_KEY_o */
        SCIM_KEY_r,              /* SCIM_KEY_p */
        SCIM_KEY_x,              /* SCIM_KEY_q */
        SCIM_KEY_o,              /* SCIM_KEY_r */
        SCIM_KEY_semicolon,      /* SCIM_KEY_s */
        SCIM_KEY_k,              /* SCIM_KEY_t */
        SCIM_KEY_f,              /* SCIM_KEY_u */
        SCIM_KEY_period,         /* SCIM_KEY_v */
        SCIM_KEY_comma,          /* SCIM_KEY_w */
        SCIM_KEY_b,              /* SCIM_KEY_x */
        SCIM_KEY_t,              /* SCIM_KEY_y */
        SCIM_KEY_slash,          /* SCIM_KEY_z */
        SCIM_KEY_underscore,     /* SCIM_KEY_braceleft */
        SCIM_KEY_bar,            /* SCIM_KEY_bar */
        SCIM_KEY_plus,           /* SCIM_KEY_braceright */
        SCIM_KEY_asciitilde,     /* SCIM_KEY_asciitilde */
    };

    if (code < SCIM_KEY_exclam || code > SCIM_KEY_asciitilde)
        return code;

    return table[code - SCIM_KEY_exclam];
}

extern "C" {
    void scim_module_init (void)
    {
        bindtextdomain (GETTEXT_PACKAGE, SCIM_HANGUL_LOCALEDIR);
        bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    }

    void scim_module_exit (void)
    {
        _scim_config.reset ();
    }

    uint32 scim_imengine_module_init (const ConfigPointer &config)
    {
        SCIM_DEBUG_IMENGINE(1) << "Initialize Hangul Engine.\n";

        _scim_config = config;

        return HANGUL_NUM_OF_FACTORY;
    }

    IMEngineFactoryPointer scim_imengine_module_create_factory (uint32 engine)
    {
        if (engine >= HANGUL_NUM_OF_FACTORY) return 0;

        HangulFactory *factory = 0;

        try {
            factory = new HangulFactory (_scim_config, hangul_factory_data [engine]);
        } catch (...) {
            delete factory;
            factory = 0;
        }

        return factory;
    }
}

HangulFactory::HangulFactory (const ConfigPointer     &config,
                              const HangulFactoryData &data)
                        
    : m_uuid                   (data.uuid),
      m_name                   (_(data.name)),
      m_composer_type          (data.composer_type),
      m_keyboard_table         (data.keyboard_table),
      m_compose_table_size     (data.compose_table_size),
      m_compose_table          (data.compose_table),
      m_always_use_jamo        (data.always_use_jamo),
      m_use_capslock           (true),
      m_use_dvorak             (false),
      m_show_candidate_comment (true)
{
    if (!config.null ()) {
        m_use_capslock =
            config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_USE_CAPSLOCK),
                          m_use_capslock);

        m_use_dvorak =
            config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_USE_DVORAK),
                          m_use_dvorak);

        m_show_candidate_comment =
            config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_SHOW_CANDIDATE_COMMENT),
                          m_show_candidate_comment);

        String str;

        str = config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_TRIGGER_KEY),
                            String ("Hangul,Alt+Alt_R+KeyRelease,Shift+space"));

        scim_string_to_key_list (m_trigger_keys, str);

        str = config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_HANGUL_HANJA_KEY),
                            String ("Hangul_Hanja,F9,Control+Control_R+KeyRelease"));

        scim_string_to_key_list (m_hangul_hanja_keys, str);

        str = config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_MANUAL_MODE_KEY),
                            String ("Control+Hangul"));

        scim_string_to_key_list (m_manual_mode_keys, str);
    }

    if (!m_trigger_keys.size ()) {
        m_trigger_keys.push_back (KeyEvent (SCIM_KEY_Hangul, 0));
        m_trigger_keys.push_back (KeyEvent (SCIM_KEY_Alt_R, SCIM_KEY_AltMask | SCIM_KEY_ReleaseMask));
        m_trigger_keys.push_back (KeyEvent (SCIM_KEY_space, SCIM_KEY_ShiftMask));
    }

    if (!m_hangul_hanja_keys.size ()) {
        m_hangul_hanja_keys.push_back (KeyEvent (SCIM_KEY_Hangul_Hanja, 0));
        m_hangul_hanja_keys.push_back (KeyEvent (SCIM_KEY_F9, 0));
        m_hangul_hanja_keys.push_back (KeyEvent (SCIM_KEY_Control_R, SCIM_KEY_ControlMask | SCIM_KEY_ReleaseMask));
    }

    if (!m_manual_mode_keys.size ()) {
        m_manual_mode_keys.push_back (KeyEvent (SCIM_KEY_Hangul, SCIM_KEY_ControlMask));
    }

    set_languages ("ko");
}

HangulFactory::~HangulFactory ()
{
}

WideString
HangulFactory::get_name () const
{
    return utf8_mbstowcs (m_name);
}

WideString
HangulFactory::get_authors () const
{
    return WideString ();
}

WideString
HangulFactory::get_credits () const
{
    return WideString ();
}

WideString
HangulFactory::get_help () const
{
    return WideString ();
}

String
HangulFactory::get_uuid () const
{
    return m_uuid;
}

String
HangulFactory::get_icon_file () const
{
    return String (SCIM_HANGUL_ICON_FILE);
}

IMEngineInstancePointer
HangulFactory::create_instance (const String &encoding, int id)
{
    return new HangulInstance (this, encoding, id);
}

HangulInstance::HangulInstance (HangulFactory *factory,
                                const String  &encoding,
                                int            id)
    : IMEngineInstanceBase (factory, encoding, id),
      m_factory (factory),
      m_prev_key (0,0),
      m_input_mode (INPUT_MODE_HANGUL),
      m_output_mode (OUTPUT_MODE_SYLLABLE)
{
    im_hangul_clear_buf ();

    switch (factory->m_composer_type) {
        case HANGUL_COMPOSER_2:
            m_hangul_composer = &HangulInstance::im_hangul_composer_2;
            break;
        case HANGUL_COMPOSER_3:
            m_hangul_composer = &HangulInstance::im_hangul_composer_3;
            break;
        default:
            m_hangul_composer = &HangulInstance::im_hangul_composer_2;
            break;
    }

    if (factory->m_always_use_jamo)
        m_output_mode |= OUTPUT_MODE_JAMO;

    char label [10];
    std::vector <WideString> labels;

    for (int i = 1; i < 10; ++i) {
        snprintf (label, 10, "%d", i);
        labels.push_back (utf8_mbstowcs (label));
    }

    m_lookup_table.set_candidate_labels (labels);
}

HangulInstance::~HangulInstance ()
{
}

bool
HangulInstance::candidate_key_event (const KeyEvent &key)
{
    switch (key.code) {
        case SCIM_KEY_Return:
        case SCIM_KEY_KP_Enter:
            select_candidate (m_lookup_table.get_cursor_pos_in_current_page ());
            break;
        case SCIM_KEY_Left:
        case SCIM_KEY_h:
        case SCIM_KEY_KP_Subtract:
        case SCIM_KEY_BackSpace:
            m_lookup_table.cursor_up ();
            update_lookup_table (m_lookup_table);
            im_hangul_update_aux_string ();
            break;
        case SCIM_KEY_Right:
        case SCIM_KEY_l:
        case SCIM_KEY_space:
        case SCIM_KEY_KP_Add:
            m_lookup_table.cursor_down ();
            update_lookup_table (m_lookup_table);
            im_hangul_update_aux_string ();
            break;
        case SCIM_KEY_Up:
        case SCIM_KEY_Page_Up:
        case SCIM_KEY_k:
            m_lookup_table.page_up ();
            update_lookup_table (m_lookup_table);
            im_hangul_update_aux_string ();
            break;
        case SCIM_KEY_Down:
        case SCIM_KEY_Page_Down:
        case SCIM_KEY_j:
        case SCIM_KEY_KP_Tab:
            m_lookup_table.page_down ();
            update_lookup_table (m_lookup_table);
            im_hangul_update_aux_string ();
            break;
        case SCIM_KEY_Escape:
            im_hangul_delete_candidates ();
            break;
        case SCIM_KEY_1: 
        case SCIM_KEY_2: 
        case SCIM_KEY_3: 
        case SCIM_KEY_4: 
        case SCIM_KEY_5: 
        case SCIM_KEY_6: 
        case SCIM_KEY_7: 
        case SCIM_KEY_8: 
        case SCIM_KEY_9: 
            select_candidate (key.code - SCIM_KEY_1);
            break;
        default:
            break;
    }

    return true;
}

bool
HangulInstance::process_key_event (const KeyEvent& key)
{
    SCIM_DEBUG_IMENGINE(2) << "process_key_event.\n";
    KeyEvent newkey;

    newkey.code = key.code;
    newkey.mask = key.mask & (SCIM_KEY_ShiftMask | SCIM_KEY_ControlMask | SCIM_KEY_AltMask | SCIM_KEY_ReleaseMask);

    /* change input mode */
    if (im_hangul_is_trigger_key (newkey)) {
        trigger_property (SCIM_PROP_INPUT_MODE);
        return true;
    }

    if (m_input_mode == INPUT_MODE_DIRECT)
        return false;

    /* toggle manual mode */
    if (im_hangul_is_manual_mode_key (newkey)) {
        trigger_property (SCIM_PROP_OUTPUT_MODE);
        return true;
    }

    /* toggle candidate table */
    if (im_hangul_is_hangul_hanja_key (newkey)) {
        if (m_lookup_table.number_of_candidates ())
            im_hangul_delete_candidates ();
        else
            im_hangul_update_candidates ();

        return true;
    }

    m_prev_key = key;

    /* ignore key release. */
    if (key.is_key_release ())
        return false;

    /* on capslock, we use Hangul Jamo */
    if (!m_factory->m_always_use_jamo && m_factory->m_use_capslock && key.code == SCIM_KEY_Caps_Lock) {
        if (!key.is_caps_lock_down ())
            m_output_mode |= OUTPUT_MODE_JAMO;
        else
            m_output_mode &= ~OUTPUT_MODE_JAMO;

        im_hangul_update_output_mode_property ();
    }

    /* ignore modifier keys */
    if (key.code == SCIM_KEY_Shift_L || key.code == SCIM_KEY_Shift_R ||
        key.code == SCIM_KEY_Control_L || key.code == SCIM_KEY_Control_R ||
        key.code == SCIM_KEY_Alt_L || key.code == SCIM_KEY_Alt_R)
        return false;

    /* candidate keys */
    if (m_lookup_table.number_of_candidates ()) {
        return candidate_key_event (key);
    }

    if (im_hangul_is_modifier_key (key)) {
        im_hangul_commit ();
        return false;
    }

    if (m_hangul_composer)
        return (this->*m_hangul_composer) (key);

    return false;
}

void
HangulInstance::move_preedit_caret (unsigned int pos)
{
}

void
HangulInstance::select_candidate (unsigned int item)
{
    SCIM_DEBUG_IMENGINE(2) << "select_candidate.\n";

    if (item < m_lookup_table.get_current_page_size ()) {
        WideString wstr = m_lookup_table.get_candidate_in_current_page (item);
        im_hangul_clear_buf ();
        commit_string (wstr);
        im_hangul_delete_candidates ();
        im_hangul_update_preedit_string ();
    }
}

void
HangulInstance::update_lookup_table_page_size (unsigned int page_size)
{
    SCIM_DEBUG_IMENGINE(2) << "update_lookup_table_page_size.\n";

    m_lookup_table.set_page_size (page_size);
}

void
HangulInstance::lookup_table_page_up ()
{
    if (!m_lookup_table.number_of_candidates () || !m_lookup_table.get_current_page_start ())
        return;

    SCIM_DEBUG_IMENGINE(2) << "lookup_table_page_up.\n";

    m_lookup_table.page_up ();

    update_lookup_table (m_lookup_table);

    im_hangul_update_aux_string ();
}

void
HangulInstance::lookup_table_page_down ()
{
    if (!m_lookup_table.number_of_candidates () ||
        m_lookup_table.get_current_page_start () + m_lookup_table.get_current_page_size () >=
          m_lookup_table.number_of_candidates ())
        return;

    SCIM_DEBUG_IMENGINE(2) << "lookup_table_page_down.\n";

    m_lookup_table.page_down ();

    update_lookup_table (m_lookup_table);

    im_hangul_update_aux_string ();
}

void
HangulInstance::reset ()
{
    SCIM_DEBUG_IMENGINE(2) << "reset.\n";

    im_hangul_commit ();
    im_hangul_delete_candidates ();
}

void
HangulInstance::focus_in ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_in.\n";

    im_hangul_register_properties ();

    if (m_lookup_table.number_of_candidates ()) {
        update_lookup_table (m_lookup_table);
        show_lookup_table ();
    } else {
        hide_lookup_table ();
    }

    im_hangul_update_aux_string ();
    im_hangul_update_preedit_string ();
}

void
HangulInstance::focus_out ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_out.\n";
}

void 
HangulInstance::trigger_property (const String &property)
{
    SCIM_DEBUG_IMENGINE(2) << "trigger_property : " << property << "\n";

    if (property == SCIM_PROP_INPUT_MODE) {
        reset ();

        if (m_input_mode == INPUT_MODE_DIRECT)
            m_input_mode = INPUT_MODE_HANGUL;
        else
            m_input_mode = INPUT_MODE_DIRECT;

        im_hangul_update_input_mode_property ();
        im_hangul_update_output_mode_property ();
    } else if (property == SCIM_PROP_OUTPUT_MODE) {
        if (m_input_mode != INPUT_MODE_DIRECT) {
            m_output_mode ^= OUTPUT_MODE_JAMO_EXT;
            im_hangul_update_output_mode_property ();
        }
    }
}

ucs4_t
HangulInstance::im_hangul_compose (ucs4_t first, ucs4_t last) const
{
    int min, max, mid;
    uint32 key;
 
    /* make key */
    key = first << 16 | last;

    /* binary search in table */
    min = 0;
    max = m_factory->m_compose_table_size - 1;

    while (max >= min) {
        mid = (min + max) / 2;
        if (m_factory->m_compose_table[mid].key < key)
            min = mid + 1;
        else if (m_factory->m_compose_table[mid].key > key)
            max = mid - 1;
        else
            return m_factory->m_compose_table[mid].code;
    }

    return 0;
}

ucs4_t
HangulInstance::im_hangul_mapping (int code, int mask) const
{
    if (m_factory->m_keyboard_table == 0)
        return 0;

    /* treat for dvorak */
    if (m_factory->m_use_dvorak)
        code = im_hangul_dvorak_to_qwerty (code);

    /* hangul jamo keysym */
    if (code >= 0x01001100 && code <= 0x010011ff)
        return code & 0x0000ffff;

    if (code >= SCIM_KEY_exclam && code <= SCIM_KEY_asciitilde) {
        /* treat capslock, as capslock is not on */
        if (mask & SCIM_KEY_CapsLockMask) {
            if (mask & SCIM_KEY_ShiftMask) {
                if (code >= SCIM_KEY_a && code <= SCIM_KEY_z)
                    code -= (SCIM_KEY_a - SCIM_KEY_A);
            } else {
                if (code >= SCIM_KEY_A && code <= SCIM_KEY_Z)
                    code += (SCIM_KEY_a - SCIM_KEY_A);
            }
        }
        return m_factory->m_keyboard_table[code - SCIM_KEY_exclam];
    }

    return 0;
}

/* composer functions (automata) */
static ucs4_t
im_hangul_choseong_to_jongseong (ucs4_t ch)
{
    static const ucs4_t table[] = {
        0x11a8,    /* choseong kiyeok       ->    jongseong kiyeok      */
        0x11a9,    /* choseong ssangkiyeok  ->    jongseong ssangkiyeok */
        0x11ab,    /* choseong nieun        ->    jongseong nieun       */
        0x11ae,    /* choseong tikeut       ->    jongseong tikeut      */
        0x0,       /* choseong ssangtikeut  ->    jongseong tikeut      */
        0x11af,    /* choseong rieul        ->    jongseong rieul       */
        0x11b7,    /* choseong mieum        ->    jongseong mieum       */
        0x11b8,    /* choseong pieup        ->    jongseong pieup       */
        0x0,       /* choseong ssangpieup   ->    jongseong pieup       */
        0x11ba,    /* choseong sios         ->    jongseong sios        */
        0x11bb,    /* choseong ssangsios    ->    jongseong ssangsios   */
        0x11bc,    /* choseong ieung        ->    jongseong ieung       */
        0x11bd,    /* choseong cieuc        ->    jongseong cieuc       */
        0x0,       /* choseong ssangcieuc   ->    jongseong cieuc       */
        0x11be,    /* choseong chieuch      ->    jongseong chieuch     */
        0x11bf,    /* choseong khieukh      ->    jongseong khieukh     */
        0x11c0,    /* choseong thieuth      ->    jongseong thieuth     */
        0x11c1,    /* choseong phieuph      ->    jongseong phieuph     */
        0x11c2     /* choseong hieuh        ->    jongseong hieuh       */
    };

    if (ch >= 0x11a8 && ch <= 0x11c2)
        return ch;
    if (ch < 0x1100 || ch > 0x1112)
        return 0;
    return table [ch - 0x1100];
}

static ucs4_t
im_hangul_jongseong_to_choseong (ucs4_t ch)
{
    static const ucs4_t table[] = {
        0x1100,    /* jongseong kiyeok          ->    choseong kiyeok       */
        0x1101,    /* jongseong ssangkiyeok     ->    choseong ssangkiyeok  */
        0x1109,    /* jongseong kiyeok-sios     ->    choseong sios         */
        0x1102,    /* jongseong nieun           ->    choseong nieun        */
        0x110c,    /* jongseong nieun-cieuc     ->    choseong cieuc        */
        0x1112,    /* jongseong nieun-hieuh     ->    choseong hieuh        */
        0x1103,    /* jongseong tikeut          ->    choseong tikeut       */
        0x1105,    /* jongseong rieul           ->    choseong rieul        */
        0x1100,    /* jongseong rieul-kiyeok    ->    choseong kiyeok       */
        0x1106,    /* jongseong rieul-mieum     ->    choseong mieum        */
        0x1107,    /* jongseong rieul-pieup     ->    choseong pieup        */
        0x1109,    /* jongseong rieul-sios      ->    choseong sios         */
        0x1110,    /* jongseong rieul-thieuth   ->    choseong thieuth      */
        0x1111,    /* jongseong rieul-phieuph   ->    choseong phieuph      */
        0x1112,    /* jongseong rieul-hieuh     ->    choseong hieuh        */
        0x1106,    /* jongseong mieum           ->    choseong mieum        */
        0x1107,    /* jongseong pieup           ->    choseong pieup        */
        0x1109,    /* jongseong pieup-sios      ->    choseong sios         */
        0x1109,    /* jongseong sios            ->    choseong sios         */
        0x110a,    /* jongseong ssangsios       ->    choseong ssangsios    */
        0x110b,    /* jongseong ieung           ->    choseong ieung        */
        0x110c,    /* jongseong cieuc           ->    choseong cieuc        */
        0x110e,    /* jongseong chieuch         ->    choseong chieuch      */
        0x110f,    /* jongseong khieukh         ->    choseong khieukh      */
        0x1110,    /* jongseong thieuth         ->    choseong thieuth      */
        0x1111,    /* jongseong phieuph         ->    choseong phieuph      */
        0x1112     /* jongseong hieuh           ->    choseong hieuh        */
    };

    if (ch < 0x11a8 || ch > 0x11c2)
        return 0;
    return table [ch - 0x11a8];
}

static void
im_hangul_jongseong_dicompose (ucs4_t ch, ucs4_t * jong, ucs4_t * cho)
{
    static const ucs4_t table[][2] = {
        { 0,      0x1100 }, /* jong kiyeok        = cho  kiyeok               */
        { 0x11a8, 0x1100 }, /* jong ssangkiyeok   = jong kiyeok + cho kiyeok  */
        { 0x11a8, 0x1109 }, /* jong kiyeok-sios   = jong kiyeok + cho sios    */
        { 0,      0x1102 }, /* jong nieun         = cho  nieun                */
        { 0x11ab, 0x110c }, /* jong nieun-cieuc   = jong nieun  + cho cieuc   */
        { 0x11ab, 0x1112 }, /* jong nieun-hieuh   = jong nieun  + cho hieuh   */
        { 0,      0x1103 }, /* jong tikeut        = cho  tikeut               */
        { 0,      0x1105 }, /* jong rieul         = cho  rieul                */
        { 0x11af, 0x1100 }, /* jong rieul-kiyeok  = jong rieul  + cho kiyeok  */
        { 0x11af, 0x1106 }, /* jong rieul-mieum   = jong rieul  + cho mieum   */
        { 0x11af, 0x1107 }, /* jong rieul-pieup   = jong rieul  + cho pieup   */
        { 0x11af, 0x1109 }, /* jong rieul-sios    = jong rieul  + cho sios    */
        { 0x11af, 0x1110 }, /* jong rieul-thieuth = jong rieul  + cho thieuth */
        { 0x11af, 0x1111 }, /* jong rieul-phieuph = jong rieul  + cho phieuph */
        { 0x11af, 0x1112 }, /* jong rieul-hieuh   = jong rieul  + cho hieuh   */
        { 0,      0x1106 }, /* jong mieum         = cho  mieum                */
        { 0,      0x1107 }, /* jong pieup         = cho  pieup                */
        { 0x11b8, 0x1109 }, /* jong pieup-sios    = jong pieup  + cho sios    */
        { 0,      0x1109 }, /* jong sios          = cho  sios                 */
        { 0x11ba, 0x1109 }, /* jong ssangsios     = jong sios   + cho sios    */
        { 0,      0x110b }, /* jong ieung         = cho  ieung                */
        { 0,      0x110c }, /* jong cieuc         = cho  cieuc                */
        { 0,      0x110e }, /* jong chieuch       = cho  chieuch              */
        { 0,      0x110f }, /* jong khieukh       = cho  khieukh              */
        { 0,      0x1110 }, /* jong thieuth       = cho  thieuth              */
        { 0,      0x1111 }, /* jong phieuph       = cho  phieuph              */
        { 0,      0x1112 }  /* jong hieuh         = cho  hieuh                */
    };

    *jong = table [ch - 0x11a8][0];
    *cho  = table [ch - 0x11a8][1];
}

bool
HangulInstance::im_hangul_composer_2 (const KeyEvent &key)
{
    ucs4_t ch;
    ucs4_t comp_ch;
    ucs4_t jong_ch;

    ch = im_hangul_mapping (key.code, key.mask);

    if (m_jongseong[0]) {
        if (im_hangul_is_choseong (ch)) {
            jong_ch = im_hangul_choseong_to_jongseong (ch);
            comp_ch = im_hangul_compose (m_jongseong[0], jong_ch);
            if (im_hangul_is_jongseong (comp_ch)) {
                m_jongseong[0] = comp_ch;
                im_hangul_push (comp_ch);
            } else {
                im_hangul_commit ();
                m_choseong[0] = ch;
                im_hangul_push (ch);
            }
            goto done;
        }
        if (im_hangul_is_jungseong (ch)) {
            ucs4_t pop, peek;
            pop = im_hangul_pop ();
            peek = im_hangul_peek ();
            if (im_hangul_is_jungseong (peek)) {
                m_jongseong[0] = 0;
                im_hangul_commit ();
                m_choseong[0] = im_hangul_jongseong_to_choseong (pop);
                m_jungseong[0] = ch;
                im_hangul_push (m_choseong[0]);
                im_hangul_push (ch);
            } else {
                ucs4_t choseong, jongseong; 
                im_hangul_jongseong_dicompose (m_jongseong[0], &jongseong, &choseong);
                m_jongseong[0] = jongseong;
                im_hangul_commit ();
                m_choseong[0] = choseong;
                m_jungseong[0] = ch;
                im_hangul_push (choseong);
                im_hangul_push (ch);
            }
            goto done;
        }
    } else if (m_jungseong[0]) {
        if (im_hangul_is_choseong (ch)) {
            if (m_choseong[0]) {
                jong_ch = im_hangul_choseong_to_jongseong (ch);
                if (im_hangul_is_jongseong (jong_ch)) {
                    m_jongseong[0] = jong_ch;
                    im_hangul_push (jong_ch);
                } else {
                    im_hangul_commit ();
                    m_choseong[0] = ch;
                    im_hangul_push (ch);
                }
            } else {
                m_choseong[0] = ch;
                im_hangul_push (ch);
            }
            goto done;
        }
        if (im_hangul_is_jungseong (ch)) {
            comp_ch = im_hangul_compose (m_jungseong[0], ch);
            if (im_hangul_is_jungseong (comp_ch)) {
                m_jungseong[0] = comp_ch;
                im_hangul_push (comp_ch);
            } else {
                im_hangul_commit ();
                m_jungseong[0] = ch;
                im_hangul_push (ch);
            }
            goto done;
        }
    } else if (m_choseong[0]) {
        if (im_hangul_is_choseong (ch)) {
            comp_ch = im_hangul_compose (m_choseong[0], ch);
            if (im_hangul_is_choseong (comp_ch)) {
                m_choseong[0] = comp_ch;
                im_hangul_push (comp_ch);
            } else {
                im_hangul_commit ();
                m_choseong[0] = ch;
                im_hangul_push (ch);
            }
            goto done;
        }
        if (im_hangul_is_jungseong (ch)) {
            m_jungseong[0] = ch;
            im_hangul_push (ch);
            goto done;
        }
    } else {
        if (im_hangul_is_choseong (ch)) {
            m_choseong[0] = ch;
            im_hangul_push (ch);
            goto done;
        }
        if (im_hangul_is_jungseong (ch)) {
            m_jungseong[0] = ch;
            im_hangul_push (ch);
            goto done;
        }
    }

    /* treat backspace */
    if (im_hangul_is_backspace_key (key)) {
        ch = im_hangul_pop ();
        if (ch == 0) return false;

        if (im_hangul_is_choseong (ch)) {
            ch = im_hangul_peek ();
            m_choseong[0] = im_hangul_is_choseong (ch) ? ch : 0;
            goto done;
        }
        if (im_hangul_is_jungseong (ch)) {
            ch = im_hangul_peek ();
            m_jungseong[0] = im_hangul_is_jungseong (ch) ? ch : 0;
            goto done;
        }
        if (im_hangul_is_jongseong (ch)) {
            ch = im_hangul_peek ();
            m_jongseong[0] = im_hangul_is_jongseong (ch) ? ch : 0;
            goto done;
        }
        return false;
    }

    im_hangul_commit ();
    return false;

done:

    im_hangul_update_preedit_string ();
    return true;
}

bool
HangulInstance::im_hangul_add_choseong  (ucs4_t ch)
{
    if (m_lindex >= 3) return false;
    m_lindex++;
    m_choseong [m_lindex] = ch;
    return true;
}

bool
HangulInstance::im_hangul_add_jungseong (ucs4_t ch)
{
    if (m_vindex >= 3) return false;
    m_vindex++;
    m_jungseong [m_vindex] = ch;
    return true;
}

bool
HangulInstance::im_hangul_add_jongseong (ucs4_t ch)
{
    if (m_tindex >= 3) return false;
    m_tindex++;
    m_jongseong [m_tindex] = ch;
    return true;
}

bool
HangulInstance::im_hangul_sub_choseong  ()
{
    m_choseong [m_lindex] = 0;
    if (m_lindex <= 0) return false;
    m_lindex--;
    return true;
}

bool
HangulInstance::im_hangul_sub_jungseong ()
{
    m_jungseong [m_vindex] = 0;
    if (m_vindex <= 0) return false;
    m_vindex--;
    return true;
}

bool
HangulInstance::im_hangul_sub_jongseong ()
{
    m_jongseong[m_tindex] = 0;
    if (m_tindex <= 0) return false;
    m_tindex--;
    return true;
}

bool
HangulInstance::im_hangul_composer_3 (const KeyEvent &key)
{
    ucs4_t ch;

    ch = im_hangul_mapping (key.code, key.mask);

    if (m_output_mode & OUTPUT_MODE_JAMO_EXT) {
        if (m_jongseong[0]) {
            if (im_hangul_is_choseong (ch)) {
                im_hangul_commit ();
                m_choseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jungseong (ch)) {
                im_hangul_commit ();
                m_jungseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jongseong (ch)) {
                if (!im_hangul_add_jongseong (ch)) {
                    im_hangul_commit ();
                    m_jongseong[0] = ch;
                }
                im_hangul_push (ch);
                goto done;
            }
        } else if (m_jungseong[0]) {
            if (im_hangul_is_choseong (ch)) {
                im_hangul_commit ();
                m_choseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jungseong (ch)) {
                if (!im_hangul_add_jungseong (ch)) {
                    im_hangul_commit ();
                    m_jungseong[0] = ch;
                }
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jongseong (ch)) {
                m_jongseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
        } else if (m_choseong[0]) {
            if (im_hangul_is_choseong (ch)) {
                if (!im_hangul_add_choseong (ch)) {
                    im_hangul_commit ();
                    m_choseong[0] = ch;
                }
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jungseong (ch)) {
                m_jungseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jongseong (ch)) {
                im_hangul_commit ();
                m_jongseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
        } else {
            if (im_hangul_is_choseong (ch)) {
                m_choseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jungseong (ch)) {
                m_jungseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jongseong (ch)) {
                m_jongseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
        }

        /* treat backspace */
        if (im_hangul_is_backspace_key (key)) {
            ch = im_hangul_pop ();
            if (ch == 0) return false;
            if (im_hangul_is_choseong (ch)) {
                im_hangul_sub_choseong ();
                goto done;
            }
            if (im_hangul_is_jungseong (ch)) {
                im_hangul_sub_jungseong ();
                goto done;
            }
            if (im_hangul_is_jongseong (ch)) {
                im_hangul_sub_jongseong ();
                goto done;
            }
            return false;
        }
    } else {
        /* choseong */
        if (im_hangul_is_choseong (ch)) {
            if (m_choseong[0] == 0) {
                m_choseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_choseong (im_hangul_peek ())) {
                ucs4_t choseong = im_hangul_compose (m_choseong[0], ch);
                if (choseong) {
                    m_choseong[0] = choseong;
                    im_hangul_push (choseong);
                    goto done;
                }
            }
            im_hangul_commit ();
            m_choseong[0] = ch;
            im_hangul_push (ch);
            goto done;
        }
        /* junseong */
        if (im_hangul_is_jungseong (ch)) {
            if (m_jungseong[0] == 0) {
                m_jungseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jungseong (im_hangul_peek ())) {
                ucs4_t jungseong = im_hangul_compose (m_jungseong[0], ch);
                if (jungseong) {
                    m_jungseong[0] = jungseong;
                    im_hangul_push (jungseong);
                    goto done;
                }
            }
            im_hangul_commit ();
            m_jungseong[0] = ch;
            im_hangul_push (ch);
            goto done;
        }
        /* jongseong */
        if (im_hangul_is_jongseong (ch)) {
            if (m_jongseong[0] == 0) {
                m_jongseong[0] = ch;
                im_hangul_push (ch);
                goto done;
            }
            if (im_hangul_is_jongseong (im_hangul_peek ())) {
                ucs4_t jongseong = im_hangul_compose (m_jongseong[0], ch);
                if (jongseong) {
                    m_jongseong[0] = jongseong;
                    im_hangul_push (jongseong);
                    goto done;
                }
            }
            im_hangul_commit ();
            m_jongseong[0] = ch;
            im_hangul_push (ch);
            goto done;
        }
        /* treat backspace */
        if (im_hangul_is_backspace_key (key)) {
            ch = im_hangul_pop ();
            if (ch == 0) return false;
            if (im_hangul_is_choseong (ch)) {
                ch = im_hangul_peek ();
                m_choseong[0] = im_hangul_is_choseong (ch) ? ch : 0;
                goto done;
            }
            if (im_hangul_is_jungseong (ch)) {
                ch = im_hangul_peek ();
                m_jungseong[0] = im_hangul_is_jungseong (ch) ? ch : 0;
                goto done;
            }
            if (im_hangul_is_jongseong (ch)) {
                ch = im_hangul_peek ();
                m_jongseong[0] = im_hangul_is_jongseong (ch) ? ch : 0;
                goto done;
            }
            return false;
        }
    }

    im_hangul_commit ();

    /* number and puctuation */
    if (ch > 0) {
        im_hangul_commit_unicode (ch);
        return true;
    }

    return false;

done:
    im_hangul_update_preedit_string ();
    return true;
}

ucs4_t
HangulInstance::im_hangul_pop  ()
{
    if (m_index < 0) return 0;
    return m_stack [m_index --];
}

ucs4_t
HangulInstance::im_hangul_peek ()
{
    if (m_index < 0) return 0;
    return m_stack [m_index];
}

void
HangulInstance::im_hangul_push (ucs4_t ch)
{
    m_stack [++m_index] = ch;
}

void
HangulInstance::im_hangul_clear_buf ()
{
    m_index = -1;
    m_stack[0] = 0;
    m_stack[1] = 0;
    m_stack[2] = 0;
    m_stack[3] = 0;
    m_stack[4] = 0;
    m_stack[5] = 0;
    m_stack[6] = 0;
    m_stack[7] = 0;
    m_stack[8] = 0;
    m_stack[9] = 0;
    m_stack[10] = 0;
    m_stack[11] = 0;

    m_lindex = 0;
    m_choseong[0] = 0;
    m_choseong[1] = 0;
    m_choseong[2] = 0;
    m_choseong[3] = 0;

    m_vindex = 0;
    m_jungseong[0] = 0;
    m_jungseong[1] = 0;
    m_jungseong[2] = 0;
    m_jungseong[3] = 0;

    m_tindex = 0;
    m_jongseong[0] = 0;
    m_jongseong[1] = 0;
    m_jongseong[2] = 0;
    m_jongseong[3] = 0;
}

WideString
HangulInstance::im_hangul_get_hangul_string ()
{
    WideString wstr;
    int i;

    if (im_hangul_is_empty ())
        return wstr;

    if (m_output_mode & OUTPUT_MODE_JAMO_EXT) {
        /* we use conjoining jamo, U+1100 - U+11FF */
        if (m_choseong[0] == 0) {
            wstr.push_back (HCF);
        } else {
            for (i = 0; i <= m_lindex; i++)
                wstr.push_back (m_choseong[i]);
        }

        if (m_jungseong[0] == 0) {
            wstr.push_back (HJF);
        } else {
            for (i = 0; i <= m_vindex; i++)
                wstr.push_back (m_jungseong[i]);
        }

        if (m_jongseong[0] != 0) {
            for (i = 0; i <= m_tindex; i++)
                wstr.push_back (m_jongseong[i]);
        }
    } else if (m_output_mode & OUTPUT_MODE_JAMO) {
        /* we use conjoining jamo, U+1100 - U+11FF */
        if (m_choseong[0] == 0) {
            wstr.push_back (HCF);
        } else {
            wstr.push_back (m_choseong[0]);
        }

        if (m_jungseong[0] == 0) {
            wstr.push_back (HJF);
        } else {
            wstr.push_back (m_jungseong[0]);
        }

        if (m_jongseong[0] != 0)
            wstr.push_back (m_jongseong[0]);
    } else {
        /* use hangul syllables (U+AC00 - U+D7AF)
         * and compatibility jamo (U+3130 - U+318F) */
        ucs4_t ch;
        ch = im_hangul_jamo_to_syllable (m_choseong[0], m_jungseong[0], m_jongseong[0]);
   
        if (ch) {
            wstr.push_back (ch);
        } else {
            if (m_choseong[0]) {
                ch = im_hangul_choseong_to_cjamo (m_choseong[0]);
                wstr.push_back (ch);
            }
            if (m_jungseong[0]) {
                ch = im_hangul_jungseong_to_cjamo (m_jungseong[0]);
                wstr.push_back (ch);
            }
            if (m_jongseong[0]) {
                ch = im_hangul_jongseong_to_cjamo (m_jongseong[0]);
                wstr.push_back (ch);
            }
        }
    }

    return wstr;
}

void
HangulInstance::im_hangul_commit ()
{
    WideString wstr = im_hangul_get_hangul_string ();
    im_hangul_clear_buf ();
    hide_preedit_string ();
    commit_string (wstr);
}

void
HangulInstance::im_hangul_commit_unicode (ucs4_t ch)
{
    WideString wstr;
    wstr.push_back (ch);
    im_hangul_clear_buf ();
    commit_string (wstr);
}

static int
get_index_of_candidate_table (ucs4_t ch)
{
    int first, last, mid;

    /* binary search */
    first = 0;
    last = sizeof (candidate_table)/sizeof (candidate_table [0]) - 1;

    while (first <= last) {
        mid = (first + last) / 2;

        if (ch == candidate_table[mid][0].ch)
            return mid;

        if (ch < candidate_table[mid][0].ch)
            last = mid - 1;
        else
            first = mid + 1;
    }

    return -1;
}

void
HangulInstance::im_hangul_update_candidates ()
{
    if (im_hangul_is_empty ())
        return;

    ucs4_t ch = 0;

    m_lookup_table.clear ();
    m_candidate_comments.clear ();

    if (m_choseong[0] != 0 && m_jungseong[0] == 0 && m_jongseong[0] == 0) {
        ch = im_hangul_choseong_to_cjamo(m_choseong[0]);
    } else {
        ch = im_hangul_jamo_to_syllable (m_choseong[0], m_jungseong[0], m_jongseong[0]);
    }

    if (ch > 0) {
        int index = get_index_of_candidate_table (ch);
        if (index != -1) {
            for (const CandidateItem *it = candidate_table [index] + 1; it->ch; ++it) {
                m_lookup_table.append_candidate (it->ch);
                m_candidate_comments.push_back (String (it->comment));
            }

            m_lookup_table.set_page_size (9);
            m_lookup_table.show_cursor ();

            update_lookup_table (m_lookup_table);
            show_lookup_table ();

            im_hangul_update_aux_string ();
        }
    }
}

void
HangulInstance::im_hangul_delete_candidates ()
{
    m_lookup_table.clear ();
    m_candidate_comments.clear ();
    hide_lookup_table ();
    hide_aux_string ();
}

void
HangulInstance::im_hangul_update_aux_string ()
{
    if (!m_factory->m_show_candidate_comment || !m_lookup_table.number_of_candidates ()) {
        hide_aux_string ();
        return;
    }

    size_t cursor = m_lookup_table.get_cursor_pos ();

    if (cursor >= m_candidate_comments.size () || !m_candidate_comments [cursor].length ()) {
        hide_aux_string ();
        return;
    }

    update_aux_string (m_lookup_table.get_candidate (cursor) + utf8_mbstowcs (String (" : ") + m_candidate_comments [cursor]));
    show_aux_string ();
}

void
HangulInstance::im_hangul_update_preedit_string ()
{
    WideString wstr = im_hangul_get_hangul_string ();

    if (wstr.length ()) {
        AttributeList attrs;
        attrs.push_back (Attribute (0, wstr.length (), SCIM_ATTR_DECORATE, SCIM_ATTR_DECORATE_REVERSE));
        show_preedit_string ();
        update_preedit_string (wstr, attrs);
    } else {
        hide_preedit_string ();
    }
}

void
HangulInstance::im_hangul_register_properties ()
{
    PropertyList proplist;

    proplist.push_back (Property (SCIM_PROP_INPUT_MODE, ""));
    proplist.push_back (Property (SCIM_PROP_OUTPUT_MODE, ""));

    register_properties (proplist);

    im_hangul_update_input_mode_property ();
    im_hangul_update_output_mode_property ();
}

void
HangulInstance::im_hangul_update_input_mode_property ()
{
    Property prop (SCIM_PROP_INPUT_MODE, "");

    if (m_input_mode == INPUT_MODE_DIRECT) {
        prop.set_label ("A");
        prop.set_tip (_("Direct input mode."));
    } else if (m_input_mode == INPUT_MODE_HANGUL) {
        prop.set_label ("한");
        prop.set_tip (_("Hangul input mode."));
    } else {
        prop.hide ();
    }

    update_property (prop);
}

void
HangulInstance::im_hangul_update_output_mode_property ()
{
    Property prop (SCIM_PROP_OUTPUT_MODE, "");
    String label;
    String tip (_("Output Mode: "));

    if (m_output_mode & OUTPUT_MODE_JAMO) {
        label += _("J");
        tip += _("Jamo");
    }

    if (m_output_mode & OUTPUT_MODE_JAMO_EXT) {
        if (label.length ()) {
            label += _("+");
            tip += _("+");
        }
        label += _("E");
        tip += _("Jamo Ext");
    }

    if (!label.length ()) {
        label += _("S");
        tip += _("Syllable");
    }

    prop.set_label (label);
    prop.set_tip (tip);

    if (m_input_mode == INPUT_MODE_DIRECT)
        prop.set_active (false);

    update_property (prop);
}

bool
HangulInstance::match_key_event (const KeyEventList &keys, const KeyEvent &key) const
{
    KeyEventList::const_iterator kit; 

    for (kit = keys.begin (); kit != keys.end (); ++kit) {
        if (key.code == kit->code && key.mask == kit->mask)
            if (!(key.mask & SCIM_KEY_ReleaseMask) || m_prev_key.code == key.code)
                return true;
    }
    return false;
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
