/* vim:set noet ts=4: */
/** 
 * scim-python
 * 
 * Copyright (c) 2007-2008 Huang Peng <shawn.p.huang@gmail.com>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: $
 */
#include "../config.h"
#include "scim-python.h"
#include <assert.h>

using namespace scim;
using namespace std;

static vector <IMEngineFactoryPointer> _factorys;
static vector <HelperInfo> _helpers;
static int _use_count = 0;

static PyObject *
scim_get_version (PyObject *self, PyObject *args)
{
	return PyString_FromString (PACKAGE_VERSION);
}

static PyMethodDef _methods[] = {
	{"get_version", scim_get_version, METH_NOARGS, "get version string of scim-python" },
	{0, }
};

PyMODINIT_FUNC
init_scim (void) 
{
	PyObject* m;
	
	m = Py_InitModule3("_scim", _methods,
						"SCIM.");
	
	if (m == NULL) {
		PyErr_Print ();
		return;
	}
	
	init_event (m);
	init_property (m);
	init_config (m);
	init_engine (m);
	init_factory (m);
	init_helper (m);
	init_attribute (m);
	init_lookup_table (m);
}

static PyObject *
Py_CallFunction (char *module, char *function, PyObject *args)
{
	PyObject *pName = NULL;
	PyObject *pModule = NULL;
	PyObject *pFunc = NULL;
	PyObject *pValue = NULL;

	pName = PyString_FromString (module);
	if (pName == NULL)
		goto _failed_out;

	pModule = PyImport_Import (pName);

	if (pModule == NULL)
		goto _failed_out;
	
	pFunc = PyObject_GetAttrString (pModule, function);
	
	if (pFunc == NULL)
		goto _failed_out;
	
	pValue = PyObject_CallObject (pFunc, args);	

	if (pValue == NULL)
		goto _failed_out;

	goto _success_out;;
	
_failed_out:
	PyErr_Print ();
_success_out:
	Py_XDECREF (pName);
	Py_XDECREF (pFunc);
	Py_XDECREF (pModule);
	
	return pValue;

}

static PyObject *
Py_CallModuleFunction (PyObject *module, char *function, PyObject *args)
{
	PyObject *pFunc = NULL;
	PyObject *pValue = NULL;

	pFunc = PyObject_GetAttrString (module, function);
	
	if (pFunc == NULL)
		goto _failed_out;
	
	pValue = PyObject_CallObject (pFunc, args);	

	if (pValue == NULL)
		goto _failed_out;

	goto _success_out;;
	
_failed_out:
	PyErr_Print ();
_success_out:
	Py_XDECREF (pFunc);
	
	return pValue;

}


extern "C" void
scim_module_init (void)
{

	char *argv[] = { 
		"scim-python", 
		NULL
	};
	
	bindtextdomain (GETTEXT_PACKAGE, SCIM_PYTHON_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	
	if (_use_count > 0) {
		goto out;
	}

	if (Py_IsInitialized ()) {
		goto out;
	}

	static int add_path = 0;
	
	if (add_path == 0) {
		char pythonpath[512] = "PYTHONPATH="SCIM_PYTHON_DATADIR;
		char *old_pythonpath = getenv ("PYTHONPATH");
		
		if (old_pythonpath != NULL) {
			strncat (pythonpath, ":", sizeof (pythonpath));
			strncat (pythonpath, old_pythonpath, sizeof (pythonpath));
		}
		putenv (strdup (pythonpath));
		add_path = 1;
	}
	
	Py_Initialize ();
	PySys_SetArgv (1, argv);

out:
	_use_count ++;
}

extern "C" void
scim_module_exit (void)
{
	if (_use_count == 0)
		return;

	_use_count --;
	if (_use_count == 0) {
		_factorys.clear ();
		Py_Finalize ();
	}
}


#ifndef DISABLE_IMENGINE
extern "C" unsigned int 
scim_imengine_module_init (const ConfigPointer &config)
{
	PyObject *pValue = NULL;
	PyObject *pArgs = NULL;
	PyObject *pyconfig = NULL;
	int size = 0;

	pyconfig = PyConfig_New (config);
	Py_DECREF (pyconfig);
	pyconfig = PyConfig_New (config);

	pArgs = Py_BuildValue ("(O)", pyconfig);
	pValue = Py_CallFunction ("engine", "query_engines", pArgs);	
	
	Py_DECREF (pArgs);
	Py_DECREF (pyconfig);
	if (PyList_Check (pValue)) {
		PyObject * pTuple = PyList_AsTuple (pValue);
		Py_DECREF (pValue);
		pValue = pTuple;
	}
	if (!PyTuple_Check (pValue))
		return 0;

	size = PyTuple_Size (pValue);

	for (int i = 0; i < size; i++) {
		IMEngineFactoryPointer p = PyIMEngineFactory::from_pyobject (PyTuple_GetItem (pValue, i));
		_factorys.push_back (p);
	}

	Py_DECREF (pValue);
	return size;
}

extern "C" IMEngineFactoryPointer 
scim_imengine_module_create_factory (uint32 engine)
{
	if ( engine < 0 || engine >= _factorys.size ())
		return IMEngineFactoryPointer (0);

	return _factorys[engine];
}
#endif //DISABLE_IMENGINE

#ifndef DISABLE_SETUPUI
#include <gtk/gtk.h>
#include <pygobject.h>

extern "C" GtkWidget * 
scim_setup_module_create_ui (void)
{
	static GtkWidget *window = NULL;
	PyObject *pValue = NULL;

	if (window != NULL)
		return window;

	
	pValue = Py_CallFunction ("setupui", "create_ui", NULL);	

	if (pValue != Py_None) {
		window = GTK_WIDGET (pygobject_get (pValue));
		gtk_object_ref (GTK_OBJECT (window));
	}

	Py_DECREF (pValue);
	return window;
}

extern "C" String
scim_setup_module_get_category (void)
{
	return String ("IMEngine");
}

extern "C" String
scim_setup_module_get_name (void)
{
	return String (_("Python"));
}

extern "C" String
scim_setup_module_get_description (void)
{
	return String (_("Setup UI for Python IM engines."));
}

extern "C" void
scim_setup_module_load_config (const ConfigPointer &config)
{
	PyObject *pValue = NULL;
	PyObject *pArgs = NULL;

	pArgs = Py_BuildValue ("(O)", PyConfig_New (config));
	pValue = Py_CallFunction ("setupui", "load_config", pArgs);
	
	if ( pValue == NULL) {
		PyErr_Clear ();
	}

	Py_XDECREF (pArgs);
	Py_XDECREF (pValue);
}

extern "C" void
scim_setup_module_save_config (const ConfigPointer &config)
{
	PyObject *pValue = NULL;
	PyObject *pArgs = NULL;

	pArgs = Py_BuildValue ("(O)", PyConfig_New (config));
	
	pValue = Py_CallFunction ("setupui", "save_config", pArgs);	
	

	Py_XDECREF (pArgs);
	Py_XDECREF (pValue);
}

extern "C" bool
scim_setup_module_query_changed ()
{
	PyObject *pValue = NULL;
	bool retval = false;

	pValue = Py_CallFunction ("setupui", "query_changed", NULL);
	
	if (pValue == Py_True) {
		retval = true;
	}

	Py_XDECREF (pValue);

	return retval;
}

#if 0
/*
 * Overide new and delete operator.
 **/
static int __count = 0;
void* operator new (size_t size)
{
	fprintf (stderr, "new count = %d\n", __count++);
	if (size < 256) {
		size = 256;
	}
	
	void *p = PyObject_Malloc (size);
	assert (p != NULL);
	return p;
}

void operator delete (void *p)
{
	fprintf (stderr, "delete count = %d\n", __count--);
	assert (p != NULL);
	PyObject_Free (p); 
}
#endif

#endif //DISABLE_SETUPUI

#ifndef DISABLE_HELPER

static PyObject *g_helper_mod = NULL;

extern "C" unsigned int 
scim_helper_module_number_of_helpers (void)
{
	PyObject *retval = NULL;
	int size = 0;
	if (g_helper_mod == NULL) {
		PyObject *name = PyString_FromString ("helper");
		g_helper_mod = PyImport_Import (name);
		Py_DECREF (name);
	}

	retval = Py_CallModuleFunction (g_helper_mod, "number_of_helpers", NULL);	

	if (retval != NULL) {
		size =  PyInt_AsLong (retval);
		Py_DECREF (retval);
	}
	
	return size;
}

extern "C" bool
scim_helper_module_get_helper_info (unsigned int index, HelperInfo &info)
{
	bool result = FALSE;
	PyObject *retval = NULL;
	PyObject *args = NULL;
	PyObject *helper_info = NULL;

	args = Py_BuildValue ("(I)", index);
	retval = Py_CallModuleFunction (g_helper_mod, "get_helper_info", args);
	
	Py_DECREF (args);
	
	if (retval == Py_None || retval == NULL) {
		result = FALSE;
		goto _out;
	}

	if (PyList_Check (retval)) {
		helper_info = PyList_AsTuple (retval);
	}
	else if (PyTuple_Check (args)) {
		helper_info = retval;
		Py_INCREF (retval);
	}

	if (helper_info != NULL && PyTuple_GET_SIZE (helper_info) == 5) {
		const char *uuid = PyString_AsString (PyTuple_GetItem (helper_info, 0));
		const char *name = PyString_AsString (PyTuple_GetItem (helper_info, 1));
		const char *icon = PyString_AsString (PyTuple_GetItem (helper_info, 2));
		const char *desc = PyString_AsString (PyTuple_GetItem (helper_info, 3));
		int opt = PyInt_AsLong (PyTuple_GetItem (helper_info, 4));
		info = HelperInfo (uuid, name, icon, desc, opt);
		result = TRUE;
	}

_out:
	Py_XDECREF (args);
	Py_XDECREF (retval);
	Py_XDECREF (helper_info);

	return result;
}

extern "C" void
scim_helper_module_run_helper (const String &uuid, const ConfigPointer &config,
						const String &display)
{
	PyObject *retval = NULL;
	PyObject *args = NULL;

	args = Py_BuildValue ("(sOs)", uuid.c_str (), PyConfig_New (config), display.c_str ());
	retval = Py_CallModuleFunction (g_helper_mod, "run_helper", args);
	
	Py_XDECREF (args);
	Py_XDECREF (retval);
	
}

#endif //DISABLE_HELPER
