/** @file scim_socket_frontend.cpp
 * implementation of class SocketFrontEnd.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_frontend.cpp,v 1.10 2003/12/31 05:09:24 suzhe Exp $
 *
 */

#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_FRONTEND
#define Uses_SCIM_SOCKET
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_STL_UTILITY
#define Uses_C_STDIO
#define Uses_C_STDLIB

#include <limits.h>
#include "scim_private.h"
#include "scim.h"
#include "scim_socket_frontend.h"

#define scim_module_init socket_LTX_scim_module_init
#define scim_module_exit socket_LTX_scim_module_exit
#define scim_frontend_module_init socket_LTX_scim_frontend_module_init
#define scim_frontend_module_run socket_LTX_scim_frontend_module_run

#define SCIM_CONFIG_FRONTEND_SOCKET_ADDRESS            "/FrontEnd/Socket/Address"
#define SCIM_CONFIG_FRONTEND_SOCKET_TIMEOUT            "/FrontEnd/Socket/Timeout"
#define SCIM_CONFIG_FRONTEND_SOCKET_CONFIG_READONLY    "/FrontEnd/Socket/ConfigReadOnly"
#define SCIM_CONFIG_FRONTEND_SOCKET_MAXCLIENTS        "/FrontEnd/Socket/MaxClients"

#define SCIM_SOCKET_FRONTEND_DEF_ADDRESS "local:/tmp/scim-socket-frontend"

using namespace scim;

static Pointer <SocketFrontEnd> _scim_frontend (0);

static int _argc;
static char **_argv;

//Module Interface
extern "C" {
    void scim_module_init (void)
    {
        SCIM_DEBUG_FRONTEND(1) << "Initializing Socket FrontEnd module...\n";
    }

    void scim_module_exit (void)
    {
        SCIM_DEBUG_FRONTEND(1) << "Exiting Socket FrontEnd module...\n";
        _scim_frontend.reset ();
    }

    void scim_frontend_module_init (const BackEndPointer &backend,
                                    const ConfigPointer &config,
                                    int argc,
                                     char **argv)
    {
        if (_scim_frontend.null ()) {
            SCIM_DEBUG_FRONTEND(1) << "Initializing Socket FrontEnd module (more)...\n";
            _scim_frontend = new SocketFrontEnd (backend, config);
            _argc = argc;
            _argv = argv;
        }
    }

    void scim_frontend_module_run (void)
    {
        if (!_scim_frontend.null ()) {
            SCIM_DEBUG_FRONTEND(1) << "Starting Socket FrontEnd module...\n";
            _scim_frontend->init (_argc, _argv);
            _scim_frontend->run ();
        }
    }
}

class _SocketServerInstanceRepLessThan
{
public:
    bool operator () (const std::pair <int, int> & lhs,
                      const std::pair <int, int> & rhs) {
        if (lhs.first < rhs.first) return true;
        else if (lhs.first == rhs.first) return lhs.second < rhs.second;
        return false;
    }
};

class _SocketClientRepLessThan
{
public:
    bool operator () (const std::pair <int, int> & lhs,
                      const std::pair <int, int> & rhs) {
        return lhs.first < rhs.first;
    }
};

SocketFrontEnd::SocketFrontEnd (const BackEndPointer &backend,
                                const ConfigPointer  &config)
    : FrontEndBase (backend),
      m_config (config),
      m_config_readonly (false),
      m_socket_timeout (-1),
      m_current_server_instance (-1)
{
    SCIM_DEBUG_FRONTEND (2) << " Constructing SocketFrontEnd object...\n";
}

SocketFrontEnd::~SocketFrontEnd ()
{
    SCIM_DEBUG_FRONTEND (2) << " Destructing SocketFrontEnd object...\n";
    if (m_socket_server.is_running ())
        m_socket_server.shutdown ();
}

void
SocketFrontEnd::show_preedit_string (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_PREEDIT_STRING);
}

void
SocketFrontEnd::show_status_string (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_STATUS_STRING);
}

void
SocketFrontEnd::show_aux_string (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_AUX_STRING);
}

void
SocketFrontEnd::show_lookup_table (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE);
}

void
SocketFrontEnd::hide_preedit_string (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_PREEDIT_STRING);
}

void
SocketFrontEnd::hide_status_string (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_STATUS_STRING);
}

void
SocketFrontEnd::hide_aux_string (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_AUX_STRING);
}

void
SocketFrontEnd::hide_lookup_table (int id)
{
    if (m_current_server_instance == id)
        m_send_transaction.put_command (SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE);
}

void
SocketFrontEnd::update_preedit_caret (int id, int caret)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET);
        m_send_transaction.put_data ((uint32) caret);
    }
}

void
SocketFrontEnd::update_preedit_string (int id,
                                       const WideString & str,
                                       const AttributeList & attrs)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING);
        m_send_transaction.put_data (str);
        m_send_transaction.put_data (attrs);
    }
}

void
SocketFrontEnd::update_status_string (int id,
                                      const WideString & str,
                                      const AttributeList & attrs)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_STATUS_STRING);
        m_send_transaction.put_data (str);
        m_send_transaction.put_data (attrs);
    }
}

void
SocketFrontEnd::update_aux_string (int id,
                                   const WideString & str,
                                   const AttributeList & attrs)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_AUX_STRING);
        m_send_transaction.put_data (str);
        m_send_transaction.put_data (attrs);
    }
}

void
SocketFrontEnd::commit_string (int id, const WideString & str)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_COMMIT_STRING);
        m_send_transaction.put_data (str);
    }
}

void
SocketFrontEnd::forward_keyevent (int id, const KeyEvent & key)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_FORWARD_KEYEVENT);
        m_send_transaction.put_data (key);
    }
}

void
SocketFrontEnd::update_lookup_table (int id, const LookupTable & table)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE);
        m_send_transaction.put_data (table);
    }
}

void
SocketFrontEnd::update_full_width_punctuation (int id, bool full)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_PUNCTUATION);
        m_send_transaction.put_data ((uint32) full);
    }
}

void
SocketFrontEnd::update_full_width_letter (int id, bool full)
{
    if (m_current_server_instance == id) {
        m_send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_LETTER);
        m_send_transaction.put_data ((uint32) full);
    }
}

void
SocketFrontEnd::init (int argc, char **argv)
{
    int max_clients = -1;

    if (!m_config.null ()) {
        String str;

        //Read settings.
        str = m_config->read (String (SCIM_CONFIG_FRONTEND_SOCKET_ADDRESS),
                                String ("default"));

        if (str == "default")
            m_socket_address = SCIM_SOCKET_FRONTEND_DEF_ADDRESS;
        else
            m_socket_address = str;

        m_config_readonly = m_config->read (String (SCIM_CONFIG_FRONTEND_SOCKET_CONFIG_READONLY), false);

        m_socket_timeout = m_config->read (String (SCIM_CONFIG_FRONTEND_SOCKET_TIMEOUT), -1);

        max_clients = m_config->read (String (SCIM_CONFIG_FRONTEND_SOCKET_MAXCLIENTS), -1);
    } else {
        m_socket_address = SCIM_SOCKET_FRONTEND_DEF_ADDRESS;
        m_config_readonly = false;
        m_socket_timeout = -1;
        max_clients = -1;
    }

    // Override the config vaule if env is set.
    const char *env = getenv ("SCIM_SOCKET_ADDRESS");
    if (env) {
        m_socket_address = String (env);
    } else {
        env = getenv ("SCIM_FRONTEND_SOCKET_ADDRESS");
        if (env)
            m_socket_address = String (env);
    }

    env = getenv ("SCIM_SOCKET_TIMEOUT");
    if (env) {
        m_socket_timeout = atoi (env);
    } else {
        env = getenv ("SCIM_FRONTEND_SOCKET_TIMEOUT");
        if (env)
            m_socket_timeout = atoi (env);
    }

    if (!m_socket_server.create (m_socket_address))
        throw FrontEndError ("SocketFrontEnd -- Cannot create SocketServer.");

    m_socket_server.set_max_clients (max_clients);

    m_socket_server.signal_connect_accept (
        slot (this, &SocketFrontEnd::socket_accept_callback));

    m_socket_server.signal_connect_receive (
        slot (this, &SocketFrontEnd::socket_receive_callback));

    m_socket_server.signal_connect_exception(
        slot (this, &SocketFrontEnd::socket_exception_callback));

    /**
     * initialize the random number generator.
     */
    srand (time (0));
}

void
SocketFrontEnd::run ()
{
    if (m_socket_server.valid ())
        m_socket_server.run ();
}

int
SocketFrontEnd::generate_key () const
{
    return rand ();
}

void
SocketFrontEnd::add_client (int client_id, int key)
{
    SCIM_DEBUG_FRONTEND (2) << " add_client (" << client_id << "," << key << ").\n";

    SocketClientRepository::iterator it
        = std::lower_bound (m_socket_client_repository.begin (),
                            m_socket_client_repository.end (),
                            std::pair <int, int> (client_id, key),
                            _SocketClientRepLessThan ());

    if (it == m_socket_client_repository.end ()) {
        m_socket_client_repository.push_back (std::pair <int, int> (client_id, key));
    } else if (it->first == client_id) {
        it->second = key;
    } else {
        m_socket_client_repository.insert (it, std::pair <int, int> (client_id, key));
    }
}

void
SocketFrontEnd::delete_client (int client_id)
{
    SCIM_DEBUG_FRONTEND (2) << " delete_client (" << client_id << ").\n";

    SocketClientRepository::iterator it
        = std::lower_bound (m_socket_client_repository.begin (),
                            m_socket_client_repository.end (),
                            std::pair <int, int> (client_id, 0),
                            _SocketClientRepLessThan ());

    if (it != m_socket_client_repository.end () && it->first == client_id) {
        m_socket_client_repository.erase (it);
    }
}

bool
SocketFrontEnd::validate_client (int client_id, int key) const
{
    SCIM_DEBUG_FRONTEND (2) << " validate_client (" << client_id << "," << key << ").\n";

    SocketClientRepository::const_iterator it
        = std::lower_bound (m_socket_client_repository.begin (),
                            m_socket_client_repository.end (),
                            std::pair <int, int> (client_id, key),
                            _SocketClientRepLessThan ());

    if (it != m_socket_client_repository.end () &&
        it->first == client_id && it->second == key)
        return true;

    return false;
}

bool
SocketFrontEnd::check_client_connection (const Socket &client) const
{
    SCIM_DEBUG_FRONTEND (1) << "check_client_connection (" << client.get_id () << ").\n";

    unsigned char buf [sizeof(uint32)];

    int nbytes = client.read_with_timeout (buf, sizeof(uint32), m_socket_timeout);

    if (nbytes == sizeof (uint32))
        return true;

    if (nbytes < 0) {
        SCIM_DEBUG_FRONTEND (2) << " Error occurred when reading socket (" << client.get_id ()
            << "):" << client.get_error_message () << "\n";
    } else {
        SCIM_DEBUG_FRONTEND (2) << " Timeout when reading socket (" << client.get_id ()
            << ").\n";
    }

    return false;
}

void
SocketFrontEnd::socket_accept_callback (SocketServer *server, const Socket &client)
{
    SCIM_DEBUG_FRONTEND (1) << "socket_accept_callback (" << client.get_id () << ").\n";
}

void
SocketFrontEnd::socket_receive_callback (SocketServer *server, const Socket &client)
{
    int id = client.get_id ();

    m_send_transaction.clear ();

    m_send_transaction.put_command (SCIM_TRANS_CMD_REPLY);

    SCIM_DEBUG_FRONTEND (1) << "socket_receive_callback (" << id << ").\n";

    // Check if the client is closed.
    if (!check_client_connection (client)) {
        SCIM_DEBUG_FRONTEND (2) << " closing client connection.\n";
        delete_client (id);
        socket_delete_all_server_instances (id);
        server->close_connection (client);
        return;
    }

    // Read transaction.
    if (m_receive_transaction.read_from_socket (client, m_socket_timeout) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND) {

        int cmd;

        SCIM_DEBUG_FRONTEND (2) << " Check request commands.\n";

        // Check if the command is valid.
        if (m_receive_transaction.get_command (cmd) &&
            cmd == SCIM_TRANS_CMD_REQUEST) {

            // Check if it's a OPEN_CONNECTION command.
            if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
                m_receive_transaction.get_command (cmd) &&
                cmd == SCIM_TRANS_CMD_OPEN_CONNECTION) {
                socket_open_connection (server, client);
                return;
            }

            uint32 key;
            // Check the client's magic key.
            if (m_receive_transaction.get_data (key) && validate_client (id, key)) {
                if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
                    m_receive_transaction.get_command (cmd)) {

                    // Dispatch the command.
                    if (cmd == SCIM_TRANS_CMD_PROCESS_KEYEVENT)
                        socket_process_key_event (id);
                    else if (cmd == SCIM_TRANS_CMD_MOVE_PREEDIT_CARET)
                        socket_move_preedit_caret (id);
                    else if (cmd == SCIM_TRANS_CMD_SELECT_LOOKUP_TABLE)
                        socket_select_lookup_table (id);
                    else if (cmd == SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE)
                        socket_update_lookup_table_page_size (id);
                    else if (cmd == SCIM_TRANS_CMD_RESET)
                        socket_reset_server_instance (id);
                    else if (cmd == SCIM_TRANS_CMD_FOCUS_IN)
                        socket_focus_in_server_instance (id);
                    else if (cmd == SCIM_TRANS_CMD_FOCUS_OUT)
                        socket_focus_out_server_instance (id);
                    else if (cmd == SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_PUNCTUATION)
                        socket_toggle_full_width_punctuation (id);
                    else if (cmd == SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_LETTER)
                        socket_toggle_full_width_letter (id);
                    else if (cmd == SCIM_TRANS_CMD_TOGGLE_INPUT_STATUS)
                        socket_toggle_input_status (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_LIST)
                        socket_get_server_factory_list (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_NAME)
                        socket_get_server_factory_name (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_AUTHORS)
                        socket_get_server_factory_authors (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_CREDITS)
                        socket_get_server_factory_credits (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_HELP)
                        socket_get_server_factory_help (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_LOCALES)
                        socket_get_server_factory_locales (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_SERVER_FACTORY_ICON_FILE)
                        socket_get_server_factory_icon_file (id);
                    else if (cmd == SCIM_TRANS_CMD_NEW_SERVER_INSTANCE)
                        socket_new_server_instance (id);
                    else if (cmd == SCIM_TRANS_CMD_DELETE_SERVER_INSTANCE)
                        socket_delete_server_instance (id);
                    else if (cmd == SCIM_TRANS_CMD_DELETE_ALL_SERVER_INSTANCES)
                        socket_delete_all_server_instances (id);
                    else if (cmd == SCIM_TRANS_CMD_FLUSH_CONFIG)
                        socket_flush_config (id);
                    else if (cmd == SCIM_TRANS_CMD_ERASE_CONFIG)
                        socket_erase_config (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_CONFIG_STRING)
                        socket_get_config_string (id);
                    else if (cmd == SCIM_TRANS_CMD_SET_CONFIG_STRING)
                        socket_set_config_string (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_CONFIG_INT)
                        socket_get_config_int (id);
                    else if (cmd == SCIM_TRANS_CMD_SET_CONFIG_INT)
                        socket_set_config_int (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_CONFIG_BOOL)
                        socket_get_config_bool (id);
                    else if (cmd == SCIM_TRANS_CMD_SET_CONFIG_BOOL)
                        socket_set_config_bool (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_CONFIG_DOUBLE)
                        socket_get_config_double (id);
                    else if (cmd == SCIM_TRANS_CMD_SET_CONFIG_DOUBLE)
                        socket_set_config_double (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_CONFIG_VECTOR_STRING)
                        socket_get_config_vector_string (id);
                    else if (cmd == SCIM_TRANS_CMD_SET_CONFIG_VECTOR_STRING)
                        socket_set_config_vector_string (id);
                    else if (cmd == SCIM_TRANS_CMD_GET_CONFIG_VECTOR_INT)
                        socket_get_config_vector_int (id);
                    else if (cmd == SCIM_TRANS_CMD_SET_CONFIG_VECTOR_INT)
                        socket_set_config_vector_int (id);
                    else if (cmd == SCIM_TRANS_CMD_LOAD_FILE)
                        socket_load_file (id);
                    else if (cmd == SCIM_TRANS_CMD_CLOSE_CONNECTION) {
                        delete_client (id);
                        socket_delete_all_server_instances (id);
                        server->close_connection (client);
                        return;
                    }

                    // Send reply to client
                    // This cmd is REPLY.
                    m_send_transaction.get_command (cmd);
                    if (m_send_transaction.get_data_type () == SCIM_TRANS_DATA_UNKNOWN)
                        m_send_transaction.put_command (SCIM_TRANS_CMD_FAIL);

                    m_send_transaction.write_to_socket (client);
                }
            }
        }
    }
}

void
SocketFrontEnd::socket_open_connection (SocketServer *server, const Socket &client)
{
    int cmd;

    int key = generate_key ();

    m_send_transaction.clear ();

    SCIM_DEBUG_FRONTEND (1) << "socket_open_connection (" << client.get_id () << ").\n";

    m_send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
    m_send_transaction.put_data ((uint32) key);

    SCIM_DEBUG_FRONTEND (2) << " Send magic key (" << key << ") to client.\n";

    // Send Feedback.
    if (m_send_transaction.write_to_socket (client)) {

        SCIM_DEBUG_FRONTEND (2) << " Check client reply.\n";
        // Check client reply.
        if (m_receive_transaction.read_from_socket (client, m_socket_timeout) &&
            m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND) {

            if (m_receive_transaction.get_command (cmd) &&
                cmd == SCIM_TRANS_CMD_REPLY) {
                if (m_receive_transaction.get_command (cmd) &&
                    cmd == SCIM_TRANS_CMD_OK) {
                    add_client (client.get_id (), key);

                    SCIM_DEBUG_FRONTEND (2) << " New connection created.\n"; 

                    return;
                }
            }
        }
    }

    // Client did not pass the registration process, close it.
    SCIM_DEBUG_FRONTEND (2) << " Failed to create new connection.\n"; 
    server->close_connection (client);
}

void
SocketFrontEnd::socket_exception_callback (SocketServer *server, const Socket &client)
{
    SCIM_DEBUG_FRONTEND (1) << "socket_exception_callback (" << client.get_id () << ").\n";

    delete_client (client.get_id ());
    socket_delete_all_server_instances (client.get_id ());
    server->close_connection (client);
}

//client_id is client's socket id
void
SocketFrontEnd::socket_get_server_factory_list (int /*client_id*/)
{
    String encoding;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_list.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (encoding)) {
        std::vector<String> uuids;

        get_server_factory_list (uuids, encoding);

        SCIM_DEBUG_FRONTEND (3) << "  Encoding (" << encoding
            << ") Num(" << uuids.size () << ").\n";

        m_send_transaction.put_data (uuids);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_server_factory_name (int /*client_id*/)
{
    String sfid;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_name.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid)) {
        WideString name = get_server_factory_name (sfid);

        m_send_transaction.put_data (name);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_server_factory_authors (int /*client_id*/)
{
    String sfid;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_authors.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid)) {
        WideString authors = get_server_factory_authors (sfid);

        m_send_transaction.put_data (authors);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_server_factory_credits (int /*client_id*/)
{
    String sfid;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_credits.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid)) {
        WideString credits = get_server_factory_credits (sfid);

        m_send_transaction.put_data (credits);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_server_factory_help (int /*client_id*/)
{
    String sfid;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_help.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid)) {
        WideString help = get_server_factory_help (sfid);

        m_send_transaction.put_data (help);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_server_factory_locales (int /*client_id*/)
{
    String sfid;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_locales.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid)) {
        String locales = get_server_factory_locales (sfid);

        SCIM_DEBUG_FRONTEND (3) << "  Locales (" << locales << ").\n";

        m_send_transaction.put_data (locales);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_server_factory_icon_file (int /*client_id*/)
{
    String sfid;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_server_factory_icon_file.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid)) {
        String iconfile = get_server_factory_icon_file (sfid);

        SCIM_DEBUG_FRONTEND (3) << "  ICON File (" << iconfile << ").\n";

        m_send_transaction.put_data (iconfile);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_new_server_instance (int client_id)
{
    String sfid;
    String encoding;

    SCIM_DEBUG_FRONTEND (2) << " socket_new_server_instance.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (sfid) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (encoding)) {
        int siid = new_server_instance (sfid, encoding);

        // ServerInstance created OK.
        if (siid >= 0) {
            SocketServerInstanceRepository::iterator it =
                std::lower_bound (m_socket_si_repository.begin (),
                                  m_socket_si_repository.end (),
                                  std::pair <int, int> (client_id, siid),
                                  _SocketServerInstanceRepLessThan ());

            if (it == m_socket_si_repository.end ())
                m_socket_si_repository.push_back (std::pair <int, int> (client_id, siid));
            else
                m_socket_si_repository.insert (it, std::pair <int, int> (client_id, siid));

            SCIM_DEBUG_FRONTEND (3) << "  ServerInstanceID (" << siid << ").\n";

            m_send_transaction.put_data ((uint32)siid);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_delete_server_instance (int client_id)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_delete_server_instance.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  ServerInstanceID (" << siid << ").\n";

        delete_server_instance ((int) siid);

        SocketServerInstanceRepository::iterator it =
            std::lower_bound (m_socket_si_repository.begin (),
                              m_socket_si_repository.end (),
                              std::pair <int, int> (client_id, siid),
                              _SocketServerInstanceRepLessThan ());

        if (it != m_socket_si_repository.end () &&
            *it == std::pair <int, int> (client_id, siid))
            m_socket_si_repository.erase (it);

        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_delete_all_server_instances (int client_id)
{
    SCIM_DEBUG_FRONTEND (2) << " socket_delete_all_server_instances.\n";

    SocketServerInstanceRepository::iterator lit =
        std::lower_bound (m_socket_si_repository.begin (),
                          m_socket_si_repository.end (),
                          std::pair <int, int> (client_id, 0),
                          _SocketServerInstanceRepLessThan ());

    SocketServerInstanceRepository::iterator uit =
        std::upper_bound (m_socket_si_repository.begin (),
                          m_socket_si_repository.end (),
                          std::pair <int, int> (client_id, INT_MAX),
                          _SocketServerInstanceRepLessThan ());

    if (lit != uit) {
        m_socket_si_repository.erase (lit, uit);
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_process_key_event (int /*client_id*/)
{
    uint32   siid;
    KeyEvent event;

    SCIM_DEBUG_FRONTEND (2) << " socket_process_key_event.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_KEYEVENT &&
        m_receive_transaction.get_data (event)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ") KeyEvent ("
            << event.code << "," << event.mask << ").\n";

        m_current_server_instance = (int) siid;

        if (process_key_event ((int) siid, event))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        else
            m_send_transaction.put_command (SCIM_TRANS_CMD_FAIL);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_move_preedit_caret (int /*client_id*/)
{
    uint32 siid;
    uint32 caret;

    SCIM_DEBUG_FRONTEND (2) << " socket_move_preedit_caret.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (caret)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid
            << ") Caret (" << caret << ").\n";

        m_current_server_instance = (int) siid;

        move_preedit_caret ((int) siid, caret); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_select_lookup_table (int /*client_id*/)
{
    uint32 siid;
    uint32 item;

    SCIM_DEBUG_FRONTEND (2) << " socket_select_lookup_table.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (item)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ") Item (" << item << ").\n";

        m_current_server_instance = (int) siid;

        select_lookup_table ((int) siid, item); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_update_lookup_table_page_size (int /*client_id*/)
{
    uint32 siid;
    uint32 size;

    SCIM_DEBUG_FRONTEND (2) << " socket_update_lookup_table_page_size.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (size)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ") PageSize (" << size << ").\n";

        m_current_server_instance = (int) siid;

        update_lookup_table_page_size ((int) siid, size); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_reset_server_instance (int /*client_id*/)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_reset_server_instance.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ").\n";

        m_current_server_instance = (int) siid;

        reset_server_instance ((int) siid); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_focus_in_server_instance (int client_id)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_focus_in_server_instance.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ").\n";

        m_current_server_instance = (int) siid;

        focus_in_server_instance ((int) siid); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_focus_out_server_instance (int /*client_id*/)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_focus_out_server_instance.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ").\n";

        m_current_server_instance = (int) siid;

        focus_out_server_instance ((int) siid); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_toggle_full_width_punctuation (int /*client_id*/)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_toggle_full_width_punctuation.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ").\n";

        m_current_server_instance = (int) siid;

        toggle_full_width_punctuation ((int) siid); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_toggle_full_width_letter (int /*client_id*/)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_toggle_full_width_letter.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ").\n";

        m_current_server_instance = (int) siid;

        toggle_full_width_letter ((int) siid); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_toggle_input_status (int /*client_id*/)
{
    uint32 siid;

    SCIM_DEBUG_FRONTEND (2) << " socket_toggle_input_status.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (siid)) {

        SCIM_DEBUG_FRONTEND (3) << "  SI (" << siid << ").\n";

        m_current_server_instance = (int) siid;

        toggle_input_status ((int) siid); 
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);

        m_current_server_instance = -1;
    }
}

void
SocketFrontEnd::socket_flush_config (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    SCIM_DEBUG_FRONTEND (2) << " socket_flush_config.\n";

    if (m_config->flush ())
        m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
}

void
SocketFrontEnd::socket_erase_config (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_erase_config.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {

        SCIM_DEBUG_FRONTEND (3) << "  Key   (" << key << ").\n";

        if (m_config->erase (key))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_config_string (int /*client_id*/)
{
    if (m_config.null ()) return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_config_string.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {
        String value;

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        if (m_config->read (key, &value)) {
            m_send_transaction.put_data (value);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_set_config_string (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;
    String value;

    SCIM_DEBUG_FRONTEND (2) << " socket_set_config_string.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (value)) {

        SCIM_DEBUG_FRONTEND (3) << "  Key   (" << key << ").\n";
        SCIM_DEBUG_FRONTEND (3) << "  Value (" << value << ").\n";

        if (m_config->write (key, value))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_config_int (int /*client_id*/)
{
    if (m_config.null ()) return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_config_int.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        int value;
        if (m_config->read (key, &value)) {
            m_send_transaction.put_data ((uint32) value);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_set_config_int (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;
    uint32 value;

    SCIM_DEBUG_FRONTEND (2) << " socket_set_config_int.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (value)) {

        SCIM_DEBUG_FRONTEND (3) << "  Key   (" << key << ").\n";
        SCIM_DEBUG_FRONTEND (3) << "  Value (" << value << ").\n";

        if (m_config->write (key, (int) value))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_config_bool (int /*client_id*/)
{
    if (m_config.null ()) return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_config_bool.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {
        bool value;

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        if (m_config->read (key, &value)) {
            m_send_transaction.put_data ((uint32) value);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_set_config_bool (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;
    uint32 value;

    SCIM_DEBUG_FRONTEND (2) << " socket_set_config_bool.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        m_receive_transaction.get_data (value)) {

        SCIM_DEBUG_FRONTEND (3) << "  Key   (" << key << ").\n";
        SCIM_DEBUG_FRONTEND (3) << "  Value (" << value << ").\n";

        if (m_config->write (key, (bool) value))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_config_double (int /*client_id*/)
{
    if (m_config.null ()) return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_config_double.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {
        double value;

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        if (m_config->read (key, &value)) {
            char buf [80];
            snprintf (buf, 79, "%lE", value);
            m_send_transaction.put_data (String (buf));
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_set_config_double (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;
    String str;

    SCIM_DEBUG_FRONTEND (2) << " socket_set_config_double.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (str)) {
        double value;
        sscanf (str.c_str (), "%lE", &value);

        SCIM_DEBUG_FRONTEND (3) << "  Key   (" << key << ").\n";
        SCIM_DEBUG_FRONTEND (3) << "  Value (" << value << ").\n";

        if (m_config->write (key, value))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_config_vector_string (int /*client_id*/)
{
    if (m_config.null ()) return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_config_vector_string.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {
        std::vector <String> vec;

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        if (m_config->read (key, &vec)) {
            m_send_transaction.put_data (vec);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_set_config_vector_string (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;
    std::vector<String> vec;

    SCIM_DEBUG_FRONTEND (2) << " socket_set_config_vector_string.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_VECTOR_STRING &&
        m_receive_transaction.get_data (vec)) {

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        if (m_config->write (key, vec))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_get_config_vector_int (int /*client_id*/)
{
    if (m_config.null ()) return;

    String key;

    SCIM_DEBUG_FRONTEND (2) << " socket_get_config_vector_int.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key)) {
        std::vector <int> vec;

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        if (m_config->read (key, &vec)) {
            std::vector <uint32> reply;

            for (uint32 i=0; i<vec.size (); ++i)
                reply.push_back ((uint32) vec[i]);

            m_send_transaction.put_data (reply);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }
    }
}

void
SocketFrontEnd::socket_set_config_vector_int (int /*client_id*/)
{
    if (m_config_readonly || m_config.null ())
        return;

    String key;
    std::vector<uint32> vec;

    SCIM_DEBUG_FRONTEND (2) << " socket_set_config_vector_int.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (key) &&
        m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_VECTOR_UINT32 &&
        m_receive_transaction.get_data (vec)) {
        std::vector<int> req;

        SCIM_DEBUG_FRONTEND (3) << "  Key (" << key << ").\n";

        for (uint32 i=0; i<vec.size (); ++i)
            req.push_back ((int) vec[i]);

        if (m_config->write (key, req))
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
    }
}

void
SocketFrontEnd::socket_load_file (int /*client_id*/)
{
    String filename;
    char *bufptr = 0;
    size_t filesize = 0;

    SCIM_DEBUG_FRONTEND (2) << " socket_load_file.\n";

    if (m_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        m_receive_transaction.get_data (filename)) {
        SCIM_DEBUG_FRONTEND (3) << "  File (" << filename << ").\n";

        if ((filesize = scim_load_file (filename, &bufptr)) > 0) {
            m_send_transaction.put_data (bufptr, filesize);
            m_send_transaction.put_command (SCIM_TRANS_CMD_OK);
        }

        delete [] bufptr;
    }
}

/*
vi:ts=4:nowrap:expandtab
*/
