/** @file scim_socket_server.cpp
 * implementation of class SocketServerFactory and SocketServerInstance.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_server.cpp,v 1.16 2004/04/18 07:01:20 suzhe Exp $
 *
 */

#define Uses_SCIM_SERVER
#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_SOCKET
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_C_STDLIB

#include "scim_private.h"
#include "scim.h"
#include "scim_socket_server.h"
#include <unistd.h>

#define scim_module_init                  socket_LTX_scim_module_init
#define scim_module_exit                  socket_LTX_scim_module_exit
#define scim_server_module_init           socket_LTX_scim_server_module_init
#define scim_server_module_create_factory socket_LTX_scim_server_module_create_factory

#define SCIM_CONFIG_SERVER_SOCKET_TIMEOUT "/Server/Socket/Timeout"
#define SCIM_CONFIG_SERVER_SOCKET_ADDRESS "/Server/Socket/Address"

#define SCIM_SOCKET_FRONTEND_DEF_ADDRESS "local:/tmp/scim-socket-frontend"

#ifndef SCIM_TEMPDIR
  #define SCIM_TEMPDIR "/tmp"
#endif

using namespace scim;

extern "C" {
    void scim_module_init (void)
    {
    }

    void scim_module_exit (void)
    {
        SocketServerFactory::destroy_global ();
    }

    unsigned int scim_server_module_init (const ConfigPointer &config)
    {
        return SocketServerFactory::init_global (config);
    }

    ServerFactoryPointer scim_server_module_create_factory (unsigned int server)
    {
        if (server >= SocketServerFactory::number_of_server_factories ())
            return 0;

        SocketServerFactory *sf = SocketServerFactory::create_server_factory (server);

        if (!sf || !sf->valid ()) {
            delete sf;
            sf = 0;
        }

        return sf;
    }
}

SocketClient  SocketServerFactory::m_socket_client;
SocketAddress SocketServerFactory::m_socket_address;
uint32        SocketServerFactory::m_socket_magic_key = 0;
int           SocketServerFactory::m_socket_timeout = -1;
volatile int  SocketServerFactory::m_socket_lock = 0;

std::vector<String> SocketServerFactory::m_peer_server_factories;

unsigned int
SocketServerFactory::init_global (const ConfigPointer &config)
{
    SCIM_DEBUG_SERVER(1) << "Init SocketServerFactory Global.\n";

    String address;

    if (!config.null ()) {
        m_socket_timeout = config->read (String (SCIM_CONFIG_SERVER_SOCKET_TIMEOUT),
                                         (int) 500);

        address = config->read (String (SCIM_CONFIG_SERVER_SOCKET_ADDRESS),
                                String (SCIM_SOCKET_FRONTEND_DEF_ADDRESS));
    }

    const char *env = getenv ("SCIM_SOCKET_ADDRESS");
    if (env) {
        address = String (env);
    } else {
        env = getenv ("SCIM_SERVER_SOCKET_ADDRESS");
        if (env)
            address = String (env);
    }

    env = getenv ("SCIM_SOCKET_TIMEOUT");
    if (env) {
        m_socket_timeout = atoi (env);
    } else {
        env = getenv ("SCIM_SERVER_SOCKET_TIMEOUT");
        if (env)
            m_socket_timeout = atoi (env);
    }

    if (address == "default")
        address = SCIM_SOCKET_FRONTEND_DEF_ADDRESS;

    m_socket_address.set_address (address);

    if (!m_socket_address.valid ())
        return 0;

    // Connect to SocketFrontEnd.
    if (!create_connection ()) {
        SCIM_DEBUG_SERVER(2) << " Cannot connect to SocketFrontEnd (" << address << ").\n";
        return 0;
    }

    SCIM_DEBUG_SERVER(2) << " Connected to SocketFrontEnd (" << address
                         << ") MagicKey (" << m_socket_magic_key << ").\n";

    // Init the connection, and get ServerFactory list.
    SocketTransaction trans;

    // Get ServerFactory list.
    init_transaction (trans);

    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_LIST);
    trans.put_data (String (""));
    if (!send_transaction (trans))
        return 0;

    int cmd;
    if (trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
        trans.get_data (m_peer_server_factories) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {

        SCIM_DEBUG_SERVER(2) << " Found " << m_peer_server_factories.size ()
            << " Server Factories.\n";

        return m_peer_server_factories.size ();
    }
    return 0;
}

bool
SocketServerFactory::create_connection ()
{
    // Connect to SocketFrontEnd.
    if (!m_socket_client.connect (m_socket_address))
        return false;

    // Init the connection, and get ServerFactory list.
    SocketTransaction trans;

    trans.put_command (SCIM_TRANS_CMD_REQUEST);
    trans.put_command (SCIM_TRANS_CMD_OPEN_CONNECTION);

    if (!trans.write_to_socket (m_socket_client))
        return false;

    int cmd;
    if (trans.read_from_socket (m_socket_client, m_socket_timeout) &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
        trans.get_data (m_socket_magic_key)) {

        trans.clear ();
        trans.put_command (SCIM_TRANS_CMD_REPLY);
        trans.put_command (SCIM_TRANS_CMD_OK);
        if (!trans.write_to_socket (m_socket_client))
            return false;
    }
    return true;
}

void
SocketServerFactory::destroy_global ()
{
    m_socket_client.close ();
}

unsigned int
SocketServerFactory::number_of_server_factories ()
{
    return m_peer_server_factories.size ();
}

SocketServerFactory *
SocketServerFactory::create_server_factory (unsigned int server)
{
    if (server < m_peer_server_factories.size ()) {
        return new SocketServerFactory (m_peer_server_factories [server]);
    }
    return 0;
}

void
SocketServerFactory::init_transaction (SocketTransaction &trans)
{
    trans.clear ();
    trans.put_command (SCIM_TRANS_CMD_REQUEST);
    trans.put_data (m_socket_magic_key);
}

bool
SocketServerFactory::send_transaction (SocketTransaction &trans)
{
    return trans.write_to_socket (m_socket_client);
}

bool
SocketServerFactory::receive_transaction (SocketTransaction &trans)
{
    return trans.read_from_socket (m_socket_client, m_socket_timeout);
}

bool
SocketServerFactory::lock_socket_client ()
{
    while (m_socket_lock > 0) NULL;
    ++ m_socket_lock;
    return true;
}

bool
SocketServerFactory::unlock_socket_client ()
{
    if (m_socket_lock <= 0) return false;
    -- m_socket_lock;
    return true;
}

SocketServerFactory::SocketServerFactory (const String &peer_uuid)
    : m_name (utf8_mbstowcs (_("Unknown"))),
      m_peer_uuid (peer_uuid),
      m_ok (false),
      m_icon_file_should_delete (false)
{
    SocketTransaction trans;
    String locales;
    int cmd;
    bool m_name_ok = false;
    bool m_locale_ok = false;

    SCIM_DEBUG_SERVER(1) << "Create SocketServerFactory " << peer_uuid << ".\n";

    // Get factory name.
    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_NAME);
    trans.put_data (m_peer_uuid);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
              trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
              trans.get_data (m_name)) {
            if (!(trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)) {
                m_name = utf8_mbstowcs (_("Unknown"));
            } else {
                m_name_ok = true;
            }
        }
    }

    // Get factory locales
    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_LOCALES);
    trans.put_data (m_peer_uuid);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (locales) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {

            SCIM_DEBUG_SERVER(2) << " Locales (" << locales << ")\n";

            set_locales (locales);
            m_locale_ok = true;
        }
    }

    init_icon_file ();

    unlock_socket_client ();

    m_ok = (m_name_ok && m_locale_ok);
}

SocketServerFactory::~SocketServerFactory ()
{
    if (m_icon_file_should_delete)
        unlink (m_icon_file.c_str ());
}

void
SocketServerFactory::init_icon_file ()
{
    SocketTransaction trans;
    String iconfile;
    int cmd;

    // Get factory icon file
    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_ICON_FILE);
    trans.put_data (m_peer_uuid);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (iconfile) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {

            // The file is not available on local machine.
            // Get the file and store it to local machine.
            if (scim_load_file (iconfile, 0) == 0) {
                init_transaction (trans);
                trans.put_command (SCIM_TRANS_CMD_LOAD_FILE);
                trans.put_data (iconfile);
                if (send_transaction (trans)) {
                    char *bufptr = 0;
                    size_t filesize = 0;

                    if (receive_transaction (trans) &&
                        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
                        trans.get_data (&bufptr, filesize) &&
                        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {

                        // Save the icon to a temporary file.
                        String tempfile;
                        String::size_type pos = iconfile.rfind (SCIM_PATH_DELIM);
                        if (pos != String::npos) {
                            tempfile = iconfile.substr (pos + 1, String::npos);
                        } else {
                            tempfile = iconfile;
                        }

                        iconfile = String (SCIM_TEMPDIR) +
                                   String (SCIM_PATH_DELIM_STRING) +
                                   m_peer_uuid + String ("-") +
                                   tempfile;

                        SCIM_DEBUG_SERVER(1) << "Creating temporary icon file: " << iconfile << "\n";

                        std::ofstream os (iconfile.c_str ());

                        if (os) {
                            os.write (bufptr, filesize);
                            os.close ();
                            // Check if the file is written correctly.
                            if (scim_load_file (iconfile, 0) == filesize) {
                                m_icon_file = iconfile;
                                m_icon_file_should_delete = true;
                            } else {
                                unlink (iconfile.c_str ());
                            }
                        }
                        delete [] bufptr;
                    }
                }
            } else { //The icon file is already in local machine.
                m_icon_file = iconfile;
                m_icon_file_should_delete = false;
            }
        }
    }
}

WideString
SocketServerFactory::get_name () const
{
    return m_name;
}

WideString
SocketServerFactory::get_authors () const
{
    SocketTransaction trans;
    int cmd;
    WideString authors = utf8_mbstowcs (_("Unknown"));

    lock_socket_client ();

    SCIM_DEBUG_SERVER(1) << "Get Authors " << m_peer_uuid << ".\n";

    // Get factory authors.
    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_AUTHORS);
    trans.put_data (m_peer_uuid);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
              trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
              trans.get_data (authors)) {
            if (!(trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)) {
                authors = utf8_mbstowcs (_("Unknown"));
            }
        }
    }

    unlock_socket_client ();

    return authors;
}

WideString
SocketServerFactory::get_credits () const
{
    SocketTransaction trans;
    int cmd;
    WideString credits = utf8_mbstowcs (_("Unknown"));

    lock_socket_client ();

    SCIM_DEBUG_SERVER(1) << "Get Credits " << m_peer_uuid << ".\n";

    // Get factory credits.
    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_CREDITS);
    trans.put_data (m_peer_uuid);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
              trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
              trans.get_data (credits)) {
            if (!(trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)) {
                credits = utf8_mbstowcs (_("Unknown"));
            }
        }
    }

    unlock_socket_client ();

    return credits;
}

WideString
SocketServerFactory::get_help () const
{
    SocketTransaction trans;
    int cmd;
    WideString help = utf8_mbstowcs (_("Unknown"));

    lock_socket_client ();

    SCIM_DEBUG_SERVER(1) << "Get Help " << m_peer_uuid << ".\n";

    // Get factory help.
    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_GET_SERVER_FACTORY_HELP);
    trans.put_data (m_peer_uuid);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
              trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
              trans.get_data (help)) {
            if (!(trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK)) {
                help = utf8_mbstowcs (_("Unknown"));
            }
        }
    }

    unlock_socket_client ();

    return help;
}

String
SocketServerFactory::get_uuid () const
{
    return m_peer_uuid;
}

String
SocketServerFactory::get_icon_file () const
{
    return m_icon_file;
}

ServerInstancePointer
SocketServerFactory::create_server_instance (const String& encoding, int id)
{
    SocketTransaction trans;
    int cmd;
    ServerInstancePointer si;
    uint32 val;
    int si_peer_id = -1;

    lock_socket_client ();

    SCIM_DEBUG_SERVER(1) << "Create Server Instance " << m_peer_uuid << ".\n";

    init_transaction (trans);
    trans.put_command (SCIM_TRANS_CMD_NEW_SERVER_INSTANCE);
    trans.put_data (m_peer_uuid);
    trans.put_data (encoding);
    if (send_transaction (trans)) {
        if (receive_transaction (trans) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY &&
            trans.get_data (val) &&
            trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_OK) {

            si_peer_id = (int) val;
        }
    }

    unlock_socket_client ();

    SCIM_DEBUG_SERVER(2) << " ServerInstance created (" << si_peer_id << ")\n";

    return new SocketServerInstance (this, encoding, id, si_peer_id);
}

SocketServerInstance::SocketServerInstance (SocketServerFactory *factory,
                                            const String& encoding,
                                            int           id,
                                            int           peer_id)
    : ServerInstanceBase (factory, encoding, id),
      m_peer_id (peer_id)
{
}

SocketServerInstance::~SocketServerInstance ()
{
    SCIM_DEBUG_SERVER(1) << "Destroy Server Instance " << m_peer_id << ".\n";

    if (m_peer_id >= 0) {
        SocketTransaction trans;
        SocketServerFactory::init_transaction (trans);

        trans.put_command (SCIM_TRANS_CMD_DELETE_SERVER_INSTANCE);
        trans.put_data (m_peer_id);

        SocketServerFactory::lock_socket_client ();
        SocketServerFactory::send_transaction (trans);
        SocketServerFactory::receive_transaction (trans);
        SocketServerFactory::unlock_socket_client ();
    }
}

bool
SocketServerInstance::process_key_event (const KeyEvent& key)
{
    if (m_peer_id < 0) return false;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "process_key_event (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_PROCESS_KEYEVENT);
    trans.put_data (m_peer_id);
    trans.put_data (key);

    return commit_transaction (trans);
}

void
SocketServerInstance::move_preedit_caret (unsigned int pos)
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "move_preedit_caret (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_MOVE_PREEDIT_CARET);
    trans.put_data (m_peer_id);
    trans.put_data ((uint32) pos);

    commit_transaction (trans);
}

void
SocketServerInstance::select_lookup_table (unsigned int item)
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "select_lookup_table (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_SELECT_LOOKUP_TABLE);
    trans.put_data (m_peer_id);
    trans.put_data ((uint32) item);

    commit_transaction (trans);
}

void
SocketServerInstance::update_lookup_table_page_size (unsigned int page_size)
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "update_lookup_table_page_size (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE);
    trans.put_data (m_peer_id);
    trans.put_data ((uint32) page_size);

    commit_transaction (trans);
}

void
SocketServerInstance::reset ()
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "reset (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_RESET);
    trans.put_data (m_peer_id);

    commit_transaction (trans);
}

void
SocketServerInstance::focus_in ()
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "focus_in (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_FOCUS_IN);
    trans.put_data (m_peer_id);

    commit_transaction (trans);
}

void
SocketServerInstance::focus_out ()
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "focus_out (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_FOCUS_OUT);
    trans.put_data (m_peer_id);

    commit_transaction (trans);
}

void
SocketServerInstance::toggle_full_width_punctuation ()
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "toggle_full_width_punctuation (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_PUNCTUATION);
    trans.put_data (m_peer_id);

    commit_transaction (trans);
}

void
SocketServerInstance::toggle_full_width_letter ()
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "toggle_full_width_letter (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_LETTER);
    trans.put_data (m_peer_id);

    commit_transaction (trans);
}

void
SocketServerInstance::toggle_input_status ()
{
    if (m_peer_id < 0) return;

    SocketTransaction trans;
    SocketServerFactory::init_transaction (trans);

    SCIM_DEBUG_SERVER(1) << "toggle_input_status (" << m_peer_id << ")\n";

    trans.put_command (SCIM_TRANS_CMD_TOGGLE_INPUT_STATUS);
    trans.put_data (m_peer_id);

    commit_transaction (trans);
}

bool
SocketServerInstance::commit_transaction (SocketTransaction &trans)
{
    if (m_peer_id < 0) return false;

    bool ret = false;

    SocketServerFactory::lock_socket_client ();

    if (SocketServerFactory::send_transaction (trans) &&
        SocketServerFactory::receive_transaction (trans)) {
        SocketServerFactory::unlock_socket_client ();
        ret = do_transaction (trans);
    } else {
        SocketServerFactory::unlock_socket_client ();
    }

    return ret;
}

bool
SocketServerInstance::do_transaction (SocketTransaction &trans)
{
    int cmd;

    SCIM_DEBUG_SERVER(2) << " Do transaction:\n";

    if (trans.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
        trans.get_command (cmd) && cmd == SCIM_TRANS_CMD_REPLY) {

        while (trans.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
               trans.get_command (cmd)) {
            switch (cmd) {
                case SCIM_TRANS_CMD_SHOW_PREEDIT_STRING:
                {
                    SCIM_DEBUG_SERVER(3) << "  show_preedit_string ()\n";
                    show_preedit_string ();
                    break;
                }
                case SCIM_TRANS_CMD_SHOW_STATUS_STRING:
                {
                    SCIM_DEBUG_SERVER(3) << "  show_status_string ()\n";
                    show_status_string ();
                    break;
                }
                case SCIM_TRANS_CMD_SHOW_AUX_STRING:
                {
                    SCIM_DEBUG_SERVER(3) << "  show_aux_string ()\n";
                    show_aux_string ();
                    break;
                }
                case SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE:
                {
                    SCIM_DEBUG_SERVER(3) << "  show_lookup_table ()\n";
                    show_lookup_table ();
                    break;
                }
                case SCIM_TRANS_CMD_HIDE_PREEDIT_STRING:
                {
                    SCIM_DEBUG_SERVER(3) << "  hide_preedit_string ()\n";
                    hide_preedit_string ();
                    break;
                }
                case SCIM_TRANS_CMD_HIDE_STATUS_STRING:
                {
                    SCIM_DEBUG_SERVER(3) << "  hide_status_string ()\n";
                    hide_status_string ();
                    break;
                }
                case SCIM_TRANS_CMD_HIDE_AUX_STRING:
                {
                    SCIM_DEBUG_SERVER(3) << "  hide_aux_string ()\n";
                    hide_aux_string ();
                    break;
                }
                case SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE:
                {
                    SCIM_DEBUG_SERVER(3) << "  hide_lookup_table ()\n";
                    hide_lookup_table ();
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET:
                {
                    uint32 caret;
                    if (trans.get_data (caret)) {
                        SCIM_DEBUG_SERVER(3) << "  update_preedit_caret (" << caret << ")\n";
                        update_preedit_caret (caret);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING:
                {
                    WideString str;
                    AttributeList attrs;
                    if (trans.get_data (str) && trans.get_data (attrs)) {
                        SCIM_DEBUG_SERVER(3) << "  update_preedit_string ()\n";
                        update_preedit_string (str, attrs);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_STATUS_STRING:
                {
                    WideString str;
                    AttributeList attrs;
                    if (trans.get_data (str) && trans.get_data (attrs)) {
                        SCIM_DEBUG_SERVER(3) << "  update_status_string ()\n";
                        update_status_string (str, attrs);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_AUX_STRING:
                {
                    WideString str;
                    AttributeList attrs;
                    if (trans.get_data (str) && trans.get_data (attrs)) {
                        SCIM_DEBUG_SERVER(3) << "  update_aux_string ()\n";
                        update_aux_string (str, attrs);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE:
                {
                    CommonLookupTable table;
                    if (trans.get_data (table)) {
                        SCIM_DEBUG_SERVER(3) << "  update_lookup_table ()\n";
                        update_lookup_table (table);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_COMMIT_STRING:
                {
                    WideString str;
                    if (trans.get_data (str)) {
                        SCIM_DEBUG_SERVER(3) << "  commit_string ()\n";
                        commit_string (str);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_FORWARD_KEYEVENT:
                {
                    KeyEvent key;
                    if (trans.get_data (key)) {
                        SCIM_DEBUG_SERVER(3) << "  forward_keyevent ()\n";
                        forward_keyevent (key);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_PUNCTUATION:
                {
                    uint32 full;
                    if (trans.get_data (full)) {
                        SCIM_DEBUG_SERVER(3) << "  update_full_width_punctuation ()\n";
                        update_full_width_punctuation ((bool) full);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_LETTER:
                {
                    uint32 full;
                    if (trans.get_data (full)) {
                        SCIM_DEBUG_SERVER(3) << "  update_full_width_letter ()\n";
                        update_full_width_letter ((bool) full);
                    }
                    break;
                }
                case SCIM_TRANS_CMD_OK:
                {
                    SCIM_DEBUG_SERVER(3) << "  ret = true\n";
                    return true;
                }
                default:
                {
                    SCIM_DEBUG_SERVER(3) << "  ret = false\n";
                    return false;
                }
            }
        }
    }
    return false;
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
