/** @file scim_table_server.h
 * definition of TableServer related classes.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_table_server.h,v 1.31 2004/04/11 10:53:26 suzhe Exp $
 */

#if !defined (__SCIM_TABLE_SERVER_H)
#define __SCIM_TABLE_SERVER_H
#include "scim_generic_table.h"

/* phrase frequency cannot larger than this value (2^16 - 1) */
#define SCIM_GT_MAX_PHRASE_FREQ            0xFFFF

/* when a phrase is input,
 * increase the freq by ((max_freq - cur_freq) >> delta)
 */
#define SCIM_GT_PHRASE_FREQ_DELTA_SHIFT    10

using namespace scim;

class TableServerFactory : public ServerFactoryBase
{
    GenericTableLibrary   m_table;

    ConfigPointer         m_config;

    std::vector<KeyEvent> m_full_width_punct_keys;
    std::vector<KeyEvent> m_full_width_letter_keys;
    std::vector<KeyEvent> m_mode_switch_keys;
    std::vector<KeyEvent> m_add_phrase_keys;
    std::vector<KeyEvent> m_del_phrase_keys;

    String                m_table_filename;

    bool                  m_is_user_table;

    bool                  m_show_prompt;

    bool                  m_show_key_hint;

    bool                  m_user_table_binary;

    bool                  m_user_phrase_first;

    bool                  m_long_phrase_first;

    time_t                m_last_time;

    friend class TableServerInstance;

public:
    TableServerFactory (const ConfigPointer &config);

    virtual ~TableServerFactory ();

    virtual WideString  get_name () const;
    virtual WideString  get_authors () const;
    virtual WideString  get_credits () const;
    virtual WideString  get_help () const;
    virtual String      get_uuid () const;
    virtual String      get_icon_file () const;

    virtual ServerInstancePointer create_server_instance (const String& encoding, int id = -1);

    bool load_table (const String &table_file, bool user_table=false);

    bool valid () const {
        return m_table.valid ();
    }

private:
    void init ();

    void save ();

    String get_sys_table_freq_file ();
    String get_sys_table_user_file ();

    void refresh ();
};

class TableServerInstance : public ServerInstanceBase
{
    Pointer <TableServerFactory>  m_factory;

    bool m_double_quotation_state;
    bool m_single_quotation_state;

    bool m_full_width_punctuation [2];
    bool m_full_width_letter [2];

    bool m_forward;
    bool m_focused;

    std::vector<String>     m_inputed_keys;
    std::vector<WideString> m_converted_strings;
    std::vector<uint32>     m_converted_indexes;

    CommonLookupTable       m_lookup_table;
    std::vector <uint32>    m_lookup_table_indexes;

    uint32                  m_inputing_caret;
    uint32                  m_inputing_key;

    IConvert                m_iconv;

    KeyEvent                m_prev_key;

    // 0 : Normal input mode
    // 1 : Add phrase mode
    // 2 : Add phrase success
    // 3 : Add phrase failed
    int                     m_add_phrase_mode;

    WideString              m_last_committed;

public:
    TableServerInstance (TableServerFactory *factory,
                         const String& encoding,
                         int id = -1);
    virtual ~TableServerInstance ();

    virtual bool process_key_event (const KeyEvent& key);
    virtual void move_preedit_caret (unsigned int pos);
    virtual void select_lookup_table (unsigned int item);
    virtual void update_lookup_table_page_size (unsigned int page_size);
    virtual void reset ();
    virtual void focus_in ();
    virtual void focus_out ();
    virtual void toggle_full_width_punctuation ();
    virtual void toggle_full_width_letter ();
    virtual void toggle_input_status ();

private:
    bool caret_left ();
    bool caret_right ();
    bool caret_home ();
    bool caret_end ();
    bool insert (char key);
    bool test_insert (char key);
    bool erase (bool backspace = true);
    bool space_hit ();
    bool enter_hit ();
    bool lookup_cursor_up ();
    bool lookup_cursor_down ();
    bool lookup_cursor_up_to_longer ();
    bool lookup_cursor_down_to_shorter ();
    bool lookup_page_up ();
    bool lookup_page_down ();
    bool lookup_select (int index, const KeyEvent &key);
    bool delete_phrase ();
    bool post_process (char key);
    void lookup_to_converted (int index);
    void commit_converted ();

    void refresh_preedit ();
    void refresh_status_string ();
    /**
     * @param show if show the lookup table
     * @param refresh if refresh the content of lookup table.
     */
    void refresh_lookup_table (bool show=true, bool refresh=true);
    void refresh_aux_string ();

    bool match_key_event (const std::vector<KeyEvent>& keyvec, const KeyEvent& key);
};

#endif
/*
vi:ts=4:nowrap:ai:expandtab
*/

