/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_frontend.cpp,v 1.25 2003/12/31 05:09:24 suzhe Exp $
 *
 */

#define Uses_SCIM_FRONTEND
#include "scim_private.h"
#include "scim.h"

namespace scim {

class __ServerInstanceRepositoryLessThanByID
{
public:
    bool operator() (const ServerInstancePointer & lhs,
                    const ServerInstancePointer & rhs) const {
        return lhs->get_id () < rhs->get_id ();
    }
    bool operator() (const ServerInstancePointer & lhs, int rhs) const {
        return lhs->get_id () < rhs;
    }
    bool operator() (int lhs, const ServerInstancePointer & rhs) const {
        return lhs < rhs->get_id ();
    }
};

class __ServerFactoryRepositoryLessThanByUUID
{
public:
    bool operator() (const ServerFactoryPointer & lhs,
                    const ServerFactoryPointer & rhs) const {
        return lhs->get_uuid () < rhs->get_uuid ();
    }
    bool operator() (const ServerFactoryPointer & lhs, const String &rhs) const {
        return lhs->get_uuid () < rhs;
    }
    bool operator() (const String &lhs, const ServerFactoryPointer & rhs) const {
        return lhs < rhs->get_uuid ();
    }
};

FrontEndBase::FrontEndBase (const BackEndPointer &backend)
    : m_backend (backend),
      m_server_instance_id_count (0)
{
    if (m_backend.null () || m_backend->number_of_servers () == 0)
        throw FrontEndError ("Invalid BackEnd!\n");

    query_server_factories ();
}

FrontEndBase::~FrontEndBase ()
{
    delete_all_server_instances ();
}

void
FrontEndBase::slot_show_preedit_string (ServerInstanceBase * si)
{
    if (si)
        show_preedit_string (si->get_id ());
}

void
FrontEndBase::slot_show_status_string (ServerInstanceBase * si)
{
    if (si)
        show_status_string (si->get_id ());
}

void
FrontEndBase::slot_show_aux_string (ServerInstanceBase * si)
{
    if (si)
        show_aux_string (si->get_id ());
}

void
FrontEndBase::slot_show_lookup_table (ServerInstanceBase * si)
{
    if (si)
        show_lookup_table (si->get_id ());
}

void
FrontEndBase::slot_hide_preedit_string (ServerInstanceBase * si)
{
    if (si)
        hide_preedit_string (si->get_id ());
}

void
FrontEndBase::slot_hide_status_string (ServerInstanceBase * si)
{
    if (si)
        hide_status_string (si->get_id ());
}

void
FrontEndBase::slot_hide_aux_string (ServerInstanceBase * si)
{
    if (si)
        hide_aux_string (si->get_id ());
}

void
FrontEndBase::slot_hide_lookup_table (ServerInstanceBase * si)
{
    if (si)
        hide_lookup_table (si->get_id ());
}

void
FrontEndBase::slot_update_preedit_caret (ServerInstanceBase * si, int caret)
{
    if (si)
        update_preedit_caret (si->get_id (), caret);
}

void
FrontEndBase::slot_update_preedit_string (ServerInstanceBase * si, const WideString & str, const AttributeList & attrs)
{
    if (si)
        update_preedit_string (si->get_id (), str, attrs);
}

void
FrontEndBase::slot_update_status_string (ServerInstanceBase * si, const WideString & str, const AttributeList & attrs)
{
    if (si)
        update_status_string (si->get_id (), str, attrs);
}

void
FrontEndBase::slot_update_aux_string (ServerInstanceBase * si, const WideString & str, const AttributeList & attrs)
{
    if (si)
        update_aux_string (si->get_id (), str, attrs);
}

void
FrontEndBase::slot_update_lookup_table (ServerInstanceBase * si, const LookupTable & table)
{
    if (si)
        update_lookup_table (si->get_id (), table);
}

void
FrontEndBase::slot_commit_string (ServerInstanceBase * si, const WideString & str)
{
    if (si)
        commit_string (si->get_id (), str);
}

void
FrontEndBase::slot_forward_keyevent (ServerInstanceBase * si, const KeyEvent & key)
{
    if (si)
        forward_keyevent (si->get_id (), key);
}

void
FrontEndBase::slot_update_full_width_punctuation (ServerInstanceBase * si, bool full)
{
    if (si)
        update_full_width_punctuation (si->get_id (), full);
}

void
FrontEndBase::slot_update_full_width_letter (ServerInstanceBase * si, bool full)
{
    if (si)
        update_full_width_letter (si->get_id (), full);
}

void
FrontEndBase::query_server_factories ()
{
    ServerFactoryPointer server;

    for (unsigned int i=0; i<m_backend->number_of_servers (); ++i) {
        server = m_backend->get_server_factory (i);
        if (!server.null ())
            m_server_factory_repository.push_back (server);
    }

    std::sort (m_server_factory_repository.begin (),
               m_server_factory_repository.end (),
               __ServerFactoryRepositoryLessThanByUUID ());
}

ServerFactoryPointer
FrontEndBase::find_server_factory (const String &uuid) const
{
    if (m_server_factory_repository.size ()) {
        ServerFactoryRepository::const_iterator it =
            std::lower_bound (m_server_factory_repository.begin (),
                              m_server_factory_repository.end (),
                              uuid,
                              __ServerFactoryRepositoryLessThanByUUID ());

        if (it != m_server_factory_repository.end () && (*it)->get_uuid () == uuid)
            return *it;
    }
    return ServerFactoryPointer (0);
}

ServerInstancePointer
FrontEndBase::find_server_instance (int id) const
{
    if (m_server_instance_repository.size ()) {
        ServerInstanceRepository::const_iterator it =
            std::lower_bound (m_server_instance_repository.begin (),
                              m_server_instance_repository.end (),
                              id,
                              __ServerInstanceRepositoryLessThanByID ());

        if (it != m_server_instance_repository.end () && (*it)->get_id () == id)
            return *it;
    }
    return ServerInstancePointer (0);
}

void
FrontEndBase::add_server_instance (const ServerInstancePointer &si)
{
    if (si.null ()) return;

    ServerInstanceRepository::iterator it =
        std::lower_bound (m_server_instance_repository.begin (),
                          m_server_instance_repository.end (),
                          si->get_id (),
                          __ServerInstanceRepositoryLessThanByID ());

    if (it != m_server_instance_repository.end ()) {
        if ((*it)->get_id () == si->get_id ())
            *it = si;
        else 
            m_server_instance_repository.insert (it, si);
    } else {
        m_server_instance_repository.push_back (si);
    }
}

void
FrontEndBase::attach_server_instance (const ServerInstancePointer &si)
{
    if (si.null ()) return;

    si->signal_connect_show_preedit_string (
        slot (this, &FrontEndBase::slot_show_preedit_string));
    si->signal_connect_show_status_string (
        slot (this, &FrontEndBase::slot_show_status_string));
    si->signal_connect_show_aux_string (
        slot (this, &FrontEndBase::slot_show_aux_string));
    si->signal_connect_show_lookup_table (
        slot (this, &FrontEndBase::slot_show_lookup_table));

    si->signal_connect_hide_preedit_string (
        slot (this, &FrontEndBase::slot_hide_preedit_string));
    si->signal_connect_hide_status_string (
        slot (this, &FrontEndBase::slot_hide_status_string));
    si->signal_connect_hide_aux_string (
        slot (this, &FrontEndBase::slot_hide_aux_string));
    si->signal_connect_hide_lookup_table (
        slot (this, &FrontEndBase::slot_hide_lookup_table));

    si->signal_connect_update_preedit_caret (
        slot (this, &FrontEndBase::slot_update_preedit_caret));
    si->signal_connect_update_preedit_string (
        slot (this, &FrontEndBase::slot_update_preedit_string));
    si->signal_connect_update_status_string (
        slot (this, &FrontEndBase::slot_update_status_string));
    si->signal_connect_update_aux_string (
        slot (this, &FrontEndBase::slot_update_aux_string));
    si->signal_connect_update_lookup_table (
        slot (this, &FrontEndBase::slot_update_lookup_table));

    si->signal_connect_commit_string (
        slot (this, &FrontEndBase::slot_commit_string));

    si->signal_connect_forward_keyevent (
        slot (this, &FrontEndBase::slot_forward_keyevent));

    si->signal_connect_update_full_width_punctuation (
        slot (this, &FrontEndBase::slot_update_full_width_punctuation));

    si->signal_connect_update_full_width_letter (
        slot (this, &FrontEndBase::slot_update_full_width_letter));
}

uint32
FrontEndBase::get_server_factory_list (std::vector<String>& uuids, const String &encoding) const
{
    ServerFactoryRepository::const_iterator it;

    uuids.clear ();

    for (it = m_server_factory_repository.begin ();
         it != m_server_factory_repository.end (); ++it) {
        if ((encoding.length () == 0 || (*it)->validate_encoding (encoding)))
            uuids.push_back ((*it)->get_uuid ());
    }

    return uuids.size ();
}

WideString
FrontEndBase::get_server_factory_name (const String &uuid) const
{
    ServerFactoryPointer sf = find_server_factory (uuid);
    if (!sf.null ())
        return sf->get_name ();

    return WideString ();
}

WideString
FrontEndBase::get_server_factory_authors (const String &uuid) const
{
    ServerFactoryPointer sf = find_server_factory (uuid);
    if (!sf.null ())
        return sf->get_authors ();

    return WideString ();
}

WideString
FrontEndBase::get_server_factory_credits (const String &uuid) const
{
    ServerFactoryPointer sf = find_server_factory (uuid);
    if (!sf.null ())
        return sf->get_credits ();

    return WideString ();
}

WideString
FrontEndBase::get_server_factory_help (const String &uuid) const
{
    ServerFactoryPointer sf = find_server_factory (uuid);
    if (!sf.null ())
        return sf->get_help ();

    return WideString ();
}

String
FrontEndBase::get_server_factory_locales (const String &uuid) const
{
    ServerFactoryPointer sf = find_server_factory (uuid);
    if (!sf.null ())
        return sf->get_locales ();

    return String ();
}

String
FrontEndBase::get_server_factory_icon_file (const String &uuid) const
{
    ServerFactoryPointer sf = find_server_factory (uuid);
    if (!sf.null ())
        return sf->get_icon_file ();

    return String ();
}

String
FrontEndBase::get_all_locales () const
{
    return m_backend->get_locales ();
}

int
FrontEndBase::new_server_instance (const String &sf_uuid, const String& encoding)
{
    ServerFactoryPointer sf = find_server_factory (sf_uuid);

    if (sf.null () || !sf->validate_encoding (encoding)) {
        SCIM_DEBUG_FRONTEND(1) << "ServerFactory " << sf_uuid
            << " does not support encoding " << encoding << "\n";
        return -1;
    }

    ServerInstancePointer si =
        sf->create_server_instance (encoding, m_server_instance_id_count);

    if (si.null ()) {
        SCIM_DEBUG_FRONTEND(1) << "ServerFactory " << sf_uuid
            << " failed to create new instance!\n";
        return -1;
    }

    ++ m_server_instance_id_count;

    if (m_server_instance_id_count < 0)
        m_server_instance_id_count = 0;

    add_server_instance (si);
    attach_server_instance (si);

    return si->get_id ();
}

bool
FrontEndBase::replace_server_instance (int si_id, const String &sf_uuid)
{
    ServerFactoryPointer sf = find_server_factory (sf_uuid);

    if (sf.null ()) return false;

    if (m_server_instance_repository.size ()) {
        ServerInstanceRepository::iterator it =
            std::lower_bound (m_server_instance_repository.begin (),
                              m_server_instance_repository.end (),
                              si_id,
                              __ServerInstanceRepositoryLessThanByID ());

        if (it != m_server_instance_repository.end () && (*it)->get_id () == si_id) {
            String encoding = (*it)->get_encoding ();
            if (sf->validate_encoding (encoding)) {
                ServerInstancePointer si = sf->create_server_instance (encoding, si_id);
                if (!si.null ()) {
                    (*it) = si;
                    attach_server_instance (*it);
                    return true;
                }
            }
        }

        SCIM_DEBUG_FRONTEND(1) << "Cannot find Server Instance " << si_id << " to replace.\n";
    }

    return false;
}

bool
FrontEndBase::delete_server_instance (int id)
{
    if (m_server_instance_repository.size ()) {
        ServerInstanceRepository::iterator it =
            std::lower_bound (m_server_instance_repository.begin (),
                              m_server_instance_repository.end (),
                              id,
                              __ServerInstanceRepositoryLessThanByID ());

        if (it != m_server_instance_repository.end () && (*it)->get_id () == id) {
            m_server_instance_repository.erase (it);
            return true;
        }
    }

    return false;
}

void
FrontEndBase::delete_all_server_instances ()
{
    m_server_instance_repository.clear ();
}

String
FrontEndBase::get_server_instance_encoding (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return si->get_encoding ();

    return String ();
}

WideString
FrontEndBase::get_server_instance_name (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return get_server_factory_name (si->get_factory_uuid ());

    return WideString ();
}

WideString
FrontEndBase::get_server_instance_authors (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return get_server_factory_authors (si->get_factory_uuid ());

    return WideString ();
}

WideString
FrontEndBase::get_server_instance_credits (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return get_server_factory_credits (si->get_factory_uuid ());

    return WideString ();
}

WideString
FrontEndBase::get_server_instance_help (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return get_server_factory_help (si->get_factory_uuid ());

    return WideString ();
}

String
FrontEndBase::get_server_instance_icon_file (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return get_server_factory_icon_file (si->get_factory_uuid ());

    return String ();
}

void
FrontEndBase::focus_in_server_instance (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->focus_in ();
}

void
FrontEndBase::focus_out_server_instance (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->focus_out ();
}

void
FrontEndBase::reset_server_instance (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->reset ();
}

bool
FrontEndBase::process_key_event (int id, const KeyEvent& key) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) return si->process_key_event (key);

    return false;
}

void
FrontEndBase::move_preedit_caret (int id, unsigned int pos) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->move_preedit_caret (pos);
}

void
FrontEndBase::select_lookup_table (int id, unsigned int item) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->select_lookup_table (item);
}

void
FrontEndBase::update_lookup_table_page_size (int id, unsigned int page_size) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->update_lookup_table_page_size (page_size);
}

void
FrontEndBase::toggle_full_width_punctuation (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->toggle_full_width_punctuation ();
}

void
FrontEndBase::toggle_full_width_letter (int id) const
{
    ServerInstancePointer si = find_server_instance (id);
    
    if (!si.null ()) si->toggle_full_width_letter ();
}

void
FrontEndBase::toggle_input_status (int id) const
{
    ServerInstancePointer si = find_server_instance (id);

    if (!si.null ()) si->toggle_input_status ();
}

} // namespace scim

/*
vi:ts=4:ai:nowrap:expandtab
*/
