/** @file scim_server_module.h
 *  @brief definition of ServerModule related classes.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_server_module.h,v 1.14 2004/03/06 15:48:11 suzhe Exp $
 */

#ifndef __SCIM_SERVER_MODULE_H
#define __SCIM_SERVER_MODULE_H

namespace scim {
/**
 * @addtogroup Server
 * @{
 */

/**
 * @brief Initialize a Server Module.
 *
 * There must be a function called "scim_server_module_init"
 * in each server module which complies with this prototype.
 * This function name can have a prefix like table_LTX_,
 * in which "table" is the module's name.
 *
 * @param config - a ConfigBase instance to maintain the configuration.
 * @return the number of factories supported by this Server Module.
 */
typedef unsigned int (*ServerModuleInitFunc) (const ConfigPointer &config);

/**
 * @brief Create a factory instance for a server,
 *
 * There must be a function called "scim_server_module_create_factory"
 * which complies with this prototype.
 * This function name can have a prefix like table_LTX_,
 * in which "table" is the module's name.
 *
 * @param server - the number of the server for which a factory instance will be created.
 * @return the pointer of the factory instance.
 */
typedef ServerFactoryPointer (*ServerModuleCreateFactoryFunc) (unsigned int server);

/**
 * @brief The class to manipulate the server modules.
 *
 * This is a wrapper of scim::Module class, which is specially
 * for manipulating the server modules.
 */
class ServerModule
{
    Module m_module;

    ServerModuleInitFunc m_server_init;
    ServerModuleCreateFactoryFunc m_server_create_factory;

    unsigned int m_number_of_servers;

    ServerModule (const ServerModule &);
    ServerModule & operator= (const ServerModule &);

public:
    /**
     * @brief Default constructor.
     */
    ServerModule ();

    /**
     * @brief Constructor.
     * @param name - the module's name, eg. "rawcode".
     * @param config - a smart pointer points to a ConfigBase instance.
     */
    ServerModule (const String &name, const ConfigPointer &config);

    /**
     * @brief Load a Server Module by its name.
     *
     * Load a module into memory.
     * If another module has been loaded into this object,
     * then the old module will be unloaded first.
     * If the old module is resident, false will be returned,
     * and the old module will be untouched.
     *
     * @param name - the name of the Server Module.
     * @param config - the ConfigBase instance to be used for storing/loading configs.
     * @return true if success.
     */
    bool load  (const String &name, const ConfigPointer &config);

    /**
     * @brief Unload the Server Module.
     * @return true if sucessfully unloaded.
     */
    bool unload ();

    /**
     * @brief Check if a module is loaded and initialized successfully.
     * @return true if a module is already loaded and initialized successfully.
     */
    bool valid () const;

    /**
     * @brief Get how many server factories supported by this module.
     *
     * @return the number of server factories.
     */
    unsigned int number_of_servers () const;

    /**
     * @brief Create a server instance for a server factory.
     *
     * @param server - the sequence number of this server factory,
     *                 must be less than the result of number_of_servers method
     *                 and greater than zero.
     * @return A smart pointer to the server instance, NULL if failed.
     */
    ServerFactoryPointer create_factory (unsigned int server) const;
};

/**
 * @brief Get a name list of currently available server modules.
 * @param mod_list - the result list will be stored here.
 * @return the number of the modules, equal to mod_list.size ().
 */
int scim_get_server_module_list (std::vector <String>& mod_list);

/** @} */

} // namespace scim

#endif //__SCIM_SERVER_MODULE_H

/*
vi:ts=4:ai:nowrap:expandtab
*/
