/**
 * @file scim_socket_transaction.h
 * @brief Socket transaction class.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket_transaction.h,v 1.16 2004/02/06 07:53:15 suzhe Exp $
 */

#ifndef __SCIM_SOCKET_TRANSACTION_H
#define __SCIM_SOCKET_TRANSACTION_H

namespace scim {

/**
 * @addtogroup Helper
 * @{
 */

/**
 * Signature of all of valid data types can be store into transaction.
 */
enum SocketTransactionDataType
{
    SCIM_TRANS_DATA_UNKNOWN,
    SCIM_TRANS_DATA_COMMAND,        //!< Send/Receive command.
    SCIM_TRANS_DATA_RAW,            //!< Send/Receive raw buffer.
    SCIM_TRANS_DATA_UINT32,         //!< Send/Receive uint32.
    SCIM_TRANS_DATA_STRING,         //!< Send/Receive String.
    SCIM_TRANS_DATA_WSTRING,        //!< Send/Receive WideString.
    SCIM_TRANS_DATA_KEYEVENT,       //!< Send/Receive KeyEvent.
    SCIM_TRANS_DATA_ATTRLIST,       //!< Send/Receive AttributeList.
    SCIM_TRANS_DATA_LOOKUP_TABLE,   //!< Send/Receive LookupTable.
    SCIM_TRANS_DATA_VECTOR_UINT32,  //!< Send/Receive vector<uint32>.
    SCIM_TRANS_DATA_VECTOR_STRING,  //!< Send/Receive vector<String>.
    SCIM_TRANS_DATA_VECTOR_WSTRING  //!< Send/Receive vector<WideString>.
};

/**
 * command types used by socket frontend server and config.
 */
const int
    //Generic commands.
    SCIM_TRANS_CMD_UNKNOWN                          = 0,

    SCIM_TRANS_CMD_REQUEST                          = 1,
    SCIM_TRANS_CMD_REPLY                            = 2,
    SCIM_TRANS_CMD_OK                               = 3,
    SCIM_TRANS_CMD_FAIL                             = 4,

    SCIM_TRANS_CMD_OPEN_CONNECTION                  = 5,
    SCIM_TRANS_CMD_CLOSE_CONNECTION                 = 6,

    SCIM_TRANS_CMD_LOAD_FILE                        = 7,

    //Commands for SocketFrontEnd<->SocketServerInstance
    SCIM_TRANS_CMD_PROCESS_KEYEVENT                 = 100,
    SCIM_TRANS_CMD_MOVE_PREEDIT_CARET               = 101,
    SCIM_TRANS_CMD_SELECT_LOOKUP_TABLE              = 102,
    SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE    = 103,
    SCIM_TRANS_CMD_RESET                            = 104,
    SCIM_TRANS_CMD_FOCUS_IN                         = 105,
    SCIM_TRANS_CMD_FOCUS_OUT                        = 106,
    SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_PUNCTUATION    = 107,
    SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_LETTER         = 108,
    SCIM_TRANS_CMD_TOGGLE_INPUT_STATUS              = 109,

    SCIM_TRANS_CMD_SHOW_PREEDIT_STRING              = 110,
    SCIM_TRANS_CMD_SHOW_STATUS_STRING               = 111,
    SCIM_TRANS_CMD_SHOW_AUX_STRING                  = 112,
    SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE                = 113,
    SCIM_TRANS_CMD_HIDE_PREEDIT_STRING              = 114,
    SCIM_TRANS_CMD_HIDE_STATUS_STRING               = 115,
    SCIM_TRANS_CMD_HIDE_AUX_STRING                  = 116,
    SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE                = 117,
    SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET             = 118,
    SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING            = 119,
    SCIM_TRANS_CMD_UPDATE_STATUS_STRING             = 120,
    SCIM_TRANS_CMD_UPDATE_AUX_STRING                = 121,
    SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE              = 122,
    SCIM_TRANS_CMD_COMMIT_STRING                    = 123,
    SCIM_TRANS_CMD_FORWARD_KEYEVENT                 = 124,
    SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_PUNCTUATION    = 125,
    SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_LETTER         = 126,

    //Commands for SocketFrontEnd<->SocketServerFactory
    SCIM_TRANS_CMD_NEW_SERVER_INSTANCE              = 200,
    SCIM_TRANS_CMD_DELETE_SERVER_INSTANCE           = 201,
    SCIM_TRANS_CMD_DELETE_ALL_SERVER_INSTANCES      = 202,

    SCIM_TRANS_CMD_GET_SERVER_FACTORY_LIST          = 203,
    SCIM_TRANS_CMD_GET_SERVER_FACTORY_NAME          = 204,
    SCIM_TRANS_CMD_GET_SERVER_FACTORY_AUTHORS       = 205,
    SCIM_TRANS_CMD_GET_SERVER_FACTORY_CREDITS       = 206,
    SCIM_TRANS_CMD_GET_SERVER_FACTORY_HELP          = 207,
    SCIM_TRANS_CMD_GET_SERVER_FACTORY_LOCALES       = 208,
    SCIM_TRANS_CMD_GET_SERVER_FACTORY_ICON_FILE     = 209,

    //Commands for SocketFrontEnd<->SocketConfig
    SCIM_TRANS_CMD_FLUSH_CONFIG                     = 300,
    SCIM_TRANS_CMD_ERASE_CONFIG                     = 301,
    SCIM_TRANS_CMD_GET_CONFIG_STRING                = 302,
    SCIM_TRANS_CMD_SET_CONFIG_STRING                = 303,
    SCIM_TRANS_CMD_GET_CONFIG_INT                   = 304,
    SCIM_TRANS_CMD_SET_CONFIG_INT                   = 305,
    SCIM_TRANS_CMD_GET_CONFIG_BOOL                  = 306,
    SCIM_TRANS_CMD_SET_CONFIG_BOOL                  = 307,
    SCIM_TRANS_CMD_GET_CONFIG_DOUBLE                = 308,
    SCIM_TRANS_CMD_SET_CONFIG_DOUBLE                = 309,
    SCIM_TRANS_CMD_GET_CONFIG_VECTOR_STRING         = 310,
    SCIM_TRANS_CMD_SET_CONFIG_VECTOR_STRING         = 311,
    SCIM_TRANS_CMD_GET_CONFIG_VECTOR_INT            = 312,
    SCIM_TRANS_CMD_SET_CONFIG_VECTOR_INT            = 313,

    //User defined area
    SCIM_TRANS_CMD_USER_DEFINED                     = 10000;


class SocketTransactionError: public Exception
{
public:
    SocketTransactionError (const String& what_arg)
        : Exception (String("scim::SocketTransaction: ") + what_arg) { }
};

/**
 * This class can pack up many data into one package
 * and send them via socket.
 */
class SocketTransaction
{
    class SocketTransactionImpl;

    SocketTransactionImpl * m_impl;

    SocketTransaction (const SocketTransaction &);
    const SocketTransaction & operator = (const SocketTransaction &);

public:
    /**
     * constructor.
     *
     * @param bufsize the initial buffer size, maybe grow afterwards.
     */
    SocketTransaction (size_t bufsize = 512);

    /**
     * destructor.
     */
    ~SocketTransaction ();

    /**
     * check if the transaction is valid.
     */
    bool valid () const;

    /**
     * write the transaction to socket.
     *
     * @param socket the socket to be written.
     * @param signature the leading signature to be written
     *        into the socket before the transaction itself,
     *        this signature maybe missing when receive the transaction.
     */
    bool write_to_socket (const Socket &socket, uint32 signature = 0) const;

    /**
     * read a transaction from socket.
     *
     * @param socket the socket to be read.
     * @param timeout time out in millisecond (1/1000 second), -1 means infinity.
     */
    bool read_from_socket (const Socket &socket, int timeout = -1);

    /**
     * store a command into transaction.
     */
    void put_command (int type);

    /**
     * store a uint32 value into transaction.
     */
    void put_data (uint32 val);

    /**
     * store a String object into transaction.
     */
    void put_data (const String &str);

    /**
     * store a WideString object into transaction.
     */
    void put_data (const WideString &str);

    /**
     * store a KeyEvent object into transaction.
     */
    void put_data (const KeyEvent &key);

    /**
     * store a AttributeList object info transaction.
     */
    void put_data (const AttributeList &attrs);

    /**
     * store a LookupTable object into transaction.
     */
    void put_data (const LookupTable &table);

    /**
     * store a std::vector<uint32> object into transaction.
     */
    void put_data (const std::vector<uint32> &vec);

    /**
     * store a std::vector<String> object into transaction.
     */
    void put_data (const std::vector<String> &vec);

    /**
     * store a std::vector<WideString> object into transaction.
     */
    void put_data (const std::vector<WideString> &vec);

    /**
     * store a raw buffer into transaction.
     */
    void put_data (const char *raw, size_t bufsize);

    /**
     * get the type of the data at current read position.
     */
    SocketTransactionDataType get_data_type () const;

    /**
     * get a command from current read position.
     */
    bool get_command (int &type);

    /**
     * get a uint32 value from current read position.
     */
    bool get_data (uint32 &val);

    /**
     * get a String from current read position.
     */
    bool get_data (String &str);

    /**
     * get a WideString from current read position.
     */
    bool get_data (WideString &str);

    /**
     * get a KeyEvent from current read position.
     */
    bool get_data (KeyEvent &key);

    /**
     * get an AttributeList from current read position.
     */
    bool get_data (AttributeList &attrs);

    /**
     * get a CommonLookupTable from current read position.
     */
    bool get_data (CommonLookupTable &table);

    /**
     * get a std::vector<uint32> from current read position.
     */
    bool get_data (std::vector<uint32> &vec);

    /**
     * get a std::vector<String> from current read position.
     */
    bool get_data (std::vector<String> &vec);

    /**
     * get a std::vector<WideString> from current read position.
     */
    bool get_data (std::vector<WideString> &vec);

    /**
     * get a raw buffer from current read position.
     * if raw == NULL then return the bufsize and skip this data.
     * *raw should be deleted afterwards (do not use free!).
     */
    bool get_data (char **raw, size_t &bufsize);

    /**
     * skip one data from current read position.
     */
    bool skip_data ();

    /**
     * rewind the current read position, then the data can be read again.
     */
    void rewind_read_pos ();

    /**
     * clear the transaction.
     */
    void clear ();
};

/** @} */

} // namespace scim

#endif //__SCIM_SOCKET_TRANSACTION_H

/*
vi:ts=4:nowrap:ai:expandtab
*/

