#include <string.h>
#include <stdio.h>

#include <glib.h>
#include <gtk/gtk.h>

#define Uses_SCIM_EVENT

#include "scim_private.h"
#include "scim.h"
#include "scimkeyselection.h"

using namespace scim;

enum {
    KEY_SELECTION_CHANGED,
    LAST_SIGNAL
};

/* GObject, GtkObject methods
 */

static void scim_key_selection_class_init              (ScimKeySelectionClass *klass);
static void scim_key_selection_init                    (ScimKeySelection      *keyselection);
static void scim_key_selection_finalize                (GObject               *object);

static void scim_key_selection_add_key_button_callback (GtkButton             *button,
                                                        ScimKeySelection      *keyselection);

static void scim_key_selection_del_key_button_callback (GtkButton             *button,
                                                        ScimKeySelection      *keyselection);

static void scim_key_selection_list_changed_callback   (GtkTreeSelection      *selection,
                                                        ScimKeySelection      *keyselection);

static KeyCode all_key_codes [] =
{
    /* Latin 1 */
    SCIM_KEY_space,
    SCIM_KEY_exclam,
    SCIM_KEY_quotedbl,
    SCIM_KEY_numbersign,
    SCIM_KEY_dollar,
    SCIM_KEY_percent,
    SCIM_KEY_ampersand,
    SCIM_KEY_apostrophe,
    SCIM_KEY_parenleft,
    SCIM_KEY_parenright,
    SCIM_KEY_asterisk,
    SCIM_KEY_plus,
    SCIM_KEY_comma,
    SCIM_KEY_minus,
    SCIM_KEY_period,
    SCIM_KEY_slash,
    SCIM_KEY_0,
    SCIM_KEY_1,
    SCIM_KEY_2,
    SCIM_KEY_3,
    SCIM_KEY_4,
    SCIM_KEY_5,
    SCIM_KEY_6,
    SCIM_KEY_7,
    SCIM_KEY_8,
    SCIM_KEY_9,
    SCIM_KEY_colon,
    SCIM_KEY_semicolon,
    SCIM_KEY_less,
    SCIM_KEY_equal,
    SCIM_KEY_greater,
    SCIM_KEY_question,
    SCIM_KEY_at,
    SCIM_KEY_A,
    SCIM_KEY_B,
    SCIM_KEY_C,
    SCIM_KEY_D,
    SCIM_KEY_E,
    SCIM_KEY_F,
    SCIM_KEY_G,
    SCIM_KEY_H,
    SCIM_KEY_I,
    SCIM_KEY_J,
    SCIM_KEY_K,
    SCIM_KEY_L,
    SCIM_KEY_M,
    SCIM_KEY_N,
    SCIM_KEY_O,
    SCIM_KEY_P,
    SCIM_KEY_Q,
    SCIM_KEY_R,
    SCIM_KEY_S,
    SCIM_KEY_T,
    SCIM_KEY_U,
    SCIM_KEY_V,
    SCIM_KEY_W,
    SCIM_KEY_X,
    SCIM_KEY_Y,
    SCIM_KEY_Z,
    SCIM_KEY_bracketleft,
    SCIM_KEY_backslash,
    SCIM_KEY_bracketright,
    SCIM_KEY_asciicircum,
    SCIM_KEY_underscore,
    SCIM_KEY_grave,
    SCIM_KEY_a,
    SCIM_KEY_b,
    SCIM_KEY_c,
    SCIM_KEY_d,
    SCIM_KEY_e,
    SCIM_KEY_f,
    SCIM_KEY_g,
    SCIM_KEY_h,
    SCIM_KEY_i,
    SCIM_KEY_j,
    SCIM_KEY_k,
    SCIM_KEY_l,
    SCIM_KEY_m,
    SCIM_KEY_n,
    SCIM_KEY_o,
    SCIM_KEY_p,
    SCIM_KEY_q,
    SCIM_KEY_r,
    SCIM_KEY_s,
    SCIM_KEY_t,
    SCIM_KEY_u,
    SCIM_KEY_v,
    SCIM_KEY_w,
    SCIM_KEY_x,
    SCIM_KEY_y,
    SCIM_KEY_z,
    SCIM_KEY_braceleft,
    SCIM_KEY_bar,
    SCIM_KEY_braceright,
    SCIM_KEY_asciitilde,

    /* function keys */
    SCIM_KEY_BackSpace,
    SCIM_KEY_Tab,
    SCIM_KEY_Linefeed,
    SCIM_KEY_Clear,
    SCIM_KEY_Return,
    SCIM_KEY_Pause,
    SCIM_KEY_Scroll_Lock,
    SCIM_KEY_Sys_Req,
    SCIM_KEY_Escape,
    SCIM_KEY_Delete,

    /* Cursor control & motion */
    SCIM_KEY_Home,
    SCIM_KEY_Left,
    SCIM_KEY_Up,
    SCIM_KEY_Right,
    SCIM_KEY_Down,
    SCIM_KEY_Page_Up,
    SCIM_KEY_Page_Down,
    SCIM_KEY_End,
    SCIM_KEY_Begin,

    /* Auxilliary Functions */
    SCIM_KEY_F1,
    SCIM_KEY_F2,
    SCIM_KEY_F3,
    SCIM_KEY_F4,
    SCIM_KEY_F5,
    SCIM_KEY_F6,
    SCIM_KEY_F7,
    SCIM_KEY_F8,
    SCIM_KEY_F9,
    SCIM_KEY_F10,
    SCIM_KEY_F11,
    SCIM_KEY_F12,

    SCIM_KEY_Shift_L,
    SCIM_KEY_Shift_R,
    SCIM_KEY_Control_L,
    SCIM_KEY_Control_R,
    SCIM_KEY_Caps_Lock,
    SCIM_KEY_Shift_Lock,

    SCIM_KEY_Meta_L,
    SCIM_KEY_Meta_R,
    SCIM_KEY_Alt_L,
    SCIM_KEY_Alt_R,
    SCIM_KEY_Super_L,
    SCIM_KEY_Super_R,
    SCIM_KEY_Hyper_L,
    SCIM_KEY_Hyper_R,

    /* International & multi-key character composition */
    SCIM_KEY_Multi_key,
    SCIM_KEY_Codeinput,
    SCIM_KEY_SingleCandidate,
    SCIM_KEY_MultipleCandidate,
    SCIM_KEY_PreviousCandidate,

    /* Japanese keyboard support */
    SCIM_KEY_Kanji,
    SCIM_KEY_Muhenkan,
    SCIM_KEY_Henkan_Mode,
    SCIM_KEY_Henkan,
    SCIM_KEY_Romaji,
    SCIM_KEY_Hiragana,
    SCIM_KEY_Katakana,
    SCIM_KEY_Hiragana_Katakana,
    SCIM_KEY_Zenkaku,
    SCIM_KEY_Hankaku,
    SCIM_KEY_Zenkaku_Hankaku,
    SCIM_KEY_Touroku,
    SCIM_KEY_Massyo,
    SCIM_KEY_Kana_Lock,
    SCIM_KEY_Kana_Shift,
    SCIM_KEY_Eisu_Shift,
    SCIM_KEY_Eisu_toggle,
    SCIM_KEY_Kanji_Bangou,
    SCIM_KEY_Zen_Koho,
    SCIM_KEY_Mae_Koho,

    /* Misc Functions */
    SCIM_KEY_Select,
    SCIM_KEY_Print,
    SCIM_KEY_Execute,
    SCIM_KEY_Insert,
    SCIM_KEY_Undo,
    SCIM_KEY_Redo,
    SCIM_KEY_Menu,
    SCIM_KEY_Find,
    SCIM_KEY_Cancel,
    SCIM_KEY_Help,
    SCIM_KEY_Break,
    SCIM_KEY_Mode_switch,
    SCIM_KEY_Num_Lock,

    /* keypad */
    SCIM_KEY_KP_Space,
    SCIM_KEY_KP_Tab,
    SCIM_KEY_KP_Enter,
    SCIM_KEY_KP_F1,
    SCIM_KEY_KP_F2,
    SCIM_KEY_KP_F3,
    SCIM_KEY_KP_F4,
    SCIM_KEY_KP_Home,
    SCIM_KEY_KP_Left,
    SCIM_KEY_KP_Up,
    SCIM_KEY_KP_Right,
    SCIM_KEY_KP_Down,
    SCIM_KEY_KP_Page_Up,
    SCIM_KEY_KP_Page_Down,
    SCIM_KEY_KP_End,
    SCIM_KEY_KP_Begin,
    SCIM_KEY_KP_Insert,
    SCIM_KEY_KP_Delete,
    SCIM_KEY_KP_Equal,
    SCIM_KEY_KP_Multiply,
    SCIM_KEY_KP_Add,
    SCIM_KEY_KP_Separator,
    SCIM_KEY_KP_Subtract,
    SCIM_KEY_KP_Decimal,
    SCIM_KEY_KP_Divide,

    SCIM_KEY_KP_0,
    SCIM_KEY_KP_1,
    SCIM_KEY_KP_2,
    SCIM_KEY_KP_3,
    SCIM_KEY_KP_4,
    SCIM_KEY_KP_5,
    SCIM_KEY_KP_6,
    SCIM_KEY_KP_7,
    SCIM_KEY_KP_8,
    SCIM_KEY_KP_9,

    SCIM_KEY_NullKey
};

static GtkWidgetClass *parent_class = NULL;

static gint key_selection_signals[LAST_SIGNAL] = { 0 };

static GType key_selection_type = 0;

void
scim_key_selection_register_type (GTypeModule *type_module)
{
    static const GTypeInfo key_selection_info =
    {
        sizeof (ScimKeySelectionClass),
        NULL,
        NULL,
        (GClassInitFunc) scim_key_selection_class_init,
        NULL,
        NULL,
        sizeof (ScimKeySelection),
        0,
        (GInstanceInitFunc) scim_key_selection_init,
    };

    if (!key_selection_type) {
        if (type_module)
            key_selection_type = g_type_module_register_type (
                                    type_module,
                                    GTK_TYPE_VBOX,
                                    "SCIM_ScimKeySelection",
                                    &key_selection_info,
                                    (GTypeFlags) 0);
        else
            key_selection_type = g_type_register_static (
                                    GTK_TYPE_VBOX,
                                    "SCIM_ScimKeySelection",
                                    &key_selection_info,
                                    (GTypeFlags) 0); 
    }
}

GType
scim_key_selection_get_type (void)
{
    if (!key_selection_type)
        scim_key_selection_register_type (NULL);

    return key_selection_type;
}

static void
scim_key_selection_class_init (ScimKeySelectionClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass*) klass;
    widget_class = (GtkWidgetClass*) klass;
    parent_class = (GtkWidgetClass*) g_type_class_peek_parent (klass);

    gobject_class->finalize = scim_key_selection_finalize;

    key_selection_signals[KEY_SELECTION_CHANGED] =
        g_signal_new ("key-selection-changed",
                    G_TYPE_FROM_CLASS (object_class),
                    G_SIGNAL_RUN_FIRST,
                    G_STRUCT_OFFSET (ScimKeySelectionClass, changed),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE, 0);

    klass->changed = NULL;
}


static GList *
create_key_names_list ()
{
    GList *list = NULL;
    String name;

    for (int i = 0; all_key_codes [i] != SCIM_KEY_NullKey; ++ i) {
        if (scim_key_to_string (name, KeyEvent (all_key_codes [i], 0))) {
            list = g_list_append (list, g_strdup (name.c_str ()));
        }
    }

    return list;
}

static void
destroy_key_names_list (GList *list)
{
    for (GList *ptr = list; ptr != NULL; ptr = g_list_next (ptr))
        g_free (ptr->data);

    g_list_free (list);
}

static void
scim_key_selection_init (ScimKeySelection *keyselection)
{
    GtkWidget *table;
    GtkWidget *hbox1;
    GtkWidget *hbox2;
    GtkWidget *frame;
    GtkWidget *label;

    GtkWidget *scrolledwindow;
    GtkWidget *button;

    GtkCellRenderer *list_cell;
    GtkTreeViewColumn *list_column;

    GList     *key_list;

    frame = gtk_frame_new (NULL);
    gtk_widget_show (frame);
    gtk_box_pack_start (GTK_BOX (keyselection), frame, TRUE, TRUE, 0);

    label = gtk_label_new (NULL);
    gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _("Selected _Keys:"));
    gtk_widget_show (label);
    gtk_frame_set_label_widget (GTK_FRAME (frame), label);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);

    // Create keys list view
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_container_add (GTK_CONTAINER(frame), scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
                                    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow),
                                         GTK_SHADOW_ETCHED_IN);

    keyselection->list_view = gtk_tree_view_new ();
    gtk_widget_show (keyselection->list_view);
    gtk_container_add (GTK_CONTAINER (scrolledwindow), keyselection->list_view);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (keyselection->list_view), FALSE);

    gtk_label_set_mnemonic_widget (GTK_LABEL (label), keyselection->list_view);

    keyselection->list_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (keyselection->list_view));
    gtk_tree_selection_set_mode (keyselection->list_selection, GTK_SELECTION_BROWSE);

    g_signal_connect (G_OBJECT (keyselection->list_selection), "changed",
                      G_CALLBACK (scim_key_selection_list_changed_callback),
                      keyselection);

    // Create key list column.
    list_cell = gtk_cell_renderer_text_new ();
    list_column = gtk_tree_view_column_new_with_attributes (
                            NULL, list_cell, "text", 0, NULL);

    gtk_tree_view_append_column (GTK_TREE_VIEW (keyselection->list_view), list_column);

    // Create key list model
    keyselection->list_model = gtk_list_store_new (1, G_TYPE_STRING);

    gtk_tree_view_set_model (GTK_TREE_VIEW (keyselection->list_view),
                             GTK_TREE_MODEL (keyselection->list_model));

    table = gtk_table_new (2, 2, FALSE);
    gtk_widget_show (table);
    gtk_box_pack_start (GTK_BOX (keyselection), table, FALSE, FALSE, 0);

    label = gtk_label_new (_("Key Code:"));
    gtk_widget_show (label);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
                      (GtkAttachOptions) (GTK_FILL),
                      (GtkAttachOptions) (0), 4, 4);
    gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

    keyselection->combo = gtk_combo_new ();
    gtk_widget_show (keyselection->combo);
    gtk_table_attach (GTK_TABLE (table), keyselection->combo, 1, 2, 0, 1,
                      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
                      (GtkAttachOptions) (0), 4, 4);
    gtk_combo_set_value_in_list (GTK_COMBO (keyselection->combo), TRUE, FALSE);
    gtk_combo_set_case_sensitive (GTK_COMBO (keyselection->combo), TRUE);
    gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (keyselection->combo)->entry), FALSE);

    label = gtk_label_new (_("Modifiers:"));
    gtk_widget_show (label);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
                      (GtkAttachOptions) (GTK_FILL),
                      (GtkAttachOptions) (0), 4, 4);
    gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox1);
    gtk_table_attach (GTK_TABLE (table), hbox1, 1, 2, 1, 2,
                      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
                      (GtkAttachOptions) (0), 4, 4);

    keyselection->toggle_ctrl = gtk_check_button_new_with_mnemonic (_("_Ctrl")); 
    gtk_widget_show (keyselection->toggle_ctrl);
    gtk_box_pack_start (GTK_BOX (hbox1), keyselection->toggle_ctrl, TRUE, TRUE, 2);

    keyselection->toggle_alt = gtk_check_button_new_with_mnemonic (_("A_lt")); 
    gtk_widget_show (keyselection->toggle_alt);
    gtk_box_pack_start (GTK_BOX (hbox1), keyselection->toggle_alt, TRUE, TRUE, 2);

    keyselection->toggle_shift = gtk_check_button_new_with_mnemonic (_("_Shift")); 
    gtk_widget_show (keyselection->toggle_shift);
    gtk_box_pack_start (GTK_BOX (hbox1), keyselection->toggle_shift, TRUE, TRUE, 2);

    keyselection->toggle_release = gtk_check_button_new_with_mnemonic (_("_Release")); 
    gtk_widget_show (keyselection->toggle_release);
    gtk_box_pack_start (GTK_BOX (hbox1), keyselection->toggle_release, TRUE, TRUE, 2);

    hbox2 = gtk_hbox_new (TRUE, 0);
    gtk_widget_show (hbox2);
    gtk_box_pack_start (GTK_BOX (keyselection), hbox2, FALSE, FALSE, 4);

    button = gtk_button_new_from_stock ("gtk-add");
    gtk_widget_show (button);
    gtk_box_pack_start (GTK_BOX (hbox2), button, TRUE, TRUE, 4);
    g_signal_connect ((gpointer) button, "clicked",
                      G_CALLBACK (scim_key_selection_add_key_button_callback),
                      keyselection);

    button = gtk_button_new_from_stock ("gtk-delete");
    gtk_widget_show (button);
    gtk_box_pack_start (GTK_BOX (hbox2), button, TRUE, TRUE, 4);
    g_signal_connect ((gpointer) button, "clicked",
                      G_CALLBACK (scim_key_selection_del_key_button_callback),
                      keyselection);

    // Create key names list
    key_list = create_key_names_list ();
    gtk_combo_set_popdown_strings (GTK_COMBO (keyselection->combo), key_list);
    destroy_key_names_list (key_list);

    keyselection->keys = NULL;
}

static void
scim_key_selection_finalize (GObject *object)
{
    ScimKeySelection *keyselection = SCIM_KEY_SELECTION (object);
    if (keyselection->keys)
        g_free (keyselection->keys);

    G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
scim_key_selection_add_key_button_callback (GtkButton        *button,
                                            ScimKeySelection *keyselection)
{
    GtkTreeIter iter;
    String key;

    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (keyselection->toggle_ctrl)))
        key += String ("Control+");
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (keyselection->toggle_alt)))
        key += String ("Alt+");
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (keyselection->toggle_shift)))
        key += String ("Shift+");

    key += String (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (keyselection->combo)->entry)));

    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (keyselection->toggle_release)))
        key += String ("+KeyRelease");

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (keyselection->list_model), &iter)) {
        gchar * keystr;

        do {
            gtk_tree_model_get (GTK_TREE_MODEL (keyselection->list_model), &iter,
                                0, &keystr, -1);

            if (keystr && String (keystr) == key)
                return;

        } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (keyselection->list_model), &iter));
    }

    gtk_list_store_append (keyselection->list_model, &iter);
    gtk_list_store_set (keyselection->list_model, &iter,
                        0, key.c_str (), -1);

    g_signal_emit_by_name (keyselection, "key-selection-changed");
}

static void
scim_key_selection_del_key_button_callback (GtkButton       *button,
                                           ScimKeySelection *keyselection)
{
    GtkTreeIter iter;
    GtkTreeModel *model;

    if (gtk_tree_selection_get_selected (keyselection->list_selection, &model, &iter)) {
        gtk_list_store_remove (keyselection->list_model, &iter);
        g_signal_emit_by_name (keyselection, "key-selection-changed");
    }
}

/* public api
 */
GtkWidget*
scim_key_selection_new (void)
{
  return GTK_WIDGET (gtk_type_new (SCIM_TYPE_KEY_SELECTION));
}

void
scim_key_selection_append_keys (ScimKeySelection *keyselection,
                               const gchar     *keys)
{
    g_return_if_fail (SCIM_IS_KEY_SELECTION (keyselection));
    g_return_if_fail (keys != NULL);

    std::vector <String> keylist;
    GtkTreeIter iter;
    GtkTreeModel *model;

    scim_split_string_list (keylist, keys);

    for (int i = 0; i < keylist.size (); ++ i) {
        gtk_list_store_append (keyselection->list_model, &iter);
        gtk_list_store_set (keyselection->list_model, &iter,
                            0, keylist [i].c_str (), -1);
    }
}

void
scim_key_selection_set_keys (ScimKeySelection *keyselection,
                             const gchar      *keys)
{
    g_return_if_fail (SCIM_IS_KEY_SELECTION (keyselection));

    gtk_list_store_clear (keyselection->list_model);
    scim_key_selection_append_keys (keyselection, keys);
}

G_CONST_RETURN gchar*
scim_key_selection_get_keys (ScimKeySelection *keyselection)
{
    g_return_val_if_fail (SCIM_IS_KEY_SELECTION (keyselection), NULL);

    if (keyselection->keys)
        g_free (keyselection->keys);

    keyselection->keys = NULL;

    GtkTreeIter iter;

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (keyselection->list_model), &iter)) {
        std::vector <String> keylist;
        gchar * keystr;

        do {
            gtk_tree_model_get (GTK_TREE_MODEL (keyselection->list_model), &iter,
                                0, &keystr, -1);

            if (keystr) keylist.push_back (keystr);
        } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (keyselection->list_model), &iter));

        if (keylist.size ())
            keyselection->keys = g_strdup (scim_combine_string_list (keylist).c_str ());
    }

    return keyselection->keys;
}

static void
scim_key_selection_list_changed_callback (GtkTreeSelection *selection,
                                          ScimKeySelection *keyselection)
{
    GtkTreeModel *model;
    GtkTreeIter   iter;
    gchar        *keystr;

    KeyEvent      keyevent;

    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        gtk_tree_model_get (model, &iter, 0, &keystr, -1);

        if (scim_string_to_key (keyevent, String (keystr))) {
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (keyselection->toggle_ctrl),
                keyevent.is_control_down ());
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (keyselection->toggle_alt),
                keyevent.is_alt_down ());
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (keyselection->toggle_shift),
                keyevent.is_shift_down ());
            gtk_toggle_button_set_active (
                GTK_TOGGLE_BUTTON (keyselection->toggle_release),
                keyevent.is_key_release ());

            keyevent.mask = 0;
            String str;

            if (scim_key_to_string (str, keyevent)) {
                gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (keyselection->combo)->entry),
                                    str.c_str ());
            }
        }
    }
}

/*****************************************************************************
 * ScimKeySelectionDialog
 *****************************************************************************/
static GtkWidgetClass *dialog_parent_class = NULL;

static GType key_selection_dialog_type = 0;

static void scim_key_selection_dialog_class_init (ScimKeySelectionDialogClass *klass);
static void scim_key_selection_dialog_init (ScimKeySelectionDialog *keyseldialog);

void
scim_key_selection_dialog_register_type (GTypeModule *type_module)
{
    static const GTypeInfo key_selection_dialog_info =
    {
        sizeof (ScimKeySelectionDialogClass),
        NULL,
        NULL,
        (GClassInitFunc) scim_key_selection_dialog_class_init,
        NULL,
        NULL,
        sizeof (ScimKeySelectionDialog),
        0,
        (GInstanceInitFunc) scim_key_selection_dialog_init,
    };

    if (!key_selection_dialog_type) {
        if (type_module)
            key_selection_dialog_type = g_type_module_register_type (
                                    type_module,
                                    GTK_TYPE_DIALOG,
                                    "SCIM_ScimKeySelectionDialog",
                                    &key_selection_dialog_info,
                                    (GTypeFlags) 0);
        else
            key_selection_dialog_type = g_type_register_static (
                                    GTK_TYPE_DIALOG,
                                    "SCIM_ScimKeySelectionDialog",
                                    &key_selection_dialog_info,
                                    (GTypeFlags) 0); 
    }
}

GType
scim_key_selection_dialog_get_type (void)
{
    if (!key_selection_dialog_type)
        scim_key_selection_dialog_register_type (NULL);

    return key_selection_dialog_type;
}

static void
scim_key_selection_dialog_class_init (ScimKeySelectionDialogClass *klass)
{
    dialog_parent_class = (GtkWidgetClass*) g_type_class_peek_parent (klass);
}

static void
scim_key_selection_dialog_init (ScimKeySelectionDialog *keyseldialog)
{
    GtkDialog *dialog;

    gtk_widget_push_composite_child ();

    dialog = GTK_DIALOG (keyseldialog);

    gtk_container_set_border_width (GTK_CONTAINER (keyseldialog), 4);
    gtk_window_set_resizable (GTK_WINDOW (keyseldialog), TRUE);

    keyseldialog->main_vbox = dialog->vbox;

    keyseldialog->keysel = scim_key_selection_new ();
    gtk_container_set_border_width (GTK_CONTAINER (keyseldialog->keysel), 4);
    gtk_widget_show (keyseldialog->keysel);
    gtk_box_pack_start (GTK_BOX (keyseldialog->main_vbox),
                        keyseldialog->keysel, TRUE, TRUE, 0);

    /* Create the action area */
    keyseldialog->action_area = dialog->action_area;

    keyseldialog->cancel_button = gtk_dialog_add_button (dialog,
                                                        GTK_STOCK_CANCEL,
                                                        GTK_RESPONSE_CANCEL);

    keyseldialog->ok_button = gtk_dialog_add_button (dialog,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK);
    gtk_widget_grab_default (keyseldialog->ok_button);

    gtk_window_set_title (GTK_WINDOW (keyseldialog),
                          _("Key Selection"));

    gtk_dialog_set_has_separator (GTK_DIALOG (dialog), TRUE);

    gtk_widget_pop_composite_child ();
}

GtkWidget*
scim_key_selection_dialog_new (const gchar *title)
{
    ScimKeySelectionDialog *keyseldialog;
  
    keyseldialog= (ScimKeySelectionDialog *) g_object_new (SCIM_TYPE_KEY_SELECTION_DIALOG, NULL);

    if (title)
        gtk_window_set_title (GTK_WINDOW (keyseldialog), title);

    return GTK_WIDGET (keyseldialog);
}

G_CONST_RETURN gchar*
scim_key_selection_dialog_get_keys (ScimKeySelectionDialog *ksd)
{
    return scim_key_selection_get_keys (SCIM_KEY_SELECTION (ksd->keysel));
}

void
scim_key_selection_dialog_set_keys (ScimKeySelectionDialog *ksd,
                                    const gchar            *keys)
{
    return scim_key_selection_set_keys (SCIM_KEY_SELECTION (ksd->keysel), keys);
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
