/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "common.h"
#include "helper.h"
#include "options.h"
#include "atcommand.h"
#include "ttyaccess.h"

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

void file_transfer_manage (int   action, char** files,
			   char* outfile, char* pipe,
			   char* mem, int slot)
{
  char* filename;
  size_t fn_size;
  struct slot_range** range;
  struct slot_range** drange; //for free slot detection
  int enable_search = 1;
  int i = 0;
  int j = 0;
  int k = 0;
  int current;

  if (mem == NULL) {
    errexit ("%s","You must define a memory to use.\n");
  }
  if (strcmp(mem,"?") == 0) {
    file_print_memlist(NULL,1);
    return;
  }

  if (action == 0) {
    errexit ("%s","You must specify a valid action.\n");
  }

  if (slot < SCMXX_SLOT_BINARY_MIN &&
      slot != SCMXX_SLOT_UNDEFINED &&
      slot != SCMXX_SLOT_ALL) {
    errexit ("%s","You must define either a valid slot, nothing or \"all\".\n");
  }

  range = NULL;
  if (action == SCMXX_ACTION_GET) {
    range = file_get_types(0);
  } else {
    range = file_get_types(1);
  }
  if (range[i] == NULL) {
    errexit("%s","ERROR: this phone does not support file exchange.\n");
  }
  while (range[i] != NULL &&
	 strcmp(range[i]->name,mem)) {
    ++i;
  }
  if (range[i] == NULL) {
    errexit("ERROR: type %s is not supported.\n",mem);
  }
  if (slot >= SCMXX_SLOT_BINARY_MIN &&
      (range[i]->min > slot || slot > range[i]->max)) { // ... test if slot is correctly set
    if (range[i]->min != range[i]->max) {
      errexit("%s values can only be %d to %d.\n",range[i]->name,range[i]->min,range[i]->max);
    } else {
      errexit("%s value can only be %d.\n",range[i]->name,range[i]->min);
    }
  }
    
  switch(action){
  default:
    errexit("%s","You must specify exactly one operation.\n");
    break;
  case SCMXX_ACTION_REMOVE: //deleting
    if (slot == SCMXX_SLOT_ALL) {
      for (current=range[i]->min;current<=range[i]->max;current++) {
	file_delete(mem, current);
      }
    } else if (slot < SCMXX_SLOT_BINARY_MIN) {
      errexit ("%s","You must define a valid slot or \"all\".\n");
    } else {
      file_delete(mem, slot);
    }
    break;
  case SCMXX_ACTION_SEND: //sending
    if (slot != SCMXX_SLOT_ALL) {
      // get the detection range (read, not write)...
      drange = file_get_types(0);
      // ...find the mem type in it...
      while (drange[j] != NULL &&
	     strcmp(drange[j]->name,mem)) {
	++j;
      }
      if (drange[j] == NULL) {
	// ...and either disable searching if not found...
	enable_search = 0;
	mem_realloc(drange,0);
	drange = range;
	j = i;
      } else {
	// ...or mak min/max a subrange of both ranges
	if (drange[j]->min < range[i]->min) {
	  drange[j]->min = range[i]->min;
	}
	if (drange[j]->max > range[i]->max) {
	  drange[j]->max = range[i]->max;
	}    
      }
    } else {
      drange = range;
      j = i;
    }

    // define current slot...
    if (slot < range[i]->min) { 
      current = range[i]->min; // ...to minimum if below...
    } else {
      current = slot; // ...or the defined slot
    }
    while (files[k] != NULL && current <= range[i]->max) {
      myprintf(0,"Using slot %d\n",current);
      file_send(files[k],mem,current);
      if (k == 0) {
	if (enable_search == 0 && slot != SCMXX_SLOT_ALL) {
	  //stop here if we cannot search for free slots
	  break;
	} else {
	  //make sure that current is in searchable range
	  if (current < drange[j]->min) {
	    current = drange[j]->min;
	  }
	}
      }
      ++k;
      if (files[k] != NULL) {
	if (slot == SCMXX_SLOT_ALL) {
	  ++current; // simply increase if all slot shall be used
	} else {
	  // only empty slots shall be used
	  current = file_detect_free_slot(mem,current,drange[j]->max);
	  if (current < 0) {
	    errexit("%s","No free slot found.\n");
	  } else {
	    myprintf(0,"Detected empty slot %d\n",current);
	  }
	}
      }
    }
    break;
  case SCMXX_ACTION_GET: //getting
    if (slot == SCMXX_SLOT_UNDEFINED) {
      slot = SCMXX_SLOT_ALL;
    }
    if (slot == SCMXX_SLOT_ALL) {
      fn_size = str_len(outfile)+numlen(range[i]->max)+1+strlen(mem);
      for (k=range[i]->min;k<=range[i]->max;k++) {
	if (outfile != NULL) {
	  if (!strcmp(outfile,"-")) {
	    file_get(outfile,mem,k,pipe);
	  } else {
	    filename = mem_alloc(fn_size+1,1);
	    snprintf(filename,fn_size+1,"%s%0*d.%s",
		     outfile,numlen(range[i]->max),k,mem);
	    file_get(filename,mem,k,pipe);
	    free(filename);
	  }
	} else {
	  errexit("%s","You must specify a file name prefix (can be \"\").\n");
	}
      }
    } else {
      file_get(outfile,mem,slot,pipe);
    }
    break;
  }
}

void file_delete (char* ftype, int slot) {
  char* ausgabe;

  at_sie_binary_write(ftype,slot,0,0);
  ausgabe = at_read_line();
  if (!strcmp(ausgabe,"OK")) {
    myprintf(0,"%s %d deleted.\n",ftype,slot);
  } else {
    errexit("%s %d NOT deleted, something went wrong(%s).\n",ftype,slot,ausgabe);
  }
  mem_realloc(ausgabe,0);
}

int file_detect_free_slot (char* ftype, int minimum, int maximum) {
  char* ausgabe;
  char* content;
  int i;
  enum return_code status;
    
  myprintf(0,"%s","Trying to find an empty slot...\n");
  i = minimum;
  while (i <= maximum) {
    at_sie_binary_read(ftype,i);
    ausgabe = at_read_line();
    status = at_line_type(ausgabe,AT_SIE_BIN_READ,&content);
    switch (status) {
    case AT_RET_OK:
      mem_realloc(ausgabe,0);
      return i;
    case AT_RET_ERROR:
      errexit("%s","\nError: unknown cause\n");
    case AT_RET_ERROR_CME:
    case AT_RET_ERROR_CMS:
      errexit("\nError: %s\n",content);
    default: 
      do {
	mem_realloc(ausgabe,0);
	ausgabe = at_read_line();
	status = at_line_type(ausgabe,AT_SIE_BIN_READ,&content);
      } while (status == AT_RET_ANSWER ||
	       status == AT_RET_OTHER);
      if (status == AT_RET_OK) {
	++i;
      } else if (status == AT_RET_ERROR) {
	errexit("%s","\nError: unknown cause\n");
      } else {
	errexit("\nError: %s\n",content);
      }
      mem_realloc(ausgabe,0);
      break;
      //no break
    }
  }
  return -1;
}

/* these are just for documenting the
 * paket size limits
 */
#define FILE_PDUSIZE_MAX_SPEC 176 //max as in spec
#define FILE_PDUSIZE_MAX_SL42 175 //max for SL42 v23
#define FILE_PDUSIZE_MIN_S45 6 //min for S45/ME45 v10

#define FILE_PDUSIZE 175
void file_send (char* file, char* ftype, int slot) {
  char* ack;

  char transfer[(2*FILE_PDUSIZE)+1];
  char buffer[(2*FILE_PDUSIZE)+1];
  int packetnum = 1;
  int lastsent = 0;

  int myfd;
  int i;
  unsigned char data;
  char datastr[3];

  memset(transfer,0,sizeof(transfer));
  memset(buffer,0,sizeof(buffer));
  memset(datastr,0,sizeof(datastr));

  //trying to open the file
  myfd = open_myFile_ro(file);
    
  //filling buffer 1 (transfer buffer)
  for(i=0;i<FILE_PDUSIZE;i++){
    if (read(myfd,&data,1)!=1) {
      break;
    }
    sprintf(transfer+(i*2),"%02x",data);
  }
  //filling buffer 2
  if (strlen(transfer) == sizeof(transfer)-1) {
    for(i=0;i<FILE_PDUSIZE;i++){
      if (read(myfd,&data,1)!=1) {
	break;
      }
      sprintf(buffer+(i*2),"%02x",data);
    }
  }

  myprintf(0,"%s","File transfer...\n");
  while (strlen(transfer)!=0) {
    //starting packet transfer
    if (strlen(buffer) > 0) {
      at_sie_binary_write(ftype,slot,packetnum,packetnum+1);
    } else {
      at_sie_binary_write(ftype,slot,packetnum,packetnum);
    }
    myprintf(0,"%s","Waiting for data request...");
    ack = at_read_line();

    if (!strncmp(ack,"> ",2)) {
      mem_realloc(ack,0);

      if (strlen(buffer) == 2*FILE_PDUSIZE ||
	  strlen(buffer) == 0) {
	//either this is the last package and/or
	//we don't have to split it up
	lastsent = strlen(transfer)/2;
      } else {
	//split the transfer size between the last two packets
	lastsent = (strlen(transfer) + strlen(buffer))/4;
	//make the first one bigger
	lastsent += ((strlen(transfer) + strlen(buffer))/2)%2;
      }

      myprintf(2,"\nBuffer1 content: %s (%d)",transfer,strlen(transfer));
      myprintf(2,"\nBuffer2 content: %s (%d)",buffer,strlen(buffer));
      if (VERBOSE_LEVEL) {
	myprintf(1,"\nSending %d Bytes in Paket %d\n",lastsent,packetnum);
      } else {
	myprintf(0,"%s","Sending data...");
      }
      //sending hex data
      if (tty_write_data(transfer,2*lastsent) == -1) {
	close(myfd);
	errexit("Error on sending data: %s\n", strerror(errno));
      }
      ack = at_read_line();
      if (!strcmp(ack,"OK")) {
	myprintf(0,"Packet %d sent\n",packetnum);
      } else {
	close(myfd);
	if (!strcmp(ack,"+CME ERROR: INV CHAR IN TEXT")) {
	  errexit("%s","ERROR: Wrong data format\n");
	} else {
	  errexit("\n\"%s\"\n",ack);
	}
      }
      mem_realloc(ack,0);

      //now enter the loop if we have to
      if (strlen(buffer) == 2*FILE_PDUSIZE) {
	strcpy(transfer,buffer);
	for (i=0;i<FILE_PDUSIZE;i++) {
	  if (read(myfd,&data,1)!=1) {
	    break;
	  }
	  sprintf(&buffer[i*2],"%02x",data);
	}
	packetnum++;
      } else {
	if (strlen(buffer) > 0) {
	  strcpy(transfer,&transfer[2*lastsent]);
	  strcpy(&transfer[strlen(transfer)],buffer);
	  memset(buffer,0,sizeof(buffer));
	  packetnum++;
	} else {
	  memset(transfer,0,1);
	}
      }
    } else {
      if (!strcmp(ack,"+CME ERROR: INVALID INDEX")) {
	mem_realloc(ack,0);
	myprintf(0,"%s","ERROR\nPacket buffer seems to be filled with something else, clearing...\n");
	at_sie_binary_write(ftype,slot,-1,-1);
	ack = at_read_line();
	if (strcmp(ack,"OK")) {
	  errexit("%s, aborting\n",ack);
	}
	mem_realloc(ack,0);
      } else {
	close(myfd);
	errexit("\n\"%s\"\n",ack);
      }
    }
  }
  if (packetnum) {
    myprintf(0,"%s","File transfer complete.\n");
  } else {
    myprintf(0,"%s","Nothing to transfer.\n");
  }
  close(myfd);
}

void file_get (char* file, char* ftype, int slot, char* pipe) {
  char* ausgabe;
  int packetnum=0;
  int maxpackets=0;
  int myfd=-1;
  unsigned int i;
  unsigned char data;
  char datastr[3];
  FILE* pipefd = NULL;

  memset(datastr,0,sizeof(datastr));

  at_sie_binary_read(ftype,slot);
  myprintf(0,"Slot %d...\n",slot);
  do{
    ausgabe = at_read_line();
    //here we have to check not only for an output of OK, but because of
    //funny behaviour of the S55 and later, also for "+CME ERROR: unknown".
    //both stand for an empty entry
    if (!(strcmp(ausgabe,"OK")) ||
	!(strcmp(ausgabe,"+CME ERROR: unknown"))) {
      mem_realloc(ausgabe,0);
      myprintf(0,"%s","Empty slot: nothing to get\n");
      close_myFile(myfd);
      return;
    } else if (strstr(ausgabe,"ERROR")!=NULL) {
      close_myFile(myfd);
      errexit("%s, aborting\n",ausgabe);			
    } else {
      strtok(ausgabe,",");
      strtok(NULL,",");
      packetnum=atoi((char *)strtok(NULL,","));
      maxpackets=atoi((char *)strtok(NULL,","));
      if (str_len(file) && myfd==-1) {
	myfd=open_myFile_rw(file);
      }
      if (str_len(pipe) && pipefd==NULL) {
	pipefd=popen(pipe,"w");
      }
      myprintf(0,"Receiving packet %d of %d...\n",packetnum,maxpackets);
      mem_realloc(ausgabe,0);
      ausgabe = at_read_line();
      for (i=0;i<strlen(ausgabe);i+=2) {	
	data=hexstr2int(ausgabe+i,2);
	if (myfd!=-1) {
	  if (write(myfd,&data,1) == -1) {
	    close_myFile(myfd);
	    if (str_len(pipe) && pipefd!=NULL) {
	      pclose(pipefd);
	    }
	    errexit("ERROR on writing to %s: %s\n",file, strerror(errno));			
	  }
	}
	if (str_len(pipe) && pipefd!=NULL) {
	  if (fwrite(&data,1,1,pipefd)!=1) {
	    close_myFile(myfd);
	    pclose(pipefd);
	    errexit("ERROR on writing to pipe \"%s\".\n",pipe);					    
	  }
	}
      }
      mem_realloc(ausgabe,0);
    }
  } while (packetnum!=maxpackets);
  close_myFile(myfd);
  if (str_len(pipe) && pipefd!=NULL && pclose(pipefd)==-1) {
    close(myfd);
    errexit("ERROR on closing pipe \"%s\".\n",pipe);			
  }
  ausgabe = at_read_line();
  if (!(strcmp(ausgabe,"OK"))) {
    mem_realloc(ausgabe,0);
    myprintf(0,"%s","File transfer complete.\n");
  } else {
    errexit("%s, aborting\n",ausgabe);
  }
}
