/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <helper.h>
#include <charsets.h>
#include "smscoding.h"

#include <string.h>

ucs4char_t* sms_data_gsm_decode (char* input, unsigned int udl,
				 int header_present)
{
  unsigned char userdata[140*2 + 1];
  gsmchar_t decoded[(140*8/7)+1];
  unsigned int state, inc, outc;
  unsigned char first, end;
  unsigned int i, counter = 0;
  uint8_t headersize = 0;

  if (input == NULL) {
    return NULL;
  }

  memset(userdata,0,sizeof(userdata));
  memset(decoded,0,sizeof(decoded));

  if (header_present) {
    headersize = hexstr2int(input,2) + 1;
  }

  for (i = headersize*2; i < strlen(input); i+=2) {
    userdata[counter++]=hexstr2int(input+i,2);
  }

  //7bit-GSM octet decoding
  state = (headersize*8)/7;
  if (headersize%7 != 0) ++state;
  inc = 0;
  outc = 0;
  while (state < udl &&
	 inc < counter &&
	 outc < sizeof(decoded)) {
    i = state%8;
    if (i == 0) {
      //character at state 0 needs no shifting
      decoded[outc] = userdata[inc] & 0x7F;
    } else {
      //shifting dependent on state (1-7)
      first = (userdata[inc+1]) << i;
      end = (userdata[inc]) >> (8-i);
      decoded[outc] = (first | end) & 0x7F;
      ++inc;
    }
    if (decoded[outc] == 0) { //special char '@'
      decoded[outc] = 128;
    }
    ++outc;
    ++state;
  }

  return convert_from_gsm(decoded);
}
ucs4char_t* sms_data_ucs2_decode (char* input, unsigned int udl,
				  int header_present)
{
  ucs4char_t* retval;
  char* temp;
  uint8_t headersize = 0;

  if (input == NULL) return NULL;
  if (header_present) {
    headersize = hexstr2int(input,2) + 1;
    if (headersize > strlen(input)) return NULL;
    input += headersize;
  }
  if (udl > strlen(input)) udl = strlen(input);

  temp = strn_dup(input,udl*2);
  retval = convert_from_ucs2_hexstring(temp);
  mem_realloc(temp,0);
  return retval;
}

ucs4char_t* sms_data_8bit_decode (char* input, unsigned int udl,
				  int header_present)
{
  ucs4char_t* retval;
  uint8_t headersize = 0;
  unsigned int i;

  if (input == NULL) return NULL;
  if (header_present) {
    headersize = hexstr2int(input,2) + 1;
    if (headersize > strlen(input)) return NULL;
    input += headersize;
  }
  if (udl > (strlen(input)/2)) udl = strlen(input)/2;

  retval = mem_alloc((udl+1)*sizeof(*retval),1);
  for (i = 0; i < udl; ++i) {
    retval[i] = hexstr2int(input+(2*i),2)&0xFF;
  }
  return retval;
}

ucs4char_t* sms_data_decode (enum sms_encoding encoding,
			     char* input, unsigned int udl,
			     int header_present)
{
  switch (encoding) {
  case SMS_CHARSET_GSM: //all encodings of uncompressed 7bit
    return sms_data_gsm_decode(input,udl,header_present);
  case SMS_CHARSET_UCS2: //all encodings of uncompressed 16bit unicode
    return sms_data_ucs2_decode(input,udl,header_present);
  case SMS_CHARSET_8BIT: //all encodings of 8bit data: ASCII assumed
    return sms_data_8bit_decode(input,udl,header_present);
  default: //never happens
    return NULL;
  }
}
