/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ttyaccess.h"
#include "common.h"
#include "helper.h"
#include "atcommand.h"
#include "config.h" //needed for TTYSPEED
#include "gtincl.h"
#include "w32compat.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#ifdef __win32__
#define WINDOWS_API
#endif

#if defined(OS2)
#define INCL_OS2
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#include <os2.h>
#include <termios.h>
#define speed_t int
#undef  B9600
#define B9600 9600
#undef  B19200
#define B19200 19200
#undef  B38400
#define B38400 38400
#undef  B57600
#define B57600 57600
#undef  B115200
#define B115200 115200
#define DOS_API
static HFILE tty_portfd;

#elif defined(WINDOWS_API)
#include <windows.h>
#define speed_t DWORD
#define B9600   CBR_9600
#define B19200  CBR_19200
#define B38400  CBR_38400
#define B57600  CBR_57600
#define B115200 CBR_115200
static HANDLE tty_portfd;

#else
#include <termios.h>
static int tty_portfd;
#endif

static speed_t tty_speed(char *newttyspeed){
  switch (atoi(newttyspeed)) {
  case 9600: return B9600;
  case 19200: return B19200;
  case 38400: return B38400;
#ifdef B57600
  case 57600: return B57600;
#endif
#ifdef B115200
  case 115200: return B115200;
#endif
  default:
    print_verbose(0,"%s","No valid baudrate defined. Using compiled in value...\n");
    return tty_speed(TTYSPEED);
  }
}

#if defined(DOS_API)
static void tty_makeraw() {
  LINECONTROL lc;
  int rc;

  lc.bDataBits = 8;
  lc.bParity = 0;
  lc.bStopBits = 0;
  lc.fTransBreak = 0;
  rc = DosDevIOCtl(tty_portfd, IOCTL_ASYNC, ASYNC_SETLINECTRL,
		   &lc, sizeof(LINECONTROL), &size, NULL, 0, NULL);
  if (rc) {
    fprintf(stderr,"%s: rc=%d\n",_("Error in setting port attributes"),rc);
    exit(EXIT_FAILURE);
  }
}
static void tty_setspeed (char* baud) {
  speed_t bps = tty_speed(args->baud);
  int rc = DosDevIOCtl(tty_portfd, IOCTL_ASYNC, ASYNC_SETBAUDRATE,
		       &bps, sizeof(ULONG), &size, NULL, 0, NULL);
  if (rc) {
    fprintf(stderr,"%s: rc=%d\n",_("Error in setting transmission speed"),rc);
    exit(EXIT_FAILURE);
  }
}
static void tty_settimeout (DCBINFO* devinfo,uint8_t dsec) {
  int timeout = 65535;
  
  if (dsec != 0) timeout = dsec*10;
  devinfo.fbTimeout = 0x02;
  devinfo.usReadTimeout = timeout;
  devinfo.usWriteTimeout = timeout;  
}
void tty_flush () {
  int rc = DosDevIOCtl(tty_portfd, IOCTL_GENERAL, DEV_FLUSHINPUT,
		       NULL, 0, NULL, NULL, 0, NULL);  
  if (!rc) rc = DosDevIOCtl(tty_portfd, IOCTL_GENERAL, DEV_FLUSHOUTPUT,
			    NULL, 0, NULL, NULL, 0, NULL);
  if (rc) {
    fprintf(stderr,"%s: rc=%d\n",_("Error in flushing buffers"),rc);
    exit(EXIT_FAILURE);
  }
}

#elif defined(WINDOWS_API)
static void tty_makeraw(DCB* dcb) {
  dcb->StopBits = ONESTOPBIT;
  dcb->Parity = NOPARITY;
  dcb->ByteSize = 8;
  dcb->fNull = FALSE;
}
static void tty_setspeed (DCB* dcb, char* baud) {
  dcb->BaudRate = tty_speed(baud);
}
static void tty_settimeout (uint8_t dsec) {
  COMMTIMEOUTS timeouts;

  if (GetCommTimeouts(tty_portfd,&timeouts)) {
    if (dsec == 0) {
      /* MFC doc tells nothing about ever-blocking read
       * so I hope that this is correct 
       */
      timeouts.ReadIntervalTimeout = 0;
      timeouts.ReadTotalTimeoutMultiplier = 0;
      timeouts.ReadTotalTimeoutConstant = 0;
    } else {
      timeouts.ReadIntervalTimeout = dsec*10;
    }
    SetCommTimeouts(tty_portfd,&timeouts);
  }
}
void tty_flush () {
  if (!PurgeComm(tty_portfd, PURGE_RXCLEAR|PURGE_TXCLEAR)) {
    print_verbose(0,"%s: %s %d\n",_("Error in flushing buffers"),
		  _("system error code"),GetLastError());
    exit(EXIT_FAILURE);
  }
}

#else
#include "config.h"
#ifdef NO_CFMAKERAW
static void tty_makeraw(struct termios* termios_p) {
  termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
  termios_p->c_oflag &= ~OPOST;
  termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
  termios_p->c_cflag &= ~(CSIZE|PARENB);
  termios_p->c_cflag |= CS8;
}
#else
#define tty_makeraw(t) (void)cfmakeraw(t)
#endif /*NO_CFMAKERAW */
static void tty_setspeed (struct termios* termios_p, char* baud) {
  cfsetispeed(termios_p, tty_speed(baud)); //input at baudrate
  cfsetospeed(termios_p, tty_speed(baud)); //ouput at baudrate
}
static void tty_settimeout (struct termios* termios_p, uint8_t dsec) {
  if (dsec == 0) {
    termios_p->c_cc[VMIN]=1; //wait for at least one character (never times out)
  } else {
    termios_p->c_cc[VMIN]=0; //return after timeout, even with nothing
  }
  termios_p->c_cc[VTIME]=dsec; //try reading for this amount of time (in deciseconds)
}
void tty_flush () {
  int rc = tcflush(tty_portfd, TCIOFLUSH);
  if (rc == -1) {
    fprintf(stderr,"%s: %s\n",_("Error in flushing buffers"),strerror(errno));    
    exit(EXIT_FAILURE);
  }
}
#endif

/** open a serial tty
 */
void tty_open(struct port_args_t* args) {
#if defined(DOS_API)
  long action;
  unsigned long size;
  int rc;
  DCBINFO devinfo;
  
  //opening the device
  print_verbose(0,_("Accessing device %s..."),args->device);
  rc = DosOpen(args->device, &tty_portfd, &action, 0, FILE_NORMAL, FILE_OPEN,
	             OPEN_ACCESS_READWRITE|OPEN_SHARE_DENYNONE, (PEAOP2)NULL);
  if (rc) {
    print_verbose(0,_("Cannot open %s"),args->device);
    print_verbose(0,": rc=%d\n",rc);
    exit(EXIT_FAILURE);
  }

  tty_setspeed(args->baud);
  tty_makeraw();

  rc = DosDevIOCtl(tty_portfd, /* file decsriptor */
		               IOCTL_ASYNC, /*asyncronous change */
		               ASYNC_GETDCBINFO, /* get device control block info */
		               NULL, /*  */
		               0, /* length of the previous parameter */
		               NULL, /* max length of data ret */
		               &devinfo, /* data to be recieved */
		               sizeof(DCBINFO), /* length of data */
		               &size); /* length of data returned */
  
  devinfo.fbFlowReplace = 0; // diable rts/cts control
  devinfo.fbCtlHndShake = 1; // Truning on DTR.
  tty_settimeout(&devinfo,args->timeout);

  rc = DosDevIOCtl(tty_portfd, IOCTL_ASYNC, ASYNC_SETDCBINFO,
		               &devinfo, /* parameters to set  */
		               sizeof(DCBINFO),&size, NULL, 0, NULL); 

#elif defined(WINDOWS_API)
  DCB dcb;
  
  //opening the device
  print_verbose(0,_("Accessing device %s..."),args->device);
  tty_portfd = CreateFile(args->device,
                          GENERIC_READ | GENERIC_WRITE,
                          0,    // must be opened with exclusive-access
                          NULL, // no security attributes
                          OPEN_EXISTING, // must use OPEN_EXISTING
                          0,    // not overlapped I/O
                          NULL); // hTemplate must be NULL for comm devices
  if (tty_portfd == INVALID_HANDLE_VALUE) {
    print_verbose(0,_("Cannot open %s"),args->device);
    print_verbose(0,": %s %d\n",_("system error code"),GetLastError());
    exit(EXIT_FAILURE);
  }
  if (GetCommState(tty_portfd,&dcb) == 0) {
    fprintf(stderr,"%s: %s %d\n",_("Error in getting port attributes"),
            _("system error code"),GetLastError());
    exit(EXIT_FAILURE);    
  }
  tty_setspeed(&dcb,args->baud);
  tty_makeraw(&dcb);
  if (SetCommState(tty_portfd,&dcb) == 0) {
    fprintf(stderr,"%s: %s %d\n",_("Error in setting port attributes"),
            _("system error code"),GetLastError());
    exit(EXIT_FAILURE);    
  }
  tty_settimeout(args->timeout);

#else
  int flags = O_RDWR | O_NOCTTY | O_NONBLOCK;
  struct termios newtio;

  /* FHS (http://www.pathname.com/fhs/pub/fhs-2.3.html#VARLOCKLOCKFILES)
   * specifies lock files for devices but the Serial HowTo (13.3)
   * also mentions the problem when it comes to DevFS (Linux).
   * Since this makes it an unreliable test, forget about it.
   */

  //opening the device
  print_verbose(0,_("Accessing device %s..."),args->device);
  if ((tty_portfd=open(args->device,flags)) == -1) {
    print_verbose(0,_("Cannot open %s"),args->device);
    print_verbose(0,": %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }
  if (fcntl(tty_portfd,F_SETFL,flags&(~O_NONBLOCK)) == -1) {
    fprintf(stderr,"%s: %s\n",_("Error in setting port attributes"),strerror(errno));
    exit(EXIT_FAILURE);
  }

  if (!args->ignorebits) {
    //getting port parameters
    if (tcgetattr(tty_portfd,&newtio) < 0){
      fprintf(stderr,"%s: %s\n",_("Error in getting port attributes"),strerror(errno));
      exit(EXIT_FAILURE);
    }
  } else {
    memset(&newtio,0,sizeof(newtio));
  }
  tty_setspeed(&newtio, args->baud);
  tty_makeraw(&newtio); //make raw, 8N1
  /* CLOCAL: ignore modem control lines
   * CREAD: enable receiver
   * ~CSTOPB: do not send two stop bits (but one)
   */
  newtio.c_cflag |= (CLOCAL | CREAD);
  newtio.c_cflag &= ~CSTOPB;
  tty_settimeout(&newtio,args->timeout);
  tty_flush();
  if(tcsetattr(tty_portfd,TCSANOW,&newtio) < 0){ //set now
    fprintf(stderr,"%s: %s\n",_("Error in setting port attributes"),strerror(errno));
    exit(EXIT_FAILURE);
  }

#endif

  print_verbose(0,"%s\n",_("done"));
  /* The following seems to be needed for C45
   * which seems to react too slow
   */
  if (args->startdelay) {
    print_verbose(0,_("Waiting for %d seconds as requested...\n"),args->startdelay);
    sleep(args->startdelay);
  }
}

int tty_write (const char* data, size_t count) {
#if defined(DOS_API)
  int rc;
  ULONG status = 0;
  ULONG instatus;

  do { 
    rc = DosWrite(tty_portfd,&data[status],count-status,&instatus);
    if (rc) return 0;
    status += instatus;
  } while (count-status > 0);

#elif defined(WINDOWS_API)
  BOOL rc;
  ULONG status = 0;
  ULONG instatus;

  do {
    rc = WriteFile(tty_portfd,&data[status],count-status,&instatus,NULL);
    if (rc == FALSE) return 0;
    status += instatus;
  } while (count-status > 0);

#else
  ssize_t status = 0;
  ssize_t instatus;

  do {
    instatus = write(tty_portfd,&data[status],count-status);
    if (instatus != -1) {
      status += instatus;
    } else {
      if (errno == EAGAIN) usleep(1);
      else return 0;
    }
  } while (count-status > 0);

#endif
  return 1;
}


/* This function returns a char* that must be freed.
 * If it returns NULL, there was an error. Look at errno for the reason.
 * If the strlen(char*)==0, reading from the device timed out. Retry or abort.
 * (the timeout might only happen when the device was not opened with O_NONBLOCK,
 *  the timeout time refers to the value that was set as c_cc[VTIME])
 */
char* tty_read_line (int (*stop_condition)(const char*,const size_t)) {	
  char* retval = NULL;
  int retvalsize = 0;
  char buffer; //buffer the read character for checks
  int counter = 0; //count the read-and-used characters
  int repeat = 0; //tell the inner loop to loop

#if defined(DOS_API)
  int rc;
  ULONG status;
 
  do {
    do {
      rc = DosRead(tty_portfd,&buffer,1,&status); /* expect a single byte */
      if (rc) {
        mem_realloc(retval,0);
        return NULL;
      }
      if (status == 0) {
        mem_realloc(retval,0);
        return str_dup("");
      } else {
        repeat = 0;
      }

#elif defined(WINDOWS_API)
  BOOL rc;
  DWORD status;
  
  do {
    do {
      rc = ReadFile(tty_portfd,&buffer,1,&status,NULL);
      if (rc == FALSE) {
        mem_realloc(retval,0);
        return NULL;
      }
      if (status == 0) {
        mem_realloc(retval,0);
        return str_dup("");
      } else {
        repeat = 0;
      }

#else
  int status;

  do {
    do {
      status = read(tty_portfd,&buffer,1);
      switch (status) {
      case -1:
        if (errno == EAGAIN) {
          usleep(1);
          repeat = 1;
        } else {
          mem_realloc(retval,0);
          return NULL;
        }
      case 0:
        mem_realloc(retval,0);
        return str_dup("");
      default:
        repeat = 0;
      }

#endif
    } while (repeat);

    // allocate space on stack
    if (counter >= retvalsize) {
      retvalsize += BUFSIZ;
      retval = mem_realloc(retval,retvalsize+1);
      memset(retval+counter,0,retvalsize-counter+1);
    }

    /* fix the '@'=0x00 (GSM character set) problem here :-(
     * we simply set the MSB to 1, so when processing:
     * only look at the last 7 bits (char&0x7F)
     */
    retval[counter++] = (buffer == 0) ? 128 : buffer;
  } while (!stop_condition(retval,counter));
  /* There are two types of possible answers:
   * "> " (2 characters) for data input requests
   * "....<CR><LF>" for all other things (even empty)
   */
  return mem_realloc(retval,strlen(retval)+1);
}

void tty_close(){
#if defined(DOS_API)
  DosClose(tty_portfd); 
#elif defined(WINDOWS_API)
  CloseHandle(tty_portfd);
#else
  close(tty_portfd);
#endif
}
