/*
 *  $Id: flowcontrol.h,v 1.5 2003/06/01 01:42:30 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * This module implements most parts of the flow control mechanisms
 *
 */

#ifndef FLOWCONTROL_H
#define FLOWCONTROL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "globals.h"
#include "sctp.h"
#include "distribution.h"
#include "reltransfer.h"

/**
 * Creates new instance of flowcontrol module and returns pointer to it
 * TODO : should parameter be unsigned short ?
 */
void *fc_new_flowcontrol(Association* asok,
                         unsigned int peer_rwnd,
                         unsigned int my_iTSN,
                         unsigned int number_of_destination_addresses,
                         unsigned int maxQueueLen);

/**
 * Deletes data occupied by a flow_control data structure
 * @param fc_instance pointer to the flow_control data structure
 */
void fc_delete_flowcontrol(Association* asok);



/**
 * this function should be called to signal to flowcontrol, that our ULP
 * has initiated a shutdown procedure. We must only send unacked data from
 * now on ! The association is about to terminate !
 */
void fc_shutdown(Association* asok);

/**
 * this function stops all currently running timers, and may be called when
 * the shutdown is imminent
 */
void fc_stop_timers(Association* asok);

/**
 * this function stops all currently running timers, and may be called when
 * the shutdown is imminent
 */
void fc_restart(Association* asok, guint32 new_rwnd, unsigned int iTSN, unsigned int maxQueueLen);

int fc_reset_cwnd(Association* asok, unsigned int pathKey);

int fc_send_all_you_can(Association* asok, gboolean doInitialRetransmit);


/**
 * function called by Reliable Transfer, when it requests retransmission
 * in SDL diagram this signal is called (Req_RTX, RetransChunks)
 * @param  all_data_acked indicates whether or not all data chunks have been acked
 * @param   new_data_acked indicates whether or not new data has been acked
 * @param   num_acked number of bytes that have been newly acked, else 0
 * @param   number_of_addresses so many addresses may have outstanding bytes
 *          actually that value may also be retrieved from the association struct (?)
 * @param   num_acked_per_address array of integers, that hold number of bytes acked for each address
 * @param   number_of_rtx_chunks number indicatin, how many chunks are to be retransmitted in on datagram
 * @param   chunks  array of pointers to data_chunk structures. These are to be retransmitted
 * @return  SCTP_SUCCESS on success
 */
int fc_fast_retransmission(Association* asok,
                     unsigned int addressKey,
                     unsigned int arwnd, unsigned int ctsna,
                     unsigned int rtx_bytes, boolean all_data_acked,
                     boolean new_data_acked, unsigned int num_acked,
                     int number_of_rtx_chunks, chunk_data ** chunks);

/**
 * function called by Reliable Transfer, after it has got a SACK chunk
 * in SDL diagram this signal is called SACK_Info
 * @param  all_data_acked indicates whether or not all data chunks have been acked
 * @param   new_data_acked indicates whether or not new data has been acked
 * @param   num_acked number of bytes that have been newly acked, else 0
 * @param   number_of_addresses so many addresses may have outstanding bytes
 *          actually that value may also be retrieved from the association struct (?)
 * @param   num_acked_per_address array of integers, that hold number of bytes acked for each address
 */
void fc_sack_info(Association* asok,
                  unsigned int addressKey,
                  unsigned int arwnd, unsigned int ctsna,
                  boolean all_data_acked, boolean new_data_acked,
                  unsigned int num_acked);


int fc_dequeue_acked_chunks(Association* asok, unsigned int ctsna);

int fc_dequeueUnackedChunk(Association* asok, unsigned int tsn);


/**
 * Function dequeues the oldest chunk from the transmit queue that has not been sent.
 * The funtion takes a lot of pointers to values that will be filled in by the function.
 * @param buf pointer to the buffer where chunk data will be copied to
 * @param len pointer to the value where length of chunk will be stored
 * @param tsn pointer to the TSN of the chunk (if it is segemted, it is one TSN of the segments)
 * @param sID pointer to the stream ID of the chunk
 * @param sSN pointer to the stream sequence number of the chunk
 * @param pID pointer to the protocol ID of the chunk
 * @param flags pointer to the flag value (unsigned char)
 * @param ctx pointer to the user context
 * @return number of unsent chunks still left in the transmit queue
 */
int fc_dequeueOldestUnsentChunk(Association* asok,
                                unsigned char *buf, unsigned int *len, unsigned int *tsn,
                                unsigned short *sID, unsigned short *sSN,unsigned int* pID,
                                unsigned char* flags, gpointer* ctx);


/**
 * function returns number of chunks, that have been submitted from the upper layer,
 * but not yet been sent ! These are waiting in the transmission queue, not the
 * retransmission queue
 */
int fc_readNumberOfUnsentChunks(Association* asok);

/**
 * function returns total number of chunks, that are currently waiting in the
 * transmit queue for sending !
 * @return size of the send queue in chunks of the current flowcontrol module, or SCTP_MODULE_NOT_FOUND
 */
int fc_readNumberOfQueuedChunks(Association* asok);



/**
 * Function returns cwnd value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current cwnd value, else -1
 */
int fc_readCWND(Association* asok, unsigned int pathKey);


/**
 * Function returns cwnd value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current cwnd2 value, else -1
 */
int fc_readCWND2(Association* asok, unsigned int pathKey);


/**
 * Function returns ssthresh value of a certain path.
 * @param path_id    path index of which we want to know the cwnd
 * @return current ssthresh value, else -1
 */
int fc_readSsthresh(Association* asok, unsigned int pathKey);



/**
 * Function returns mtu value of a certain path.
 * @param path_id    path index of which we want to know the mtu
 * @return current MTU value, else -1
 */
int fc_readMTU(Association* asok, unsigned int pathKey);


/**
 * Function returns the partial bytes acked value of a certain path.
 * @param path_id    path index of which we want to know the PBA
 * @return current PBA value, else -1
 */
int fc_readPBA(Association* asok, unsigned int pathKey);


/**
 * Function returns the outstanding byte count value of a certain path.
 * @param path_id    path index of which we want to know the outstanding_bytes_per_address
 * @return current outstanding_bytes_per_address value, else -1
 */
int fc_readOutstandingBytesPerAddress(Association* asok, unsigned int pathKey);

int fc_decrease_outstanding_bytes(Association* asok, unsigned int addressKey, unsigned int numOfBytes);

/**
 * Function returns the outstanding byte count value of this association.
 * @return current outstanding_bytes value, else -1
 */
int fc_readOutstandingBytes(Association* asok);


int fc_get_maxSendQueue(Association* asok, unsigned int * queueLen);


int fc_set_maxSendQueue(Association* asok, unsigned int maxQueueLen);

int fc_addDestinationAddress(Association* asok,
                             unsigned int newKey,
                             int newIndex);

int fc_delDestinationAddress(Association* asok,
                             unsigned int delKey,
                             int delIndex,
                             unsigned int altKey,
                             int altIndex);

#endif
