/*
 *  $Id: pathmanagement.h,v 1.7 2004/01/13 16:13:14 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This module does faultmanagement for paths.
 *
 * function prefix: pm_
 */

#ifndef PATHMANAGEMENT_H
#define PATHMANAGEMENT_H

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#include "distribution.h"


/* The states of pathmanagement, also used for network status change */
#define  PM_ACTIVE     0
#define  PM_INACTIVE   1
#define  PM_PATH_CONFIRMED             8
#define  PM_PATH_UNCONFIRMED           9


#define PM_INITIAL_HB_INTERVAL  30000
#define PM_PMTUD_INTERVAL  33000
#define PM_PMTUD_INCREASE_INTERVAL  660000


/******************** Functions to answer peer HB requests ****************************************/
/**
 * Function is called to perform a on demand HB on a certain path
 */
int pm_doHB(Association* asok, unsigned int pathKey);


/**
 * pm_heartbeat is called when a heartbeat was received from the peer.
 *  params: heartbeatChunk: the heartbeat chunk
 */
int pm_heartbeat(Association* asok, SCTP_heartbeat * heartbeatChunk, unsigned int addressKey);


/******************** Signals *********************************************************************/

/*------------------- Signals from the Unix-Interface --------------------------------------------*/

/**
 * pm_heartbeatTimer is called when the heartbeat timer expires.
 *  params: timerID:            ID of timer
 *          associationIDvoid:  pointer to the association-ID
 *          pathIDvoid:         pointer to the path-ID
 */
void pm_heartbeatTimer(TimerID timerID, void *associationIDvoid, void *pathKeyvoid);



/**
 * pm_heartbeatAck is called when a heartbeat acknowledgement was received from the peer.
 *  params: heartbeatChunk: the heartbeat chunk
 */
int pm_heartbeatAck(Association* asok, SCTP_heartbeat * heartbeatChunk);

/*------------------- Signals from SCTP internal modules -----------------------------------------*/

/**
 * pm_chunksAcked is called by reliable transfer whenever chunks have been acknowledged.
 *  Params: pathID:      path-ID
 *          newRTO:      the newly determined RTO in milliseconds
 *                       newRTO = 0 ==> no RTO measurements done
 */
int  pm_chunksAcked(Association* asok, unsigned int  pathKey, unsigned int newRTO);

/**
 * function to be called every time we send data to a path
 * to keep it from becoming "idle"
 */
void pm_chunksSentOn(Association* asok, unsigned int pathKey);


/**
 * pm_chunksRetransmitted is called by reliable transfer whenever chunks have been
 *  retransmitted.
 *  Params: pathID:      path-ID
 */
gboolean pm_chunksRetransmitted(Association* asok, unsigned int pathKey);




/**
 * pm_rto_backoff is called by reliable transfer when the T3 retransmission timer expires.
 * Each call of this function doubles the RTO (timer back off).
 * @params pathKey  path for which to do the backoff
 */
void pm_rto_backoff(Association* asok, unsigned int pathKey);

/*------------------- Helper Function  ----------------------------------------------------------*/

unsigned int pm_getTime(void);


/*------------------- Functions called by the ULP ------------------------------------------------*/


/**
 * pm_enableHB is called when ULP wants to enable heartbeat.
 * @params pathKey  path for which to enable the HB
 * @params hearbeatIntervall  heartbeat intervall in msecs
 */
int pm_enableHB(Association* asok, unsigned int pathKey, unsigned int hearbeatIntervall);



/**
 * pm_disableAllHB is called when on shutdown to disable all heartbeats.
 */
int pm_disableAllHB(Association* asok);



/**
 * pm_disableHB is called when ULP wants to disable heartbeat.
 * @params pathKey  path for which to disable the HB
 */
int pm_disableHB(Association * asok, unsigned int pathKey);


int pm_checkValidPathKey(Association* asok, unsigned int pathKey);



/**
 * pm_getPrimaryPathKey returns the key identifying the primary path.
 *  Params: returns      primary path, or 0 as error
 */
unsigned int pm_getPrimaryPathKey(Association *asok);


/**
 * pm_setPrimaryPath sets the primary path.
 * @params pathKey  path key for the new primary path
 * @return
 */
int  pm_setPrimaryPathKey(Association* asok, unsigned int pathKey);

int  pm_getMaxPathRetransmisions(Association* asok);

int  pm_setRtoInitial(Association* asok, int new_rto_initial);

int  pm_getRtoInitial(Association* asok);


int  pm_setRtoMin(Association* asok, int new_rto_min);

int  pm_getRtoMin(Association* asok);

int  pm_setRtoMax(Association* asok, int new_rto_max);

int  pm_getRtoMax(Association* asok);


int  pm_setMaxPathRetransmisions(Association* asok, int new_max);

/*------------------- Functions called by ULP to read pathmanagement state info ------------------*/


/**
 * pm_readRTO returns the currently set RTO value in msecs for a certain path.
 * @param pathKey   key of the address/path
 * @return  path's current RTO
*/
int pm_readRTO(Association* asok, unsigned int pathKey);


/**
 * pm_readSRTT returns the currently set SRTT value for a certain path.
 * @param pathKey  key of the address/path
 * @return  path's current smoothed round trip time, or SCTP_UNSPECIFIED_ERROR, SCTP_PARAMETER_PROBLEM
 */
int pm_readSRTT(Association* asok, unsigned int pathKey);

/**
 * pm_readRttVar returns the currently measured value for Round-Trip
 * time variation of a certain path.
 * @param pathKey    key of the address/path
 * @return  path's current RTTvar in msecs, SCTP_UNSPECIFIED_ERROR or SCTP_PARAMETER_PROBLEM
*/
int pm_readRttVar(Association* asok, unsigned int pathKey);


/**
 * pm_readState returns the current state of the path.
 * @param pathKey  key of the questioned address
 * @return state of path (PM_INACTIVE / PM_ACTIVE), or  SCTP_PARAMETER_PROBLEM
 */
int pm_readState(Association* asok, unsigned int pathKey);



int pm_addDestinationAddress(Association* asok,
                             unsigned int newKey,
                             int newIndex);

int pm_delDestinationAddress(Association* asok,
                             unsigned int delKey,
                             int delIndex);

/*------------------- Functions called to create, init and delete pathman-instances --------------*/

/* pm_setPaths modufies number of paths and sets the primary path.
   This is required for association setup, where the local ULP provides
   only one path and the peer may provide additional paths.
   This function also initializes the path structures and starts the heartbeat timer for each
   path. For this reason it is recommended to call this function when communication up is called.
   Params: primaryPathID:      path-ID
           noOfPaths           number of paths
*/
int pm_setPaths(Association* asok, int noOfPaths, unsigned int primaryPathKey);


/**
 * pm_pathConfirmed returns the current confirmation state of the path.
 * @param pathID  path-ID
 * @return TRUE if path has been confirmed, FALSE if path has not been confirmed
*/
gboolean pm_pathConfirmed(Association* asok, unsigned int pathKey);

/**
 * pm_newPathman creates a new instance of path-management. Path management does heartbeating,
 * and path monitoring as well as RTO management
 * @param asok               pointer to the new association
 * @param numberOfPaths      number of paths of the association
 * @param primaryPathKey     initial primary path key
 * @return pointer to the newly created path-management instance
 */
void *pm_newPathman(Association* asok, int numberOfPaths, unsigned int primaryPathKey);



/**
 * Deletes the instance pointed to by pathmanPtr.
 * @param   pathmanPtr pointer to the instance that is to be deleted
 */
void pm_deletePathman(Association* asok);




/*------------------- Functions called to get/change MTU information --------------*/

/**
 * pm_readAsocMTU returns the current MTU of the asociation
 * which is the lowest path mtu
 * @return Current MTU or SCTP_PARAMETER_PROBLEM
 */
int pm_readAsocMTU(Association* asok);


/**
 * pm_readPathMTU returns the current pathMTU of the path.
 * @param pathKey  key of the questioned address
 * @return Current Path MTU or SCTP_PARAMETER_PROBLEM
 */
int pm_readPathSCTPMTU(Association* asok, unsigned int pathKey);


gboolean pm_pathMTUDMustBeDone(Association *asok, int pathKey);


void pm_pathMTUDWasDone(Association *asok, int pathKey);

#ifdef DO_PMTUD
void pm_adjustPMTU(union sockunion * Address, unsigned short remotePort, unsigned short localPort, unsigned short newMTU);
#endif /* DO_PMTUD */


#endif
