/* ScummVM - Scumm Interpreter
 * Copyright (C) 2001  Ludvig Strigeus
 * Copyright (C) 2001-2006 The ScummVM project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $URL: https://svn.sourceforge.net/svnroot/scummvm/scummvm/tags/release-0-9-0/engines/simon/charset.cpp $
 * $Id: charset.cpp 22825 2006-06-02 02:35:04Z kirben $
 *
 */

#include "common/stdafx.h"

#include "simon/simon.h"
#include "simon/intern.h"

namespace Simon {

void SimonEngine::print_char_helper_1(const byte *src, uint len) {
	uint idx;

	if (_textWindow == NULL)
		return;

	while (len-- != 0) {
		if (getGameType() == GType_FF) {
			if (getBitFlag(93)) {
				if (_curWindow == 3) {
					if ((_newLines >= _textWindow->scrollY) && (_newLines < (_textWindow->scrollY + 3)))
						windowPutChar(*src);
					if (*src == '\n')		// Do two top lines of text only
						_newLines++;
					src++;
				}
			} else {
				if (getBitFlag(94)) {
					if (_curWindow == 3) {
						if (_newLines == (_textWindow->scrollY + 7))
							windowPutChar(*src);
						if (*src == '\n')	// Do two top lines of text only
							_newLines++;
						src++;
					}
				} else {
					if (getBitFlag(92))
						delay(50);
					windowPutChar(*src++);
				}
			}
		} else {
			if (*src != 12 && _textWindow->iconPtr != NULL &&
					_fcsData1[idx = getWindowNum(_textWindow)] != 2) {
	
				_fcsData1[idx] = 2;
				_fcsData2[idx] = 1;
			}

			windowPutChar(*src++);
		}
	}
}

void SimonEngine::print_char_helper_5(WindowBlock *window) {
	uint index = getWindowNum(window);
	tidyIconArray(index);
	_fcsData1[index] = 0;
}

void SimonEngine::tidyIconArray(uint i) {
	WindowBlock *window;

	if (_fcsData2[i]) {
		mouseOff();
		window = _windowArray[i];
		drawIconArray(i, window->iconPtr->itemRef, window->iconPtr->line, window->iconPtr->classMask);
		_fcsData2[i] = 0;
		mouseOn();
	}
}

void SimonEngine::renderStringAmiga(uint vga_sprite_id, uint color, uint width, uint height, const char *txt) {
	VgaPointersEntry *vpe = &_vgaBufferPointers[2];
	byte *src, *dst, *dst_org, chr;
	uint count;

	if (vga_sprite_id >= 100) {
		vga_sprite_id -= 100;
		vpe++;
	}

	src = dst = vpe->vgaFile2;

	count = 499;
	if (vga_sprite_id == 1)
		count *= 2;

	src += vga_sprite_id * 8;
	dst += READ_BE_UINT32(src);
	WRITE_BE_UINT16(dst + 4, height);
	WRITE_BE_UINT16(dst + 6, width);

	uint charsize = width/8 * height;
	memset(dst, 0, count);
	dst_org = dst;
	int delta = 0;
	while ((chr = *txt++) != 0) {
		int tmp = chr;
		if (chr == 10) {
			dst_org += width * 10;
			dst = dst_org;
			delta = 0;
		} else if ((tmp -= '!') < 0) {
			delta += 6;
			if (delta > 8)
			{
				delta -= 8;
				dst_org++;
			}
		} else {
			byte *img = src + chr * 41;
			int CTR = img[40];
			int D3 = 8 - delta;
			for (int D2 = 9; D2 != 0; D2--)
			{
				byte *cur_dst = dst_org;
				for (int D7 = 2; D7 != 0; D7--)
				{
					chr = *img >> delta;
					if (chr)
					{
						if (color & 1) *(cur_dst + charsize * 0) |= chr;
						if (color & 2) *(cur_dst + charsize * 1) |= chr;
						if (color & 4) *(cur_dst + charsize * 2) |= chr;
						if (color & 8) *(cur_dst + charsize * 3) |= chr;
					}
					if ((D3 >= CTR) && (chr = *img++ << (D3)))
					{
						if (color & 1) *(cur_dst + charsize * 0) |= chr;
						if (color & 2) *(cur_dst + charsize * 1) |= chr;
						if (color & 4) *(cur_dst + charsize * 2) |= chr;
						if (color & 8) *(cur_dst + charsize * 3) |= chr;
					}
					color++;
				}
				chr = *img >> delta;
				if (chr)
				{
					*(cur_dst + charsize * 0) |= chr;
					*(cur_dst + charsize * 1) |= chr;
					*(cur_dst + charsize * 2) |= chr;
					*(cur_dst + charsize * 3) |= chr;
				}
				if ((D3 >= CTR) && (chr = *img++ << (D3)))
				{
					*(cur_dst + charsize * 0) |= chr;
					*(cur_dst + charsize * 1) |= chr;
					*(cur_dst + charsize * 2) |= chr;
					*(cur_dst + charsize * 3) |= chr;
				}
				cur_dst += width/8;
			}
			delta += CTR;
			if (delta > 8)
			{
				delta -= 8;
				dst_org++;
			}
		}
	}
}

void SimonEngine::renderString(uint vga_sprite_id, uint color, uint width, uint height, const char *txt) {
	VgaPointersEntry *vpe = &_vgaBufferPointers[2];
	byte *src, *dst, *p, *dst_org, chr;
	const int textHeight = (getGameType() == GType_FF) ? 15: 10;
	uint count = 0;

	if (vga_sprite_id >= 100) {
		vga_sprite_id -= 100;
		vpe++;
	}

	src = dst = vpe->vgaFile2;

	if (getGameType() == GType_FF) {
		if (vga_sprite_id == 1)
			count = 45000;
	} else {
		count = 4000;
		if (vga_sprite_id == 1)
			count *= 2;
	}

	p = dst + vga_sprite_id * 8;

	if (getGameType() == GType_FF) {
		WRITE_LE_UINT16(p + 4, height);
		WRITE_LE_UINT16(p + 6, width);
		// We need to adjust the offset to the next buffer to be right
		// after this one. By default, each buffer is only 9000 bytes
		// long. A two-line string can very well be more than twice
		// that size!
		//
		// The original seems to make an exception for sprite id 1, but
		// even the first conversation option can be a long line. For
		// some reason, I cannot reproduce the text corruption with the
		// original interpreter, though, so maybe we're missing some
		// detail here. Let's hope it's safe to always adjust the
		// buffer size anyway.
		WRITE_LE_UINT16(p + 8, READ_LE_UINT32(p) + width * height);
	} else {
		WRITE_BE_UINT16(p + 4, height);
		WRITE_BE_UINT16(p + 6, width);
	}
	dst += readUint32Wrapper(p);

	if (count != 0)
		memset(dst, 0, count);

	if (_language == Common::HB_ISR)
		dst += width - 1; // For Hebrew, start at the right edge, not the left.

	dst_org = dst;
	while ((chr = *txt++) != 0) {
		if (chr == 10) {
			dst_org += width * textHeight;
			dst = dst_org;
		} else if ((chr -= ' ') == 0) {
			dst += (_language == Common::HB_ISR ? -6 : 6); // Hebrew moves to the left, all others to the right
		} else {
			byte *img_hdr, *img;
			uint i, img_width, img_height;

			if (getGameType() == GType_FF) {
				img_hdr = src + 96 + chr * 8;
				img_height = READ_LE_UINT16(img_hdr + 4);
				img_width = READ_LE_UINT16(img_hdr + 6);
				img = src + READ_LE_UINT32(img_hdr);
			} else {
				img_hdr = src + 48 + chr * 4;
				img_height = img_hdr[2];
				img_width = img_hdr[3];
				img = src + READ_LE_UINT16(img_hdr);
			}

			if (_language == Common::HB_ISR)
				dst -= img_width - 1; // For Hebrew, move from right edge to left edge of image.
			byte *cur_dst = dst;

			assert(img_width > 0 && img_width < 50 && img_height > 0 && img_height < 50);

			do {
				for (i = 0; i != img_width; i++) {
					chr = *img++;
					if (chr) {
						if (chr == 0xF)
							chr = 207;
						else
							chr += color;
						cur_dst[i] = chr;
					}
				}
				cur_dst += width;
			} while (--img_height);

			if (_language != Common::HB_ISR) // Hebrew character movement is done higher up
				dst += img_width - 1;
		}
	}
}

static const byte feebleFontSize[208] = {
	8, 2, 5, 7, 8, 8, 8, 2, 4, 4, 8, 8, 3, 8, 2, 9,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 3, 5, 8, 5, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 9, 4, 4, 9,
	4, 8, 8, 8, 8, 8, 7, 8, 8, 4, 5, 7, 3, 8, 8, 8,
	8, 8, 8, 7, 7, 8, 8, 8, 8, 8, 8, 5, 2, 5, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 4, 4, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 4, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

uint SimonEngine::getFeebleFontSize(byte chr) {
	return feebleFontSize[chr - 32];
}

void SimonEngine::showMessageFormat(const char *s, ...) {
	char buf[STRINGBUFLEN];
	char *str;
	va_list va;

	va_start(va, s);
	vsnprintf(buf, STRINGBUFLEN, s, va);
	va_end(va);

	if (!_fcsData1[_curWindow]) {
		openTextWindow();
		if (!_showMessageFlag) {
			_windowArray[0] = _textWindow;
			if (getGameType() == GType_FF)
				showmessage_helper_3(_textWindow->textColumn, _textWindow->width);
			else
				showmessage_helper_3(_textWindow->textLength, _textWindow->textMaxLength);
		}
		_showMessageFlag = true;
		_fcsData1[_curWindow] = 1;
	}

	for (str = buf; *str; str++)
		showmessage_print_char(*str);
}

void SimonEngine::showmessage_print_char(byte chr) {
	if (chr == 12) {
		_numLettersToPrint = 0;
		_printCharCurPos = 0;
		_printCharPixelCount = 0;
		print_char_helper_1(&chr, 1);
		print_char_helper_5(_textWindow);
	} else if (chr == 0 || chr == ' ' || chr == 10) {
		bool fit;

		// Note that in FF, _printCharCurPos may be greater than
		// _printCharMaxPos. In Simon, that is probably prevented by
		// testing if _printCharCurPos == _printCharMaxPos below.

		if (getGameType() == GType_FF) {
			fit = _printCharMaxPos > _printCharCurPos + _printCharPixelCount;
		} else {
			fit = _printCharMaxPos - _printCharCurPos >= _printCharPixelCount;
		}

		if (fit) {
			_printCharCurPos += _printCharPixelCount;
			print_char_helper_1(_lettersToPrintBuf, _numLettersToPrint);

			if (_printCharCurPos == _printCharMaxPos) {
				_printCharCurPos = 0;
			} else {
				if (chr)
					print_char_helper_1(&chr, 1);
				if (chr == 10)
					_printCharCurPos = 0;
				else if (chr != 0)
					_printCharCurPos += (getGameType() == GType_FF) ? feebleFontSize[chr - 32] : 1;
			}
		} else {
			const byte newline_character = 10;
			_printCharCurPos = _printCharPixelCount;
			print_char_helper_1(&newline_character, 1);
			print_char_helper_1(_lettersToPrintBuf, _numLettersToPrint);
			if (chr == ' ') {
				print_char_helper_1(&chr, 1);
				_printCharCurPos += (getGameType() == GType_FF) ? feebleFontSize[chr - 32] : 1;
			} else {
				print_char_helper_1(&chr, 1);
				_printCharCurPos = 0;
			}
		}
		_numLettersToPrint = 0;
		_printCharPixelCount = 0;
	} else {
		_lettersToPrintBuf[_numLettersToPrint++] = chr;
		_printCharPixelCount += (getGameType() == GType_FF) ? feebleFontSize[chr - 32] : 1;
	}
}

void SimonEngine::openTextWindow() {
	if (_textWindow)
		return;

	if (getGameType() == GType_FF)
		_textWindow = openWindow(64, 96, 384, 172, 1, 0, 15);
	else
		_textWindow = openWindow(8, 144, 24, 6, 1, 0, 15);
}

void SimonEngine::showmessage_helper_3(uint a, uint b) {
	_printCharCurPos = a;
	_printCharMaxPos = b;
	_printCharPixelCount = 0;
	_numLettersToPrint = 0;
	_newLines = 0;
}

void SimonEngine::windowPutChar(WindowBlock *window, byte c, byte b) {
	byte width = 6;

	if (c == 12) {
		clearWindow(window);
	} else if (c == 13 || c == 10) {
		video_putchar_newline(window);
	} else if ((c == 1 && _language != Common::HB_ISR) || (c == 8)) {
		if (_language == Common::HB_ISR) {
			if (b >= 64 && b < 91)
				width = _hebrewCharWidths [b - 64];

			if (window->textLength != 0) {
				window->textLength--;
				window->textColumnOffset += width;
				if (window->textColumnOffset >= 8) {
					window->textColumnOffset -= 8;
					window->textColumn--;
				}
			}
		} else {
			int8 val = (c == 8) ? 6 : 4;

			if (window->textLength != 0) {
				window->textLength--;
				window->textColumnOffset -= val;
				if ((int8)window->textColumnOffset < val) {
					window->textColumnOffset += 8;
					window->textColumn--;
				}
			}
		}
	} else if (c >= 32) {
		if (getGameType() == GType_FF) {
			video_putchar_drawchar(window, window->textColumn + window->x, window->textRow + window->y, c);
			window->textColumn += feebleFontSize[c - 32];
			return;
		}

		if (window->textLength == window->textMaxLength) {
			video_putchar_newline(window);
		} else if (window->textRow == window->height) {
			video_putchar_newline(window);
			window->textRow--;
		}

		if (_language == Common::HB_ISR) {
			if (c >= 64 && c < 91)
				width = _hebrewCharWidths [c - 64];
			window->textColumnOffset -= width;
			if (window->textColumnOffset >= width) {
				window->textColumnOffset += 8;
				window->textColumn++;
			}
			video_putchar_drawchar(window, (window->width + window->x - window->textColumn) * 8, window->textRow * 8 + window->y, c);
			window->textLength++;
		} else {
			video_putchar_drawchar(window, (window->textColumn + window->x) * 8, window->textRow * 8 + window->y, c);

			window->textLength++;
			window->textColumnOffset += 6;
			if (c == 'i' || c == 'l')
				window->textColumnOffset -= 2;

			if (window->textColumnOffset >= 8) {
				window->textColumnOffset -= 8;
				window->textColumn++;
			}
		}
	}
}

void SimonEngine::video_putchar_newline(WindowBlock *window) {
	if (getGameType() == GType_FF) {
		if (_noOracleScroll == 0) {
			if (window->height < window->textRow + 30) {
				if (!getBitFlag(94)) {
					_noOracleScroll = 1;
					if (getBitFlag(92)) {
						_noOracleScroll = 0;
						checkLinkBox();
						scrollOracle();
						linksUp();
						window->scrollY++;
						_oracleMaxScrollY++;
					} else {
						_oracleMaxScrollY++;
						checkLinkBox();
					}
				}
			} else {
				window->textRow += 15;
				checkLinkBox();
			}
		} else {
			_oracleMaxScrollY++;
			checkLinkBox();
		}
	} else {
		if (window->textRow != window->height)
			window->textRow++;
	}

	window->textColumn = 0;
	window->textColumnOffset = 0;
	window->textLength = 0;
}

#ifdef PALMOS_68K
static const byte *russian_video_font;
static const byte *polish_video_font;
static const byte *french_video_font;
static const byte *german_video_font;
static const byte *hebrew_video_font;
static const byte *italian_video_font;
static const byte *spanish_video_font;
static const byte *video_font;
#else
static const byte russian_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 0, 100, 40, 48, 40, 100, 0,
	0, 0, 96, 48, 40, 40, 112, 0,
	60, 68, 68, 60, 36, 68, 68, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	72, 84, 84, 116, 84, 84, 72, 0,
	0, 0, 60, 68, 60, 36, 100, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	16, 32, 0, 120, 112, 64, 56, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	0, 0, 96, 48, 40, 40, 112, 0,
	0, 0, 112, 88, 20, 20, 56, 0,
	0, 0, 120, 4, 28, 4, 120, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	0, 0, 84, 84, 84, 84, 56, 4,
	56, 68, 68, 124, 68, 68, 68, 0,
	124, 36, 32, 56, 36, 36, 120, 0,
	120, 36, 36, 56, 36, 36, 120, 0,
	124, 36, 32, 32, 32, 32, 112, 0,
	56, 40, 40, 40, 40, 40, 124, 68,
	124, 36, 32, 56, 32, 36, 124, 0,
	84, 84, 84, 56, 84, 84, 84, 0,
	56, 68, 4, 24, 4, 68, 56, 0,
	68, 68, 76, 84, 100, 68, 68, 0,
	100, 40, 40, 48, 40, 36, 100, 0,
	28, 36, 36, 36, 36, 36, 100, 0,
	68, 108, 84, 68, 68, 68, 68, 0,
	68, 68, 68, 124, 68, 68, 68, 0,
	56, 68, 68, 68, 68, 68, 56, 0,
	124, 68, 68, 68, 68, 68, 68, 0,
	120, 36, 36, 56, 32, 32, 112, 0,
	56, 68, 64, 64, 64, 68, 56, 0,
	124, 84, 16, 16, 16, 16, 56, 0,
	100, 36, 36, 28, 4, 4, 56, 0,
	56, 84, 84, 84, 56, 16, 56, 0,
	108, 40, 16, 16, 40, 40, 108, 0,
	72, 72, 72, 72, 72, 72, 60, 4,
	76, 72, 72, 56, 8, 8, 28, 0,
	84, 84, 84, 84, 84, 84, 60, 0,
	84, 84, 84, 84, 84, 84, 56, 4,
	56, 68, 4, 28, 4, 68, 56, 0,
	0, 0, 68, 100, 84, 84, 100, 0,
	0, 0, 72, 84, 116, 84, 72, 0,
	0, 0, 60, 68, 60, 36, 100, 0,
	0, 0, 120, 4, 24, 4, 120, 0,
	0, 0, 100, 40, 48, 40, 100, 0,
	60, 68, 68, 60, 36, 68, 68, 0,
	0, 0, 56, 4, 60, 68, 60, 0,
	60, 64, 32, 56, 68, 68, 56, 0,
	48, 72, 80, 120, 68, 68, 56, 0,
	0, 0, 120, 4, 56, 64, 60, 0,
	56, 4, 4, 60, 68, 68, 56, 0,
	0, 0, 56, 68, 120, 64, 56, 0,
	40, 0, 56, 68, 120, 64, 56, 0,
	0, 0, 84, 84, 56, 84, 84, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	0, 0, 68, 68, 68, 68, 60, 0,
	56, 0, 68, 68, 68, 68, 60, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 28, 36, 36, 36, 100, 0,
	0, 0, 68, 108, 84, 68, 68, 0,
	0, 0, 56, 68, 68, 68, 56, 0,
	0, 0, 68, 68, 124, 68, 68, 0,
	0, 0, 124, 68, 68, 68, 68, 0,
	0, 0, 120, 36, 36, 56, 32, 112,
	0, 0, 60, 64, 64, 64, 60, 0,
	0, 0, 124, 84, 16, 16, 56, 0,
	0, 0, 68, 68, 60, 4, 56, 0,
	48, 16, 56, 84, 84, 56, 16, 56,
	0, 0, 68, 40, 16, 40, 68, 0,
	0, 0, 72, 72, 72, 72, 60, 4,
	0, 0, 76, 72, 72, 56, 8, 28,
	0, 0, 84, 84, 84, 84, 60, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte polish_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 0, 112, 136, 248, 128, 112, 8,
	0, 16, 120, 128, 112, 8, 240, 0,
	192, 64, 64, 96, 192, 64, 224, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 32, 112, 136, 136, 136, 112, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	0, 32, 112, 136, 128, 136, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	0, 32, 240, 136, 136, 136, 136, 0,
	80, 0, 136, 136, 136, 136, 112, 0,
	0, 32, 248, 144, 32, 72, 248, 0,
	8, 32, 248, 144, 32, 72, 248, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	0, 0, 112, 8, 120, 136, 120, 4,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	224, 64, 64, 96, 192, 72, 248, 0,
	16, 120, 128, 112, 8, 136, 112, 0,
	248, 72, 64, 112, 64, 72, 248, 16,
	32, 248, 16, 32, 64, 136, 248, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte french_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	32, 80, 0, 112, 136, 136, 112, 0,
	32, 80, 0, 112, 8, 248, 120, 0,
	112, 136, 128, 128, 136, 112, 32, 96,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	112, 136, 128, 128, 136, 112, 32, 96,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	32, 80, 0, 144, 144, 144, 104, 0,
	32, 16, 0, 112, 248, 128, 112, 0,
	32, 80, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 192, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 0, 112, 136, 128, 112, 32, 96,
	160, 0, 192, 64, 64, 64, 224, 0,
	32, 16, 0, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte german_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 0, 112, 8, 120, 136, 120, 0,
	80, 0, 112, 136, 136, 136, 112, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	96, 144, 144, 160, 144, 144, 160, 128,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	80, 0, 112, 136, 248, 136, 136, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	136, 112, 136, 136, 136, 136, 112, 0,
	80, 0, 136, 136, 136, 136, 112, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 192, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 48, 72, 64, 72, 48, 16, 48,
	0, 80, 0, 96, 32, 40, 48, 0,
	32, 16, 0, 152, 144, 144, 232, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte hebrew_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 144, 0, 96, 144, 144, 104, 0,
	0, 144, 0, 96, 144, 144, 96, 0,
	0, 144, 0, 144, 144, 144, 96, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 112, 136, 240, 136, 136, 240, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	16, 32, 0, 120, 112, 64, 56, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	5, 5, 4, 6, 5, 3, 4, 5,
	6, 3, 5, 5, 4, 6, 5, 3,
	4, 6, 5, 6, 6, 6, 5, 5,
	5, 6, 5, 6, 6, 6, 6, 6,
	112, 136, 8, 16, 32, 0, 32, 0,
	0, 0, 144, 80, 160, 144, 144, 0,
	0, 0, 224, 32, 32, 32, 240, 0,
	0, 0, 224, 32, 96, 160, 160, 0,
	0, 0, 248, 16, 16, 16, 16, 0,
	0, 0, 240, 16, 16, 144, 144, 0,
	0, 0, 192, 64, 64, 64, 64, 0,
	0, 0, 224, 64, 32, 64, 64, 0,
	0, 0, 240, 144, 144, 144, 144, 0,
	0, 0, 184, 168, 136, 136, 112, 0,
	0, 0, 192, 64, 0, 0, 0, 0,
	0, 0, 240, 16, 16, 16, 16, 16,
	0, 0, 224, 16, 16, 16, 224, 0,
	128, 128, 224, 32, 32, 32, 192, 0,
	0, 0, 248, 72, 72, 72, 120, 0,
	0, 0, 176, 208, 144, 144, 176, 0,
	0, 0, 192, 64, 64, 64, 64, 64,
	0, 0, 96, 32, 32, 32, 224, 0,
	0, 0, 248, 72, 72, 72, 48, 0,
	0, 0, 80, 80, 80, 80, 224, 0,
	0, 0, 248, 72, 104, 8, 8, 8,
	0, 0, 248, 72, 104, 8, 248, 0,
	0, 0, 216, 72, 48, 16, 16, 16,
	0, 0, 144, 80, 32, 16, 240, 0,
	0, 0, 240, 16, 144, 160, 128, 128,
	0, 0, 240, 16, 16, 16, 16, 0,
	0, 0, 168, 168, 200, 136, 112, 0,
	0, 0, 240, 80, 80, 80, 208, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 48, 72, 64, 72, 48, 16, 48,
	0, 80, 0, 96, 32, 40, 48, 0,
	32, 16, 0, 152, 144, 144, 232, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte italian_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 0, 112, 8, 120, 136, 120, 0,
	80, 0, 112, 136, 136, 136, 112, 0,
	32, 16, 0, 112, 136, 136, 112, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	32, 16, 0, 192, 64, 64, 224, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	32, 16, 0, 112, 136, 136, 112, 0,
	32, 16, 0, 112, 248, 128, 112, 0,
	32, 80, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 96, 32, 40, 48, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 0, 112, 136, 128, 112, 32, 96,
	160, 0, 192, 64, 64, 64, 224, 0,
	32, 16, 0, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte spanish_video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	80, 0, 112, 8, 120, 136, 120, 0,
	80, 0, 112, 136, 136, 136, 112, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	96, 144, 144, 160, 144, 144, 160, 128,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	32, 64, 0, 112, 248, 128, 112, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 64, 0, 112, 8, 248, 120, 0,
	32, 64, 0, 192, 64, 64, 224, 0,
	32, 64, 0, 112, 136, 136, 112, 0,
	32, 64, 0, 144, 144, 144, 104, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	80, 160, 0, 240, 136, 136, 136, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	32, 0, 32, 64, 128, 136, 112, 0,
	32, 0, 32, 32, 112, 112, 32, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};

static const byte video_font[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	32, 112, 112, 32, 32, 0, 32, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 144, 0, 96, 144, 144, 104, 0,
	0, 144, 0, 96, 144, 144, 96, 0,
	0, 144, 0, 144, 144, 144, 96, 0,
	0, 16, 40, 16, 42, 68, 58, 0,
	48, 48, 96, 0, 0, 0, 0, 0,
	0, 4, 8, 8, 8, 8, 4, 0,
	0, 32, 16, 16, 16, 16, 32, 0,
	0, 0, 20, 8, 62, 8, 20, 0,
	0, 112, 136, 240, 136, 136, 240, 0,
	0, 0, 0, 0, 0, 48, 48, 96,
	0, 0, 0, 240, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 48, 48, 0,
	16, 32, 0, 120, 112, 64, 56, 0,
	112, 136, 152, 168, 200, 136, 112, 0,
	32, 96, 32, 32, 32, 32, 112, 0,
	112, 136, 8, 48, 64, 136, 248, 0,
	112, 136, 8, 48, 8, 136, 112, 0,
	16, 48, 80, 144, 248, 16, 56, 0,
	248, 128, 240, 8, 8, 136, 112, 0,
	48, 64, 128, 240, 136, 136, 112, 0,
	248, 136, 8, 16, 32, 32, 32, 0,
	112, 136, 136, 112, 136, 136, 112, 0,
	112, 136, 136, 120, 8, 16, 96, 0,
	0, 0, 48, 48, 0, 48, 48, 0,
	32, 16, 0, 112, 8, 248, 120, 0,
	32, 80, 0, 144, 144, 144, 104, 0,
	80, 0, 144, 144, 144, 144, 104, 0,
	32, 80, 0, 112, 248, 128, 112, 0,
	112, 136, 8, 16, 32, 0, 32, 0,
	32, 80, 0, 192, 64, 64, 224, 0,
	112, 136, 136, 248, 136, 136, 136, 0,
	240, 72, 72, 112, 72, 72, 240, 0,
	48, 72, 128, 128, 128, 72, 48, 0,
	224, 80, 72, 72, 72, 80, 224, 0,
	248, 72, 64, 112, 64, 72, 248, 0,
	248, 72, 64, 112, 64, 64, 224, 0,
	48, 72, 128, 152, 136, 72, 56, 0,
	136, 136, 136, 248, 136, 136, 136, 0,
	248, 32, 32, 32, 32, 32, 248, 0,
	24, 8, 8, 8, 136, 136, 112, 0,
	200, 72, 80, 96, 80, 72, 200, 0,
	224, 64, 64, 64, 64, 72, 248, 0,
	136, 216, 168, 168, 136, 136, 136, 0,
	136, 200, 168, 152, 136, 136, 136, 0,
	112, 136, 136, 136, 136, 136, 112, 0,
	240, 72, 72, 112, 64, 64, 224, 0,
	112, 136, 136, 136, 136, 168, 112, 8,
	240, 72, 72, 112, 72, 72, 200, 0,
	112, 136, 128, 112, 8, 136, 112, 0,
	248, 168, 32, 32, 32, 32, 112, 0,
	136, 136, 136, 136, 136, 136, 120, 0,
	136, 136, 136, 80, 80, 32, 32, 0,
	136, 136, 136, 136, 168, 216, 136, 0,
	136, 136, 80, 32, 80, 136, 136, 0,
	136, 136, 136, 112, 32, 32, 112, 0,
	248, 136, 16, 32, 64, 136, 248, 0,
	0, 14, 8, 8, 8, 8, 14, 0,
	0, 128, 64, 32, 16, 8, 4, 0,
	0, 112, 16, 16, 16, 16, 112, 0,
	0, 48, 72, 64, 72, 48, 16, 48,
	0, 80, 0, 96, 32, 40, 48, 0,
	32, 16, 0, 152, 144, 144, 232, 0,
	0, 0, 112, 8, 120, 136, 120, 0,
	192, 64, 80, 104, 72, 72, 112, 0,
	0, 0, 112, 136, 128, 136, 112, 0,
	24, 16, 80, 176, 144, 144, 112, 0,
	0, 0, 112, 136, 248, 128, 112, 0,
	48, 72, 64, 224, 64, 64, 224, 0,
	0, 0, 104, 144, 144, 112, 136, 112,
	192, 64, 80, 104, 72, 72, 200, 0,
	64, 0, 192, 64, 64, 64, 224, 0,
	8, 0, 8, 8, 8, 8, 136, 112,
	192, 64, 72, 80, 96, 80, 200, 0,
	192, 64, 64, 64, 64, 64, 224, 0,
	0, 0, 144, 216, 168, 136, 136, 0,
	0, 0, 240, 136, 136, 136, 136, 0,
	0, 0, 112, 136, 136, 136, 112, 0,
	0, 0, 176, 72, 72, 112, 64, 224,
	0, 0, 104, 144, 144, 112, 16, 56,
	0, 0, 176, 72, 72, 64, 224, 0,
	0, 0, 120, 128, 112, 8, 240, 0,
	64, 64, 240, 64, 64, 72, 48, 0,
	0, 0, 144, 144, 144, 144, 104, 0,
	0, 0, 136, 136, 136, 80, 32, 0,
	0, 0, 136, 136, 168, 216, 144, 0,
	0, 0, 136, 80, 32, 80, 136, 0,
	0, 0, 136, 136, 136, 112, 32, 192,
	0, 0, 248, 144, 32, 72, 248, 0,
	32, 80, 0, 96, 144, 144, 96, 0,
	0, 14, 8, 48, 8, 8, 14, 0,
	0, 8, 8, 8, 8, 8, 8, 0,
	0, 112, 16, 12, 16, 16, 112, 0,
	0, 0, 0, 0, 0, 0, 248, 0,
	252, 252, 252, 252, 252, 252, 252, 252,
	240, 240, 240, 240, 240, 240, 240, 240,
};
#endif

static const byte feeble_video_font[] = {
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	128,128,128,128,128,128,128,0,0,128,0,0,0,
	144,144,144,0,0,0,0,0,0,0,0,0,0,
	0,72,252,72,72,252,72,0,0,0,0,0,0,
	124,146,144,144,124,18,18,18,146,124,16,16,0,
	62,98,146,100,8,16,32,76,146,140,0,0,0,
	0,112,136,136,80,32,82,138,132,122,0,0,0,
	128,128,128,0,0,0,0,0,0,0,0,0,0,
	32,64,128,128,128,128,128,128,64,32,0,0,0,
	128,64,32,32,32,32,32,32,64,128,0,0,0,
	16,146,84,56,56,84,146,16,0,0,0,0,0,
	0,0,16,16,16,254,16,16,16,0,0,0,0,
	0,0,0,0,0,0,0,0,0,64,64,128,0,
	0,0,0,0,0,254,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,128,0,0,0,
	0,0,1,2,4,8,16,32,64,128,128,0,0,
	120,132,132,132,132,132,132,132,132,120,0,0,0,
	16,48,16,16,16,16,16,16,16,56,0,0,0,
	120,132,132,4,4,8,16,32,64,252,0,0,0,
	124,130,130,2,2,28,2,2,130,124,0,0,0,
	4,12,20,36,68,132,254,4,4,4,0,0,0,
	254,128,128,252,2,2,2,2,130,124,0,0,0,
	62,64,128,252,130,130,130,130,130,124,0,0,0,
	254,2,2,2,2,4,8,16,16,16,0,0,0,
	120,132,132,132,120,132,132,132,132,120,0,0,0,
	124,130,130,130,130,130,126,2,2,124,0,0,0,
	0,0,0,0,128,0,0,0,128,0,0,0,0,
	0,0,0,0,0,64,0,0,0,64,64,128,0,
	0,0,16,32,64,128,64,32,16,0,0,0,0,
	0,0,0,254,0,0,254,0,0,0,0,0,0,
	0,0,128,64,32,16,32,64,128,0,0,0,0,
	60,66,130,2,4,8,16,16,0,16,16,0,0,
	60,66,130,154,166,166,156,128,128,126,0,0,0,
	124,130,130,130,130,254,130,130,130,130,0,0,0,
	252,130,130,130,252,130,130,130,130,252,0,0,0,
	124,130,130,128,128,128,128,130,130,124,0,0,0,
	252,130,130,130,130,130,130,130,130,252,0,0,0,
	254,128,128,128,128,252,128,128,128,254,0,0,0,
	254,128,128,128,128,252,128,128,128,128,0,0,0,
	124,130,130,128,128,134,130,130,130,124,0,0,0,
	130,130,130,130,130,254,130,130,130,130,0,0,0,
	224,64,64,64,64,64,64,64,64,224,0,0,0,
	30,2,2,2,2,2,130,130,130,124,0,0,0,
	130,130,132,136,144,224,144,136,132,130,0,0,0,
	128,128,128,128,128,128,128,128,128,254,0,0,0,
	130,198,170,146,130,130,130,130,130,130,0,0,0,
	130,130,194,162,146,138,134,130,130,130,0,0,0,
	124,130,130,130,130,130,130,130,130,124,0,0,0,
	252,130,130,130,130,252,128,128,128,128,0,0,0,
	124,130,130,130,130,130,130,130,130,126,2,2,0,
	252,130,130,130,130,252,130,130,130,130,0,0,0,
	124,130,128,128,124,2,2,130,130,124,0,0,0,
	254,16,16,16,16,16,16,16,16,16,0,0,0,
	130,130,130,130,130,130,130,130,130,124,0,0,0,
	130,130,130,130,130,130,130,68,40,16,0,0,0,
	130,130,130,130,130,130,130,146,170,198,0,0,0,
	130,68,40,16,16,16,16,40,68,130,0,0,0,
	130,130,130,130,68,40,16,16,16,16,0,0,0,
	254,2,2,4,8,16,32,64,128,254,0,0,0,
	224,128,128,128,128,128,128,128,128,224,0,0,0,
	128,128,64,32,16,8,4,2,1,0,0,0,0,
	224,32,32,32,32,32,32,32,32,224,0,0,0,
	160,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,255,0,0,0,
	32,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,62,66,130,130,130,134,122,0,0,0,
	128,128,128,252,130,130,130,130,130,252,0,0,0,
	0,0,0,124,130,128,128,128,130,124,0,0,0,
	2,2,2,126,130,130,130,130,130,126,0,0,0,
	0,0,0,124,130,130,254,128,128,126,0,0,0,
	28,32,32,248,32,32,32,32,32,32,0,0,0,
	0,0,0,62,66,130,130,130,130,126,2,2,124,
	128,128,128,252,130,130,130,130,130,130,0,0,0,
	64,0,0,64,192,64,64,64,64,224,0,0,0,
	16,0,0,16,16,16,16,16,16,16,16,16,224,
	128,128,128,132,136,144,224,144,136,132,0,0,0,
	192,64,64,64,64,64,64,64,64,64,0,0,0,
	0,0,0,252,146,146,146,146,146,146,0,0,0,
	0,0,0,156,162,194,130,130,130,130,0,0,0,
	0,0,0,124,130,130,130,130,130,124,0,0,0,
	0,0,0,252,130,130,130,130,130,252,128,128,128,
	0,0,0,126,130,130,130,130,130,126,2,2,2,
	0,0,0,156,162,194,128,128,128,128,0,0,0,
	0,0,0,124,128,128,120,4,4,248,0,0,0,
	32,32,32,248,32,32,32,32,32,28,0,0,0,
	0,0,0,130,130,130,130,130,130,126,0,0,0,
	0,0,0,130,130,130,130,68,40,16,0,0,0,
	0,0,0,146,146,146,146,146,146,124,0,0,0,
	0,0,0,130,68,40,16,40,68,130,0,0,0,
	0,0,0,130,130,130,130,130,130,126,2,2,124,
	0,0,0,254,4,8,16,32,64,254,0,0,0,
	48,64,64,64,64,128,64,64,64,64,48,0,0,
	128,128,128,128,128,128,128,128,128,128,128,128,0,
	192,32,32,32,32,16,32,32,32,32,192,0,0,
	152,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,254,0,0,
	124,130,130,128,128,128,128,130,130,124,16,16,48,
	0,40,0,130,130,130,130,130,130,126,0,0,0,
	16,32,0,124,130,130,254,128,128,126,0,0,0,
	16,40,0,62,66,130,130,130,134,122,0,0,0,
	0,40,0,62,66,130,130,130,134,122,0,0,0,
	16,8,0,62,66,130,130,130,134,122,0,0,0,
	8,20,8,62,66,130,130,130,134,122,0,0,0,
	0,0,0,124,130,128,128,128,130,124,16,16,48,
	16,40,0,124,130,130,254,128,128,126,0,0,0,
	0,40,0,124,130,130,254,128,128,126,0,0,0,
	16,8,0,124,130,130,254,128,128,126,0,0,0,
	0,160,0,64,192,64,64,64,64,224,0,0,0,
	64,160,0,64,192,64,64,64,64,224,0,0,0,
	128,64,0,64,192,64,64,64,64,224,0,0,0,
	40,0,124,130,130,130,254,130,130,130,0,0,0,
	16,40,124,130,130,130,254,130,130,130,0,0,0,
	40,0,254,128,128,252,128,128,128,254,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	16,40,0,124,130,130,130,130,130,124,0,0,0,
	0,40,0,124,130,130,130,130,130,124,0,0,0,
	32,16,0,124,130,130,130,130,130,124,0,0,0,
	16,40,0,130,130,130,130,130,130,126,0,0,0,
	16,8,0,130,130,130,130,130,130,126,0,0,0,
	0,40,0,130,130,130,130,130,130,126,2,2,124,
	40,0,124,130,130,130,130,130,130,124,0,0,0,
	40,0,130,130,130,130,130,130,130,124,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	8,16,0,62,66,130,130,130,134,122,0,0,0,
	64,128,0,64,192,64,64,64,64,224,0,0,0,
	16,32,0,124,130,130,130,130,130,124,0,0,0,
	8,16,0,130,130,130,130,130,130,126,0,0,0,
	20,40,0,156,162,194,130,130,130,130,0,0,0,
	20,40,130,194,162,146,138,134,130,130,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	16,16,0,16,16,32,64,128,130,132,120,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	128,0,0,128,128,128,128,128,128,128,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
	60,66,130,130,130,132,132,130,130,130,156,128,128,
	0,0,0,0,0,0,0,0,0,0,0,0,0,
};

void SimonEngine::video_putchar_drawchar(WindowBlock *window, uint x, uint y, byte chr) {
	const byte *src;
	byte color, *dst;
	uint h, w, i;

	if (_noOracleScroll)
		return;

	_lockWord |= 0x8000;

	dst = getFrontBuf() + y * _dxSurfacePitch + x + window->textColumnOffset;

	if (getGameType() == GType_FF) {
		h = 13;
		w =  feebleFontSize[chr - 0x20];

		src = feeble_video_font + (chr - 0x20) * 13;
	} else {
		h = 8;
		w = 6;

		switch(_language) {
		case Common::RU_RUS:
			src = russian_video_font + (chr - 0x20) * 8;
			break;
		case Common::PL_POL:
			src = polish_video_font + (chr - 0x20) * 8;
			break;
		case Common::HB_ISR:
			src = hebrew_video_font + (chr - 0x20) * 8;
			break;
		case Common::ES_ESP:
			src = spanish_video_font + (chr - 0x20) * 8;
			break;
		case Common::IT_ITA:
			src = italian_video_font + (chr - 0x20) * 8;
			break;
		case Common::FR_FRA:
			src = french_video_font + (chr - 0x20) * 8;
			break;
		case Common::DE_DEU:
			src = german_video_font + (chr - 0x20) * 8;
			break;
		case Common::EN_ANY:
			src = video_font + (chr - 0x20) * 8;
			break;
		default:
			error("video_putchar_drawchar: Unknown language %d\n", _language);
		}
	}

	color = window->text_color;

	do {
		int8 b = *src++;
		i = 0;
		do {
			if (b < 0) {
				if (getGameType() == GType_FF) {
					if (dst[i] == 0)
						dst[i] = color;
				} else {
					dst[i] = color;
				}
			}

			b <<= 1;
		} while (++i != w);
		dst += _dxSurfacePitch;
	} while (--h);

	_lockWord &= ~0x8000;
}

} // End of namespace Simon

#ifdef PALMOS_68K
#include "scumm_globals.h"

_GINIT(Simon_Charset)
_GSETPTR(Simon::russian_video_font, GBVARS_RUSSIANVIDEOFONT_INDEX, byte, GBVARS_SIMON)
//_GSETPTR(Simon::polish_video_font, GBVARS_POLISHVIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GSETPTR(Simon::french_video_font, GBVARS_FRENCHVIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GSETPTR(Simon::german_video_font, GBVARS_GERMANVIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GSETPTR(Simon::hebrew_video_font, GBVARS_HEBREWVIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GSETPTR(Simon::italian_video_font, GBVARS_ITALIANVIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GSETPTR(Simon::spanish_video_font, GBVARS_SPANISHVIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GSETPTR(Simon::video_font, GBVARS_VIDEOFONT_INDEX, byte, GBVARS_SIMON)
_GEND

_GRELEASE(Simon_Charset)
_GRELEASEPTR(GBVARS_RUSSIANVIDEOFONT_INDEX, GBVARS_SIMON)
//_GRELEASEPTR(GBVARS_POLISHVIDEOFONT_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_FRENCHVIDEOFONT_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_GERMANVIDEOFONT_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_HEBREWVIDEOFONT_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_ITALIANVIDEOFONT_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_SPANISHVIDEOFONT_INDEX, GBVARS_SIMON)
_GRELEASEPTR(GBVARS_VIDEOFONT_INDEX, GBVARS_SIMON)
_GEND

#endif
