#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <cstring>
#include <cctype>
#include <cerrno>

#include <sys/stat.h>
#include <sys/types.h>
#include <zlib.h>


#ifdef HAVE_MMAP
#  include <fcntl.h>
#  include <sys/mman.h>
#  include <unistd.h>
#elif defined(_WIN32)
#  include <windows.h>
#endif
#include <algorithm>
#include <deque>

#define NEW_CMP

#include "distance.h"

#include "lib.h"



#ifdef MEASURE_TIME
double Lib::middle_lookup_time, Lib::measure_time;
#endif

// Notice: read src/tools/DICTFILE_FORMAT for the dictionary 
// file's format information!

class MapFile {
public:
  MapFile(void) : data(NULL) {}
  ~MapFile();
  bool open(const gchar *file_name, gulong file_size=gulong(-1));
  inline gchar *begin(void) { return data; }
private:
  gchar *data;
  gulong size;
#ifdef HAVE_MMAP
  int mmap_fd;
#elif defined(_WIN32)
  HANDLE hFile;
  HANDLE hFileMap;
#endif
};

inline bool MapFile::open(const gchar *file_name, gulong file_size)
{
  size=file_size;
#ifdef HAVE_MMAP
  if ((mmap_fd = ::open(file_name, O_RDONLY)) < 0) {
    //g_print("Open file %s failed!\n",fullfilename);
    return false;
  }
  if (gulong(-1)==file_size) {
    struct stat stat_info;
    if (fstat(mmap_fd, &stat_info)==-1)
      return false;
    file_size=stat_info.st_size;
  }
  data = (gchar *)mmap( NULL, file_size, PROT_READ, MAP_SHARED, mmap_fd, 0);
  if ((void *)data == (void *)(-1)) {
    //g_print("mmap file %s failed!\n",idxfilename);
    data=NULL;
    return false;
  }
#else
#  ifdef _WIN32
  hFile = CreateFile(file_name, GENERIC_READ, 0, NULL, OPEN_ALWAYS, 
		     FILE_ATTRIBUTE_NORMAL, 0);
  if (gulong(-1)==file_size)
    file_size=GetFileSize(hFile, NULL);
  
  hFileMap = CreateFileMapping(hFile, NULL, PAGE_READONLY, 0,  
			       file_size, NULL);
  data = (gchar *)MapViewOfFile(hFileMap, FILE_MAP_READ, 0, 0, file_size);
#  else
  gsize read_len;
  if (!g_file_get_contents(file_name, &data, &read_len, NULL))
    return false;

  if (read_len!=file_size)
    return false;		
#  endif
#endif

  return true;
}

inline MapFile::~MapFile()
{
  if (!data)
    return;
#ifdef HAVE_MMAP
  munmap(data, size);
  close(mmap_fd);
#else
#  ifdef _WIN32
  UnmapViewOfFile(data);
  CloseHandle(hFileMap);
  CloseHandle(hFile);
#  else
		g_free(data);
#  endif
#endif			
}

inline bool bIsVowel(gchar inputchar)
{
  gchar ch = g_ascii_toupper(inputchar);
  return( ch=='A' || ch=='E' || ch=='I' || ch=='O' || ch=='U' );
}

bool bIsPureEnglish(const gchar *str) 
{ 
  // i think this should work even when it is UTF8 string :).
  for (int i=0; str[i]!=0; i++) 
    //if(str[i]<0)
    //if(str[i]<32 || str[i]>126) // tab equal 9,so this is not OK.
    // Better use isascii() but not str[i]<0 while char is default unsigned in arm
    if (!isascii(str[i])) 
            return false;            
  return true;	
}

bool bContainRule(const char* sWord)
{
  for (int i=0; sWord[i]!=0; i++)   
    if (sWord[i]=='*' || sWord[i]=='?')   
      return true;
   
  return false;
}

inline gint stardict_strcmp(const gchar *s1, const gchar *s2) 
{
#ifndef NEW_CMP
  gint a=g_ascii_strcasecmp(s1, s2);
  if (a == 0)
    return strcmp(s1, s2);
  else
    return a;
#else
  gint res=0;
  while (*s1 && *s2) {
    res=g_unichar_tolower(g_utf8_get_char(s1))-g_unichar_tolower(g_utf8_get_char(s2));
    if (res!=0)
      return res;
    s1=g_utf8_next_char(s1);
    s2=g_utf8_next_char(s2);
  }

  if (*s1 && !*s2)
    return 1;
  if (!*s1 && *s2)
    return -1;
  return 0;
#endif
}

bool DictInfo::load_from_ifo_file(const gchar *ifofilename,
				  bool istreedict)
{
  ifo_file_name=ifofilename;
  gchar *buffer;
  if (!g_file_get_contents(ifofilename, &buffer, NULL, NULL))
    return false;
  
#define TREEDICT_MAGIC_DATA "StarDict's treedict ifo file\nversion=2.4.2\n"
#define DICT_MAGIC_DATA "StarDict's dict ifo file\nversion=2.4.2\n"
  const gchar *magic_data=istreedict ? TREEDICT_MAGIC_DATA : DICT_MAGIC_DATA;
  if (!g_str_has_prefix(buffer, magic_data)) {
    g_free(buffer);
    return false;
  }

  gchar *p1,*p2,*p3;
  
  p1 = buffer + strlen(magic_data)-1;

  p2 = strstr(p1,"\nwordcount=");
  if (!p2) {
    g_free(buffer);
    return false;
  }
  
  p3 = strchr(p2+ sizeof("\nwordcount=")-1,'\n');
  gchar *tmpstr = (gchar *)g_memdup(p2+sizeof("\nwordcount=")-1, p3-(p2+sizeof("\nwordcount=")-1)+1);
  tmpstr[p3-(p2+sizeof("\nwordcount=")-1)] = '\0';
  wordcount = atol(tmpstr);
  g_free(tmpstr);

  if (istreedict) {
    p2 = strstr(p1,"\ntdxfilesize=");
    if (!p2) {
      g_free(buffer);
      return false;
    }
    p3 = strchr(p2+ sizeof("\ntdxfilesize=")-1,'\n');
    tmpstr = (gchar *)g_memdup(p2+sizeof("\ntdxfilesize=")-1, p3-(p2+sizeof("\ntdxfilesize=")-1)+1);
    tmpstr[p3-(p2+sizeof("\ntdxfilesize=")-1)] = '\0';
    index_file_size = atol(tmpstr);
    g_free(tmpstr);
  } else {
  
    p2 = strstr(p1,"\nidxfilesize=");
    if (!p2) {
      g_free(buffer);
      return false;
    }
    
    p3 = strchr(p2+ sizeof("\nidxfilesize=")-1,'\n');
    tmpstr = (gchar *)g_memdup(p2+sizeof("\nidxfilesize=")-1, p3-(p2+sizeof("\nidxfilesize=")-1)+1);
    tmpstr[p3-(p2+sizeof("\nidxfilesize=")-1)] = '\0';
    index_file_size = atol(tmpstr);
    g_free(tmpstr);
  }
	
  p2 = strstr(p1,"\nbookname=");

  if (!p2) {
    g_free(buffer);
    return false;
  }

  p2 = p2 + sizeof("\nbookname=") -1;
  p3 = strchr(p2, '\n');
  bookname.assign(p2, p3-p2);

  p2 = strstr(p1,"\nauthor=");
  if (p2) {
    p2 = p2 + sizeof("\nauthor=") -1;
    p3 = strchr(p2, '\n');
    author.assign(p2, p3-p2);
  }

  p2 = strstr(p1,"\nemail=");
  if (p2) {
    p2 = p2 + sizeof("\nemail=") -1;
    p3 = strchr(p2, '\n');
    email.assign(p2, p3-p2);
  }

  p2 = strstr(p1,"\nwebsite=");
  if (p2) {
    p2 = p2 + sizeof("\nwebsite=") -1;
    p3 = strchr(p2, '\n');
    website.assign(p2, p3-p2);
  }

  p2 = strstr(p1,"\ndate=");
  if (p2) {
    p2 = p2 + sizeof("\ndate=") -1;
    p3 = strchr(p2, '\n');
    date.assign(p2, p3-p2);
  }

  p2 = strstr(p1,"\ndescription=");
  if (p2) {
    p2 = p2 + sizeof("\ndescription=")-1;
    p3 = strchr(p2, '\n');
    description.assign(p2, p3-p2);
  }

  p2 = strstr(p1,"\nsametypesequence=");
  if (p2) {		
    p2+=sizeof("\nsametypesequence=")-1;
    p3 = strchr(p2, '\n');
    sametypesequence.assign(p2, p3-p2);
  }

  g_free(buffer);

  return true;		
}
//===================================================================
DictBase::DictBase()
{
	sametypesequence = NULL;
	dictfile = NULL;
	dictdzfile = NULL;
	cache_cur =0;
}

DictBase::~DictBase()
{
	g_free(sametypesequence);
	if (dictfile)
		fclose(dictfile);
	if (dictdzfile)
		dict_data_close(dictdzfile);
}

gchar* DictBase::GetWordData(guint32 idxitem_offset, guint32 idxitem_size)
{
  for (int i=0; i<WORDDATA_CACHE_NUM; i++)	
    if (cache[i].data && cache[i].offset == idxitem_offset)
			return cache[i].data;
			
  if (dictfile)
		fseek(dictfile, idxitem_offset, SEEK_SET);
	
	gchar *data;
	if (sametypesequence) {
    gchar *origin_data = (gchar *)g_malloc(idxitem_size);
		
		if (dictfile)
      fread(origin_data, idxitem_size, 1, dictfile);
		else
      dict_data_read(dictdzfile, origin_data, idxitem_offset, idxitem_size);
	
    guint32 data_size;
		gint sametypesequence_len;
		sametypesequence_len = strlen(sametypesequence);
		//there have sametypesequence_len char being omitted.
    data_size = idxitem_size + sizeof(guint32) + sametypesequence_len;
		//if the last item's size is determined by the end up '\0',then +=sizeof(gchar);
    //if the last item's size is determined by the head guint32 type data,then +=sizeof(guint32);
		switch (sametypesequence[sametypesequence_len-1]) {
			case 'm':
			case 't':
			case 'y':
    case 'l':
    case 'g':
				data_size += sizeof(gchar);
				break;
			case 'W':
			case 'P':
      data_size += sizeof(guint32);
				break;
		}			
		data = (gchar *)g_malloc(data_size);
		gchar *p1,*p2;
    p1 = data + sizeof(guint32);
		p2 = origin_data;
    guint32 sec_size;
		//copy the head items.
    for (int i=0; i<sametypesequence_len-1; i++) {
      *p1=sametypesequence[i];
      p1+=sizeof(gchar);
			switch (sametypesequence[i]) {
				case 'm':
				case 't':
				case 'y':
      case 'l':
      case 'g':
					sec_size = strlen(p2)+1;
					memcpy(p1, p2, sec_size);
	p1+=sec_size;
	p2+=sec_size;
					break;
				case 'W':
				case 'P':
	sec_size = *reinterpret_cast<guint32 *>(p2);
	sec_size += sizeof(guint32);
					memcpy(p1, p2, sec_size);
	p1+=sec_size;
	p2+=sec_size;
					break;
			}							
		}	
		//calculate the last item 's size.
		sec_size = idxitem_size - (p2-origin_data);
    *p1=sametypesequence[sametypesequence_len-1];
    p1+=sizeof(gchar);
		switch (sametypesequence[sametypesequence_len-1]) {
			case 'm':
			case 't':
			case 'y':
    case 'l':
    case 'g':
				memcpy(p1, p2, sec_size);
				p1 += sec_size;				
      *p1='\0';//add the end up '\0';
				break;
			case 'W':
			case 'P':
      *reinterpret_cast<guint32 *>(p1)=sec_size;
      p1 += sizeof(guint32);
				memcpy(p1, p2, sec_size);
				break;
		}		
		g_free(origin_data);		
    *reinterpret_cast<guint32 *>(data)=data_size;
  } else {		
    data = (gchar *)g_malloc(idxitem_size + sizeof(guint32));
		if (dictfile)
      fread(data+sizeof(guint32), idxitem_size, 1, dictfile);		
		else
      dict_data_read(dictdzfile, data+sizeof(guint32), idxitem_offset, idxitem_size);
    *reinterpret_cast<guint32 *>(data)=idxitem_size;
	}	
		g_free(cache[cache_cur].data);

	cache[cache_cur].data = data;
	cache[cache_cur].offset = idxitem_offset;
	cache_cur++;
	if (cache_cur==WORDDATA_CACHE_NUM)
    cache_cur = 0;
    return data;
}


//===================================================================
Lib::Lib()
{
    wordcount=0;
	bookname = NULL;
	idxfile = NULL;
	wordlist = NULL;
	idxdatabuffer = NULL;
}

Lib::~Lib()
{
#ifdef MEASURE_TIME
  if (middle_lookup_time!=0.) {
    g_message("average time of lookup: %lf\n", middle_lookup_time);
    g_message("times of mesure: %.0lf\n", measure_time);
    middle_lookup_time=0.;
  }
#endif
		g_free(bookname);
	if (idxfile) {
		fclose(idxfile);
			g_free(wordoffset);
  } else {    
			g_free(wordlist);
	}
}

bool Lib::load_wordoffset_from_cache(const std::string& idxname, const std::string& cache_dir)
{
  std::deque<std::string> pn;
  pn.push_back(idxname+".oft");
  if (!cache_dir.empty()) {
    std::string f2(idxname);
    std::string::size_type pos=f2.rfind(G_DIR_SEPARATOR);
    if (pos!=std::string::size_type(-1))
      f2.erase(0, pos+1);
    f2=cache_dir+G_DIR_SEPARATOR+f2+".oft";
    pn.push_back(f2);
  }
  
  while (!pn.empty()) {
    std::string name=pn.back();
    pn.pop_back();
    MapFile m;
    if (!m.open(name.c_str()))
      continue;
#ifdef NEW_CMP  
#  define MAGIC_CACHE_DATA "StarDict's Cache, Version: 0.04nc"
#else
#  define MAGIC_CACHE_DATA "StarDict's Cache, Version: 0.04"
#endif
    if (strncmp(m.begin(), MAGIC_CACHE_DATA, sizeof(MAGIC_CACHE_DATA)-1)!=0)
      continue;
    char *p=m.begin()+sizeof(MAGIC_CACHE_DATA)-1;
#ifdef NEW_CMP
    CharList::size_type as=*((CharList::size_type *)p);
    p+=sizeof(CharList::size_type);
    alphabet.resize(as);
    for (CharList::iterator q=alphabet.begin(); q!=alphabet.end(); ++q) {
      q->ch=*((gunichar *)p);
      p+=sizeof(gunichar);
      q->index=*((glong *)p);
      p+=sizeof(glong);
    }    
#endif
    wordoffset = (WordCoord *)g_malloc((wordcount+1)*sizeof(WordCoord));
#if 1
    for (glong i=0; i<wordcount+1; ++i) {
      wordoffset[i].offset=*((glong *)p);
      p+=sizeof(glong);
      wordoffset[i].size=*((glong *)p);
      p+=sizeof(glong);
    }
#else
    //bad solution, because of aligment of data
    memcpy(wordoffset, p, (wordcount+1)*sizeof(WordCoord));
#endif
#ifdef DEBUG
    g_message("load from cache\n");
#endif
    return true;
  }

    
  return false;
}

void Lib::save_wordoffset_to_cache(const std::string& idxname, const std::string& cache_dir)
{
  std::deque<std::string> pn;
  pn.push_back(idxname+".oft");
  if (!cache_dir.empty()) {
    std::string f2(idxname);
    std::string::size_type pos=f2.rfind(G_DIR_SEPARATOR);
    if (pos!=std::string::size_type(-1))
      f2.erase(0, pos+1);
    f2=cache_dir+G_DIR_SEPARATOR+f2+".oft";
    pn.push_back(f2);
  }
  
  while (!pn.empty()) {
    std::string name=pn.back();
    pn.pop_back();
    FILE *f=fopen(name.c_str(), "wb");
    if (f==NULL)
      continue;
    fwrite(MAGIC_CACHE_DATA, sizeof(MAGIC_CACHE_DATA)-1, 1, f);
    
#ifdef NEW_CMP
    
    CharList::size_type as=alphabet.size();
    fwrite(&as, sizeof(as), 1, f);
    for (CharList::iterator q=alphabet.begin(); q!=alphabet.end(); ++q) {
      fwrite(&(q->ch), sizeof(q->ch), 1, f);
      fwrite(&(q->index), sizeof(q->index), 1, f);
    }    
#endif

#if 1
    for (glong i=0; i<wordcount+1; ++i) {
      fwrite(&(wordoffset[i].offset), sizeof(glong), 1, f);
      fwrite(&(wordoffset[i].size), sizeof(glong), 1, f);
    }
#else
    //bad solution, because of data's aligment
    fwrite(wordoffset, 1, (wordcount+1)*sizeof(WordCoord), f);
#endif
    fclose(f);
#ifdef DEBUG
    g_message("saved to cache\n");
#endif
    return;
  }
}

bool Lib::load(const char *ifofilename, const std::string& cache_dir)
{	
  gulong idxfilesize;
  if (!load_ifofile(ifofilename, &idxfilesize))
    return false;

  std::string fullfilename(ifofilename);
  fullfilename.replace(fullfilename.length()-sizeof("ifo")+1, 
		       sizeof("ifo")-1, "dict.dz");
	
  if (g_file_test(fullfilename.c_str(), G_FILE_TEST_EXISTS)) {
    dictdzfile = dict_data_open(fullfilename.c_str(), 0);
    if (!dictdzfile) {
      //g_print("open file %s failed!\n",fullfilename);
      return false;
    }
  } else {
    fullfilename.erase(fullfilename.length()-sizeof(".dz")+1, 
		       sizeof(".dz")-1);
    dictfile = fopen(fullfilename.c_str(),"rb");
    if (!dictfile) {
      //g_print("open file %s failed!\n",fullfilename);
      return false;
    }
  }

  fullfilename=ifofilename;
  fullfilename.replace(fullfilename.length()-sizeof("ifo")+1, 
		       sizeof("ifo")-1, "idx.gz");
  
  if (g_file_test(fullfilename.c_str(), G_FILE_TEST_EXISTS)) {
    gzFile in;	
    in = gzopen(fullfilename.c_str(),"rb");
    if (in == NULL) {
      //g_print("Open file %s failed!\n",fullfilename);
      return false;
    }
		
    idxdatabuffer = (gchar *)g_malloc(idxfilesize);
		
    gulong len;
    len = gzread(in, idxdatabuffer, idxfilesize);
    if (len < 0)
      return false;
    gzclose(in);
    if (len != idxfilesize)
      return false;
  } else {
    fullfilename.erase(fullfilename.length()-sizeof(".gz")+1, sizeof(".gz")-1);
    if (!load_wordoffset_from_cache(fullfilename, cache_dir)) {
      MapFile map_file;
      if (!map_file.open(fullfilename.c_str(), idxfilesize))
	return false;
      idxdatabuffer=map_file.begin();
    
      loadwordoffset();
      save_wordoffset_to_cache(fullfilename, cache_dir);
      idxdatabuffer=NULL;
    }
    if (!(idxfile = fopen(fullfilename.c_str(), "rb"))) {
      if (wordoffset) {
	g_free(wordoffset);
	wordoffset = NULL;
      }
      return false;
    }
    cur_wordindex = -2;	// so it is always invalid in GetWord();
#ifdef DEBUG
    g_print("bookname: %s , wordcount %u\n",bookname, wordcount);
#endif
    return true;
  }

  loadwordlist();
#ifdef DEBUG
  g_print("bookname: %s , wordcount %u\n",bookname, wordcount);
#endif
  return true;
}

bool Lib::load_ifofile(const char *ifofilename, gulong *idxfilesize)
{
  DictInfo dict_info;
  if (!dict_info.load_from_ifo_file(ifofilename, false))
		return false;
	
  *idxfilesize = dict_info.index_file_size;	
  wordcount = dict_info.wordcount;
  bookname = g_strdup(dict_info.bookname.c_str());

  if (!dict_info.sametypesequence.empty())
    sametypesequence=g_strdup(dict_info.sametypesequence.c_str());

	return true;
}

inline bool less_for_compare(const char *lh, const char *rh) {
  return stardict_strcmp(lh, rh)<0;
}
#ifdef NEW_CMP
struct CharacterArea1 {
  gunichar ch;
  guint32 begin, end;
  friend bool operator<(const CharacterArea1& lh, const CharacterArea1& rh) {
    glong res=g_unichar_tolower(lh.ch)-g_unichar_tolower(rh.ch);
    if (res==0)
      return lh.begin<rh.begin;
    return res<0;
  }
};
#endif
void Lib::loadwordlist()
{
#ifdef NEW_CMP  
  std::vector<CharacterArea1> alphabet_areas;
  CharacterArea1 cur;
  gunichar ch;  
#endif
  wordlist = (gchar **)g_malloc((wordcount+1)*sizeof(gchar *));
	gchar *p1 = idxdatabuffer;
#ifdef NEW_CMP
  ch=cur.ch=g_unichar_tolower(g_utf8_get_char(p1));
  cur.begin=0;
  alphabet_areas.push_back(cur);  
#endif
  guint32 i;
  for (i=0; i<wordcount; i++) {
#ifdef NEW_CMP
    ch = g_unichar_tolower(g_utf8_get_char(p1));
    if (ch!=cur.ch) {//character change
      alphabet_areas.back().end=i;
      cur.ch=ch;
      cur.begin=i;
      alphabet_areas.push_back(cur);
    }
#endif
    wordlist[i] = p1;
    p1 += strlen(p1) +1 + 2*sizeof(guint32);
  }
  wordlist[wordcount] = p1;
#ifdef NEW_CMP
  alphabet_areas.back().end=i;

  if (!alphabet_areas.empty()) {
    std::sort(alphabet_areas.begin(), alphabet_areas.end());
    guint32 cur_beg=0, cur_size;
    for (std::vector<CharacterArea1>::iterator ptr=alphabet_areas.begin();
	 ptr!=alphabet_areas.end(); ++ptr) {      
      cur_size=ptr->end-ptr->begin;

      ptr->begin=cur_beg;
      cur_beg+=cur_size;
    }
  }
  std::vector<CharacterArea1>::iterator ptr=alphabet_areas.begin();
  Character character;
  character.ch=ptr->ch;
  character.index=ptr->begin;
  alphabet.push_back(character);
  ++ptr;
  while(ptr!=alphabet_areas.end()) {
    if (character.ch!=ptr->ch) {
      character.ch=ptr->ch;
      character.index=ptr->begin;
      this->alphabet.push_back(character);
    }
    ++ptr;
  }

  std::sort(wordlist, wordlist+wordcount, less_for_compare);
#endif
}

#ifdef NEW_CMP
class StarDictCompare {
public:
  explicit StarDictCompare(const gchar *data_value) : data(data_value) {}
  bool operator()(const Lib::WordCoord & lh, const Lib::WordCoord & rh) {
    return stardict_strcmp(data+lh.offset, data+rh.offset)<0;
  }
private:
  const gchar *data;
};
#endif	

void Lib::loadwordoffset()
{
#ifdef NEW_CMP  
  std::vector<CharacterArea1> alphabet_areas;
  CharacterArea1 cur;
  gunichar ch;
#endif

  wordoffset = (WordCoord *)g_malloc((wordcount+1)*sizeof(WordCoord));
  gchar *p1 = idxdatabuffer;
#ifdef NEW_CMP
  ch=cur.ch=g_unichar_tolower(g_utf8_get_char(p1));
  cur.begin=0;
  alphabet_areas.push_back(cur);  
#endif
  guint32 i;
  for (i=0; i<wordcount; i++) {
#ifdef NEW_CMP
    ch = g_unichar_tolower(g_utf8_get_char(p1));
    if (ch!=cur.ch) {//character change
      alphabet_areas.back().end=i;
      cur.ch=ch;
      cur.begin=i;
      alphabet_areas.push_back(cur);
    }
#endif
    wordoffset[i].offset = p1 - idxdatabuffer;
    wordoffset[i].size = strlen(p1);
    //g_print("%s\n", p1);
    p1 += wordoffset[i].size +1 + 2*sizeof(guint32);
    // We can check the word len < 256 here.
    // or we can save the max word length, then wordentry_buf=g_malloc(max_wordlen);
    // but wordentry_buf[256] should be enough for most case. Get the max length will slow down the loading a little.
  }
  wordoffset[wordcount].offset = p1 - idxdatabuffer;
  wordoffset[wordcount].size = 0;
#ifdef NEW_CMP
  alphabet_areas.back().end=i;

  if (!alphabet_areas.empty()) {
    std::sort(alphabet_areas.begin(), alphabet_areas.end());
    guint32 cur_beg=0, cur_size;
    for (std::vector<CharacterArea1>::iterator ptr=alphabet_areas.begin();
	 ptr!=alphabet_areas.end(); ++ptr) {           
      cur_size=ptr->end-ptr->begin;     
      ptr->begin=cur_beg;
      cur_beg+=cur_size;
    }    
  }
  std::vector<CharacterArea1>::iterator ptr=alphabet_areas.begin();
  Character character;
  character.ch=ptr->ch;
  character.index=ptr->begin;
  alphabet.push_back(character);
  ++ptr;
  while(ptr!=alphabet_areas.end()) {
    if (character.ch!=ptr->ch) {
      character.ch=ptr->ch;
      character.index=ptr->begin;
      alphabet.push_back(character);
    }
    ++ptr;
  }

  StarDictCompare stardict_compare(idxdatabuffer);
  std::sort(wordoffset, wordoffset+wordcount, stardict_compare);
#endif
}

bool Lib::Lookup(const char* sWord,glong *pIndex)
{
#ifdef MEASURE_TIME
  clock_t t=clock();
#endif
  bool bFound=false;
  glong iTo=wordcount-1;

  if (stardict_strcmp(sWord, GetWord(0))<0) {
    *pIndex = 0;
  } else if (stardict_strcmp(sWord, GetWord(iTo)) >0) {
    *pIndex = INVALID_INDEX;
  } else {
    glong iThisIndex=0;
    glong iFrom=0;
#ifdef NEW_CMP
#if 1
    Character first;
    first.ch=g_unichar_tolower(g_utf8_get_char(sWord));
    CharList::iterator p=std::lower_bound(alphabet.begin(), alphabet.end(), first);
    iFrom=p->index;
#if 0
    char buf[7];
    buf[g_unichar_to_utf8(p->ch, buf)]='\0';
    printf("Lookup: %s\n", buf);
#endif
    ++p;
    if (p!=alphabet.end())
      iTo=p->index;    
#endif
#endif
    gint cmpint;
    while (iFrom<=iTo) {
      iThisIndex=(iFrom+iTo)/2;

      cmpint = stardict_strcmp(sWord, GetWord(iThisIndex));

      if (cmpint>0)
				iFrom=iThisIndex+1;
      else if (cmpint<0)
				iTo=iThisIndex-1;
      else {	
	bFound=true;
	break;
			}
    	}

    if (!bFound) {
			/*glong len = g_utf8_strlen(sWord, -1);
			gchar *last_str = g_utf8_offset_to_pointer(sWord, len-1);
			gunichar last = g_utf8_get_char(last_str);
			if (((g_unichar_isspace(last) || g_unichar_ispunct(last)) || g_unichar_isdigit(last))
				&& (g_ascii_strncasecmp(sWord, GetWord(iTo), (last_str - sWord))==0))
				*pIndex = iTo;      //previous
			else 
            	*pIndex = iFrom;    //next
			*/
			*pIndex = iFrom;    //next
    } else
			*pIndex = iThisIndex;		
	}		

#ifdef MEASURE_TIME
  t=clock()-t;
  middle_lookup_time*=measure_time;
  measure_time+=1;
  middle_lookup_time+=double(t)/CLOCKS_PER_SEC;
  middle_lookup_time/=measure_time;
#endif

  return bFound;
}

bool Lib::LookupWithRule(GPatternSpec *pspec, glong *aIndex, int iBuffLen)
{
    int iIndexCount=0;
  
  for(guint32 i=0; i<wordcount && iIndexCount<iBuffLen-1; i++)
    if (g_pattern_match_string(pspec, GetWord(i)))        
            aIndex[iIndexCount++]=i;
            
    aIndex[iIndexCount]= -1; // -1 is the end.	
	
  return (iIndexCount>0);
}

gchar *Lib::GetWord(glong index)
{
	if (idxfile) {
    fseek(idxfile, wordoffset[index].offset, SEEK_SET);
		cur_wordindex = index;
		
    fread(wordentry_buf, wordoffset[index].size+1, 1, idxfile);
		//g_print("%s\n", wordentry_buf);
    fread(&wordentry_offset, sizeof(guint32), 1, idxfile);
		wordentry_offset = g_ntohl(wordentry_offset);
    fread(&wordentry_size, sizeof(guint32), 1, idxfile);
		wordentry_size = g_ntohl(wordentry_size);
		return wordentry_buf;
  } else {
		return wordlist[index];
	}
}

gchar *
Lib::GetWordData(glong index)
{
	if (idxfile) {
    if (index != cur_wordindex) {
			cur_wordindex = index;
      fseek(idxfile, wordoffset[index].offset+wordoffset[index].size+1, SEEK_SET);
      fread(&wordentry_offset, sizeof(guint32), 1, idxfile);
			wordentry_offset = g_ntohl(wordentry_offset);
      fread(&wordentry_size, sizeof(guint32), 1, idxfile);
			wordentry_size = g_ntohl(wordentry_size);			
		}		
		return DictBase::GetWordData(wordentry_offset, wordentry_size);
  } else {
    gchar *p1 = wordlist[index]+strlen(wordlist[index])+sizeof(gchar);
    guint32 offset=*reinterpret_cast<guint32 *>(p1);
		offset = g_ntohl(offset);
    p1 += sizeof(guint32);
    guint32 size=*reinterpret_cast<guint32 *>(p1);
		size = g_ntohl(size);
		return DictBase::GetWordData(offset, size);
	}
}

//===================================================================
Libs::Libs()
{	
  iMaxFuzzyDistance  = MAX_FUZZY_DISTANCE; //need to read from cfg.
}

void Libs::Clear(void)
{
  for (std::vector<Lib *>::iterator p=oLib.begin(); p!=oLib.end(); ++p)
    delete *p;
  oLib.clear();
}

Libs::~Libs()
{	
  Clear();
}

/********************************************************************/
inline glong Libs::iLength(int iLib)
{
    return (oLib[iLib])->length();
}

/********************************************************************/

void Libs::LoadDir(const std::string& dirname, const GSList *order_list, const GSList *disable_list, const std::string& cache_dir)
{	
  GDir *dir = g_dir_open(dirname.c_str(), 0, NULL);
  if (dir) {

    const gchar *filename;
    while ((filename = g_dir_read_name(dir))!=NULL) {	
      std::string fullfilename=dirname+G_DIR_SEPARATOR+filename;
      if (g_file_test(fullfilename.c_str(), G_FILE_TEST_IS_DIR)) {
	LoadDir(fullfilename, order_list, disable_list, cache_dir);
      } else if (g_str_has_suffix(filename, ".ifo")) {
	const GSList *tmplist1 = order_list;
	bool loaded = false;
	while (tmplist1) {
	  if (strcmp((gchar *)(tmplist1->data), fullfilename.c_str()) == 0) {
	    loaded = true;
	    break;
	  }
	  tmplist1 = g_slist_next(tmplist1);
	}
	if (loaded)
	  continue;
	
	const GSList *tmplist2 = disable_list;
	bool disabled = false;
	while (tmplist2) {
	  if (strcmp((gchar *)(tmplist2->data), fullfilename.c_str()) == 0) {
	    disabled = true;
	    break;
	  }
	  tmplist2 = g_slist_next(tmplist2);
	}
	if (disabled)
	  continue;
	
	Lib *lib = new Lib;
	if (lib->load(fullfilename.c_str(), cache_dir))
	  oLib.push_back(lib);
	else
	  delete lib;					
      }
    }		
    g_dir_close(dir);
  }	
}

void Libs::Load(const std::string& dicts_dir, const GSList *order_list, const GSList *disable_list)
{
#ifdef MEASURE_TIME
  clock_t t=clock();
#endif
  std::string cache_dir;
#ifndef _WIN32
  cache_dir=std::string(g_get_home_dir())+"/.stardict/cache";
  if (mkdir(cache_dir.c_str(), S_IRWXU)==-1 && errno!=EEXIST)
    g_warning("Can not create %s\n", cache_dir.c_str());
#endif
  
  const GSList *tmplist1 = order_list;
  while (tmplist1) {
    gchar *idxfilename = (gchar *)(tmplist1->data);
    tmplist1 = g_slist_next(tmplist1);
    bool disabled = false;
    const GSList *tmplist2 = disable_list;
    while (tmplist2) {
      if (strcmp((gchar *)(tmplist2->data), idxfilename)==0) {
	disabled = true;
	break;
      }
      tmplist2 = g_slist_next(tmplist2);
    }
    if (disabled)
      continue;
    
    Lib *lib = new Lib;
    if (lib->load(idxfilename, cache_dir))
      oLib.push_back(lib);
    else
      delete lib;	
  }
  
#ifndef _WIN32       
  std::string home_dir=std::string(g_get_home_dir())+"/.stardict/dic";
  LoadDir(home_dir, order_list, disable_list, cache_dir);
#endif
  LoadDir(dicts_dir, order_list, disable_list, cache_dir);
#ifdef MEASURE_TIME
  t=clock()-t;
  g_message("Time of loading all dictionaries: %.3lf sec\n", double(t)/CLOCKS_PER_SEC);
#endif
}

void Libs::GetDictInfoFromDir(const std::string& dirname, const GSList *order_list, const GSList *disable_list, DictInfoList& dict_info_list)
{
  GDir *dir = g_dir_open(dirname.c_str(), 0, NULL);
  if (dir) {
    const gchar *filename;
    while ((filename = g_dir_read_name(dir))!=NULL) {	
      std::string fullfilename=dirname+G_DIR_SEPARATOR+filename;
      if (g_file_test(fullfilename.c_str(), G_FILE_TEST_IS_DIR)) {
	GetDictInfoFromDir(fullfilename, order_list, disable_list, dict_info_list);
      } else if (g_str_has_suffix(filename, ".ifo")) {
	const GSList *tmplist1 = order_list;
	bool loaded = false;
	while (tmplist1) {
	  if (strcmp((gchar *)(tmplist1->data), fullfilename.c_str()) == 0) {
	    loaded = true;
	    break;
	  }
	  tmplist1 = g_slist_next(tmplist1);
	}
	if (loaded)
	  continue;
	
	const GSList *tmplist2 = disable_list;
	bool disabled = false;
	while (tmplist2) {
	  if (strcmp((gchar *)(tmplist2->data), fullfilename.c_str()) == 0) {
	    disabled = true;
	    break;
	  }
	  tmplist2 = g_slist_next(tmplist2);
	}
	if (disabled)
	  continue;
	DictInfo di;
	if (di.load_from_ifo_file(fullfilename.c_str(), false)) {
	  dict_info_list.push_back(di);
#ifdef DEBUG	  
	  g_print("Load info from: %s\n", fullfilename.c_str());
#endif	 
	}
      }
    }		
    g_dir_close(dir);
  }
}

void Libs::GetDictInfoList(const std::string& dicts_dir, const GSList *order_list, const GSList *disable_list, DictInfoList& dict_info_list)
{
  const GSList *tmplist1 = order_list;
  while (tmplist1) {
    gchar *idxfilename = (gchar *)(tmplist1->data);
    tmplist1 = g_slist_next(tmplist1);
    bool disabled = false;
    const GSList *tmplist2 = disable_list;
    while (tmplist2) {
      if (strcmp((gchar *)(tmplist2->data), idxfilename)==0) {
	disabled = true;
	break;
      }
      tmplist2 = g_slist_next(tmplist2);
    }
    if (disabled)
      continue;
    
    DictInfo di;
    if (di.load_from_ifo_file(idxfilename, false))
      dict_info_list.push_back(di);
  }
  
#ifndef _WIN32       
  std::string home_dir=std::string(g_get_home_dir())+"/.stardict/dic";
  GetDictInfoFromDir(home_dir, order_list, disable_list, dict_info_list);
#endif
  GetDictInfoFromDir(dicts_dir, order_list, disable_list, dict_info_list);
}

gchar *Libs::poGetCurrentWord(glong * iCurrent)
{
  gchar * poCurrentWord = NULL;
  gchar *word;
  for (int iLib=0; iLib<oLib.size(); iLib++) {
    if (iCurrent[iLib]==INVALID_INDEX)
      continue;
    if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<0)
      continue;
    if ( poCurrentWord == NULL ) {
      poCurrentWord = poGetWord(iCurrent[iLib],iLib);
    } else {
      word = poGetWord(iCurrent[iLib],iLib);
      
      if (stardict_strcmp(poCurrentWord, word) > 0 )	
	poCurrentWord = word;
    }
  }
  return poCurrentWord;
}

gchar *
Libs::poGetNextWord(const gchar *sWord, glong * iCurrent)
{
  // the input can be:
  // (word,iCurrent),read word,write iNext to iCurrent,and return next word. used by TopWin::NextCallback();
  // (NULL,iCurrent),read iCurrent,write iNext to iCurrent,and return next word. used by AppCore::ListWords();
  gchar * poCurrentWord = NULL;
  gint iCurrentLib=0;
  
  gchar *word;
  for (int iLib=0; iLib<oLib.size(); iLib++) {
    if (sWord)
      oLib[iLib]->Lookup(sWord, &(iCurrent[iLib]));
    if (iCurrent[iLib]==INVALID_INDEX)
      continue;
    if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<0)
      continue;
    if ( poCurrentWord == NULL ) {
      poCurrentWord = poGetWord(iCurrent[iLib],iLib);
      iCurrentLib = iLib;
    }
    else {
			word = poGetWord(iCurrent[iLib],iLib);

			if (  stardict_strcmp(poCurrentWord, word) > 0 ) {
				poCurrentWord = word;
				iCurrentLib = iLib;
			}
        }
    }
    if (poCurrentWord) {
        iCurrent[iCurrentLib]++;
		for (int iLib=0;iLib<oLib.size();iLib++) {
			if (iLib == iCurrentLib)
				continue;
			if (iCurrent[iLib]==INVALID_INDEX)
	            continue;
			if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<0)
				continue;
			if (strcmp(poCurrentWord, poGetWord(iCurrent[iLib],iLib)) == 0 )
				iCurrent[iLib]++;
		}
		poCurrentWord = poGetCurrentWord(iCurrent);
	}
    return poCurrentWord;
}


gchar *
Libs::poGetPreWord(glong * iCurrent)
{
	// used by TopWin::PreviousCallback(); the iCurrent is cached by AppCore::TopWinWordChange();
    gchar * poCurrentWord = NULL;
    int iCurrentLib=0;

	gchar *word;
    for (int iLib=0;iLib<oLib.size();iLib++) {
		if (iCurrent[iLib]==INVALID_INDEX)
            continue;
		if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<=0)
			continue;
		if ( poCurrentWord == NULL ) {
			poCurrentWord = poGetWord(iCurrent[iLib]-1,iLib);
			iCurrentLib = iLib;
		}
		else {
			word = poGetWord(iCurrent[iLib]-1,iLib);

			if (stardict_strcmp(poCurrentWord, word) < 0 ) {

				poCurrentWord = word;
				iCurrentLib = iLib;
			}
		}
	}
	
    if (poCurrentWord) {
        iCurrent[iCurrentLib]--;
		for (int iLib=0;iLib<oLib.size();iLib++) {
			if (iLib == iCurrentLib)
				continue;
			if (iCurrent[iLib]==INVALID_INDEX)
	            continue;
			if ( iCurrent[iLib]>=iLength(iLib) || iCurrent[iLib]<=0)
				continue;
			if (strcmp(poCurrentWord, poGetWord(iCurrent[iLib]-1,iLib)) == 0 )
				iCurrent[iLib]--;
		}
	}
    return poCurrentWord;
}

bool Libs::LookupSimilarWord(const gchar* sWord, glong & iWordIndex, int iLib)
{
    glong iIndex;
  bool bFound=false;
	gchar *casestr;

	if (!bFound) {
		// to lower case.
		casestr = g_utf8_strdown(sWord, -1);
		if (strcmp(casestr, sWord)) {
			if(oLib[iLib]->Lookup(casestr,&iIndex))
				bFound=true;
		}
		g_free(casestr);
		// to upper case.
		if (!bFound) {
			casestr = g_utf8_strup(sWord, -1);
			if (strcmp(casestr, sWord)) {
				if(oLib[iLib]->Lookup(casestr,&iIndex))
					bFound=true;
			}
			g_free(casestr);
		}	
		// to upper the first character.
		if (!bFound) {
			gchar *nextchar = g_utf8_next_char(sWord);
			gchar *firstchar = g_utf8_strup(sWord, nextchar - sWord);
			casestr = g_strdup_printf("%s%s", firstchar, nextchar);
			g_free(firstchar);
			if (strcmp(casestr, sWord)) {
				if(oLib[iLib]->Lookup(casestr,&iIndex))
					bFound=true;
			}
			g_free(casestr);
		}	
	}
	
  if (bIsPureEnglish(sWord)) {		
        // If not Found , try other status of sWord.
        int iWordLen=strlen(sWord);
    bool isupcase;
		
		gchar *sNewWord = (gchar *)g_malloc(iWordLen + 1);

        //cut one char "s" or "d"
        if(!bFound && iWordLen>1) {
      isupcase = sWord[iWordLen-1]=='S' || !strncmp(&sWord[iWordLen-2],"ED",2);
      if (isupcase || sWord[iWordLen-1]=='s' || !strncmp(&sWord[iWordLen-2],"ed",2)) {
	            strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-1]='\0'; // cut "s" or "d"
	if (oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }
        
        //cut "ly"
        if(!bFound && iWordLen>2) {
			isupcase = !strncmp(&sWord[iWordLen-2],"LY",2);
			if (isupcase || (!strncmp(&sWord[iWordLen-2],"ly",2))) {
	            strcpy(sNewWord,sWord);
    	        sNewWord[iWordLen-2]='\0';  // cut "ly"
	if (iWordLen>5 && sNewWord[iWordLen-3]==sNewWord[iWordLen-4]
	    && !bIsVowel(sNewWord[iWordLen-4]) && 
	    bIsVowel(sNewWord[iWordLen-5])) {//doubled
		      
    	            sNewWord[iWordLen-3]='\0';
        	        if( oLib[iLib]->Lookup(sNewWord,&iIndex) )
            	        bFound=true;
                	else {
						if (isupcase || g_ascii_isupper(sWord[0])) {
							casestr = g_ascii_strdown(sNewWord, -1);
							if (strcmp(casestr, sNewWord)) {
								if(oLib[iLib]->Lookup(casestr,&iIndex))
									bFound=true;
							}
							g_free(casestr);
						}
						if (!bFound)
							sNewWord[iWordLen-3]=sNewWord[iWordLen-4];  //restore
					}					                    	
	            }
    	        if (!bFound) {
					if (oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        	bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}
        		}
			}
		}
        
        //cut "ing"
        if(!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3],"ING",3);
			if (isupcase || !strncmp(&sWord[iWordLen-3],"ing",3) ) {
	            strcpy(sNewWord,sWord);
    	        sNewWord[iWordLen-3]='\0';
        	    if ( iWordLen>6 && (sNewWord[iWordLen-4]==sNewWord[iWordLen-5])
	     && !bIsVowel(sNewWord[iWordLen-5]) && 
	     bIsVowel(sNewWord[iWordLen-6])) {  //doubled	  
    	            sNewWord[iWordLen-4]='\0';
        	        if (oLib[iLib]->Lookup(sNewWord,&iIndex))
            	        bFound=true;
                	else {
						if (isupcase || g_ascii_isupper(sWord[0])) {
							casestr = g_ascii_strdown(sNewWord, -1);
							if (strcmp(casestr, sNewWord)) {
								if(oLib[iLib]->Lookup(casestr,&iIndex))
									bFound=true;
							}
							g_free(casestr);
						}
						if (!bFound)
                    		sNewWord[iWordLen-4]=sNewWord[iWordLen-5];  //restore
					}
	            }
    	        if( !bFound ) {
					if (oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        	bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}						
				}
            	if(!bFound)
	            {
					if (isupcase)
						strcat(sNewWord,"E"); // add a char "E"
					else
    	            	strcat(sNewWord,"e"); // add a char "e"
        	        if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	        bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}						
            	}
			}
        }

        //cut two char "es"
        if(!bFound && iWordLen>3) {
			isupcase = (!strncmp(&sWord[iWordLen-2],"ES",2) && 
		  (sWord[iWordLen-3] == 'S' || 
		   sWord[iWordLen-3] == 'X' || 
		   sWord[iWordLen-3] == 'O' || 
		   (iWordLen >4 && sWord[iWordLen-3] == 'H' && 
		    (sWord[iWordLen-4] == 'C' || 
		     sWord[iWordLen-4] == 'S'))));
			if (isupcase || 
				(!strncmp(&sWord[iWordLen-2],"es",2) && 
	   (sWord[iWordLen-3] == 's' || sWord[iWordLen-3] == 'x' || 
	    sWord[iWordLen-3] == 'o' || 
	    (iWordLen >4 && sWord[iWordLen-3] == 'h' && 
	     (sWord[iWordLen-4] == 'c' || sWord[iWordLen-4] == 's'))))) {
            	strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-2]='\0';
    	        if(oLib[iLib]->Lookup(sNewWord,&iIndex))
        	        bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
	        }
		}

        //cut "ed"
    if (!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-2],"ED",2);
      if (isupcase || !strncmp(&sWord[iWordLen-2],"ed",2)) {
	            strcpy(sNewWord,sWord);
    	        sNewWord[iWordLen-2]='\0';
	if (iWordLen>5 && (sNewWord[iWordLen-3]==sNewWord[iWordLen-4])
	    && !bIsVowel(sNewWord[iWordLen-4]) && 
	    bIsVowel(sNewWord[iWordLen-5])) {//doubled	            
    	            sNewWord[iWordLen-3]='\0';
	  if (oLib[iLib]->Lookup(sNewWord,&iIndex))
            	        bFound=true;
                	else {
						if (isupcase || g_ascii_isupper(sWord[0])) {
							casestr = g_ascii_strdown(sNewWord, -1);
							if (strcmp(casestr, sNewWord)) {
								if(oLib[iLib]->Lookup(casestr,&iIndex))
									bFound=true;
							}
							g_free(casestr);
						}
						if (!bFound)
                    		sNewWord[iWordLen-3]=sNewWord[iWordLen-4];  //restore
					}
            	}
	if (!bFound) {
					if (oLib[iLib]->Lookup(sNewWord,&iIndex))
                		bFound=true;
					else if (isupcase || g_ascii_isupper(sWord[0])) {
						casestr = g_ascii_strdown(sNewWord, -1);
						if (strcmp(casestr, sNewWord)) {
							if(oLib[iLib]->Lookup(casestr,&iIndex))
								bFound=true;
						}
						g_free(casestr);
					}
				}
			}
        }

        // cut "ied" , add "y".
    if (!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3],"IED",3);
      if (isupcase || (!strncmp(&sWord[iWordLen-3],"ied",3))) {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-3]='\0';
				if (isupcase)
					strcat(sNewWord,"Y"); // add a char "Y"
				else
    	        	strcat(sNewWord,"y"); // add a char "y"
	if (oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }
        
		// cut "ies" , add "y".
    if (!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3],"IES",3);
      if (isupcase || (!strncmp(&sWord[iWordLen-3],"ies",3))) {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-3]='\0';
				if (isupcase)
					strcat(sNewWord,"Y"); // add a char "Y"
				else
    	        	strcat(sNewWord,"y"); // add a char "y"
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }

		// cut "er".
    if (!bFound && iWordLen>2) {
			isupcase = !strncmp(&sWord[iWordLen-2],"ER",2);
      if (isupcase || (!strncmp(&sWord[iWordLen-2],"er",2))) {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-2]='\0';
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }

		// cut "est".
    if (!bFound && iWordLen>3) {
			isupcase = !strncmp(&sWord[iWordLen-3], "EST", 3);
      if (isupcase || (!strncmp(&sWord[iWordLen-3],"est", 3))) {
				strcpy(sNewWord,sWord);
	            sNewWord[iWordLen-3]='\0';
        	    if(oLib[iLib]->Lookup(sNewWord,&iIndex))
            	    bFound=true;
				else if (isupcase || g_ascii_isupper(sWord[0])) {
					casestr = g_ascii_strdown(sNewWord, -1);
					if (strcmp(casestr, sNewWord)) {
						if(oLib[iLib]->Lookup(casestr,&iIndex))
							bFound=true;
					}
					g_free(casestr);
				}
			}
        }
		
		g_free(sNewWord);
    }	
	
	if (bFound)
    	iWordIndex = iIndex;
	else {
		//don't change iWordIndex here.
		//when LookupSimilarWord all failed too, we want to use the old LookupWord index to list words.
		//iWordIndex = INVALID_INDEX;
	}

  return bFound;
}

bool Libs::SimpleLookupWord(const gchar* sWord, glong & iWordIndex, int iLib)
{
  bool bFound = oLib[iLib]->Lookup(sWord, &iWordIndex);
	if (!bFound)
		bFound = LookupSimilarWord(sWord, iWordIndex, iLib);
	return bFound;
}

inline bool operator<(const Libs::Fuzzystruct & lh, const Libs::Fuzzystruct & rh) {    
  if (lh.iMatchWordDistance!=rh.iMatchWordDistance)
    return lh.iMatchWordDistance<rh.iMatchWordDistance;

  if (lh.pMatchWord && rh.pMatchWord)
    return stardict_strcmp(lh.pMatchWord, rh.pMatchWord)<0;

  
  return false;
}

bool Libs::LookupWithFuzzy(const gchar *sWord, Fuzzystruct oFuzzystruct[], gint fuzzystruct_amount, TProgressFunc ProgressFunc)
{
  if (sWord[0] == '\0')
    return false;
           
  for (int i=0; i<fuzzystruct_amount; i++) {
    oFuzzystruct[i].pMatchWord = NULL;
    oFuzzystruct[i].iMatchWordDistance = iMaxFuzzyDistance;
  }
  int iMaxDistance = iMaxFuzzyDistance;
  int iDistance;
  bool Found = false;
  EditDistance oEditDistance;

  glong iCheckWordLen;
  int sCheckLen;
  const char *sCheck;
  gunichar *ucs4_str1,*ucs4_str2;
  glong ucs4_str2_len;
  char *sLowerCheckWord;
  gchar *sLowerWord = g_utf8_strdown(sWord, -1);	
  ucs4_str2 = g_utf8_to_ucs4_fast(sLowerWord,-1,&ucs4_str2_len);
  g_free(sLowerWord);
  for (int iLib=0; iLib<oLib.size(); iLib++) {
    if (ProgressFunc)
      ProgressFunc();

    if (stardict_strcmp(sWord, poGetWord(0,iLib))>=0 && 
	stardict_strcmp(sWord, poGetWord(iLength(iLib)-1,iLib))<=0) {
      //there are Chinese dicts and English dicts...        
      const int iwords = iLength(iLib);
      for (int index=0; index<iwords; index++) {
	sCheck = poGetWord(index,iLib);
	// tolower and skip too long or too short words
	sCheckLen = strlen(sCheck);
	iCheckWordLen = g_utf8_strlen(sCheck, sCheckLen);
	if (iCheckWordLen-ucs4_str2_len>=iMaxDistance || 
	    ucs4_str2_len-iCheckWordLen>=iMaxDistance)
	  continue;
	sLowerCheckWord = g_utf8_strdown(sCheck, sCheckLen);
	if (iCheckWordLen > ucs4_str2_len)
	  (*g_utf8_offset_to_pointer(sLowerCheckWord, ucs4_str2_len)) = '\0';
	ucs4_str1 = g_utf8_to_ucs4_fast(sLowerCheckWord, -1,NULL);
	g_free(sLowerCheckWord);
	iDistance = oEditDistance.CalEditDistance(ucs4_str1,ucs4_str2,iMaxDistance);
	g_free(ucs4_str1);								
	if (iDistance<iMaxDistance && iDistance < ucs4_str2_len) {
	  // when ucs4_str2_len=1,2 we need less fuzzy.
	  Found = true;
	  bool bAlreadyInList = false;
	  int iMaxDistanceAt=0;
	  for (int j=0; j<fuzzystruct_amount; j++) {
	    if (oFuzzystruct[j].pMatchWord && 
		strcmp(oFuzzystruct[j].pMatchWord,sCheck)==0 ) {//already in list
	      bAlreadyInList = true;
	      break;
	    }
	    //find the position,it will certainly be found (include the first time) as iMaxDistance is set by last time.
	    if (oFuzzystruct[j].iMatchWordDistance == iMaxDistance ) {
	      iMaxDistanceAt = j;
	    }
	  }
	  if (!bAlreadyInList) {
	    if (oFuzzystruct[iMaxDistanceAt].pMatchWord)
	      g_free(oFuzzystruct[iMaxDistanceAt].pMatchWord);
	    oFuzzystruct[iMaxDistanceAt].pMatchWord = g_strdup(sCheck);
	    oFuzzystruct[iMaxDistanceAt].iMatchWordDistance = iDistance;
	    // calc new iMaxDistance
	    iMaxDistance = iDistance;
	    for (int j=0; j<fuzzystruct_amount; j++) {
	      if (oFuzzystruct[j].iMatchWordDistance > iMaxDistance)
		iMaxDistance = oFuzzystruct[j].iMatchWordDistance;
	    } // calc new iMaxDistance
	  }   // add to list
	}   // find one
      }   // each word
    }   // ok for search
  }   // each lib
  g_free(ucs4_str2);
	
  if (Found)// sort with distance
    std::sort(oFuzzystruct, oFuzzystruct+fuzzystruct_amount);
  
  return Found;
}

gint Libs::LookupWithRule(const gchar *word, TProgressFunc ProgressFunc, gchar **ppMatchWord)
{	
  glong aiIndex[MAX_MATCH_ITEM_PER_LIB+1];
  gint iMatchCount = 0;
  GPatternSpec *pspec = g_pattern_spec_new(word);
	
  for (int iLib=0; iLib<oLib.size(); iLib++) {
    //if(oLibs.LookdupWordsWithRule(pspec,aiIndex,MAX_MATCH_ITEM_PER_LIB+1-iMatchCount,iLib)) 
    // -iMatchCount,so save time,but may got less result and the word may repeat.
    
    if (oLib[iLib]->LookupWithRule(pspec,aiIndex, MAX_MATCH_ITEM_PER_LIB+1)) {
      if (ProgressFunc)
	ProgressFunc();
      for (int i=0; aiIndex[i]!=-1; i++) {
	gchar * sMatchWord = poGetWord(aiIndex[i],iLib);
	bool bAlreadyInList = false;
	for (int j=0; j<iMatchCount; j++) {
	  if (strcmp(ppMatchWord[j],sMatchWord)==0) {//already in list
	    bAlreadyInList = true;
	    break;
	  }
	}
	if (!bAlreadyInList)
	  ppMatchWord[iMatchCount++] = g_strdup(sMatchWord);
      }
    }
  }
  g_pattern_spec_free(pspec);

  if (iMatchCount)// sort it.
    std::sort(ppMatchWord, ppMatchWord+iMatchCount, less_for_compare); 

  return iMatchCount;
}

