#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifdef __MSW__
# include <windows.h>
#endif

#include <GL/gl.h>

#include "../include/string.h"

#include "matrixmath.h"
#include "sfm.h"

#include "sound.h"
#include "sarreality.h"
#include "obj.h"
#include "objsound.h"
#include "objutils.h"
#include "sar.h"
#include "config.h"


#ifdef __MSW__
static double rint(double x);
#endif	/* __MSW__ */

int SARObjIsAllocated(
	sar_object_struct **ptr, int total,
        int n
);
sar_object_struct *SARObjGetPtr(
        sar_object_struct **ptr, int total,
        int n
);
int SARGetObjectNumberFromPointer(
	sar_scene_struct *scene,
	sar_object_struct **ptr, int total,
        sar_object_struct *obj_ptr
);
int SARIsTextureAllocated(sar_scene_struct *scene, int n);
sar_object_struct *SARObjMatchPointerByName(
	sar_scene_struct *scene,
	sar_object_struct **ptr, int total,
        const char *name, int *obj_num
);

v3d_texture_ref_struct *SARGetTextureRefByName(
	sar_scene_struct *scene, const char *name
);
int SARGetTextureRefNumberByName(
	sar_scene_struct *scene, const char *name
);

int SARObjLandingGearState(sar_object_struct *obj_ptr);

sar_obj_part_struct *SARObjGetPartPtr(
	sar_object_struct *obj_ptr, int type, int skip
);
sar_obj_rotor_struct *SARObjGetRotorPtr(
        sar_object_struct *obj_ptr, int n, int *total
);
sar_obj_hoist_struct *SARObjGetHoistPtr(
	sar_object_struct *obj_ptr, int n, int *total
);
int SARObjGetOnBoardPtr(
        sar_object_struct *obj_ptr,
	int **crew, int **passengers, int **passengers_max,
	float **passengers_mass,
	int **passengers_leave_pending, int **passengers_drop_pending
);
sar_external_fueltank_struct *SARObjGetFuelTankPtr(
        sar_object_struct *obj_ptr, int n, int *total
);

sar_visual_model_struct *SARVisualModelNew(
	sar_scene_struct *scene,
	const char *filename, const char *name
);
void *SARVisualModelNewList(sar_visual_model_struct *vmodel);
int SARVisualModelGetRefCount(sar_visual_model_struct *vmodel);  
void SARVisualModelRef(sar_visual_model_struct *vmodel);
void SARVisualModelUnref(
	sar_scene_struct *scene, sar_visual_model_struct *vmodel
);
void SARVisualModelCallList(sar_visual_model_struct *vmodel);
static void SARVisualModelDelete(sar_visual_model_struct *vmodel);
void SARVisualModelDeleteAll(sar_scene_struct *scene);

sar_cloud_layer_struct *SARCloudLayerCreate(
	sar_scene_struct *scene,
	int tile_width, int tile_height,
        float range, float altitude,
        const char *tex_name
);
void SARCloudLayerDestroy(
	sar_scene_struct *scene,
	sar_cloud_layer_struct *cloud_layer_ptr
);

sar_cloud_bb_struct *SARCloudBBCreate(
        sar_scene_struct *scene,   
        int tile_width, int tile_height,
        float x, float y, float z,   /* In meters. */
        float width, float height,    /* In meters. */
        const char *tex_name
);
void SARCloudBBDestroy(sar_cloud_bb_struct *cloud_bb_ptr);

int SARObjAddToGroundList(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);
void SARObjRemoveFromGroundList(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);

int SARObjAddToHumanRescueList(
	sar_scene_struct *scene, sar_object_struct *obj_ptr
);
void SARObjRemoveFromHumanRescueList(
        sar_scene_struct *scene, sar_object_struct *obj_ptr
);

int SARObjAddContactBoundsSpherical(
	sar_object_struct *obj_ptr,
	sar_obj_flags_t crash_flags, int crash_type,
	float contact_radius
);
int SARObjAddContactBoundsCylendrical(
        sar_object_struct *obj_ptr,
        sar_obj_flags_t crash_flags, int crash_type,
        float contact_radius,
	float contact_h_min, float contact_h_max
);
int SARObjAddContactBoundsRectangular(
        sar_object_struct *obj_ptr,
        sar_obj_flags_t crash_flags, int crash_type,
	float contact_x_min, float contact_x_max,
	float contact_y_min, float contact_y_max,
	float contact_z_min, float contact_z_max
);

int SARObjCreateIntercept(
        sar_scene_struct *scene,
        sar_obj_intercept_struct ***ptr, int *total,
        sar_obj_flags_t flags,
        float x, float y, float z,
        float radius,
        float urgency,
        const char *name
);
static sar_obj_part_struct *SARObjCreatePartNexus(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total,
        int type                /* One of SAR_OBJ_PART_TYPE_*. */
);
sar_obj_part_struct *SARObjCreatePart(
	sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total,
        int type		/* One of SAR_OBJ_PART_TYPE_*. */
);
sar_obj_part_struct *SARObjCreateAirBrake(
        sar_scene_struct *scene,
	sar_obj_part_struct ***ptr, int *total
);
sar_obj_part_struct *SARObjCreateDoorRescue(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
);
sar_obj_part_struct *SARObjCreateLandingGear(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
);
int SARObjCreateExternalFuelTanks(
        sar_scene_struct *scene,
        sar_external_fueltank_struct ***ptr, int *total
);
int SARObjCreateRotor(
	sar_scene_struct *scene,
	sar_obj_rotor_struct ***ptr, int *total
);
sar_light_struct *SARObjCreateLight(
	sar_scene_struct *scene,
        sar_light_struct ***ptr, int *total
);
int SARObjCreate(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
        int type
);

void SARObjDeleteIntercepts(
	sar_scene_struct *scene,
	sar_obj_intercept_struct ***ptr, int *total
);
void SARObjDeleteLights(
	sar_scene_struct *scene,
	sar_light_struct ***ptr, int *total
);
void SARObjDeleteParts(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
);
void SARObjDeleteExternalFuelTanks(
        sar_scene_struct *scene,
        sar_external_fueltank_struct ***ptr, int *total
);
void SARObjDeleteRotors(
	sar_scene_struct *scene,
	sar_obj_rotor_struct ***ptr, int *total
);
void SARObjDelete(
	void *core_ptr,
	sar_object_struct ***ptr, int *total, int n
);

void SARObjGenerateTilePlane(
	float min, float max,
	float tile_width, float tile_height
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


#ifdef __MSW__
static double rint(double x)
{
	if((double)((double)x - (int)x) > (double)0.5)
	    return((double)((int)x + (int)1));
	else
	    return((double)((int)x));
}
#endif	/* __MSW__ */


/*
 *	Returns true if object n is allocated in the given array.
 */
int SARObjIsAllocated(
	sar_object_struct **ptr, int total,
	int n
)
{
	if((ptr == NULL) || (n < 0) || (n >= total))
	    return(0);
	else if(ptr[n] == NULL)
	    return(0);
	else
	    return(1);
}

/*
 *	Returns pointer to object n if it is valid and not NULL.
 *
 *	Returns NULL on error.
 */
sar_object_struct *SARObjGetPtr(
        sar_object_struct **ptr, int total,
        int n
)
{
        if((ptr == NULL) || (n < 0) || (n >= total))
            return(NULL);
        else
            return(ptr[n]);
}


/*
 *	Returns the number of the object pointer in the given pointer
 *	array or -1 if no match.
 */
int SARGetObjectNumberFromPointer(
	sar_scene_struct *scene,
	sar_object_struct **ptr, int total,
	sar_object_struct *obj_ptr
)
{
	int i;

	if(obj_ptr == NULL)
	    return(-1);

	for(i = 0; i < total; i++)
	{
	    if(ptr[i] == obj_ptr)
		return(i);
	}

	return(-1);
}

/*
 *	Checks if the texture is allocated on the scene.
 */
int SARIsTextureAllocated(sar_scene_struct *scene, int n)
{
	if((scene == NULL) || (n < 0) || (n >= scene->total_texture_refs))
	    return(0);
	else if(scene->texture_ref[n] == NULL)
	    return(0);
	else
	    return(1);
}

/*
 *	Returns the pointer to the object matching the given name or
 *	NULL if no match was made.
 */
sar_object_struct *SARObjMatchPointerByName(
	sar_scene_struct *scene,
        sar_object_struct **ptr, int total,
        const char *name, int *obj_num
) 
{
	int i;
	sar_object_struct *obj_ptr;


	if(obj_num != NULL)
	    *obj_num = -1;

	/* Go through each object. */
	for(i = 0; i < total; i++)
	{
	    obj_ptr = ptr[i];
	    if((obj_ptr != NULL) ? (obj_ptr->name == NULL) : 1)
		continue;

	    /* Names match? */
	    if(!strcasecmp(obj_ptr->name, name))
	    {
		if(obj_num != NULL)
		    *obj_num = i;

		return(obj_ptr);
	    }
	}

	return(NULL);
}

/*
 *	Returns a pointer to the texture reference structure matching
 *	the given name. Can return NULL for failed match.
 */
v3d_texture_ref_struct *SARGetTextureRefByName(
        sar_scene_struct *scene, const char *name
)
{
	int i, total;
	v3d_texture_ref_struct *t, **ptr;


	if((scene == NULL) || (name == NULL))
	    return(NULL);

	ptr = scene->texture_ref;
	total = scene->total_texture_refs;

	for(i = 0; i < total; i++)
	{
	    t = ptr[i];
	    if((t != NULL) ? (t->name == NULL) : 1)
		continue;

	    if(!strcasecmp(t->name, name))
		return(t);
	}

	return(NULL);
}

/*
 *	Same as SARGetTextureRefByName() except that it returns
 *	the index number of the texture or -1 on no match.
 */
int SARGetTextureRefNumberByName(
	sar_scene_struct *scene, const char *name
)
{
        int i, total;
        v3d_texture_ref_struct *t, **ptr;


        if((scene == NULL) || (name == NULL))
            return(-1);

        ptr = scene->texture_ref;
        total = scene->total_texture_refs;

        for(i = 0; i < total; i++)
        {
            t = ptr[i];
            if((t != NULL) ? (t->name == NULL) : 1)
                continue;

            if(!strcasecmp(t->name, name))
                return(i);
        }

        return(-1);
}


/*
 *	Returns -1 if the object does not have landing gears, 0 if the
 *	landing gears exist but are up, or 1 if the landing gears are
 *	down.
 */
int SARObjLandingGearState(sar_object_struct *obj_ptr)
{
	sar_object_aircraft_struct *obj_aircraft_ptr;

	if(obj_ptr == NULL)
	    return(-1);

	obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
	return((obj_aircraft_ptr != NULL) ?
	    obj_aircraft_ptr->landing_gear_state : -1
	);
}

/*
 *	Returns the pointer to the part structure on the object that
 *	matches the given part type. The skip indicates the number
 *	of part structures of the specified type to be skipped.
 */
sar_obj_part_struct *SARObjGetPartPtr(
        sar_object_struct *obj_ptr, int type, int skip
)
{
	int i, j, m = 0;
        sar_object_aircraft_struct *obj_aircraft_ptr;
	sar_obj_part_struct *part_ptr, **list = NULL;


	if(obj_ptr == NULL)
	    return(NULL);


	/* Begin checking object type and get the object parts list. */

	/* Aircraft. */
        obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
        if(obj_aircraft_ptr != NULL)
        {
	    list = obj_aircraft_ptr->part;
	    m = obj_aircraft_ptr->total_parts;
	}


	/* Got list of object parts? */
	if(list != NULL)
	{
	    for(i = 0, j = 0; i < m; i++)
	    {
		part_ptr = list[i];
		if(part_ptr == NULL)
		    continue;
		if(part_ptr->type != type)
		    continue;

		/* Increment j, the skip count. If the skip count is still
		 * at or below the requested number of times to skip n,
		 * then skip this part.
		 */
		j++;
		if(j <= skip)
		    continue;

		return(part_ptr);
	    }
        }

	return(NULL);
}

/*
 *	Returns pointer to the n th index rotor and optional total
 *	on the given object. Can return NULL if none exist.
 *
 *	If object only has one, then pass n as 0.
 */
sar_obj_rotor_struct *SARObjGetRotorPtr(
        sar_object_struct *obj_ptr, int n, int *total
)
{
	sar_object_aircraft_struct *obj_aircraft_ptr;


	if(total != NULL)
	    *total = 0;

	if(obj_ptr == NULL)
	    return(NULL);

	obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
	if(obj_aircraft_ptr != NULL)
	{
	    if(total != NULL)
		*total = obj_aircraft_ptr->total_rotors;
	    if((obj_aircraft_ptr->rotor != NULL) &&
	       (n >= 0) && (n < obj_aircraft_ptr->total_rotors)
	    )
		return(obj_aircraft_ptr->rotor[n]);
	}

	return(NULL);
}

/*
 *      Returns pointer to the n th index door and optional total
 *      on the given object. Can return NULL if none exist.
 *      
 *      If object only has one, then pass n as 0.
 */
sar_obj_hoist_struct *SARObjGetHoistPtr(
        sar_object_struct *obj_ptr, int n, int *total
)
{
        sar_object_aircraft_struct *obj_aircraft_ptr;


        if(total != NULL)
            *total = 0;

        if(obj_ptr == NULL)
            return(NULL);

	obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
	if(obj_aircraft_ptr != NULL)
	{
	    if(total != NULL)
		*total = 1;
	    if(n == 0)
		return(&obj_aircraft_ptr->hoist);
	}

        return(NULL);
}

/*
 *	Returns pointers to the object's value for crew and passengers
 *	(and related information).
 *
 *	Returns -1 on error, -2 if the object's type does not have 
 *	passengers, and 0 on success.
 */
int SARObjGetOnBoardPtr(
        sar_object_struct *obj_ptr,                   
        int **crew, int **passengers, int **passengers_max,
	float **passengers_mass,
        int **passengers_leave_pending, int **passengers_drop_pending
)
{
        sar_object_aircraft_struct *obj_aircraft_ptr;


	if(crew != NULL)
	    *crew = NULL;
        if(passengers != NULL)
            *passengers = NULL;
        if(passengers_max != NULL)
            *passengers_max = NULL;
        if(passengers_mass != NULL)
            *passengers_mass = NULL;
        if(passengers_leave_pending != NULL)
            *passengers_leave_pending = NULL;
        if(passengers_drop_pending != NULL)
            *passengers_drop_pending = NULL;

        if(obj_ptr == NULL)
            return(-1);

        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
            if(obj_aircraft_ptr != NULL)
            {
		if(crew != NULL)
		    *crew = &obj_aircraft_ptr->crew;
                if(passengers != NULL)  
                    *passengers = &obj_aircraft_ptr->passengers;
                if(passengers_max != NULL)  
                    *passengers_max = &obj_aircraft_ptr->passengers_max;
                if(passengers_mass != NULL)
                    *passengers_mass = &obj_aircraft_ptr->passengers_mass;
                if(passengers_leave_pending != NULL)
                    *passengers_leave_pending = &obj_aircraft_ptr->passengers_leave_pending;
                if(passengers_drop_pending != NULL)
                    *passengers_drop_pending = &obj_aircraft_ptr->passengers_drop_pending;

		return(0);
            }
            break;

/* Add support for other objects that can have passengers here. */

	  default:
	    return(-2);
  	    break;
        }

	return(-1);
}

/*
 *      Returns pointer to the n th external reserved fuel tank and 
 *	optional total on the given object. Can return NULL if none exist.
 *
 *      If object only has one, then pass n as 0.
 */
sar_external_fueltank_struct *SARObjGetFuelTankPtr(
        sar_object_struct *obj_ptr, int n, int *total
)
{
        sar_object_aircraft_struct *obj_aircraft_ptr;

        if(total != NULL)
            *total = 0;

        if(obj_ptr == NULL)
            return(NULL);

        switch(obj_ptr->type)
        {
          case SAR_OBJ_TYPE_AIRCRAFT:
            obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
            if(obj_aircraft_ptr != NULL)
            {
                if(total != NULL)
                    *total = obj_aircraft_ptr->total_external_fueltanks;

                if((obj_aircraft_ptr->external_fueltank != NULL) &&
                   (n >= 0) && (n < obj_aircraft_ptr->total_external_fueltanks)
                )
                    return(obj_aircraft_ptr->external_fueltank[n]);
            }
            break;
                    
        }
        return(NULL);
}


/*
 *	Creates a new visual model structure on the given scene structure
 *	or returns an existing one if filename and name both match the
 *	name of an existing visual model on the scene structure.
 *
 *	If filename or name is NULL then a new visual model will be
 *	created and will be treated as `unique' meaning that it will not
 *	be returned for match with any other call to SARVisualModelNew()
 *	(since the filename and name are unmatchable).
 *
 *	A new visual model will be given one ref count, while an existing
 *	visual model will have its ref count incremented by one. There
 *	will never be a case where a returned visual model will have only
 *	0 or negative ref counts. Ref counts on returned visual models
 *	will always be 1 or greater.
 *
 *	Can return NULL on error.
 */
sar_visual_model_struct *SARVisualModelNew(
        sar_scene_struct *scene,
        const char *filename, const char *name
)
{
	int i;
	sar_visual_model_struct *vmodel;

	if(scene == NULL)
	    return(NULL);

	/* Both filename and name given? */
	if((filename != NULL) && (name != NULL))
	{
	    /* Check for existing visual model structure on scene
	     * structure.
	     */
	    for(i = 0; i < scene->total_visual_models; i++)
	    {
		vmodel = scene->visual_model[i];
		if(vmodel == NULL)
		    continue;

		/* Visual model on scene structure has both filename
		 * and name set?
		 */
		if((vmodel->filename != NULL) && (vmodel->name != NULL))
		{
		    /* Given filename and name matches? */
		    if(!strcmp(vmodel->filename, filename) &&
                       !strcmp(vmodel->name, name)
		    )
		    {
			/* Increment ref count for matched visual model
			 * and return its pointer.
			 */
			vmodel->ref_count++;
			return(vmodel);
		    }
		}
	    }
	}


	/* Need to create a new visual model on scene if this point has
	 * reached.
	 */

	/* Sanitize total. */
        if(scene->total_visual_models < 0) 
            scene->total_visual_models = 0;

	/* Look for available pointer. */
	for(i = 0; i < scene->total_visual_models; i++)
	{
	    if(scene->visual_model[i] == NULL)
		break;
	}
	if(i < scene->total_visual_models)
	{
	    /* Got available index i. */
	}
	else
	{
	    /* Need to allocate more pointers. */
	    i = scene->total_visual_models;
	    scene->total_visual_models = i + 1;
	    scene->visual_model = (sar_visual_model_struct **)realloc(
		scene->visual_model,
		scene->total_visual_models * sizeof(sar_visual_model_struct *)
	    );
	    if(scene->visual_model == NULL)
	    {
		scene->total_visual_models = 0;
		return(NULL);
	    }
	    else
	    {
		scene->visual_model[i] = NULL;
	    }
	}

	/* Allocate new structure. */
	scene->visual_model[i] = vmodel = SAR_VISUAL_MODEL(calloc(
	    1, sizeof(sar_visual_model_struct)
	));
	if(vmodel != NULL)
	{
	    vmodel->load_state = SAR_VISUAL_MODEL_NOT_LOADED;

	    /* Give one initial ref count for new visual models. */
	    vmodel->ref_count = 1;

	    /* Copy file name if available. */
	    if(filename != NULL)
		vmodel->filename = STRDUP(filename);

	    /* Copy model name if available. */
	    if(name != NULL)
		vmodel->name = STRDUP(name);

	    /* Reset pointer to GL display list. */
	    vmodel->data = NULL;
	}

	return(vmodel);
}

/*
 *	(Re)generates a list on the given visual model, can return NULL
 *	on failure.
 *
 *	The return value is really a GLuint (not a void *).
 */
void *SARVisualModelNewList(sar_visual_model_struct *vmodel)
{
	GLuint list;

	if(vmodel == NULL)
	    return(NULL);

	/* Get existing GL list if any. */
	list = (GLuint)vmodel->data;
	if(list > 0)
	{
	    /* Already has an allocated GL list for some odd reason,
	     * oh well, delete it and then create a new GL list.
	     */
	    glDeleteLists(list, 1);
	    list = 0;
	    vmodel->data = NULL;
	}

	/* Create new GL list. */
	list = glGenLists(1);

	/* Set new GL list as the data pointer on the visual model
	 * structure.
	 */
	vmodel->data = (void *)list;

	return((void *)list);
}

/*
 *	Returns number of ref counts on visual model.
 *
 *	Can return -1 if visual model is NULL.
 */
int SARVisualModelGetRefCount(sar_visual_model_struct *vmodel)
{
        if(vmodel != NULL)
            return(vmodel->ref_count);
	else
	    return(-1);
} 

/*
 *	Increases the ref count on the given visual model by one.
 */
void SARVisualModelRef(sar_visual_model_struct *vmodel)
{
	if(vmodel != NULL)
	    vmodel->ref_count++;
}

/*
 *	Decrease the ref count on the given visual model by one.
 *
 *	If (after decrement) the ref count drops to 0 then the given
 *	visual model will be destroyed and its pointer should not
 *	be referenced again.
 *
 *	It's a general sense to not reference any pointer value again from
 *	any one variable after it's passed to this function. The scene
 *	structure's visual_model list is ultimatly to hold the pointer
 *	the visual model all the way to when it is down to its last
 *	ref count of 0.
 *
 *	If scene is not NULL then any visual model pointers on the
 *	scene's visual models list matching the given vmodel pointer
 *	will be set to NULL.
 */
void SARVisualModelUnref(
	sar_scene_struct *scene, sar_visual_model_struct *vmodel
)
{
	if(vmodel != NULL)
	{
	    /* No existing reference counts to unref? */
	    if(vmodel->ref_count < 1)
	    {
		/* Print warning. */
		fprintf(
		    stderr,
"SARVisualModelUnref(): Cannot unref visual model `%s', no additional refcounts.\n",
		    vmodel->name
		);
		return;
	    }
	    else
	    {
		/* Reduce reference count by 1. */
		vmodel->ref_count--;
	    }
	    /* Sanitize reference count. */
	    if(vmodel->ref_count < 0)
	        vmodel->ref_count = 0;

	    /* Has reference count reached 0? */
	    if(vmodel->ref_count == 0)
	    {
		/* Reference counts have reached 0 on this visual model,
		 * so once that happens the visual model structure needs
		 * to be deallocated and removed from the scene
	         * structure's vmodel_list.
		 */

		/* Delete this visual model, deallocating its GL list and
		 * the visual model structure itself.
		 */
		SARVisualModelDelete(vmodel);

		/* Pointer vmodel is now invalid, but do not set it to
		 * NULL just yet as it needs to be used to check for 
		 * references to it on the scene structure.
		 */
		if(scene != NULL)
		{
		    int i;

		    for(i = 0; i < scene->total_visual_models; i++)
		    {
			if(scene->visual_model[i] == vmodel)
			    scene->visual_model[i] = NULL;
		    }
		}

	    }	/* Has reference count reached 0? */
	}
}

/*
 *      Calls the GL list on the visual model if it is not 0.  
 */
void SARVisualModelCallList(sar_visual_model_struct *vmodel)
{
        GLuint list;

        if(vmodel == NULL)
            return;

        list = (GLuint)vmodel->data;
        if(list > 0)
            glCallList(list);
}

/*
 *	Deallocates all resources for the given visual model and the
 *	visual model structure itself.
 *
 *	This function is intended to be called from SARVisualModelUnref().
 */
static void SARVisualModelDelete(sar_visual_model_struct *vmodel)
{
	GLuint list;

	if(vmodel == NULL)
	    return;

	if(vmodel->ref_count > 0)
	    fprintf(
		stderr,
 "SARVisualModelDelete(): Warning: Visual model `%s' still has %i ref_counts.\n",
		vmodel->name, vmodel->ref_count
	    );

	/* Free filename. */
	free(vmodel->filename);
	vmodel->filename = NULL;

	/* Free name. */
	free(vmodel->name);
	vmodel->name = NULL;

	/* Delete the GL list. */
	list = (GLuint)vmodel->data;
	if(list > 0)
	    glDeleteLists(list, 1);

	/* Free structure itself. */
	free(vmodel);
}

/*
 *	Dealloates all the visual models on the given scene structure
 *	and deletes the GL list on each visual model structure.
 */
void SARVisualModelDeleteAll(sar_scene_struct *scene)
{
	int i;
	sar_visual_model_struct *vmodel;

	if(scene == NULL)
	    return;

	/* Itterate through visual models recorded on the scene
	 * structure.
	 */
	for(i = 0; i < scene->total_visual_models; i++)
	{
	    vmodel = scene->visual_model[i];
	    if(vmodel == NULL)
		continue;

	    /* Delete the visual model, note that its ref counts
	     * should be 0 at this point. The visual model will be
	     * deallocated after this call.
	     */
	    SARVisualModelDelete(vmodel);
	    scene->visual_model[i] = vmodel = NULL;
	}

	free(scene->visual_model);
	scene->visual_model = NULL;
	scene->total_visual_models = 0;
}


/*
 *	Allocates a new cloud layer structure.
 */
sar_cloud_layer_struct *SARCloudLayerCreate(
	sar_scene_struct *scene,
	int tile_width, int tile_height,
	float range,			/* Tiling range in meters. */
	float altitude,		/* Altitude in meters. */
	const char *tex_name
)
{
	GLuint list;
	float min, max;
	v3d_texture_ref_struct *t;
	sar_visual_model_struct **vmodel;
	sar_cloud_layer_struct *cloud_layer_ptr = (sar_cloud_layer_struct *)
	    calloc(1, sizeof(sar_cloud_layer_struct));


	if(cloud_layer_ptr == NULL)
	    return(NULL);

	cloud_layer_ptr->tile_width = tile_width;
	cloud_layer_ptr->tile_height = tile_height;
	cloud_layer_ptr->range = MAX(range, MAX(tile_width, tile_height));
	cloud_layer_ptr->z = altitude;

	/* Match texture by name. */
	t = SARGetTextureRefByName(scene, tex_name);

	/* Calculate entire size limits. */
	min = -cloud_layer_ptr->range;
	max = cloud_layer_ptr->range;

	/* Get pointer to visual model. */
        vmodel = &cloud_layer_ptr->visual_model;

	/* Unref visual model if its pointer to reference is not NULL. */
	if((*vmodel) != NULL)
	{
	    SARVisualModelUnref(scene, *vmodel);
	    (*vmodel) = NULL;
	}

	/* Create new visual model. */
	(*vmodel) = SARVisualModelNew(
	    scene, NULL, NULL
	);

	/* (Re)generate GL list on visual model structure. */
	list = (GLuint)SARVisualModelNewList(*vmodel);
	if(list != 0)
	{
	    /* Mark as loading. */
	    (*vmodel)->load_state = SAR_VISUAL_MODEL_LOADING;

	    /* Begin recording GL list. */
	    glNewList(list, GL_COMPILE);
	    {
		/* Select texture if defined. */
		V3DTextureSelect(t);

		/* Do not set color. */

		/* Close range textured tiles. */
		SARObjGenerateTilePlane(
		    min, max,			/* Min and max. */
		    (float)tile_width, (float)tile_height	/* Tiling size. */
		);
	    }
	    glEndList();

            /* Mark as done loading. */
            (*vmodel)->load_state = SAR_VISUAL_MODEL_LOADED;
	}

	return(cloud_layer_ptr);
}

/*
 *	Deallocates all substructures of the cloud layer and
 *	the cloud layer structure itself.
 *
 *	If scene is NULL then the visual model on the cloud layer cannot
 *	be unref'ed if it has one. Sometimes scene will be NULL when this
 *	is called for deallocating a preset weather data entry which has
 *	not visual model.
 */
void SARCloudLayerDestroy(
	sar_scene_struct *scene,
	sar_cloud_layer_struct *cloud_layer_ptr
)
{
	if(cloud_layer_ptr != NULL)
	{
	    SARVisualModelUnref(scene, cloud_layer_ptr->visual_model);
	    cloud_layer_ptr->visual_model = NULL;

	    free(cloud_layer_ptr->tex_name);
	    free(cloud_layer_ptr);
	}
}

/*
 *	Allocates a new cloud `billboard' object structure.
 */
sar_cloud_bb_struct *SARCloudBBCreate(
        sar_scene_struct *scene,
        int tile_width, int tile_height,
        float x, float y, float z,	/* In meters. */
	float width, float height,	/* In meters. */
        const char *tex_name
)
{
        sar_cloud_bb_struct *cloud_bb_ptr = (sar_cloud_bb_struct *)
            calloc(1, sizeof(sar_cloud_bb_struct));
 
        
        if(cloud_bb_ptr == NULL)
            return(NULL);
         
        cloud_bb_ptr->tile_width = tile_width;
        cloud_bb_ptr->tile_height = tile_height;

	cloud_bb_ptr->tex_name = STRDUP(tex_name);

	/* Match texture number on scene. */
	cloud_bb_ptr->tex_num = SARGetTextureRefNumberByName(
	    scene, tex_name
	);

        cloud_bb_ptr->x = x;
        cloud_bb_ptr->y = y;
        cloud_bb_ptr->z = z;

        cloud_bb_ptr->width = width;
        cloud_bb_ptr->height = height;

	cloud_bb_ptr->lightning_min_int = 0;	/* No lightning. */
	cloud_bb_ptr->lightning_on_int = SAR_DEF_LIGHTNING_ON_INT;

	cloud_bb_ptr->lightning_on_left = 0;
	cloud_bb_ptr->lightning_last = 0;

	memset(
	    &cloud_bb_ptr->lightning_point[0], 0x00,
	    SAR_LIGHTNING_POINTS_MAX * sizeof(sar_position_struct)
	);

	return(cloud_bb_ptr);
}

/*      
 *      Deallocates all substructures of the cloud `billboard' and
 *      the cloud `billboard' structure itself.
 */
void SARCloudBBDestroy(sar_cloud_bb_struct *cloud_bb_ptr)
{
        if(cloud_bb_ptr != NULL)
        {
            free(cloud_bb_ptr->tex_name);
            free(cloud_bb_ptr);
        }
        return;
}


/*
 *	Checks if the object is in the scene's ground list.
 *	If not, then it will add it to the list. Returns 0 on success
 *	or negative on error/already in list.
 */
int SARObjAddToGroundList(
	sar_scene_struct *scene,  
        sar_object_struct *obj_ptr
)
{
	int i;


	if((scene == NULL) ||
           (obj_ptr == NULL)
	)
	    return(-1);

	if(scene->total_ground_objects < 0)
	    scene->total_ground_objects = 0;

	/* Check if already in list. */
	for(i = 0; i < scene->total_ground_objects; i++)
	{
	    if(obj_ptr == scene->ground_object[i])
		return(-2);
	}

	/* Not in list, so add to list. */
        for(i = 0; i < scene->total_ground_objects; i++)
        {
            if(scene->ground_object[i] == NULL)
            {
		/* Found an empty entry. */
		scene->ground_object[i] = obj_ptr;
		return(0);
	    }
        }

	/* Need to allocate more entries. */
	i = scene->total_ground_objects;
	scene->total_ground_objects++;

	scene->ground_object = (sar_object_struct **)realloc(
	    scene->ground_object,
	    scene->total_ground_objects * sizeof(sar_object_struct *)
	);
	if(scene->ground_object == NULL)
	{
	    scene->total_ground_objects = 0;
	    return(-1);
	}

	scene->ground_object[i] = obj_ptr;

	return(0);
}

/*
 *	Removes the object from the ground list if it is in there.
 */
void SARObjRemoveFromGroundList(
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr
)
{
	int i;


        if((scene == NULL) || (obj_ptr == NULL))
            return;

        for(i = 0; i < scene->total_ground_objects; i++)
        {
            if(obj_ptr == scene->ground_object[i])
                scene->ground_object[i] = NULL;
        }
}

/*
 *	Adds object pointer to the humans need rescue list on the scene:
 */
int SARObjAddToHumanRescueList(
	sar_scene_struct *scene,
        sar_object_struct *obj_ptr
)
{
        int i;
            
            
        if((scene == NULL) ||
           (obj_ptr == NULL)
        )
            return(-1);
            
        if(scene->total_human_need_rescue_objects < 0)
            scene->total_human_need_rescue_objects = 0;

        /* Check if already in list. */
        for(i = 0; i < scene->total_human_need_rescue_objects; i++)
        {
            if(obj_ptr == scene->human_need_rescue_object[i])
                return(-2);
        }

        /* Not in list, so add to list. */
        for(i = 0; i < scene->total_human_need_rescue_objects; i++)
        {
            if(scene->human_need_rescue_object[i] == NULL)
            {
                /* Found an empty entry. */
                scene->human_need_rescue_object[i] = obj_ptr;
                return(0);
            }
        }

        /* Need to allocate more entries. */
        i = scene->total_human_need_rescue_objects;
        scene->total_human_need_rescue_objects++;

        scene->human_need_rescue_object = (sar_object_struct **)realloc(
            scene->human_need_rescue_object,
            scene->total_human_need_rescue_objects * sizeof(sar_object_struct *)
        );
        if(scene->human_need_rescue_object == NULL)
        {
            scene->total_human_need_rescue_objects = 0;
            return(-1);
        }

        scene->human_need_rescue_object[i] = obj_ptr;

        return(0);
}

/*
 *	Removes object from humans that need rescue list if it is in
 *	there.
 */
void SARObjRemoveFromHumanRescueList(
	sar_scene_struct *scene,
	sar_object_struct *obj_ptr
)
{
        int i;

        if((scene == NULL) || (obj_ptr == NULL))
            return;

        for(i = 0; i < scene->total_human_need_rescue_objects; i++)
        {
            if(obj_ptr == scene->human_need_rescue_object[i])
                scene->human_need_rescue_object[i] = NULL;
        }
}


/*
 *	Allocate (as needed) contact bounds structure to object
 *	for shape SAR_CONTACT_SHAPE_SPHERICAL.
 *
 *      Returns non-zero on error.
 */
int SARObjAddContactBoundsSpherical(
        sar_object_struct *obj_ptr,
	sar_obj_flags_t crash_flags, int crash_type,
	float contact_radius
)
{
	sar_contact_bounds_struct *cb;


	if(obj_ptr == NULL)
	    return(-1);

	/* Allocate as needed. */
	if(obj_ptr->contact_bounds == NULL)
	    obj_ptr->contact_bounds = SAR_CONTACT_BOUNDS(
		calloc(1, sizeof(sar_contact_bounds_struct))
	    );

	cb = obj_ptr->contact_bounds;
	if(cb == NULL)
	    return(-1);

	cb->crash_flags = crash_flags;
	cb->crash_type = crash_type;
	cb->contact_shape = SAR_CONTACT_SHAPE_SPHERICAL;

	cb->contact_radius = (float)MAX(contact_radius, 0.0);

	return(0);
}

/*
 *      Allocate (as needed) contact bounds structure to object
 *      for shape SAR_CONTACT_SHAPE_CYLENDRICAL.
 *
 *      Returns non-zero on error.
 */
int SARObjAddContactBoundsCylendrical(
        sar_object_struct *obj_ptr,
	sar_obj_flags_t crash_flags, int crash_type,
        float contact_radius,
        float contact_h_min, float contact_h_max 
)
{
        sar_contact_bounds_struct *cb;


        if(obj_ptr == NULL)
            return(-1);

        /* Allocate as needed. */
        if(obj_ptr->contact_bounds == NULL)
            obj_ptr->contact_bounds = SAR_CONTACT_BOUNDS(
                calloc(1, sizeof(sar_contact_bounds_struct))
	    );

        cb = obj_ptr->contact_bounds;
        if(cb == NULL)
            return(-1);

        cb->crash_flags = crash_flags;
        cb->crash_type = crash_type;
        cb->contact_shape = SAR_CONTACT_SHAPE_CYLENDRICAL;

        cb->contact_radius = (float)MAX(contact_radius, 0.0);
	cb->contact_h_min = contact_h_min;
        cb->contact_h_max = contact_h_max;

	/* Flip height as needed. */
	if(cb->contact_h_min > cb->contact_h_max)
	{
	    float h = cb->contact_h_min;

	    cb->contact_h_min = cb->contact_h_max;
	    cb->contact_h_max = h;
	}

	return(0);
}

/*
 *      Allocate (as needed) contact bounds structure to object
 *      for shape SAR_CONTACT_SHAPE_RECTANGULAR.
 *
 *	Returns non-zero on error.
 */
int SARObjAddContactBoundsRectangular(
        sar_object_struct *obj_ptr,
	sar_obj_flags_t crash_flags, int crash_type,
        float contact_x_min, float contact_x_max,
        float contact_y_min, float contact_y_max,
        float contact_z_min, float contact_z_max 
)
{
        sar_contact_bounds_struct *cb;


        if(obj_ptr == NULL)
            return(-1);

        /* Allocate as needed. */
        if(obj_ptr->contact_bounds == NULL)
            obj_ptr->contact_bounds = SAR_CONTACT_BOUNDS(
                calloc(1, sizeof(sar_contact_bounds_struct))
	    );

        cb = obj_ptr->contact_bounds;
        if(cb == NULL)
            return(-1);

        cb->crash_flags = crash_flags;
        cb->crash_type = crash_type;
        cb->contact_shape = SAR_CONTACT_SHAPE_RECTANGULAR;

        cb->contact_x_min = contact_x_min;
	cb->contact_x_max = contact_x_max;
        if(cb->contact_x_min > cb->contact_x_max)
        {
            float d = cb->contact_x_min;

            cb->contact_x_min = cb->contact_x_max;
            cb->contact_x_max = d;
        }

	cb->contact_y_min = contact_y_min;
	cb->contact_y_max = contact_y_max;
        if(cb->contact_y_min > cb->contact_y_max)
        {
            float d = cb->contact_y_min;

            cb->contact_y_min = cb->contact_y_max;
            cb->contact_y_max = d;
        }

	cb->contact_z_min = contact_z_min;
	cb->contact_z_max = contact_z_max;
        if(cb->contact_z_min > cb->contact_z_max)
        {
            float d = cb->contact_z_min;

            cb->contact_z_min = cb->contact_z_max;
            cb->contact_z_max = d;
        }

	/* Calculate trig values based on object's current heading. */
	cb->cos_heading = (float)cos(-obj_ptr->dir.heading);
        cb->sin_heading = (float)sin(-obj_ptr->dir.heading);

        return(0);
}


/*
 *	Adds an intercept to the object. Returns the allocated intercept
 *	index number or -1 on error.
 */
int SARObjCreateIntercept(
	sar_scene_struct *scene,
	sar_obj_intercept_struct ***ptr, int *total,
	sar_obj_flags_t flags,
	float x, float y, float z,
	float radius,
	float urgency,
	const char *name
)
{
	int n;
	sar_obj_intercept_struct *intercept_ptr;


	if((ptr == NULL) || (total == NULL))
	    return(-1);

	/* Append intercept. */
	n = MAX(*total, 0);
	*total = n + 1;
	*ptr = (sar_obj_intercept_struct **)realloc(
	    *ptr,
	    (*total) * sizeof(sar_obj_intercept_struct *)
	);
	if(*ptr == NULL)
	{
	    *total = 0;
	    return(-1);
	}

	/* Allocate structure. */
	(*ptr)[n] = intercept_ptr = (sar_obj_intercept_struct *)calloc(
	    1, sizeof(sar_obj_intercept_struct)
	);
	if(intercept_ptr == NULL)
	    return(-1);

	/* Set values. */
	intercept_ptr->flags = flags;
	intercept_ptr->x = x;
	intercept_ptr->y = y;
	intercept_ptr->z = z;
	intercept_ptr->radius = (float)MAX(radius, 0.0);
	intercept_ptr->urgency = (float)CLIP(urgency, 0.0, 1.0);
	intercept_ptr->name = STRDUP(name);

	return(n);
}


/*
 *	Nexus for creating a new part in the given object parts list.
 */
static sar_obj_part_struct *SARObjCreatePartNexus(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total,
	int type		/* One of SAR_OBJ_PART_TYPE_*. */
)
{
        int i, n;
        sar_obj_part_struct *part_ptr;


        if((ptr == NULL) || (total == NULL))
            return(NULL);

        if((*total) < 0)
            *total = 0;

	/* Look for an available pointer in the given list. */
        for(i = 0; i < *total; i++)
        {
            if((*ptr)[i] == NULL)
                break;
        }
	/* Found available pointer in the given list? */
        if(i < *total)
        {
	    n = i;
        }
        else
        {
	    /* Increase total and allocate a new pointer. */
            n = *total;
            *total = n + 1;

            *ptr = (sar_obj_part_struct **)realloc(
                *ptr,
                (*total) * sizeof(sar_obj_part_struct *)
            );
            if(*ptr == NULL)
            {
                *total = 0;
                return(NULL);
            }
        }

	/* Allocate a new part pointer. */
        (*ptr)[n] = part_ptr = (sar_obj_part_struct *)calloc(
            1, sizeof(sar_obj_part_struct)
        );
	if(part_ptr != NULL)
	{
	    part_ptr->type = type;
	}

	return(part_ptr);
}

/*
 *      Creates a new part of the specified type in the given list of
 *	object part structures.
 *
 *      Can return NULL on error.
 */
sar_obj_part_struct *SARObjCreatePart(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total,
        int type                /* One of SAR_OBJ_PART_TYPE_*. */
)
{
        return(SARObjCreatePartNexus(
            scene, ptr, total, type
        ));
}

/*
 *      Creates a new air brake as a part in the given list of object
 *      part structures.
 *
 *      Can return NULL on error.
 */
sar_obj_part_struct *SARObjCreateAirBrake(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
)
{
	return(SARObjCreatePartNexus(
	    scene, ptr, total, SAR_OBJ_PART_TYPE_AIR_BRAKE
	));
}

/*
 *      Creates a new rescue door as a part in the given list of object
 *      part structures.
 *
 *      Can return NULL on error.
 */
sar_obj_part_struct *SARObjCreateDoorRescue(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
)
{
        return(SARObjCreatePartNexus(
            scene, ptr, total, SAR_OBJ_PART_TYPE_DOOR_RESCUE
        ));
}

/*      Creates a new rescue door as a part in the given list of object
 *      part structures.
 *
 *      Can return NULL on error.
 */
sar_obj_part_struct *SARObjCreateLandingGear(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
)
{
        return(SARObjCreatePartNexus(
            scene, ptr, total, SAR_OBJ_PART_TYPE_LANDING_GEAR
        ));
}

/*
 *      Allocates a new external fuel tank structure, returning
 *      the new fuel tank index number or -1 on error.
 */
int SARObjCreateExternalFuelTanks(
        sar_scene_struct *scene,
        sar_external_fueltank_struct ***ptr, int *total
)
{        
        int i, n;
        sar_external_fueltank_struct *eft_ptr;


        if((ptr == NULL) ||
           (total == NULL)
        )
            return(-1);

        if(*total < 0)
            (*total) = 0;

        for(i = 0; i < (*total); i++)
        {   
            if((*ptr)[i] == NULL)
                break;
        }
        if(i < (*total))
        {
            n = i;
        }
        else
        {
            n = (*total);
            (*total) = (*total) + 1;

            (*ptr) = (sar_external_fueltank_struct **)realloc(
                *ptr,
                (*total) * sizeof(sar_external_fueltank_struct *)
            );
            if((*ptr) == NULL)
            {
                (*total) = 0;
                return(-1);
            }
        }

        (*ptr)[n] = (sar_external_fueltank_struct *)calloc(
            1,
            sizeof(sar_external_fueltank_struct)
        );
        eft_ptr = (*ptr)[n];
        if(eft_ptr == NULL)
        {
            return(-1);
        }

        /* Reset values. */
	eft_ptr->flags = 0;
	eft_ptr->visual_model = NULL;

        return(n);
}

/*
 *      Allocates a new rotor structure, returning
 *      the new rotor index number or -1 on error.
 */
int SARObjCreateRotor(
        sar_scene_struct *scene,
        sar_obj_rotor_struct ***ptr, int *total
) 
{
        int i, n;
        sar_obj_rotor_struct *rotor_ptr;


        if((ptr == NULL) ||
           (total == NULL)
        )
            return(-1);

        if(*total < 0)
            *total = 0;

        for(i = 0; i < *total; i++)
        {
            if((*ptr)[i] == NULL)
                break;
        }
        if(i < *total)
        {
            n = i;
        }
        else
        {
            n = *total;
            *total = *total + 1;

            *ptr = (sar_obj_rotor_struct **)realloc(
                *ptr,
                *total * sizeof(sar_obj_rotor_struct *)
            );
            if(*ptr == NULL) 
            {
                *total = 0;
                return(-1);
            }
        }

        (*ptr)[n] = (sar_obj_rotor_struct *)calloc( 
            1,
            sizeof(sar_obj_rotor_struct)
        );
        rotor_ptr = (*ptr)[n];
        if(rotor_ptr == NULL)
        {
            return(-1);
        }

        /* Reset values. */
        rotor_ptr->flags = 0;
 
        rotor_ptr->dir.heading = (float)(0.0 * PI);
	rotor_ptr->dir.pitch = (float)(0.5 * PI);
	rotor_ptr->dir.bank = (float)(0.0 * PI);

        rotor_ptr->anim_pos = (sar_grad_anim_t)-1;

        rotor_ptr->visual_model = NULL;

        return(n);
}

/*
 *      Allocates a new light structure, returning the pointer to the
 *      light structure or -1 on error.
 */
sar_light_struct *SARObjCreateLight(
        sar_scene_struct *scene,
        sar_light_struct ***ptr, int *total
)
{
        int i, n;
        sar_light_struct *light;


        if((ptr == NULL) || (total == NULL))
            return(NULL);

        if(*total < 0)
            *total = 0;

        for(i = 0; i < *total; i++)
        {
            if((*ptr)[i] == NULL)
                break;
        }
        if(i < *total)
        {
            n = i;
        }
        else
        {
            n = MAX(*total, 0);
            *total = n + 1;
            *ptr = (sar_light_struct **)realloc(
                *ptr,
                (*total) * sizeof(sar_light_struct *)
            );
            if(*ptr == NULL)
            {
                *total = 0;
                return(NULL);
            }
        }

        (*ptr)[n] = light = (sar_light_struct *)calloc(
            1, sizeof(sar_light_struct)
        );
        if(light == NULL)
            return(NULL);

	return(light);
}


/*
 *	Creates a new object of the specified type, returning its
 *	index number in the given array or -1 on error.
 *
 *	The object structure's member `data' will be allocated as
 *	well determined by its type.
 */
int SARObjCreate(
	sar_scene_struct *scene,
	sar_object_struct ***ptr, int *total,
	int type
)
{
	int i, n, len;
	sar_object_struct *obj_ptr;


	if((scene == NULL) ||
           (ptr == NULL) ||
           (total == NULL) ||
	   (type <= SAR_OBJ_TYPE_GARBAGE)
	)
	    return(-1);

	if(*total < 0)
	    *total = 0;

	for(i = 0; i < *total; i++)
	{
	    if((*ptr)[i] == NULL)
		break;

	    if((*ptr)[i]->type <= SAR_OBJ_TYPE_GARBAGE)
		break;
	}
	if(i < *total)
	{
	    /* Got already allocated structure. */
	    n = i;
	}
	else
	{
	    /* Need to allocate a new object structure. */
	    n = (*total);
	    (*total) = (*total) + 1;

	    (*ptr) = (sar_object_struct **)realloc(
		*ptr,
		(*total) * sizeof(sar_object_struct *)
	    );
	    if((*ptr) == NULL)
	    {
		(*total) = 0;
		return(-1);
	    }

	    (*ptr)[n] = NULL;
	}

	/* Allocate object structure as needed. */
	if((*ptr)[n] == NULL)
	{
	    (*ptr)[n] = (sar_object_struct *)calloc(
		1,
		sizeof(sar_object_struct)
	    );
	    if((*ptr)[n] == NULL)
	    {
		return(-1);
	    }
	}

	obj_ptr = (*ptr)[n];


	/* Set object type. */
	obj_ptr->type = type;

	/* Mark birth time. */
	obj_ptr->birth_time_ms = cur_millitime;
	obj_ptr->birth_time_sec = cur_systime;

	/* Allocate substructure (if len is 0 then implies has no
	 * substructure).
	 */
	switch(type)
	{
	  case SAR_OBJ_TYPE_GARBAGE:
            len = 0;
	    break;

	  case SAR_OBJ_TYPE_STATIC:
            len = 0;
	    break;

	  case SAR_OBJ_TYPE_AUTOMOBILE:
            len = 0;
	    break;

	  case SAR_OBJ_TYPE_WATERCRAFT:
	    len = 0;
	    break;

	  case SAR_OBJ_TYPE_AIRCRAFT:
	    len = sizeof(sar_object_aircraft_struct);
	    break;

	  case SAR_OBJ_TYPE_GROUND:
	    len = sizeof(sar_object_ground_struct);
	    break;

	  case SAR_OBJ_TYPE_RUNWAY:
	    len = sizeof(sar_object_runway_struct);
	    break;

	  case SAR_OBJ_TYPE_HELIPAD:
	    len = sizeof(sar_object_helipad_struct);
            break;

	  case SAR_OBJ_TYPE_HUMAN:
	    len = sizeof(sar_object_human_struct);
	    break;


	  case SAR_OBJ_TYPE_SMOKE:
            len = sizeof(sar_object_smoke_struct);
            break;

          case SAR_OBJ_TYPE_FIRE:
            len = sizeof(sar_object_fire_struct);;
            break;

          case SAR_OBJ_TYPE_EXPLOSION:
            len = sizeof(sar_object_explosion_struct);
            break;

	  case SAR_OBJ_TYPE_FUELTANK:
	    len = sizeof(sar_object_fueltank_struct);
            break;


	  case SAR_OBJ_TYPE_PREMODELED:
	    len = sizeof(sar_object_premodeled_struct);
	    break;

	  default:
	    len = 0;
	    break;
	}

	/* If size of substructure is positive then allocate it. */
	if(len > 0)
	    obj_ptr->data = calloc(1, len);
	else
	    obj_ptr->data = NULL;


	/* If object type is SAR_OBJ_TYPE_GROUND, then add to ground
	 * list on scene structure.
	 */
	if((scene != NULL) &&
           (obj_ptr->type == SAR_OBJ_TYPE_GROUND)
	)
	    SARObjAddToGroundList(scene, obj_ptr);


	return(n);
}


/*
 *	Deletes all intercept structures in the given array.
 */
void SARObjDeleteIntercepts(
	sar_scene_struct *scene,
	sar_obj_intercept_struct ***ptr, int *total
)
{
	int i;
	sar_obj_intercept_struct *intercept_ptr;


        if((ptr == NULL) || (total == NULL))
            return;

	for(i = 0; i < (*total); i++)
	{
	    intercept_ptr = (*ptr)[i];
	    if(intercept_ptr == NULL)
		continue;

	    free(intercept_ptr->name);
	    free(intercept_ptr);
	}

	free(*ptr);
	(*ptr) = NULL;
	(*total) = 0;
}

/*
 *	Deletes all structures and resources in the array of light
 *	structures.
 */
void SARObjDeleteLights(
	sar_scene_struct *scene,
	sar_light_struct ***ptr, int *total
)
{
        int i;
        sar_light_struct *light;


        if((ptr == NULL) || (total == NULL))
            return;

        for(i = 0; i < *total; i++)
        {
            light = (*ptr)[i];
            if(light == NULL)
                continue;

            free(light);
        }

        free(*ptr);
        *ptr = NULL;
        *total = 0;
}


/*
 *	Delete all object part structures in the given list, including 
 *	the list itself.
 */
void SARObjDeleteParts(
        sar_scene_struct *scene,
        sar_obj_part_struct ***ptr, int *total
)
{
        int i;
        sar_obj_part_struct *part_ptr;


        if((ptr == NULL) || (total == NULL))
            return;

	/* Iterate through all object part structures in the given list. */
        for(i = 0; i < *total; i++)
        {
            part_ptr = (*ptr)[i];
            if(part_ptr == NULL)
                continue;

	    /* Delete all resources on this object part structure. */
            SARVisualModelUnref(scene, part_ptr->visual_model);
            part_ptr->visual_model = NULL;

	    /* Deallocate structure itself. */
            free(part_ptr);
        }

	/* Deallocate the list itself. */
        free(*ptr);
        (*ptr) = NULL;
        (*total) = 0;
}


/*
 *	Deletes all structures and resources in the array of external
 *	fueltank structures.
 */
void SARObjDeleteExternalFuelTanks(
	sar_scene_struct *scene,
        sar_external_fueltank_struct ***ptr, int *total
)
{
	int i;
	sar_external_fueltank_struct *eft_ptr;

        for(i = 0; i < (*total); i++)
        {
            eft_ptr = (*ptr)[i];
            if(eft_ptr == NULL)
                continue;  

            SARVisualModelUnref(scene, eft_ptr->visual_model);
	    eft_ptr->visual_model = NULL;

            free(eft_ptr);
        }

        free(*ptr);
        (*ptr) = NULL;

        (*total) = 0;
}

/*                  
 *      Deletes all structures and resources in the array of landing
 *      gear structures.
 */                     
void SARObjDeleteRotors(
	sar_scene_struct *scene,
        sar_obj_rotor_struct ***ptr, int *total
)                   
{ 
        int i;
        sar_obj_rotor_struct *rotor_ptr;


        if((ptr == NULL) ||
           (total == NULL)
        )
            return;

        for(i = 0; i < *total; i++)
        {
            rotor_ptr = (*ptr)[i];
            if(rotor_ptr == NULL)
                continue;

	    SARVisualModelUnref(scene, rotor_ptr->visual_model);
	    rotor_ptr->visual_model = NULL;

            free(rotor_ptr);
        }

        free(*ptr);
        *ptr = NULL;

        *total = 0;     
}                     

/*
 *	Deletes object n from the array, setting it to NULL.
 */
void SARObjDelete(
	void *core_ptr,
        sar_object_struct ***ptr, int *total, int n
)
{
	sar_core_struct *cp = core_ptr;
	sar_scene_struct *scene;
	snd_recorder_struct *recorder;


	if(cp == NULL)
	    return;

	scene = cp->scene;
	if(scene == NULL)
	{
	    fprintf(
		stderr,
		"SARObjDelete(): Error: Scene is NULL.\n"
	    );
	    return;
	}

	recorder = cp->recorder;

	if(SARObjIsAllocated(*ptr, *total, n))
	{
	    int i;
	    sar_visual_model_struct **vmodel = NULL;
	    void **sound_play = NULL;
	    sar_object_struct *obj_ptr, *obj_ptr2;
	    sar_object_aircraft_struct *obj_aircraft_ptr;
	    sar_object_runway_struct *obj_runway_ptr;
	    sar_object_helipad_struct *obj_helipad_ptr;
	    sar_object_ground_struct *obj_ground_ptr;
	    sar_object_smoke_struct *obj_smoke_ptr;
	    sar_object_explosion_struct *obj_explosion_ptr;
	    sar_object_fire_struct *obj_fire_ptr;
            sar_object_chemical_spray_struct *obj_chemical_spray_ptr;
	    sar_object_fueltank_struct *obj_fueltank_ptr;
	    sar_object_human_struct *obj_human_ptr;
	    sar_object_premodeled_struct *obj_premodeled_ptr;
	    sar_obj_hoist_struct *hoist_ptr;


	    obj_ptr = (*ptr)[n];

#define DO_UNREF_VISUAL_MODEL		\
{ if(vmodel != NULL) {			\
 if(*vmodel != NULL)			\
 {					\
  SARVisualModelUnref(scene, *vmodel);	\
  *vmodel = NULL;			\
 }					\
} }

#define DO_STOP_SOUND			\
{ if(sound_play != NULL) {		\
 if(*sound_play != NULL)		\
 {					\
  SoundStopPlay(recorder, *sound_play);	\
  *sound_play = NULL;			\
 }					\
} }

#define DO_FREE_STORAGE			\
{ if(storage != NULL) {			\
  free(*storage);			\
  *storage = NULL;			\
} }

	    /* Deallocate substructures. */
	    if(obj_ptr->data != NULL)
	    {
		switch(obj_ptr->type)
		{
		  case SAR_OBJ_TYPE_AIRCRAFT:
		    obj_aircraft_ptr = SAR_OBJ_GET_AIRCRAFT(obj_ptr);
		    if(obj_aircraft_ptr == NULL)
			break;

		    /* Engine sound play object and source index. */
		    sound_play = &obj_aircraft_ptr->engine_inside_sndplay;
		    DO_STOP_SOUND
		    obj_aircraft_ptr->engine_inside_sndsrc = -1;

                    sound_play = &obj_aircraft_ptr->engine_outside_sndplay;
                    DO_STOP_SOUND
		    obj_aircraft_ptr->engine_outside_sndsrc = -1;

		    /* Repeating warning sound play object and source index. */
                    sound_play = &obj_aircraft_ptr->stall_sndplay;
                    DO_STOP_SOUND
                    obj_aircraft_ptr->stall_sndsrc = -1;

                    sound_play = &obj_aircraft_ptr->overspeed_sndplay;
                    DO_STOP_SOUND
                    obj_aircraft_ptr->overspeed_sndsrc = -1;

		    /* Cockpit visual model. */
		    vmodel = &obj_aircraft_ptr->visual_model_cockpit;
		    DO_UNREF_VISUAL_MODEL

		    /* Delete all parts and their visual models. */
		    SARObjDeleteParts(
			scene,
			&obj_aircraft_ptr->part,
			&obj_aircraft_ptr->total_parts
		    );

		    /* Delete all rotors and their visual models. */
		    SARObjDeleteRotors(
                        scene,
                        &obj_aircraft_ptr->rotor,
                        &obj_aircraft_ptr->total_rotors
                    );

		    /* Delete all external fuel tanks and their visual models. */
		    SARObjDeleteExternalFuelTanks(
                        scene,
                        &obj_aircraft_ptr->external_fueltank,
                        &obj_aircraft_ptr->total_external_fueltanks
                    );

		    /* Delete all hoists. */
		    for(i = 0; 1; i++)
		    {
			hoist_ptr = SARObjGetHoistPtr(obj_ptr, i, NULL);
			if(hoist_ptr == NULL)
			    break;

			/* Free list of occupants on hoist struct. */
			free(hoist_ptr->occupant);
			hoist_ptr->occupant = NULL;
			hoist_ptr->total_occupants = 0;
		    }

		    /* Delete all way pointer intercepts. */
		    SARObjDeleteIntercepts(
			scene,
			&obj_aircraft_ptr->intercept,
			&obj_aircraft_ptr->total_intercepts
		    );

		    /* Flight dynamics model. */
		    if((scene->realm != NULL) &&
                       (obj_aircraft_ptr->fdm != NULL)
		    )
		    {
			SFMModelDelete(scene->realm, obj_aircraft_ptr->fdm);
			/* SFM fdm destroy callback will call our callback
			 * function to set this pointer to NULL.
			 */
/*
			obj_aircraft_ptr->fdm = NULL;
 */
		    }
		    break;

		  case SAR_OBJ_TYPE_GROUND:
		    obj_ground_ptr = SAR_OBJ_GET_GROUND(obj_ptr);
                    if(obj_ground_ptr == NULL)
                        break;

		    /* Heightfield z points map. */
		    free(obj_ground_ptr->z_point_value);
		    obj_ground_ptr->z_point_value = NULL;

		    obj_ground_ptr->grid_points_x = 0;
		    obj_ground_ptr->grid_points_y = 0;
		    obj_ground_ptr->grid_points_total = 0;

		    break;

	          case SAR_OBJ_TYPE_RUNWAY:
		    obj_runway_ptr = SAR_OBJ_GET_RUNWAY(obj_ptr);
		    if(obj_runway_ptr == NULL)
			break;

		    free(obj_runway_ptr->north_label);
		    free(obj_runway_ptr->south_label);
                    vmodel = &obj_runway_ptr->north_label_vmodel;
                    DO_UNREF_VISUAL_MODEL
                    vmodel = &obj_runway_ptr->south_label_vmodel;
                    DO_UNREF_VISUAL_MODEL
                    vmodel = &obj_runway_ptr->threshold_vmodel;
                    DO_UNREF_VISUAL_MODEL
                    vmodel = &obj_runway_ptr->td_marker_vmodel;
                    DO_UNREF_VISUAL_MODEL
		    vmodel = &obj_runway_ptr->midway_marker_vmodel;
		    DO_UNREF_VISUAL_MODEL
                    vmodel = &obj_runway_ptr->north_displaced_threshold_vmodel;
                    DO_UNREF_VISUAL_MODEL
                    vmodel = &obj_runway_ptr->south_displaced_threshold_vmodel;
                    DO_UNREF_VISUAL_MODEL
		    break;

                  case SAR_OBJ_TYPE_HELIPAD:
                    obj_helipad_ptr = SAR_OBJ_GET_HELIPAD(obj_ptr);
                    if(obj_helipad_ptr == NULL)
                        break;

                    free(obj_helipad_ptr->label);
                    vmodel = &obj_helipad_ptr->label_vmodel;
                    DO_UNREF_VISUAL_MODEL

                    break;

                  case SAR_OBJ_TYPE_SMOKE:
                    obj_smoke_ptr = SAR_OBJ_GET_SMOKE(obj_ptr);
                    if(obj_smoke_ptr == NULL)
                        break;

		    free(obj_smoke_ptr->unit);
		    obj_smoke_ptr->unit = NULL;
		    obj_smoke_ptr->total_units = 0;

                    break;

                  case SAR_OBJ_TYPE_FIRE:
		    obj_fire_ptr = SAR_OBJ_GET_FIRE(obj_ptr);
		    if(obj_fire_ptr == NULL)
			break;

                    break;

		  case SAR_OBJ_TYPE_EXPLOSION:
                    obj_explosion_ptr = SAR_OBJ_GET_EXPLOSION(obj_ptr);
                    if(obj_explosion_ptr == NULL)
                        break;

		    break;

		  case SAR_OBJ_TYPE_CHEMICAL_SPRAY:
		    obj_chemical_spray_ptr = SAR_OBJ_GET_CHEMICAL_SPRAY(obj_ptr);
		    if(obj_chemical_spray_ptr == NULL)
			break;

		    obj_chemical_spray_ptr->owner = -1;
		    obj_chemical_spray_ptr->tex_num = -1;

		    break;

		  case SAR_OBJ_TYPE_FUELTANK:
		    obj_fueltank_ptr = SAR_OBJ_GET_FUELTANK(obj_ptr);
		    if(obj_fueltank_ptr == NULL)
			break;

		    break;


		  case SAR_OBJ_TYPE_HUMAN:
		    obj_human_ptr = SAR_OBJ_GET_HUMAN(obj_ptr);
		    if(obj_human_ptr == NULL)
			break;

		    free(obj_human_ptr->mesg_enter);
		    break;

		  case SAR_OBJ_TYPE_PREMODELED:
		    obj_premodeled_ptr = SAR_OBJ_GET_PREMODELED(obj_ptr);
                    if(obj_premodeled_ptr == NULL)
                        break;

		    break;

		  default:
		    break;
		}

		free(obj_ptr->data);
		obj_ptr->data = NULL;
	    }

	    /* Deallocate standard visual models. */
	    vmodel = &obj_ptr->visual_model_shadow;
	    DO_UNREF_VISUAL_MODEL
            vmodel = &obj_ptr->visual_model_dawn;       /* Dawn. */
            DO_UNREF_VISUAL_MODEL
            vmodel = &obj_ptr->visual_model_night;      /* Night. */
            DO_UNREF_VISUAL_MODEL
            vmodel = &obj_ptr->visual_model_dusk;       /* Dusk. */
            DO_UNREF_VISUAL_MODEL
            vmodel = &obj_ptr->visual_model_far;	/* Far. */
            DO_UNREF_VISUAL_MODEL
            vmodel = &obj_ptr->visual_model;            /* Day. */
            DO_UNREF_VISUAL_MODEL

	    /* Deallocate contact bounds structure. */
	    if(obj_ptr->contact_bounds != NULL)
	    {
		free(obj_ptr->contact_bounds);
		obj_ptr->contact_bounds = NULL;
	    }

	    /* Deallocate all lights. */
	    SARObjDeleteLights(
		scene,
		&obj_ptr->light, &obj_ptr->total_lights
	    );

	    /* Deallocate all sound sources. */
	    for(i = 0; i < obj_ptr->total_sndsrcs; i++)
		SARSoundSourceDelete(obj_ptr->sndsrc[i]);
	    free(obj_ptr->sndsrc);
	    obj_ptr->sndsrc = NULL;
	    obj_ptr->total_sndsrcs = 0;


	    /* Deallocate name. */
	    free(obj_ptr->name);
	    obj_ptr->name = NULL;

	    /* Deallocate object structure itself. */
	    free(obj_ptr);
	    (*ptr)[n] = NULL;


	    /* Object has now been destroyed, now we need to check if
	     * any other objects are referencing this object. If so we
	     * need to reset those references.
	     */

	    /* Check if any explosion objects are referencing this object. */
	    for(i = 0; i < (*total); i++)
	    {
		obj_ptr2 = (*ptr)[i];
		if(obj_ptr2 == NULL)
		    continue;

		obj_explosion_ptr = SAR_OBJ_GET_EXPLOSION(obj_ptr2);
		if(obj_explosion_ptr != NULL)
		{
		    if(obj_explosion_ptr->ref_object == n)
			obj_explosion_ptr->ref_object = -1;
		}
	    }

	    /* Unreference this object from the scene structure so that
	     * no other resources check the scene structure and mistake the
	     * object as still being allocated.
	     */
	    if(scene != NULL)
	    {
		/* Was this referenced as the player object? */
		if(obj_ptr == scene->player_obj_ptr)
		    scene->player_obj_ptr = NULL;

		if(n == scene->player_obj_num)
		    scene->player_obj_num = -1;

		/* Was this referenced as the camera target? */
		if(n == scene->camera_target)
		    scene->camera_target = -1;

		/* Remove this object from ground list as needed. */
		SARObjRemoveFromGroundList(scene, obj_ptr);

		/* Remove this object from human need rescue list as needed. */
		SARObjRemoveFromHumanRescueList(scene, obj_ptr);
	    }

#undef DO_UNREF_VISUAL_MODEL
#undef DO_STOP_SOUND
#undef DO_FREE_STORAGE
	}
}

/*
 *	Generates an array of tiles. The call to begin a list and
 *	set up color and texture should already have been made before
 *	calling this function.
 */
void SARObjGenerateTilePlane(
	float min, float max,	/* In meters. */
	float tile_width, float tile_height
)
{
	float x, y, x2, y2;


	/* Begin gl instructions. */
        glBegin(GL_QUADS);

	/* Quadrant 1 is `upper right' and quadrants are numbed
	 * clockwise.
	 */

	/* Quads in quadrant 1. */
	for(y = 0; y < max; y += tile_height)
        {
            for(x = 0; x < max; x += tile_width)
            {
                x2 = x + tile_width;
                y2 = y + tile_height;
                {
                    glNormal3f(0.0f, 1.0f, 0.0f);
                    glTexCoord2f(0.0f, 1.0f - 0.0f);
                    glVertex3f((GLfloat)x, (GLfloat)0.0f, (GLfloat)-y);
                    glTexCoord2f(1.0f, 1.0f - 0.0f);
                    glVertex3f((GLfloat)x2, (GLfloat)0.0f, (GLfloat)-y);
                    glTexCoord2f(1.0f, 1.0f - 1.0f);
                    glVertex3f((GLfloat)x2, (GLfloat)0.0f, (GLfloat)-y2);
                    glTexCoord2f(0.0f, 1.0f - 1.0f);
                    glVertex3f((GLfloat)x, (GLfloat)0.0f, (GLfloat)-y2);
                }
            }
        }
        /* Quads in quadrant 2. */
        for(y = 0; y > min; y -= tile_height)
        {
            for(x = 0; x < max; x += tile_width)
            {
                x2 = x + tile_width;
                y2 = y - tile_height;
                {
                    glNormal3f(0.0f, 1.0f, 0.0f);
                    glTexCoord2f(0.0f, 1.0f - 0.0f);
                    glVertex3f((GLfloat)x, 0.0f, (GLfloat)-y2);
                    glTexCoord2f(1.0f, 1.0f - 0.0f);
                    glVertex3f((GLfloat)x2, 0.0f, (GLfloat)-y2);
                    glTexCoord2f(1.0f, 1.0f - 1.0f);
                    glVertex3f((GLfloat)x2, 0.0f, (GLfloat)-y);
                    glTexCoord2f(0.0f, 1.0f - 1.0f);
                    glVertex3f((GLfloat)x, 0.0f, (GLfloat)-y);
               }
           }
       }
       /* Quads in quadrant 3. */
       for(y = 0; y > min; y -= tile_height)
       {
           for(x = 0; x > min; x -= tile_width)
           {
               x2 = x - tile_width;
               y2 = y - tile_height;
               {
                   glNormal3f(0.0f, 1.0f, 0.0f);
                   glTexCoord2f(1.0f, 1.0f - 1.0f);
                   glVertex3f((GLfloat)x, 0.0f, (GLfloat)-y);
                   glTexCoord2f(0.0f, 1.0f - 1.0f);
                   glVertex3f((GLfloat)x2, 0.0f, (GLfloat)-y);
                   glTexCoord2f(0.0f, 1.0f - 0.0f);
                   glVertex3f((GLfloat)x2, 0.0f, (GLfloat)-y2);
                   glTexCoord2f(1.0f, 1.0f - 0.0f);
                   glVertex3f((GLfloat)x, 0.0f, (GLfloat)-y2);
               }
            }
        }
        /* Quads in quadrant 4. */
        for(y = 0; y < max; y += tile_height)
        {
            for(x = 0; x > min; x -= tile_width)
            {
                x2 = x - tile_width;
                y2 = y + tile_height;
                {
                   glNormal3f(0.0f, 1.0f, 0.0f);
                   glTexCoord2f(1.0f, 1.0f - 1.0f);
                   glVertex3f((GLfloat)x, 0.0f, (GLfloat)-y2);
                   glTexCoord2f(0.0f, 1.0f - 1.0f);
                   glVertex3f((GLfloat)x2, 0.0f, (GLfloat)-y2);
                   glTexCoord2f(0.0f, 1.0f - 0.0f);
                   glVertex3f((GLfloat)x2, 0.0f, (GLfloat)-y);
                   glTexCoord2f(1.0f, 1.0f - 0.0f);
                   glVertex3f((GLfloat)x, 0.0f, (GLfloat)-y);
                }
            }
        }
	glEnd();
}
