#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "menu.h"
#include "obj.h"
#include "mission.h"
#include "simutils.h"
#include "sar.h"
#include "sarmenuop.h"
#include "sarmenucb.h"
#include "sarmenuoptions.h"
#include "missionio.h"
#include "sarsimbegin.h"
#include "sarmenucodes.h"
#include "config.h"


/* Menu Procedures */
static void SARMenuButtonCBTest(sar_core_struct *core_ptr);
static void SARMenuCBBeginMission(sar_core_struct *core_ptr);
static void SARMenuCBViewLastLog(sar_core_struct *core_ptr);
static void SARMenuCBMissionMap(sar_core_struct *core_ptr);
static void SARMenuCBMissionBrief(sar_core_struct *core_ptr);
static void SARMenuCBBeginFreeFlight(sar_core_struct *core_ptr);
static void SARMenuCBFreeFlightAircraftInfo(sar_core_struct *core_ptr);
static void SARMenuCBMapOp(sar_core_struct *core_ptr, int id_code);

/* Menu Callbacks */
void SARMenuButtonCB(void *object, int id_code, void *client_data);
void SARMenuListSelectCB(
        void *object, int id_code, void *client_data,
	int item_num, void *item_ptr, void *item_data
);
void SARMenuListActivateCB(
        void *object, int id_code, void *client_data,
        int item_num, void *item_ptr, void *item_data
);
void SARMenuSwitchCB(
	void *object, int id_code, void *client_data,
	Boolean state
);
void SARMenuSpinCB(
	void *object, int id_code, void *client_data,
	char *value
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)


/*
 *      Default spacings (in pixels):
 *      
 *      These values should match and coenciding values in menu.c. 
 */
#define DEF_XMARGIN             5
#define DEF_YMARGIN             5

#define DEF_WIDTH               100
#define DEF_HEIGHT              100


/*
 *	Test callback procedure, used for debugging purposes when
 *	SAR_OP_TEST is given as the operation in SARMenuButtonCB().
 */
static void SARMenuButtonCBTest(sar_core_struct *core_ptr)
{
	int i;
	sar_menu_struct *mission_log_menu;


        if(core_ptr == NULL)
            return;

        /* Get mission log map menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION_LOG_MAP);
        mission_log_menu = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Update the map object's map and markings with the information
	 * on the mission log file.
	 */
        SARMissionLoadMissionLogToMenuMap(
            core_ptr, mission_log_menu,
            fname.mission_log
        );

        /* Switch to log map menu. */
        SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_MISSION_LOG_MAP);
}

/*
 *	Begin mission callback procedure.
 */
static void SARMenuCBBeginMission(sar_core_struct *core_ptr)
{
        int i;
        sar_menu_struct *m;
	const sar_menu_list_item_struct *item;
	const char *filename;
        sar_menu_list_item_data_struct *d;

  
        if(core_ptr == NULL)
            return;

        /* Get pointer to menu Mission. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Get selected mission on the missions list */
        item = SARMenuListGetSelectedItem(m, SAR_OP_MISSION_LIST_SELECT);
        d = SAR_MENU_LIST_ITEM_DATA(
	    (item != NULL) ? item->client_data : NULL
	);
        filename = (d != NULL) ? d->filename : NULL;
        free(core_ptr->cur_mission_file);
        core_ptr->cur_mission_file = STRDUP(filename);

	/* Switch to mission simulation and leave menu system. */
	SARSimBeginMission(
	    core_ptr,
	    core_ptr->cur_mission_file
	);
}

/*
 *	Switch to last log map procedure.
 */
static void SARMenuCBViewLastLog(sar_core_struct *core_ptr)
{
        int i;
        sar_menu_struct *mission_log_menu;


        if(core_ptr == NULL)
            return;

        /* Get mission log map menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION_LOG_MAP);
        mission_log_menu = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Update the map object's map and markings with the information
         * on the mission log file.
         */
        SARMissionLoadMissionLogToMenuMap(
            core_ptr, mission_log_menu,
            fname.mission_log
        );

        /* Switch to log map menu. */
        SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_MISSION_LOG_MAP);
}

/*
 *	Switch to mission map procedure.
 */
static void SARMenuCBMissionMap(sar_core_struct *core_ptr)
{
	int i;
	const char *filename;
	sar_menu_struct *m;
	sar_menu_list_item_struct *item;
	sar_menu_list_item_data_struct *d;


        if(core_ptr == NULL)
            return;

        /* Get pointer to mission menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Get selected mission on missions list */
        item = SARMenuListGetSelectedItem(m, SAR_OP_MISSION_LIST_SELECT);
	d = SAR_MENU_LIST_ITEM_DATA(
	    (item != NULL) ? item->client_data : NULL
	);
        filename = (d != NULL) ? d->filename : NULL;

        /* Get pointer to mission map menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION_MAP);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

	/* Load scene's map and markings to the map object from the given
	 * scene file.
	 */
	SARMissionLoadSceneToMenuMap(
	    core_ptr, m, filename
	);

	/* Switch to mission map menu. */
        SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_MISSION_MAP);
}

/*
 *	Switch to mission briefing procedure.
 */
static void SARMenuCBMissionBrief(sar_core_struct *core_ptr)
{
	int i;
	const char *filename;
	char *mission_desc;
        sar_menu_struct *m;
        sar_menu_list_item_struct *item;
	sar_menu_list_item_data_struct *d;


        if(core_ptr == NULL)
            return;

        /* Get pointer to mission menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Get selected mission on missions list */
        item = SARMenuListGetSelectedItem(m, SAR_OP_MISSION_LIST_SELECT);
        d = SAR_MENU_LIST_ITEM_DATA(
	    (item != NULL) ? item->client_data : NULL
	);
        filename = (d != NULL) ? d->filename : NULL;


        /* Get pointer to mission brief menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION_BRIEF);
        m = (i > -1) ? core_ptr->menu[i] : NULL;


	/* Get mission description (briefing) */
	mission_desc = SARMissionLoadDescription(filename);
	if((SARMenuGetObjectByID(m, SAR_OP_MISSION_BRIEF_MESG, &i) != NULL) &&
	   (mission_desc != NULL)
	)
	    SARMenuMessageBoxSet(
		core_ptr->display, m, i, mission_desc, False
	    );
	free(mission_desc);

        /* Switch to mission brief menu. */
        SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_MISSION_BRIEF);
}

/*
 *	Begin free flight procedure.
 */
static void SARMenuCBBeginFreeFlight(sar_core_struct *core_ptr)
{
	int i;
	const char *filename;
	sar_menu_struct *m;
	sar_menu_list_item_struct *item;
	sar_menu_list_item_data_struct *d;
	int spin_num, spin_sel_item;
	sar_menu_switch_struct *sw;
	sar_menu_spin_struct *spin_ptr;
	sar_position_struct *start_pos = NULL;
	sar_direction_struct *start_dir = NULL;
	const char *weather_preset_name = NULL;
	Boolean free_flight_system_time = False;


	if(core_ptr == NULL)
	    return;

	/* Begin collecting information from menu objects in order to
	 * pass to the begin free flight simulation function.
	 */

	/* Get pointer to menu Free Flight: Select Scenery */
	i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT);
	m = (i > -1) ? core_ptr->menu[i] : NULL;

	/* Get selected scenery from scenery list */
	item = SARMenuListGetSelectedItem(m, SAR_OP_FREE_FLIGHT_SCENERY_LIST);
	d = SAR_MENU_LIST_ITEM_DATA(
	    (item != NULL) ? item->client_data : NULL
	);
 	filename = (d != NULL) ? d->filename : NULL;
	free(core_ptr->cur_scene_file);
	core_ptr->cur_scene_file = STRDUP(filename);

        /* Get starting location from locations list */
	item = SARMenuListGetSelectedItem(m, SAR_OP_FREE_FLIGHT_LOCATIONS_LIST);
        d = SAR_MENU_LIST_ITEM_DATA(
            (item != NULL) ? item->client_data : NULL
        );
 	if(d != NULL)
	{
            start_pos = &d->pos;
	    start_dir = &d->dir;
	}


        /* Get pointer to menu Free Flight: Select Aircraft */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Get selected aircraft from aircrafts list */
	item = SARMenuListGetSelectedItem(m, SAR_OP_FREE_FLIGHT_AIRCRAFTS_LIST);
        d = SAR_MENU_LIST_ITEM_DATA(
            (item != NULL) ? item->client_data : NULL
        );
        filename = (d != NULL) ? d->filename : NULL;
        free(core_ptr->cur_player_model_file);
        core_ptr->cur_player_model_file = STRDUP(filename);


	/* Get pointer to menu Free Flight: Weather */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_WEATHER);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

	/* Weather preset spin */
	spin_ptr = SAR_MENU_SPIN(SARMenuGetObjectByID(
	    m, SAR_OP_MENU_FREE_FLIGHT_WEATHER_CONDITION,
	    &spin_num
	));
	weather_preset_name = SARMenuSpinGetCurrentValue(
	    m, spin_num, &spin_sel_item
	);

	/* Free flight system time switch */
	sw = SAR_MENU_SWITCH(SARMenuGetObjectByID(
            m, SAR_OP_MENU_FREE_FLIGHT_SYSTEM_TIME,
            NULL
        ));
	free_flight_system_time = (sw != NULL) ? sw->state : False;


	/* Switch to free flight simulation and leave menu system. */
	SARSimBeginFreeFlight(
	    core_ptr,
	    core_ptr->cur_scene_file,
	    core_ptr->cur_player_model_file,
	    start_pos, start_dir,
	    weather_preset_name, free_flight_system_time
	);
}

/*
 *	Switch to aircraft information menu procedure.
 */
static void SARMenuCBFreeFlightAircraftInfo(sar_core_struct *core_ptr)
{
	int i, ov_num;
	const char *filename;
	sar_menu_struct *m;
        sar_menu_list_item_struct *item;
        sar_menu_list_item_data_struct *d;
	sar_menu_objview_struct *ov_ptr;


        if(core_ptr == NULL)
            return;

        /* Get pointer to menu Free Flight: Select Aircraft */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

        /* Get selected aircraft from aircrafts list */
        item = SARMenuListGetSelectedItem(m, SAR_OP_FREE_FLIGHT_AIRCRAFTS_LIST);
        d = SAR_MENU_LIST_ITEM_DATA(
	    (item != NULL) ? item->client_data : NULL
	);
        filename = (d != NULL) ? d->filename : NULL;


        /* Get pointer to menu Free Flight: Aircraft information */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT_INFO);
        m = (i > -1) ? core_ptr->menu[i] : NULL;

	/* Get pointer to aircraft information 3d object viewer object
	 * on the menu.
	 */
	ov_num = -1;
	ov_ptr = NULL;
	if(m != NULL)
	{
	    ov_num = 0;

	    if((ov_num >= 0) && (ov_num < m->total_objects))
	    {
		void *p = m->object[ov_num];
		if((p != NULL) ?
		    (*(int *)p == SAR_MENU_OBJECT_TYPE_OBJVIEW) : False
		)
		    ov_ptr = SAR_MENU_OBJVIEW(p);
	    }
	}
	if(ov_ptr != NULL)
	{
            /* Mark input as busy while loading new aircraft information,
	     * note that the input will be set busy again when we
	     * switch menus but that is okay.
	     */
            GWSetInputBusy(core_ptr->display);

	    /* Set up aircraft information 3d object view to display
	     * new aircraft.
	     */
	    SARMenuObjViewLoad(
		core_ptr->display, m,
		ov_ptr, ov_num, filename
	    );
	    SARMenuObjViewRebufferImage(
		core_ptr->display, m,
		ov_ptr, ov_num
	    );
	}

	/* Switch to aircraft information menu. */
	SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT_INFO);
}


/*
 *	Map menu button callback.
 */
static void SARMenuCBMapOp(sar_core_struct *core_ptr, int id_code)
{
	gw_display_struct *display;
	int i, ptype, map_num = -1;
	int width, height, w, h;
	void *p;
        sar_menu_struct *m;
	sar_menu_map_struct *map_ptr = NULL;

	float mtop_coeff;
	int	scroll_inc = 20;	/* Scroll increment in pixels. */
	int	tex_w, tex_h,	/* All these units in pixels. */
		scroll_x_min, scroll_x_max,
		scroll_y_min, scroll_y_max;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	if(display == NULL)
	    return;

        /* Get current menu (assuming it has a map object on it. */
	m = SARGetCurrentMenuPtr(core_ptr);
	if(m == NULL)
	    return;

        /* Get first map object on the menu. */
        for(i = 0; i < m->total_objects; i++)
        {
            p = m->object[i];
            if(p == NULL)
                continue;
            else
                ptype = (*(int *)p);

            if(ptype == SAR_MENU_OBJECT_TYPE_MAP)
            {
                map_num = i;
                map_ptr = (sar_menu_map_struct *)p;
                break;
            }
        }
        if(map_ptr == NULL)
            return;

/* Calculates scroll bounds by first fetching latest mtop_coeff
 * value and updates variables tex_w, tex_h, scroll_x_min, scroll_y_min,
 * scroll_x_max, and scroll_y_max.
 */
#define GET_SCROLL_BOUNDS	\
{ \
 mtop_coeff = map_ptr->m_to_pixels_coeff; \
 \
 /* Get texture size in pixels. */ \
 tex_w = (int)(map_ptr->bg_tex_width * mtop_coeff); \
 tex_h = (int)(map_ptr->bg_tex_height * mtop_coeff); \
 \
 /* Calculate scroll bounds in pixels. */ \
 scroll_x_min = (w / 2) - (tex_w / 2); \
 scroll_x_max = (tex_w / 2) - (w / 2); \
 \
 scroll_y_min = (h / 2) - (tex_h / 2); \
 scroll_y_max = (tex_h / 2) - (h / 2); \
}

	/* Get bounds of display. */
	GWContextGet(
	    display, GWContextCurrent(display),
	    NULL, NULL,
	    NULL, NULL,
	    &width, &height
	);

	/* Get bounds of map object. */
        w = (int)(map_ptr->width * width);
        if(w <= 0)
            w = (int)(width - (2 * DEF_XMARGIN));
        if(w <= 0)
            w = DEF_XMARGIN;

        if(map_ptr->height > 0)
            h = (int)(map_ptr->height * height);
        else
            h = DEF_HEIGHT;
        if(h <= 0)
            h = DEF_YMARGIN;


	/* Calculate scroll bounds. */
	GET_SCROLL_BOUNDS

	/* Handle by id code. */
	switch(id_code)
	{
	  case SAR_OP_MISSION_MAP_LEFT: case SAR_OP_MISSION_LOG_MAP_LEFT:
/*
	    map_ptr->scroll_x = CLIP(
		map_ptr->scroll_x + scroll_inc,
		scroll_x_min, scroll_x_max
	    );
 */
	    map_ptr->scroll_x = map_ptr->scroll_x + scroll_inc;
	    break;

          case SAR_OP_MISSION_MAP_RIGHT: case SAR_OP_MISSION_LOG_MAP_RIGHT:
/*
            map_ptr->scroll_x = CLIP(
                map_ptr->scroll_x - scroll_inc,
                scroll_x_min, scroll_x_max
            );
 */
	    map_ptr->scroll_x = map_ptr->scroll_x - scroll_inc;
            break;

          case SAR_OP_MISSION_MAP_UP: case SAR_OP_MISSION_LOG_MAP_UP:
/*
            map_ptr->scroll_y = CLIP(
                map_ptr->scroll_y - scroll_inc,
                scroll_y_min, scroll_y_max
            );
 */
	    map_ptr->scroll_y = map_ptr->scroll_y - scroll_inc;
            break;

          case SAR_OP_MISSION_MAP_DOWN: case SAR_OP_MISSION_LOG_MAP_DOWN:
/*
            map_ptr->scroll_y = CLIP(
                map_ptr->scroll_y + scroll_inc,
                scroll_y_min, scroll_y_max
            );
 */
	    map_ptr->scroll_y = map_ptr->scroll_y + scroll_inc;
            break;

          case SAR_OP_MISSION_MAP_ZOOM_IN: case SAR_OP_MISSION_LOG_MAP_ZOOM_IN:
	    /* This portion of code came from menumap.c's drag zoom
	     * event handling.
	     */
	    {
                    float scroll_x_coeff, scroll_y_coeff;
                    float prev_m_to_pixels_coeff =
                        map_ptr->m_to_pixels_coeff;

                    /* Calculate new zoom. */
                    map_ptr->m_to_pixels_coeff = (float)CLIP(
                        map_ptr->m_to_pixels_coeff +
                            (scroll_inc *
                            map_ptr->m_to_pixels_coeff / 100),
                        0.002,          /* Zoomed out max. */
                        0.050           /* Zoomed in max. */
                    );

                    /* Need to calculate last scroll position coeff
                     * relative to the previous zoom dimension, then
                     * set the new scroll position by multiplying
                     * the scroll position coeff just calculated by
                     * the new zoom dimension.
                     */
                    if(map_ptr->bg_tex_width > 0)
                        scroll_x_coeff = (float)map_ptr->scroll_x /
                            (map_ptr->bg_tex_width *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_x_coeff = 0.0;
                    map_ptr->scroll_x = (int)(map_ptr->bg_tex_width *
                        map_ptr->m_to_pixels_coeff * scroll_x_coeff);

                    if(map_ptr->bg_tex_height > 0)
                        scroll_y_coeff = -(float)map_ptr->scroll_y /
                            (map_ptr->bg_tex_height *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_y_coeff = 0.0;

                    map_ptr->scroll_y = -(int)(map_ptr->bg_tex_height *
                        map_ptr->m_to_pixels_coeff * scroll_y_coeff);

                    /* Need to calculate scroll bounds again. */
                    GET_SCROLL_BOUNDS

                    /* Reclip scroll bounds. */
/* Its safe to scroll anywhere, plus we can see offmap markings.
                    map_ptr->scroll_x = CLIP(
                        map_ptr->scroll_x,
                        scroll_x_min, scroll_x_max
                    );
                    map_ptr->scroll_y = CLIP(
                        map_ptr->scroll_y,
                        scroll_y_min, scroll_y_max
                    );
 */
	    }
	    break;

          case SAR_OP_MISSION_MAP_ZOOM_OUT: case SAR_OP_MISSION_LOG_MAP_ZOOM_OUT:
            /* This portion of code came from menumap.c's drag zoom  
             * event handling.
             */
            {
                    float scroll_x_coeff, scroll_y_coeff;
                    float prev_m_to_pixels_coeff =
                        map_ptr->m_to_pixels_coeff;

                    /* Calculate new zoom. */
                    map_ptr->m_to_pixels_coeff = (float)CLIP(
                        map_ptr->m_to_pixels_coeff -
                            (scroll_inc *
                            map_ptr->m_to_pixels_coeff / 100),
                        0.002,          /* Zoomed out max. */
                        0.050           /* Zoomed in max. */
                    );

                    /* Need to calculate last scroll position coeff
                     * relative to the previous zoom dimension, then
                     * set the new scroll position by multiplying
                     * the scroll position coeff just calculated by
                     * the new zoom dimension.
                     */
                    if(map_ptr->bg_tex_width > 0)
                        scroll_x_coeff = (float)map_ptr->scroll_x /
                            (map_ptr->bg_tex_width *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_x_coeff = 0.0;
                    map_ptr->scroll_x = (int)(map_ptr->bg_tex_width *
                        map_ptr->m_to_pixels_coeff * scroll_x_coeff);

                    if(map_ptr->bg_tex_height > 0)
                        scroll_y_coeff = -(float)map_ptr->scroll_y /
                            (map_ptr->bg_tex_height * 
                                prev_m_to_pixels_coeff);
                    else
                        scroll_y_coeff = 0.0;

                    map_ptr->scroll_y = -(int)(map_ptr->bg_tex_height *
                        map_ptr->m_to_pixels_coeff * scroll_y_coeff);

                    /* Need to calculate scroll bounds again. */   
                    GET_SCROLL_BOUNDS

                    /* Reclip scroll bounds. */
/* Its safe to scroll anywhere, plus we can see offmap markings.
                    map_ptr->scroll_x = CLIP(
                        map_ptr->scroll_x,
                        scroll_x_min, scroll_x_max
                    );
                    map_ptr->scroll_y = CLIP(
                        map_ptr->scroll_y,
                        scroll_y_min, scroll_y_max
                    );
 */
            }
            break;

	  case SAR_OP_MISSION_LOG_MAP_EVENT_NEXT:
	    SARMenuMapMarkingSelectNext(map_ptr, True);
	    break;

	  case SAR_OP_MISSION_LOG_MAP_EVENT_PREV:
            SARMenuMapMarkingSelectPrev(map_ptr, True);
	    break;
	}

	/* Redraw map object. */
	SARMenuDrawObject(
	    display, m, map_num
	);
	GWSwapBuffer(display);

#undef GET_SCROLL_BOUNDS
}


/*
 *	Menu button callback.
 */
void SARMenuButtonCB(void *object, int id_code, void *client_data)
{
	sar_core_struct *core_ptr = SAR_CORE(client_data);
	if((core_ptr == NULL) || (object == NULL))
	    return;

	switch(id_code)
	{
	  case SAR_OP_MENU_TO_MAIN:
	    SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_MAIN);
	    break;

	  case SAR_OP_MENU_EXIT:
            runlevel = 1;
            break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT);
	    break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT);
            break;

          case SAR_OP_MENU_TO_FREE_FLIGHT_WEATHER:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_WEATHER);
            break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT_INFO:
	    SARMenuCBFreeFlightAircraftInfo(core_ptr);
            break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN:
            SARMenuCBBeginFreeFlight(core_ptr);
            break;

	  case SAR_OP_MENU_TO_MISSION:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_MISSION);
	    break;

	  case SAR_OP_MENU_TO_MISSION_LOG_MAP:
	    SARMenuCBViewLastLog(core_ptr);
	    break;

          case SAR_OP_MENU_TO_MISSION_MAP:
	    SARMenuCBMissionMap(core_ptr);
            break;

          case SAR_OP_MENU_TO_MISSION_BRIEF:
	    SARMenuCBMissionBrief(core_ptr);
            break;

	  case SAR_OP_MENU_TO_MISSION_BEGIN:
	    SARMenuCBBeginMission(core_ptr);
	    break;

	  case SAR_OP_MISSION_MAP_LEFT:
	  case SAR_OP_MISSION_MAP_RIGHT:
	  case SAR_OP_MISSION_MAP_UP:
	  case SAR_OP_MISSION_MAP_DOWN:
	  case SAR_OP_MISSION_MAP_ZOOM_IN:
          case SAR_OP_MISSION_MAP_ZOOM_OUT:
          case SAR_OP_MISSION_LOG_MAP_LEFT:
          case SAR_OP_MISSION_LOG_MAP_RIGHT:
          case SAR_OP_MISSION_LOG_MAP_UP:
          case SAR_OP_MISSION_LOG_MAP_DOWN:
          case SAR_OP_MISSION_LOG_MAP_ZOOM_IN:
          case SAR_OP_MISSION_LOG_MAP_ZOOM_OUT:
	  case SAR_OP_MISSION_LOG_MAP_EVENT_NEXT:
	  case SAR_OP_MISSION_LOG_MAP_EVENT_PREV:
            SARMenuCBMapOp(core_ptr, id_code);   
            break;

	  case SAR_OP_MENU_TO_CAMPAIGN:
/*
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_CAMPAIGN);
 */
            GWOutputMessage(
                core_ptr->display, GWOutputMessageTypeWarning,
                "Campaign Support Not Finished",
 "Sorry, campaign support is still under construction.",
"The coding work for campaign support is still under construction\n\
and not yet playable, please choose to either fly a mission or\n\
free flight."
            );

	    break;

	  case SAR_OP_MENU_TO_OPTIONS:
	    SARMenuOptionsFetch(core_ptr);
	    SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS);
	    break;

          case SAR_OP_MENU_TO_OPTIONS_SIMULATION:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS_SIMULATION);
            break;

          case SAR_OP_MENU_TO_OPTIONS_CONTROLLER:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER);
            break;

	  case SAR_OP_MENU_TO_OPTIONS_CONTROLLER_JS_BTN:
	    SARMenuSwitchToMenu(
		core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER_JS_BTN
	    );
            break;

          case SAR_OP_MENU_TO_OPTIONS_CONTROLLER_TEST:
            SARMenuSwitchToMenu(
                core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER_TEST
            );
	    break;

          case SAR_OP_MENU_TO_OPTIONS_GRAPHICS:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS_GRAPHICS);
            break;

          case SAR_OP_MENU_TO_OPTIONS_GRAPHICS_INFO:
	    SARMenuOptionsGraphicsInfoRefresh(core_ptr);
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS_GRAPHICS_INFO);
            break;

          case SAR_OP_MENU_TO_OPTIONS_SOUND:
	    SARMenuOptionsSoundRefresh(core_ptr);
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS_SOUND);
            break;

          case SAR_OP_MENU_TO_OPTIONS_SOUND_INFO:
	    SARMenuOptionsSoundInfoRefresh(core_ptr);
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_OPTIONS_SOUND_INFO);
            break;

	  case SAR_OP_TEST:
	    SARMenuButtonCBTest(core_ptr);
	    break;

	  default:
	    fprintf(stderr,
 "SARMenuButtonCB(): Internal error: Unsupported menu operation code `%i'.\n",
		id_code
	    );
	    break;
	}
}


/*
 *	Menu list select callback.
 */
void SARMenuListSelectCB(
        void *object, int id_code, void *client_data,
        int item_num, void *item_ptr, void *item_data
)
{
	gw_display_struct *display;
	sar_core_struct *core_ptr = SAR_CORE(client_data);
	sar_menu_struct *m;
	sar_menu_list_struct *list = SAR_MENU_LIST(object);
	sar_menu_list_item_data_struct *d = SAR_MENU_LIST_ITEM_DATA(item_data);
	sar_option_struct *opt;
	if((list == NULL) || (d == NULL) || (core_ptr == NULL))
	    return;

	display = core_ptr->display;
	opt = &core_ptr->option;

	/* Get current selected menu and name. */
	m = SARGetCurrentMenuPtr(core_ptr);
	if(m == NULL)
	    return;

	switch(id_code)
	{
	  /* Mission: Select Mission */
	  case SAR_OP_MISSION_LIST_SELECT:
	    opt->last_selected_mission = item_num;
	    break;

	  /* Free Flight: Select Scenery */
	  case SAR_OP_FREE_FLIGHT_SCENERY_LIST:
	    /* A new scenery was selected on the list, so the scenery's
	     * locations list needs to be updated.
	     */
	    if(d->filename != NULL)
	    {
		const char *filename = d->filename;
		int i, loc_list_num = -1;
		void *o;
		sar_menu_list_struct *loc_list = NULL;

		for(i = 0; i < m->total_objects; i++)
		{
		    o = m->object[i];
		    if(!SAR_MENU_IS_LIST(o))
			continue;

		    loc_list = SAR_MENU_LIST(o);
		    if(loc_list->id_code == SAR_OP_FREE_FLIGHT_LOCATIONS_LIST)
		    {
			loc_list_num = i;
			break;
		    }
		}
		if((loc_list_num > -1) && (loc_list != NULL))
		{
		    /* Load locations from scene, existing locations
		     * on the locations list will be deleted.
		     */
		    SARSceneLoadLocationsToList(
		        core_ptr, m, loc_list, loc_list_num,
		        filename
		    );

		    /* Select first item on locations list. */
		    if(loc_list->total_items > 0)
		        loc_list->selected_item = 0;

		    /* Need to redraw the locations list. */
		    SARMenuDrawObject(display, m, loc_list_num);
		    GWSwapBuffer(display);
		}

		opt->last_selected_ffscene = item_num;
	    }
	    break;

	  /* Free Flight: Select Scenery Location */
	  case SAR_OP_FREE_FLIGHT_LOCATIONS_LIST:
	    break;

	  /* Free Flight: Select Aircraft */
	  case SAR_OP_FREE_FLIGHT_AIRCRAFTS_LIST:
            opt->last_selected_ffaircraft = item_num;
	    break;
	}
}

/*
 *      Menu list activate callback.
 */
void SARMenuListActivateCB(
        void *object, int id_code, void *client_data,
        int item_num, void *item_ptr, void *item_data
)
{
        gw_display_struct *display;
        sar_core_struct *core_ptr = SAR_CORE(client_data);
        sar_menu_struct *m;
        sar_menu_list_struct *list = SAR_MENU_LIST(object);
        sar_menu_list_item_data_struct *d = SAR_MENU_LIST_ITEM_DATA(item_data);
        sar_option_struct *opt;
        if((list == NULL) || (d == NULL) || (core_ptr == NULL))
            return;

        display = core_ptr->display;
        opt = &core_ptr->option;

        /* Get current selected menu and name. */
        m = SARGetCurrentMenuPtr(core_ptr);
        if(m == NULL)
            return;

        switch(id_code)
        {
          /* Mission: Select Mission */
          case SAR_OP_MISSION_LIST_SELECT:
            SARMenuCBMissionBrief(core_ptr);
            break;

          /* Free Flight: Select Scenery */
          case SAR_OP_FREE_FLIGHT_SCENERY_LIST:
	    break;

          /* Free Flight: Select Scenery Location */
          case SAR_OP_FREE_FLIGHT_LOCATIONS_LIST:
            SARMenuSwitchToMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT);
            break;

          /* Free Flight: Select Aircraft */
          case SAR_OP_FREE_FLIGHT_AIRCRAFTS_LIST:
            SARMenuCBBeginFreeFlight(core_ptr);
            break;
	}

}

/*
 *      Menu standard switch object callback.
 */
void SARMenuSwitchCB(
	void *object, int id_code, void *client_data,
	Boolean state
)
{
        sar_core_struct *core_ptr = SAR_CORE(client_data);
        sar_menu_switch_struct *sw = (sar_menu_switch_struct *)object;
	sar_option_struct *opt;

        if((core_ptr == NULL) || (sw == NULL))
            return;

	opt = &core_ptr->option;

        /* Handle by menu object id code. */
        switch(id_code)
        {
          case SAR_OP_MENU_FREE_FLIGHT_SYSTEM_TIME:
            opt->system_time_free_flight = sw->state;
            break;

        }
}

/*
 *	Menu standard spin object callback.
 */
void SARMenuSpinCB(
	void *object, int id_code, void *client_data,
	char *value
)
{
        sar_core_struct *core_ptr = SAR_CORE(client_data);
        sar_menu_spin_struct *spin_ptr = (sar_menu_spin_struct *)object;
	sar_option_struct *opt;

        if((core_ptr == NULL) || (spin_ptr == NULL))
            return;

	opt = &core_ptr->option;

	/* Handle by menu object id code. */
	switch(id_code)
	{
	  case SAR_OP_MENU_FREE_FLIGHT_WEATHER_CONDITION:
            opt->last_selected_ffweather = spin_ptr->cur_value;
	    break;

        }
}
