#include <stdlib.h>
#include <string.h>

#include "menu.h"
#include "sar.h"
#include "sarmenuop.h"


void SARMenuSwitchToMenu(sar_core_struct *core_ptr, const char *name);
sar_menu_list_struct *SARMenuGetList(
        sar_menu_struct *m, int skip, int *list_num
);
sar_menu_spin_struct *SARMenuGetSpin(
        sar_menu_struct *m, int skip, int *spin_num
);
sar_menu_list_item_struct *SARMenuListGetSelectedItem(
        sar_menu_struct *m, int id_code
);
char *SARMenuGetCurrentMenuName(sar_core_struct *core_ptr);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)


/*
 *	Switches to the menu specified by name.
 *
 *	If name is NULL then the current menu on the core structure
 *	is set to -1, indicating no menu is selected and implies
 *	switching to simulation.
 */
void SARMenuSwitchToMenu(sar_core_struct *core_ptr, const char *name)
{
	int new_menu_num, old_menu_num;
	gw_display_struct *display;
        const sar_option_struct *opt;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	opt = &core_ptr->option;

        /* Record previous menu number. */
        old_menu_num = core_ptr->cur_menu;

	/* Is given menu name NULL? */
	if(name == NULL)
	{
	    /* No menu name given, this implies we are going out of the
	     * menu system and entering simulation.
	     */
	    sar_menu_struct *old_menu = (SARIsMenuAllocated(core_ptr, old_menu_num) ?
                core_ptr->menu[old_menu_num] : NULL
            );

	    /* Unload image on old menu. */
/*	    if(opt->prioritize_memory && (old_menu != NULL)) */
	    if(old_menu != NULL)
		SARMenuUnloadBackgroundImage(old_menu);

	    /* Unselect current menu on core structure, indicating that
	     * no menu is selected and that we are not in the menus.
	     */
	    core_ptr->cur_menu = -1;

	    /* Skip redrawing, since redrawing would be done rather
	     * frequently during the simulation.
	     */
/*	    GWPostRedraw(display); */

	    return;
	}

	/* Get menu number to change to by matching the menu who's name
	 * is the same as the given name.
	 */
	new_menu_num = SARMatchMenuByName(core_ptr, name);
	/* Got match and matched menu number is different from the one
	 * currently selected?
	 */
	if((new_menu_num > -1) && (new_menu_num != core_ptr->cur_menu))
	{
            sar_menu_struct *old_menu = (SARIsMenuAllocated(core_ptr, old_menu_num) ?
		core_ptr->menu[old_menu_num] : NULL
	    );
	    sar_menu_struct *new_menu = (SARIsMenuAllocated(core_ptr, new_menu_num) ?
		core_ptr->menu[new_menu_num] : NULL
	    );
	    const char *old_bg_image_path, *new_bg_image_path;


	    /* Mark input as busy while deallocating/allocating resources
	     * during changing menus.
	     */
            GWSetInputBusy(display);


	    /* Get background image paths on both new and old menus, use
	     * NULL if respective menu is NULL.
	     */
	    old_bg_image_path = (const char *)((old_menu != NULL) ?
		old_menu->bg_image_path : NULL
	    );
            new_bg_image_path = (const char *)((new_menu != NULL) ?
                new_menu->bg_image_path : NULL
            );

	    /* Check if background images are the same on both old and
	     * new menus by comparing the background image paths (if
	     * defined).
	     */
	    if(((old_bg_image_path != NULL) &&
                (new_bg_image_path != NULL)) ?
	         ((old_menu->bg_image != NULL) ?
		  !strcmp(old_bg_image_path, new_bg_image_path) : 0)
		: 0
	    )
	    {
		/* Image paths are the same, so unload the image on the
		 * new menu and transfer the image from the old menu to
		 * the new menu. Mark the old menu's image NULL after
		 * transfer so that it dosen't get referenced again.
		 * Note that both new_menu and old_menu are known to be
		 * not NULL from the above check.
		 */
		SARMenuUnloadBackgroundImage(new_menu);
		if(new_menu->bg_image == NULL)
		{
		    new_menu->bg_image = old_menu->bg_image;
		    old_menu->bg_image = NULL;
		}
	    }
	    else
	    {
		/* Image paths are different, so unload background image
		 * on old menu and load new background image on new menu.
		 */
/*		if(opt->prioritize_memory && (old_menu != NULL)) */
		if(old_menu != NULL)
		    SARMenuUnloadBackgroundImage(old_menu);

		if(opt->menu_backgrounds &&
		   ((new_menu != NULL) ? (new_menu->bg_image == NULL) : 0)
		)
		    SARMenuLoadBackgroundImage(new_menu);
	    }

	    /* Is the new menu valid? */
	    if(new_menu != NULL)
	    {
		/* New menu is valid, so switch to it. */
		int n;
		sar_menu_struct *m = new_menu;

		/* Reset all progress bars on the new menu. */
		for(n = 0; n < m->total_objects; n++)
		{
		    if(!SAR_MENU_IS_PROGRESS(m->object[n]))
			continue;

		    SARMenuProgressSet(
			display, m, n,
			0.0f, False
		    );
		}

		/* Update index number of the selected menu on the core
		 * structure.
		 */
	        core_ptr->cur_menu = new_menu_num;
	    }

	    /* Redraw. */
	    GWPostRedraw(display);

	    /* Mark input as ready. */
	    GWSetInputReady(display);
	}
}

/*
 *	Returns the pointer to the list object on the menu,
 *	skipping the indicated number of lists.
 *
 *	Can return NULL for no match.
 */
sar_menu_list_struct *SARMenuGetList(
	sar_menu_struct *m, int skip, int *list_num
)     
{
        int i;
        void *ptr;
        sar_menu_list_struct *list_ptr;


        if(m == NULL)
            return(NULL);

	if(list_num != NULL)
	    *list_num = -1;
 
        for(i = 0; i < m->total_objects; i++)
        {       
            ptr = m->object[i];
            if(ptr == NULL)
                continue;

            if(*(int *)ptr != SAR_MENU_OBJECT_TYPE_LIST)
                continue;

            /* Get pointer to list object. */
            list_ptr = ptr;

            /* Skip this list object? */
            if(skip > 0)
            {
                skip--;
                continue;
            }
	    else
	    {
		if(list_num != NULL)
		    *list_num = i;
                return(list_ptr);
	    }
	}

	return(NULL);
}

/*
 *      Returns the pointer to the spin object on the menu,
 *      skipping the indicated number of spin objects.
 *
 *	Can return NULL for no match.
 */
sar_menu_spin_struct *SARMenuGetSpin(
        sar_menu_struct *m, int skip, int *spin_num
)
{
        int i;
        void *ptr;
	sar_menu_spin_struct *spin_ptr;


        if(m == NULL)
            return(NULL);

        if(spin_num != NULL)
            *spin_num = -1;

        for(i = 0; i < m->total_objects; i++)
        {
            ptr = m->object[i];
            if(ptr == NULL)
                continue;
 
            if(*(int *)ptr != SAR_MENU_OBJECT_TYPE_SPIN)
                continue;

            /* Get pointer to spin object. */
            spin_ptr = ptr;

            /* Skip this spin object? */
            if(skip > 0)
            {
                skip--;
                continue;
            }
            else
            {
                if(spin_num != NULL)
                    *spin_num = i;

                return(spin_ptr);
            }
        }

        return(NULL);
}

/*
 *	Returns the pointer to the selected list item on the list object
 *	on the menu, skipping the indicated number of list objects.
 */
sar_menu_list_item_struct *SARMenuListGetSelectedItem(
	sar_menu_struct *m, int id_code
)
{
	int i;
	sar_menu_list_struct *list = SAR_MENU_LIST(
	    SARMenuGetObjectByID(m, id_code, NULL)
	);
	if(list == NULL)
	    return(NULL);

	i = list->selected_item;
	if((i >= 0) && (i < list->total_items))
	    return(list->item[i]);
	else
	    return(NULL);
}

/*
 *	Returns the pointer to the name of the currently selected
 *	menu or NULL on failure.
 */
char *SARMenuGetCurrentMenuName(sar_core_struct *core_ptr)
{
	int cur_menu;
	sar_menu_struct *m;


	if(core_ptr == NULL)
	    return(NULL);

	cur_menu = core_ptr->cur_menu;
	if(SARIsMenuAllocated(core_ptr, cur_menu))
	    m = core_ptr->menu[cur_menu];
	else
	    return(NULL);

	return(m->name);
}
