#include <stdio.h>  
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <sys/types.h>

#ifdef __MSW__
# include <windows.h>
#else
# include <sys/time.h>
# include <unistd.h>
#endif
#include <GL/gl.h>

#include "../include/strexp.h"
#include "../include/string.h"

#include "menu.h"
#include "obj.h"
#include "sar.h"   
#include "sardraw.h"
#include "config.h"


#ifdef __MSW__
static double rint(double x);
#endif	/* __MSW__ */

Boolean SARGetGLVersion(int *major, int *minor, int *release);
char *SARGetGLVendorName(void);
char *SARGetGLRendererName(void);
char **SARGelGLExtensionNames(int *strc);

int SARIsMenuAllocated(sar_core_struct *core_ptr, int n);
int SARMatchMenuByName(
        sar_core_struct *core_ptr,
        const char *name
);
sar_menu_struct *SARGetCurrentMenuPtr(sar_core_struct *core_ptr);
void SARDeleteListItemData(sar_menu_list_item_struct *item_ptr);

char *SARTimeOfDayString(sar_core_struct *core_ptr, float t);
char *SARDeltaTimeString(sar_core_struct *core_ptr, time_t t);

void SARDrawMapProcessHits(
        sar_core_struct *core_ptr, GLint hits, GLuint buffer[]
);

void SARSetGlobalTextColorBrightness(
	sar_core_struct *core_ptr, float g
);

void SARReportGLError(
	sar_core_struct *core_ptr, GLenum error_code
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define IS_STRING_EMPTY(s)      (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)


#ifdef __MSW__
static double rint(double x)
{
	if((double)((double)x - (int)x) > (double)0.5)
	    return((double)((int)x + (int)1));
	else
	    return((double)((int)x));
}
#endif	/* __MSW__ */


/*
 *	Returns the version numbers for the GL version, returns
 *	GL_TRUE on success or GL_FALSE on failure.
 *
 *	If any of the inputs are NULL, then that respective input will
 *	not be set.
 */
Boolean SARGetGLVersion(int *major, int *minor, int *release)
{
	const GLubyte *strptr;
	char **strv;
	int strc;


	/* Reset inputs. */
	if(major != NULL)
	    *major = 0;
        if(minor != NULL)
            *minor = 0;
        if(release != NULL)
            *release = 0;

	/* Get statically allocated '.' character separated string
	 * containing the version numbers. Format can be either
	 * "major.minor" or "major.minor.release".
	 */
	strptr = glGetString(GL_VERSION);
	if(strptr == NULL)
	    return(GL_FALSE);

	/* Explode version string at the '.' characters. */
	strv = strchrexp((const char *)strptr, '.', &strc);
	if(strv == NULL)
	    return(GL_FALSE);

	if((strc > 0) && (major != NULL))
	    *major = atoi(strv[0]);

        if((strc > 1) && (minor != NULL))
            *minor = atoi(strv[1]);

        if((strc > 2) && (release != NULL))
            *release = atoi(strv[2]);

	StringFreeArray(strv, strc);

	return(GL_TRUE);
}

/*
 *      Returns a dynamically allocated string containing the GL
 *      vendor name. The returned string must be free()'ed by the
 *      calling function.
 *
 *      Can return NULL on error.
 */
char *SARGetGLVendorName(void)
{
	/* Return copy of string describing the GL vendor name. */
	return(STRDUP((const char *)glGetString(GL_VENDOR)));
}

/*
 *	Returns a dynamically allocated string containing the GL
 *	renderer name. The returned string must be free()'ed by the
 *	calling function.
 *
 *	Can return NULL on error.
 */
char *SARGetGLRendererName(void)
{
	/* Return copy of string describing the GL renderer name. */
	return(STRDUP((const char *)glGetString(GL_RENDERER)));
}


/*
 *	Returns a dynamically allocated array of strings containing
 *	the list of GL extension names available.
 *
 *	If strc is not NULL then it will be set to the number of
 *	strings returned.
 *
 *	Can return NULL on error.
 */
char **SARGelGLExtensionNames(int *strc)
{
        const GLubyte *strptr;
	char **strv;
	int lstrc;


	/* Reset inputs. */
	if(strc != NULL)
	    *strc = 0;

	/* Get statically allocated space separated string containing
	 * the GL extensions list.
	 */
        strptr = glGetString(GL_EXTENSIONS);
	if(strptr == NULL)
	    return(NULL);

	/* Explode space separated string. */
	strv = strexp((char *)strptr, &lstrc);
	if(strv == NULL)
	    return(NULL);

	/* Update number of strings reciefved (hence number of
	 * extensions).
	 */
	if(strc != NULL)
	    *strc = lstrc;

	return(strv);
}


/*
 *      Is menu allocated on the core structure.
 */
int SARIsMenuAllocated(sar_core_struct *core_ptr, int n)
{
        if(core_ptr == NULL)
            return(0);
        else if((n < 0) || (n >= core_ptr->total_menus))
            return(0);
        else if(core_ptr->menu[n] == NULL)
            return(0);
        else
            return(1);
}

/*
 *      Matches a menu on the core structure that matches the given
 *      name (case insensitive). Can return -1 for no match.
 */
int SARMatchMenuByName(
        sar_core_struct *core_ptr,
        const char *name
)
{
        int i;
        sar_menu_struct *menu_ptr;
        
        
        if((core_ptr == NULL) ||
           (name == NULL)
        )
            return(-1);
            
        for(i = 0; i < core_ptr->total_menus; i++)
        {
            menu_ptr = core_ptr->menu[i];
            if(menu_ptr == NULL)
                continue;

            if(menu_ptr->name == NULL)
                continue;
            
            if(!strcasecmp(menu_ptr->name, name))
                return(i);
        }
 
        return(-1);
}

/*
 *	Returns a pointer to the currently selected menu on the given
 *	core structure. Can return NULL on error or if there is no menu
 *	selected.
 */
sar_menu_struct *SARGetCurrentMenuPtr(sar_core_struct *core_ptr)
{
        int n;
        
        
        if(core_ptr == NULL)
            return(NULL);
        else
            n = core_ptr->cur_menu;
         
        if((n < 0) || (n >= core_ptr->total_menus))
            return(NULL);
        else
            return(core_ptr->menu[n]);
}

/*
 *	Deletes the client data structure specified on the menu
 *	list object item pointer.
 */
void SARDeleteListItemData(sar_menu_list_item_struct *item_ptr)
{
        sar_menu_list_item_data_struct *d = SAR_MENU_LIST_ITEM_DATA(
	    (item_ptr != NULL) ? item_ptr->client_data : NULL
	);
        if(d == NULL)
            return;

	free(d->filename);
	free(d->name);
	free(d);

	item_ptr->client_data = NULL;
}

/*
 *	Returns a statically allocated string containing the time
 *	of day in %h:%m format or by whatever format specified on the
 *	core structure if the core_ptr is not NULL.
 *
 *	Time t is given in seconds from midnight.
 */
char *SARTimeOfDayString(sar_core_struct *core_ptr, float t)
{
	static char str[80];

#define HTOS(h)	((h) * 3600.0f)

	/* Sanitize time. */
	while(t >= HTOS(24.0f))
	    t -= HTOS(24.0f);
	while(t < HTOS(0.0f))
	    t += HTOS(24.0f);

	*str = '\0';

	/* No core structure given? */
	if(core_ptr == NULL)
	{
	    int	h = (int)MAX((t / 3600.0f), 0.0f),
		m = (int)MAX(((int)((int)t / 60) % 60), 0.0f);

	    sprintf(str, "%i:%2.2i", h, m);
	}
	else
	{
            int h = (int)MAX((t / 3600.0f), 0.0f),
                m = (int)MAX(((int)((int)t / 60) % 60), 0.0f);

            sprintf(str, "%i:%2.2i", h, m);
        }

#undef HTOS
	return(str);
}

/*
 *      Returns a statically allocated string verbosly stating the
 *      delta time t in accordance to the format specified by the
 *	core_ptr. If core_ptr is NULL then a generic format will be
 *	assumed.
 *
 *      This function never returns NULL.
 */
char *SARDeltaTimeString(sar_core_struct *core_ptr, time_t t)
{
#define len 256
        static char s[len];

        if(t <= 0)
            return("none");

        if(t < 60)
        {
            sprintf(s, "%ld second%s",
                t,
                (t == 1) ? "" : "s"
            );
        }
        else if(t < 3600) 
        {
            t = (time_t)rint((double)t / 60.0);
            sprintf(s, "%ld minute%s",
                t,
                (t == 1) ? "" : "s"
            );
        }
        else if(t < 86400)
        {
            t = (time_t)rint((double)t / 3600.0);
            sprintf(s, "%ld hour%s",
                t,
                (t == 1) ? "" : "s"
            );
        }

        return(s);
#undef len
}


/*
 *	Updates the global hud and message text color based on the
 *	given gamma g which must be in the domain of 0.0 to 1.0.
 */
void SARSetGlobalTextColorBrightness(
        sar_core_struct *core_ptr, float g
)
{
        sar_color_struct *c;
	sar_option_struct *opt;

	if(core_ptr == NULL)
	    return;

	opt = &core_ptr->option;

	/* Set message text color. */
        c = &opt->message_color;
        c->a = 1.0f;
        c->r = g;
        c->g = g;
        c->b = g;

        /* Set new hud color. */
        c = &opt->hud_color;
        if(g > 0.5f)
        {
            float ng = (g - 0.5f) / 0.5f;
            c->a = 1.0f;
            c->r = ng;
            c->g = 1.0f;
            c->b = ng;
        }
        else
        {
            c->a = 1.0f;
            c->r = 0.0f;
            c->g = g * 2.0f;
            c->b = 0.0f;
        }
}

/*
 *	Reports the given GL error code, does not check if the
 *	error is actually an error and prints explicitly even if
 *	global options specify not to print errors.
 */
void SARReportGLError(
        sar_core_struct *core_ptr, GLenum error_code 
)
{
	const char *error_mesg = NULL, *error_type = NULL;

	switch(error_code)
	{
	  case GL_INVALID_ENUM:
	    error_type = "GL_INVALID_ENUM";
	    error_mesg =
		"Invalid GLenum argument (possibly out of range)";
	    break;

	  case GL_INVALID_VALUE:
	    error_type = "GL_INVALID_VALUE";
	    error_mesg =
                "Numeric argument out of range";
            break;

	  case GL_INVALID_OPERATION:
	    error_type = "GL_INVALID_OPERATION";
	    error_mesg =
                "Operation illegal in current state";
            break;

	  case GL_STACK_OVERFLOW:
	    error_type = "GL_STACK_OVERFLOW";
	    error_mesg =
                "Command would cause stack overflow";
            break;

	  case GL_STACK_UNDERFLOW:
	    error_type = "GL_STACK_UNDERFLOW";
	    error_mesg =
                "Command would cause a stack underflow";
            break;

          case GL_OUT_OF_MEMORY:
            error_type = "GL_OUT_OF_MEMORY";
            error_mesg =
                "Not enough memory left to execute command";
            break;

	  default:
	    error_type = "*UNKNOWN*";
	    error_mesg = "Undefined GL error";
	    break;
	}

	fprintf(
	    stderr,
	    "GL Error code %i: %s: %s\n",
	    error_code, error_type, error_mesg
	);
}
