/*
 * Copyright (C) 2009-2016 Alistair Neil <info@dazzleships.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package lib;

import java.awt.Color;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTextArea;
import javax.swing.LookAndFeel;
import javax.swing.UIManager;
import javax.swing.event.MenuEvent;

/**
 *
 * @author Alistair Neil <info@dazzleships.net>
 */
public class GTKFixes {

    /**
     * Fixes incorrect JMenu selectionForeground highlighting
     *
     * @param jms Array of JMenus or single JMenu
     */
    public static void fixGtkMenuSelectionColor(JMenu... jms) {
        // If gtk laf not installed then do nothing
        if (!UIManager.getLookAndFeel().getID().contentEquals("GTK")) {
            return;
        }
        for (final JMenu jm : jms) {
            final Color fgfixed = new Color(((Color) UIManager.get("MenuItem.selectionForeground")).getRGB());
            final Color fgnormal = (Color) UIManager.get("Menu.foreground");
            jm.addMenuListener(new javax.swing.event.MenuListener() {

                @Override
                public void menuSelected(MenuEvent e) {
                    jm.setForeground(fgfixed);
                }

                @Override
                public void menuDeselected(MenuEvent e) {
                    jm.setForeground(fgnormal);
                }

                @Override
                public void menuCanceled(MenuEvent e) {
                    jm.setForeground(fgnormal);
                }
            });
        }
    }

    /**
     * Fixes incorrect JMenu selectionForeground highlighting
     *
     * @param jmi Array of JMenuItems or single JMenuItem
     */
    public static void fixGtkMenuItemFgColor(JMenuItem... jmi) {
        // If gtk laf not installed then do nothing
        if (!UIManager.getLookAndFeel().getID().contentEquals("GTK")) {
            return;
        }
        for (final JMenuItem jm : jmi) {
            final Color fgfixed = new Color(UIManager.getColor("MenuItem.foreground").getRGB());
            jm.setForeground(fgfixed);
        }
    }

    public static void fixGtkTextAreaColor(JTextArea jta) {
        if (OSFunction.isLinux()) {
            jta.setForeground(UIManager.getColor("JLabel.foreground"));
            jta.setBackground(UIManager.getColor("JLabel.background"));
        }
    }

    /**
     * Swing menus are looking pretty bad on Linux when the GTK LaF is used (See
     * bug #6925412). It will most likely never be fixed anytime soon so this
     * method provides a workaround for it. It uses reflection to change the GTK
     * style objects of Swing so popup menu borders have a minimum thickness of
     * 1 and menu separators have a minimum vertical thickness of 1. This code
     * was taken from K's Cluttered Loft (Klaus Reimer) at http://www.ailis.de
     * and as far as im aware is in the public domain as it has no attached
     * license if this proves incorrect then I will remove it immediately.
     */
    public static void installGtkPopupBugWorkaround() {
        // Get current look-and-feel implementation class
        LookAndFeel laf = UIManager.getLookAndFeel();
        Class<?> lafClass = laf.getClass();

        // We do reflection from here on. Failure is silently ignored. The
        // workaround is simply not installed when something goes wrong here
        try {
            // Access the GTK style factory
            Field field = lafClass.getDeclaredField("styleFactory");
            boolean accessible = field.isAccessible();
            field.setAccessible(true);
            Object styleFactory = field.get(laf);
            field.setAccessible(accessible);

            // Fix the horizontal and vertical thickness of popup menu style
            Object style = getGtkStyle(styleFactory, new JPopupMenu(),
                    "POPUP_MENU");
            fixGtkThickness(style, "yThickness");
            fixGtkThickness(style, "xThickness");

            // Fix the vertical thickness of the popup menu separator style
            style = getGtkStyle(styleFactory, new JSeparator(),
                    "POPUP_MENU_SEPARATOR");
            fixGtkThickness(style, "yThickness");
        } catch (Exception ex) {
            // Silently ignored. Workaround can't be applied.
            Logger.getGlobal().logp(Level.INFO, GTKFixes.class.getName(),
                    "installGtkPopupBugWorkaround", "Exception", ex);
        }
    }

    /**
     * Called internally by installGtkPopupBugWorkaround to fix the thickness of
     * a GTK style field by setting it to a minimum value of 1. This code was
     * taken from K's Cluttered Loft (Klaus Reimer) at http://www.ailis.de and
     * as far as im aware is in the public domain as it has no attached license
     * if this proves incorrect then I will remove it immediately.
     *
     * @param style The GTK style object.
     * @param fieldName The field name.
     * @throws Exception When reflection fails.
     */
    public static void fixGtkThickness(Object style, String fieldName)
            throws Exception {
        Field field = style.getClass().getDeclaredField(fieldName);
        boolean accessible = field.isAccessible();
        field.setAccessible(true);
        field.setInt(style, Math.max(1, field.getInt(style)));
        field.setAccessible(accessible);
    }

    /**
     * Called internally by installGtkPopupBugWorkaround. Returns a specific GTK
     * style object.
     *
     * @param styleFactory The GTK style factory.
     * @param component The target component of the style.
     * @param regionName The name of the target region of the style.
     * @return The GTK style.
     * @throws Exception When reflection fails. This code was taken from K's
     * Cluttered Loft (Klaus Reimer) at http://www.ailis.de and as far as im
     * aware is in the public domain as it has no attached license if this
     * proves incorrect then I will remove it immediately.
     */
    public static Object getGtkStyle(Object styleFactory,
            JComponent component, String regionName) throws Exception {
        // Create the region object
        Class<?> regionClass = Class.forName("javax.swing.plaf.synth.Region");
        Field field = regionClass.getField(regionName);
        Object region = field.get(regionClass);

        // Get and return the style
        Class<?> styleFactoryClass = styleFactory.getClass();
        Method method = styleFactoryClass.getMethod("getStyle",
                new Class<?>[]{JComponent.class, regionClass});
        boolean accessible = method.isAccessible();
        method.setAccessible(true);
        Object style = method.invoke(styleFactory, component, region);
        method.setAccessible(accessible);
        return style;
    }

}
