import gtk
from math import cos, sin
import math
import cairo

TWO_PI = math.pi * 2


X = 0
Y = 1
_getx = lambda x, radius, angle: x + cos(angle) * radius
_gety = lambda y, radius, angle: y + sin(angle) * radius
_getxy = lambda center, radius, angle: (_getx(center[X], radius, angle), _gety(center[Y], radius, angle))

class Slice(object):
    def __init__(self, center, radius, offset, apperture):
        # the center of the slice
        self.center = center
        # the length of the radius
        self.radius = radius
        # the angular offset
        self.offset = offset
        # relative angle of the slice
        self.apperture = apperture
        # apperture counting from 'o' to 'offset'
        self.angle1 = offset
        # final apperture
        self.angle2 = offset + apperture
        # big point
        self.corner1 = _getxy(center, radius, self.angle1)
        # small point
        self.corner2 = _getxy(center, radius, self.angle2)

    def split(self, parts):
        apperture = self.apperture/parts
        new_Slice = lambda x: Slice(self.center, self.radius, self.offset + x, apperture)
        val = 0
        for x in range(parts):
            yield new_Slice(val)
            val += apperture

    def split2(self, parts):
        apperture = self.apperture/parts
        new_Slice = lambda x: Slice(self.center, self.radius, self.offset, apperture + x)
        val = 0
        for x in range(parts):
            yield new_Slice(val)
            val += apperture
    

def color_to_rgba(color, alpha=1.0):
    """
    Converts a GdkColor to a tuple in the rgba format.
    You can pass an optional argument for the alpha,
    since it defaults to 1.0.
    """
    return (color.red/65535.0,
            color.green/65535.0,
            color.blue/65535.0,
            alpha)

def create_fade_pattern(p1, p2, r, g, b, a=1.0):
    """Creates a pattern that is fading."""
    patt = cairo.LinearGradient(p1[0], p1[1], p2[0], p2[1])
    patt.add_color_stop_rgba(0.0, r, g, b, 0)
    patt.add_color_stop_rgba(0.7, r, g, b, .8)
    patt.add_color_stop_rgba(0.0, r, g, b, 0)
    
    return patt

def sketch_arc(ctx, s):
    ctx.move_to(*s.center)
    ctx.line_to(*s.corner1)
    ctx.arc(s.center[X], s.center[Y], s.radius, s.angle1, s.angle2)
    ctx.line_to(s.center[X], s.center[Y])

def sketch_radius(ctx, center, corner):
    ctx.move_to(*center)
    ctx.line_to(*corner)

def _draw_interior(ctx, s, color):
    # Draw interior
    ctx.set_source_color(color)
    sketch_arc(ctx, s)
    ctx.close_path()
    ctx.fill()


def _draw_interior_fade(ctx, s, color):
    apperture_1 = s.apperture * 0.90
    s1 = Slice(s.center, s.radius, s.offset, apperture_1)
    s2 = Slice(s.center, s.radius, s.offset + apperture_1, s.apperture - apperture_1)
    
    # Draw interior
    ctx.set_source_color(color)
    sketch_arc(ctx, s1)
    ctx.close_path()
    ctx.fill()

    # Draw fade
    rgba = list(color_to_rgba(color))
    rgba[-1] = 0.30
    for small_s in s2.split2(8):
        #ctx.set_source_color(color)
        ctx.set_source_rgba(*rgba)
        sketch_arc(ctx, small_s)
        ctx.close_path()
        ctx.fill()

def draw_slice(ctx, center, radius, offset, percentage, color, border_color, fade=False):
    # Obtain the apperture of the slice
    offset *= math.pi * 2
    apperture = percentage * math.pi * 2
    s = Slice(center, radius, offset, apperture)
    if fade:
        _draw_interior_fade(ctx, s, color)
    else:
        _draw_interior(ctx, s, color)

    # Draw border
    ctx.set_source_color(border_color)
    sketch_radius(ctx, s.center, s.corner1)
    ctx.stroke()

class CairoWidget(gtk.DrawingArea):
    """Base class for using cairo graphics"""
    def __init__(self):
        super(CairoWidget, self).__init__()
        self.connect("expose-event",
                     lambda widget, evt: \
                     self.draw(self.window.cairo_create()))
        
    def get_border_color(self):
        return self.style.dark[gtk.STATE_NORMAL]
    
    def get_normal_background_color(self):
        return self.style.base[gtk.STATE_NORMAL]

    def get_selected_background_color(self):
        return self.style.base[gtk.STATE_SELECTED]

    def get_empty_background_color(self):
        return self.style.mid[gtk.STATE_NORMAL]

    def get_background_color(self):
        return self.style.bg[gtk.STATE_NORMAL]

    def draw(self, ctx):
        raise NotImplementedError("You must implement 'draw(ctx)' method")

class CairoPieChart(CairoWidget):
    """Draws a group of slices that represent musics"""
    def __init__(self, values, getter):
        super(CairoPieChart, self).__init__()
        self.values = values
        self.selected = []
        self.getter = getter
        self.values = values
    

    def draw_slice(self, ctx, center, radius, offset, percentage, color):
        draw_slice(ctx, center, radius, offset, percentage, color, self.get_border_color())

    def draw_background(self, center, radius, ctx):
        ctx.set_source_color(self.get_background_color())
        ctx.arc(center[X], center[Y], radius, 0, TWO_PI)
        ctx.fill()

    def draw_border(self, center, radius, ctx):
        ctx.set_source_color(self.get_border_color())
        ctx.arc(center[X], center[Y], radius, 0, TWO_PI)
        ctx.stroke()

    def get_color_for(self, index):
        selected = index in self.selected

        if selected:
            color = self.get_selected_background_color()
        else:
            color = self.get_normal_background_color()
        return color
    
    def draw(self, ctx):
        rect = self.allocation
        
        radius = min(rect.width / 2.0, rect.height / 2.0) - 5
        if radius <= 0:
            return
        
        x = rect.width / 2.0
        y = rect.height / 2.0
        center = (x, y)
        ctx.set_line_width(0.75)

        # the small radius is 20% of the radius
        small_radius = radius * 0.20
        

        # Calculate total from all the values on the source
        total = 0
        for val in self.values:
            total += self.getter(val)
        if total < self.total:
            total = self.total
        total = float(total)

        offset = 0.0
        
        # cycle for each slice
        for index, val in enumerate(self.values):
            # get the percentage each slice occupies
            apperture = self.getter(val) / total
            
            color = self.get_color_for(index)
            self.draw_slice(ctx, center, radius, offset, apperture, color)
            offset += apperture
        
        if offset < 1:
            color = self.get_empty_background_color()
            self.draw_slice(ctx, center, radius, offset, 1.0 - offset, color)

        self.draw_background(center, small_radius, ctx)
        self.draw_border(center, small_radius, ctx)
        self.draw_border(center, radius, ctx)


def bling(chart):
    
    chart.values[0] += 0.1
    chart.values[1] -= 0.1
    if chart.values[1] < 0 or chart.values[0] > 22:
        chart.values[0] = 22
        chart.values[1] = 0
    chart.queue_draw()
    
    return chart.values[1] > 0


def main():
    import gobject
    window = gtk.Window()
    chart = CairoPieChart([10,12], lambda x: x)
    chart.total = 30
    chart.selected = [0]
    chart.show()
    chart.set_size_request(256, 256)

    btn = gtk.Button("Hi")
    btn.show()
    vbox = gtk.VBox()
    vbox.show()
    vbox.add(btn)
    vbox.add(chart)
    window.add(vbox)
    gobject.timeout_add(100, bling, chart)
    
    #window.add(chart)
    window.connect("destroy", gtk.main_quit)
    window.show()
    gtk.main()

if __name__ == '__main__':
    main()
