/****************************************************************************
 * Driver for Solarflare network controllers
 *           (including support for SFE4001 10GBT NIC)
 *
 * Copyright 2005-2006: Fen Systems Ltd.
 * Copyright 2005-2008: Solarflare Communications Inc,
 *                      9501 Jeronimo Road, Suite 250,
 *                      Irvine, CA 92618, USA
 *
 * Initially developed by Michael Brown <mbrown@fensystems.co.uk>
 * Maintained by Solarflare Communications <linux-net-drivers@solarflare.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 ****************************************************************************
 */

#include <linux/socket.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <net/ip.h>
#include <net/checksum.h>
#include "net_driver.h"
#include "rx.h"
#include "efx.h"
#include "falcon.h"
#include "selftest.h"
#include "workarounds.h"


/* Number of RX descriptors pushed at once. */
#define EFX_RX_BATCH  8

/* Size of buffer allocated for skb header area. */
#define EFX_SKB_HEADERS  64u

/*
 * rx_alloc_method - RX buffer allocation method
 *
 * This driver supports two methods for allocating and using RX buffers:
 * each RX buffer may be backed by an skb or by an order-n page.
 *
 * When LRO is in use then the second method has a lower overhead,
 * since we don't have to allocate then free skbs on reassembled frames.
 *
 * Values:
 *   - RX_ALLOC_METHOD_AUTO = 0
 *   - RX_ALLOC_METHOD_SKB  = 1
 *   - RX_ALLOC_METHOD_PAGE = 2
 *
 * The heuristic for %RX_ALLOC_METHOD_AUTO is a simple hysteresis count
 * controlled by the parameters below.
 *
 *   - Since pushing and popping descriptors are separated by the rx_queue
 *     size, so the watermarks should be ~rxd_size.
 *   - The performance win by using page-based allocation for LRO is less
 *     than the performance hit of using page-based allocation of non-LRO,
 *     so the watermarks should reflect this.
 *
 * Per channel we maintain a single variable, updated by each channel:
 *
 *   rx_alloc_level += (lro_performed ? RX_ALLOC_FACTOR_LRO :
 *                      RX_ALLOC_FACTOR_SKB)
 * Per NAPI poll interval, we constrain rx_alloc_level to 0..MAX (which
 * limits the hysteresis), and update the allocation strategy:
 *
 *   rx_alloc_method = (rx_alloc_level > RX_ALLOC_LEVEL_LRO ?
 *                      RX_ALLOC_METHOD_PAGE : RX_ALLOC_METHOD_SKB)
 */
static int rx_alloc_method = RX_ALLOC_METHOD_PAGE;

#define RX_ALLOC_LEVEL_LRO 0x2000
#define RX_ALLOC_LEVEL_MAX 0x3000
#define RX_ALLOC_FACTOR_LRO 1
#define RX_ALLOC_FACTOR_SKB (-2)

/* This is the percentage fill level below which new RX descriptors
 * will be added to the RX descriptor ring.
 */
static unsigned int rx_refill_threshold = 90;

/* This is the percentage fill level to which an RX queue will be refilled
 * when the "RX refill threshold" is reached.
 */
static unsigned int rx_refill_limit = 95;

/*
 * RX maximum head room required.
 *
 * This must be at least 1 to prevent overflow and at least 2 to allow
 * pipelined receives.
 */
#define EFX_RXD_HEAD_ROOM 2

/* Macros for zero-order pages (potentially) containing multiple RX buffers */
#define RX_DATA_OFFSET(_data)				\
	(((unsigned long) (_data)) & (PAGE_SIZE-1))
#define RX_BUF_OFFSET(_rx_buf)				\
	RX_DATA_OFFSET((_rx_buf)->data)

#define RX_PAGE_SIZE(_efx)				\
	(PAGE_SIZE * (1u << (_efx)->rx_buffer_order))


#if !defined(EFX_NOT_UPSTREAM) || defined(EFX_USE_GENERIC_LRO)

/**************************************************************************
 *
 * Linux generic LRO handling
 *
 **************************************************************************
 */

static int efx_lro_get_skb_hdr(struct sk_buff *skb, void **ip_hdr,
			       void **tcpudp_hdr, u64 *hdr_flags, void *priv)
{
	struct efx_channel *channel = (struct efx_channel *)priv;
	struct iphdr *iph;
	struct tcphdr *th;

	iph = (struct iphdr *)skb->data;
	if (skb->protocol != htons(ETH_P_IP) || iph->protocol != IPPROTO_TCP)
		goto fail;

	th = (struct tcphdr *)(skb->data + iph->ihl * 4);

	*tcpudp_hdr = th;
	*ip_hdr = iph;
	*hdr_flags = LRO_IPV4 | LRO_TCP;

	channel->rx_alloc_level += RX_ALLOC_FACTOR_LRO;
	return 0;
fail:
	channel->rx_alloc_level += RX_ALLOC_FACTOR_SKB;
	return -1;
}

static int efx_get_frag_hdr(struct skb_frag_struct *frag, void **mac_hdr,
			    void **ip_hdr, void **tcpudp_hdr, u64 *hdr_flags,
			    void *priv)
{
	struct efx_channel *channel = (struct efx_channel *)priv;
	struct ethhdr *eh;
	struct iphdr *iph;

	/* We support EtherII and VLAN encapsulated IPv4 */
	eh = (struct ethhdr *)(page_address(frag->page) + frag->page_offset);
	*mac_hdr = eh;

	if (eh->h_proto == htons(ETH_P_IP)) {
		iph = (struct iphdr *)(eh + 1);
	} else {
		struct vlan_ethhdr *veh = (struct vlan_ethhdr *)eh;
		if (veh->h_vlan_encapsulated_proto != htons(ETH_P_IP))
			goto fail;

		iph = (struct iphdr *)(veh + 1);
	}
	*ip_hdr = iph;

	/* We can only do LRO over TCP */
	if (iph->protocol != IPPROTO_TCP)
		goto fail;

	*hdr_flags = LRO_IPV4 | LRO_TCP;
	*tcpudp_hdr = (struct tcphdr *)((u8 *) iph + iph->ihl * 4);

	channel->rx_alloc_level += RX_ALLOC_FACTOR_LRO;
	return 0;
 fail:
	channel->rx_alloc_level += RX_ALLOC_FACTOR_SKB;
	return -1;
}

int efx_lro_init(struct net_lro_mgr *lro_mgr, struct efx_nic *efx)
{
	size_t s = sizeof(struct net_lro_desc) * EFX_MAX_LRO_DESCRIPTORS;
	struct net_lro_desc *lro_arr;

	/* Allocate the LRO descriptors structure */
	lro_arr = kzalloc(s, GFP_KERNEL);
	if (lro_arr == NULL)
		return -ENOMEM;

	lro_mgr->lro_arr = lro_arr;
	lro_mgr->max_desc = EFX_MAX_LRO_DESCRIPTORS;
	lro_mgr->max_aggr = EFX_MAX_LRO_AGGR;
	lro_mgr->frag_align_pad = EFX_PAGE_SKB_ALIGN;

	lro_mgr->get_skb_header = efx_lro_get_skb_hdr;
	lro_mgr->get_frag_header = efx_get_frag_hdr;
	lro_mgr->dev = efx->net_dev;

	lro_mgr->features = LRO_F_NAPI;

	/* We can pass packets up with the checksum intact */
	lro_mgr->ip_summed = CHECKSUM_UNNECESSARY;

	lro_mgr->ip_summed_aggr = CHECKSUM_UNNECESSARY;

	return 0;
}

void efx_lro_fini(struct net_lro_mgr *lro_mgr)
{
	kfree(lro_mgr->lro_arr);
	lro_mgr->lro_arr = NULL;
}

#endif /* EFX_USE_GENERIC_LRO */

/**
 * efx_init_rx_buffer_skb - create new RX buffer using skb-based allocation
 *
 * @rx_queue:		Efx RX queue
 * @rx_buf:		RX buffer structure to populate
 *
 * This allocates memory for a new receive buffer, maps it for DMA,
 * and populates a struct efx_rx_buffer with the relevant
 * information.  Return a negative error code or 0 on success.
 */
static inline int efx_init_rx_buffer_skb(struct efx_rx_queue *rx_queue,
					 struct efx_rx_buffer *rx_buf)
{
	struct efx_nic *efx = rx_queue->efx;
	struct net_device *net_dev = efx->net_dev;
	int skb_len = efx->rx_buffer_len;

	rx_buf->skb = netdev_alloc_skb(net_dev, skb_len);
	if (unlikely(!rx_buf->skb))
		return -ENOMEM;

	/* Adjust the SKB for padding and checksum */
	skb_reserve(rx_buf->skb, NET_IP_ALIGN);
	rx_buf->len = skb_len - NET_IP_ALIGN;
	rx_buf->data = (char *)rx_buf->skb->data;

	rx_buf->skb->ip_summed = CHECKSUM_UNNECESSARY;

	/* Map for DMA */
	rx_buf->dma_addr = pci_map_single(efx->pci_dev,
					  rx_buf->data, rx_buf->len,
					  PCI_DMA_FROMDEVICE);

	if (unlikely(pci_dma_mapping_error(rx_buf->dma_addr))) {
		/* Free the SKB */
		dev_kfree_skb_any(rx_buf->skb);
		rx_buf->skb = NULL;
		return -EIO;
	}

	return 0;
}

/**
 * efx_init_rx_buffer_page - create new RX buffer using page-based allocation
 *
 * @rx_queue:		Efx RX queue
 * @rx_buf:		RX buffer structure to populate
 *
 * This allocates memory for a new receive buffer, maps it for DMA,
 * and populates a struct efx_rx_buffer with the relevant
 * information.  Return a negative error code or 0 on success.
 */
static inline int efx_init_rx_buffer_page(struct efx_rx_queue *rx_queue,
					  struct efx_rx_buffer *rx_buf)
{
	struct efx_nic *efx = rx_queue->efx;
	int bytes, space, offset;

	bytes = efx->rx_buffer_len - EFX_PAGE_IP_ALIGN;

	/* If there is space left in the previously allocated page,
	 * then use it. Otherwise allocate a new one */
	rx_buf->page = rx_queue->buf_page;
	if (rx_buf->page == NULL) {
		dma_addr_t dma_addr;

		rx_buf->page = alloc_pages(__GFP_COLD | __GFP_COMP | GFP_ATOMIC,
					   efx->rx_buffer_order);
		if (unlikely(rx_buf->page == NULL))
			return -ENOMEM;

		/* Map the entire page for DMA */
		dma_addr = pci_map_page(efx->pci_dev, rx_buf->page,
					0, RX_PAGE_SIZE(efx),
					PCI_DMA_FROMDEVICE);

		if (unlikely(pci_dma_mapping_error(dma_addr))) {
			__free_pages(rx_buf->page, efx->rx_buffer_order);
			rx_buf->page = NULL;
			return -EIO;
		}

		rx_queue->buf_page = rx_buf->page;
		rx_queue->buf_dma_addr = dma_addr;
		rx_queue->buf_data = ((char *) page_address(rx_buf->page) +
				      EFX_PAGE_IP_ALIGN);
	}

	offset = RX_DATA_OFFSET(rx_queue->buf_data);
	rx_buf->len = bytes;
	rx_buf->dma_addr = rx_queue->buf_dma_addr + offset;
	rx_buf->data = rx_queue->buf_data;

	/* Try to pack multiple buffers per page */
	if (efx->rx_buffer_order == 0) {
		/* The next buffer starts on the next 512 byte boundary */
		rx_queue->buf_data += ((bytes + 0x1ff) & ~0x1ff);
		offset += ((bytes + 0x1ff) & ~0x1ff);

		space = RX_PAGE_SIZE(efx) - offset;
		if (space >= bytes) {
			/* Refs dropped on kernel releasing each skb */
			get_page(rx_queue->buf_page);
			goto out;
		}
	}

	/* This is the final RX buffer for this page, so mark it for
	 * unmapping */
	rx_queue->buf_page = NULL;
	rx_buf->unmap_addr = rx_queue->buf_dma_addr;

 out:
	return 0;
}

/* This allocates memory for a new receive buffer, maps it for DMA,
 * and populates a struct efx_rx_buffer with the relevant
 * information.
 */
static inline int efx_init_rx_buffer(struct efx_rx_queue *rx_queue,
				     struct efx_rx_buffer *new_rx_buf)
{
	int rc = 0;

	if (rx_queue->channel->rx_alloc_push_pages) {
		new_rx_buf->skb = NULL;
		rc = efx_init_rx_buffer_page(rx_queue, new_rx_buf);
		rx_queue->alloc_page_count++;
	} else {
		new_rx_buf->page = NULL;
		rc = efx_init_rx_buffer_skb(rx_queue, new_rx_buf);
		rx_queue->alloc_skb_count++;
	}

	if (unlikely(rc < 0))
		EFX_LOG_RL(rx_queue->efx, "%s RXQ[%d] =%d\n", __func__,
			   rx_queue->queue, rc);
	return rc;
}

static inline void efx_unmap_rx_buffer(struct efx_nic *efx,
				       struct efx_rx_buffer *rx_buf)
{
	if (rx_buf->page) {
		EFX_BUG_ON_PARANOID(rx_buf->skb);
		if (rx_buf->unmap_addr) {
			pci_unmap_page(efx->pci_dev, rx_buf->unmap_addr,
				       RX_PAGE_SIZE(efx), PCI_DMA_FROMDEVICE);
			rx_buf->unmap_addr = 0;
		}
	} else if (likely(rx_buf->skb)) {
		pci_unmap_single(efx->pci_dev, rx_buf->dma_addr,
				 rx_buf->len, PCI_DMA_FROMDEVICE);
	}
}

static inline void efx_free_rx_buffer(struct efx_nic *efx,
				      struct efx_rx_buffer *rx_buf)
{
	if (rx_buf->page) {
		__free_pages(rx_buf->page, efx->rx_buffer_order);
		rx_buf->page = NULL;
	} else if (likely(rx_buf->skb)) {
		dev_kfree_skb_any(rx_buf->skb);
		rx_buf->skb = NULL;
	}
}

static inline void efx_fini_rx_buffer(struct efx_rx_queue *rx_queue,
				      struct efx_rx_buffer *rx_buf)
{
	/* Unmap for DMA */
	efx_unmap_rx_buffer(rx_queue->efx, rx_buf);

	/* Free the skb/page */
	efx_free_rx_buffer(rx_queue->efx, rx_buf);
}

/**
 * efx_fast_push_rx_descriptors - push new RX descriptors quickly
 * @rx_queue:		RX descriptor queue
 * @retry:              Recheck the fill level
 * This will aim to fill the RX descriptor queue up to
 * @rx_queue->@fast_fill_limit. If there is insufficient atomic
 * memory to do so, the caller should retry.
 */
static int __efx_fast_push_rx_descriptors(struct efx_rx_queue *rx_queue,
					  int retry)
{
	struct efx_rx_buffer *rx_buf;
	unsigned fill_level, index;
	int i, space, rc = 0;

	/* Calculate current fill level.  Do this outside the lock,
	 * because most of the time we'll end up not wanting to do the
	 * fill anyway.
	 */
	fill_level = (rx_queue->added_count - rx_queue->removed_count);
	EFX_BUG_ON_PARANOID(fill_level >
			    rx_queue->efx->type->rxd_ring_mask + 1);

	/* Don't fill if we don't need to */
	if (fill_level >= rx_queue->fast_fill_trigger)
		return 0;

	/* Record minimum fill level */
	if (unlikely(fill_level < rx_queue->min_fill))
		if (fill_level)
			rx_queue->min_fill = fill_level;

	/* Acquire RX add lock.  If this lock is contended, then a fast
	 * fill must already be in progress (e.g. in the refill
	 * tasklet), so we don't need to do anything
	 */
	if (!spin_trylock_bh(&rx_queue->add_lock))
		return -1;

 retry:
	/* Recalculate current fill level now that we have the lock */
	fill_level = (rx_queue->added_count - rx_queue->removed_count);
	EFX_BUG_ON_PARANOID(fill_level >
			    rx_queue->efx->type->rxd_ring_mask + 1);
	space = rx_queue->fast_fill_limit - fill_level;
	if (space < EFX_RX_BATCH)
		goto out_unlock;

	EFX_TRACE(rx_queue->efx, "RX queue %d fast-filling descriptor ring from"
		  " level %d to level %d using %s allocation\n",
		  rx_queue->queue, fill_level, rx_queue->fast_fill_limit,
		  rx_queue->channel->rx_alloc_push_pages ? "page" : "skb");

	do {
		for (i = 0; i < EFX_RX_BATCH; ++i) {
			index = (rx_queue->added_count &
				 rx_queue->efx->type->rxd_ring_mask);
			rx_buf = efx_rx_buffer(rx_queue, index);
			rc = efx_init_rx_buffer(rx_queue, rx_buf);
			if (unlikely(rc))
				goto out;
			++rx_queue->added_count;
		}
	} while ((space -= EFX_RX_BATCH) >= EFX_RX_BATCH);

	EFX_TRACE(rx_queue->efx, "RX queue %d fast-filled descriptor ring "
		  "to level %d\n", rx_queue->queue,
		  rx_queue->added_count - rx_queue->removed_count);

 out:
	/* Send write pointer to card. */
	falcon_notify_rx_desc(rx_queue);

	/* If the fast fill is running inside from the refill tasklet, then
	 * for SMP systems it may be running on a different CPU to
	 * RX event processing, which means that the fill level may now be
	 * out of date. */
	if (unlikely(retry && (rc == 0)))
		goto retry;

 out_unlock:
	/* Release RX add lock */
	spin_unlock_bh(&rx_queue->add_lock);

	return rc;
}

/**
 * efx_fast_push_rx_descriptors - push new RX descriptors quickly
 * @rx_queue:		RX descriptor queue
 *
 * This will aim to fill the RX descriptor queue up to
 * @rx_queue->@fast_fill_limit.  If there is insufficient memory to do so,
 * it will schedule a work item to immediately continue the fast fill
 */
void efx_fast_push_rx_descriptors(struct efx_rx_queue *rx_queue)
{
	int rc;

	rc = __efx_fast_push_rx_descriptors(rx_queue, 0);
	if (unlikely(rc)) {
		/* Schedule the work item to run immediately. The hope is
		 * that work is immediately pending to free some memory
		 * (e.g. an RX event or TX completion)
		 */
		efx_schedule_slow_fill(rx_queue, 0);
	}
}

void efx_rx_work(struct work_struct *data)
{
	struct efx_rx_queue *rx_queue;
	int rc;

#if !defined(EFX_USE_KCOMPAT) || !defined(EFX_NEED_WORK_API_WRAPPERS)
	rx_queue = container_of(data, struct efx_rx_queue, work.work);
#else
	rx_queue = container_of(data, struct efx_rx_queue, work);
#endif

	if (unlikely(!rx_queue->channel->enabled))
		return;

	EFX_TRACE(rx_queue->efx, "RX queue %d worker thread executing on CPU "
		  "%d\n", rx_queue->queue, raw_smp_processor_id());

	++rx_queue->slow_fill_count;
	/* Push new RX descriptors, allowing at least 1 jiffy for
	 * the kernel to free some more memory. */
	rc = __efx_fast_push_rx_descriptors(rx_queue, 1);
	if (rc)
		efx_schedule_slow_fill(rx_queue, 1);
}

static inline void efx_rx_packet__check_len(struct efx_rx_queue *rx_queue,
					    struct efx_rx_buffer *rx_buf,
					    int len, int *discard,
					    int *leak_packet)
{
	struct efx_nic *efx = rx_queue->efx;
	unsigned max_len = rx_buf->len - efx->type->rx_buffer_padding;

	if (likely(len <= max_len))
		return;

	/* The packet must be discarded, but this is only a fatal error
	 * if the caller indicated it was
	 */
	*discard = 1;

	if ((len > rx_buf->len) && EFX_WORKAROUND_8071(efx)) {
		EFX_ERR_RL(efx, " RX queue %d seriously overlength "
			   "RX event (0x%x > 0x%x+0x%x). Leaking\n",
			   rx_queue->queue, len, max_len,
			   efx->type->rx_buffer_padding);
		/* If this buffer was skb-allocated, then the meta
		 * data at the end of the skb will be trashed. So
		 * we have no choice but to leak the fragment.
		 */
		*leak_packet = (rx_buf->skb != NULL);
		efx_schedule_reset(efx, RESET_TYPE_RX_RECOVERY);
	} else {
		EFX_ERR_RL(efx, " RX queue %d overlength RX event "
			   "(0x%x > 0x%x)\n", rx_queue->queue, len, max_len);
	}

	rx_queue->channel->n_rx_overlength++;
}

#if !defined(EFX_NOT_UPSTREAM) || defined(EFX_USE_GENERIC_LRO)

/* Pass a received packet up through the generic LRO stack
 *
 * Handles driverlink veto, and passes the fragment up via
 * the appropriate LRO method
 */
static inline void efx_rx_packet_lro(struct efx_channel *channel,
				     struct efx_rx_buffer *rx_buf)
{
	struct efx_nic *efx = channel->efx;
	struct net_lro_mgr *lro_mgr = &channel->lro_mgr;
	void *priv = channel;
	int veto;

	/* It would be faster if we had access to packets at the
	 * other side of generic LRO. Unfortunately, there isn't
	 * an obvious interface to this, so veto packets before LRO */
	veto = EFX_DL_CALLBACK(efx, rx_packet, rx_buf->data, rx_buf->len);
	if (unlikely(veto)) {
		EFX_LOG(efx, "LRO RX vetoed by driverlink %s driver\n",
			efx->dl_cb_dev.rx_packet->driver->name);
		/* Free the buffer now */
		efx_free_rx_buffer(efx, rx_buf);
		return;
	}

	/* Pass the skb/page into the LRO engine */
	if (rx_buf->page) {
		struct skb_frag_struct frags;

		frags.page = rx_buf->page;
		frags.page_offset = RX_BUF_OFFSET(rx_buf);
		frags.size = rx_buf->len;

		lro_receive_frags(lro_mgr, &frags, rx_buf->len,
				  rx_buf->len, priv, 0);

		EFX_BUG_ON_PARANOID(rx_buf->skb);
		rx_buf->page = NULL;
	} else {
		EFX_BUG_ON_PARANOID(!rx_buf->skb);

		lro_receive_skb(lro_mgr, rx_buf->skb, priv);
		rx_buf->skb = NULL;
	}
}
#endif

/* Allocate and construct an SKB around a struct page.*/
static inline struct sk_buff *efx_rx_mk_skb(struct efx_rx_buffer *rx_buf,
					    struct efx_nic *efx,
					    int hdr_len)
{
	struct sk_buff *skb;

	/* Allocate an SKB to store the headers */
	skb = netdev_alloc_skb(efx->net_dev, hdr_len + EFX_PAGE_SKB_ALIGN);
	if (unlikely(skb == NULL)) {
		EFX_ERR_RL(efx, "RX out of memory for skb\n");
		return NULL;
	}

	EFX_BUG_ON_PARANOID(skb_shinfo(skb)->nr_frags);
	EFX_BUG_ON_PARANOID(rx_buf->len < hdr_len);

	skb->ip_summed = CHECKSUM_UNNECESSARY;
	skb_reserve(skb, EFX_PAGE_SKB_ALIGN);

	skb->len = rx_buf->len;
	skb->truesize = rx_buf->len + sizeof(struct sk_buff);
	memcpy(skb->data, rx_buf->data, hdr_len);
	skb->tail += hdr_len;

	/* Append the remaining page onto the frag list */
	if (unlikely(rx_buf->len > hdr_len)) {
		struct skb_frag_struct *frag = skb_shinfo(skb)->frags;
		frag->page = rx_buf->page;
		frag->page_offset = RX_BUF_OFFSET(rx_buf) + hdr_len;
		frag->size = skb->len - hdr_len;
		skb_shinfo(skb)->nr_frags = 1;
		skb->data_len = frag->size;
	} else {
		__free_pages(rx_buf->page, efx->rx_buffer_order);
		skb->data_len = 0;
	}

	/* Ownership has transferred from the rx_buf to skb */
	rx_buf->page = NULL;

	/* Move past the ethernet header */
	skb->protocol = eth_type_trans(skb, efx->net_dev);

	return skb;
}

#if defined(EFX_USE_KCOMPAT) && defined(EFX_USE_FASTCALL)
void fastcall efx_rx_packet(struct efx_rx_queue *rx_queue,
			    unsigned int index, unsigned int len,
			    int checksummed, int discard)
#else
void efx_rx_packet(struct efx_rx_queue *rx_queue, unsigned int index,
		   unsigned int len, int checksummed, int discard)
#endif
{
	struct efx_nic *efx = rx_queue->efx;
	struct efx_rx_buffer *rx_buf;
	int leak_packet = 0;

	rx_buf = efx_rx_buffer(rx_queue, index);
	EFX_BUG_ON_PARANOID(!rx_buf->data);
	EFX_BUG_ON_PARANOID(rx_buf->skb && rx_buf->page);
	EFX_BUG_ON_PARANOID(!(rx_buf->skb || rx_buf->page));

	/* This allows the refill path to post another buffer.
	 * EFX_RXD_HEAD_ROOM ensures that the slot we are using
	 * isn't overwritten yet.
	 */
	rx_queue->removed_count++;

	/* Validate the length encoded in the event vs the descriptor pushed */
	efx_rx_packet__check_len(rx_queue, rx_buf, len,
				 &discard, &leak_packet);

	EFX_TRACE(efx, "RX queue %d received id %x at %llx+%x %s%s\n",
		  rx_queue->queue, index,
		  (unsigned long long)rx_buf->dma_addr, len,
		  (checksummed ? " [SUMMED]" : ""),
		  (discard ? " [DISCARD]" : ""));

	/* Discard packet, if instructed to do so */
	if (unlikely(discard)) {
#if defined(EFX_NOT_UPSTREAM) && defined(SFC_PCAP_SUPPORT)
		efx_capture_packet(efx, buf_addr, len);
#endif
		if (unlikely(leak_packet))
			rx_queue->channel->n_skbuff_leaks++;
		else
			/* We haven't called efx_unmap_rx_buffer yet,
			 * so fini the entire rx_buffer here */
			efx_fini_rx_buffer(rx_queue, rx_buf);
		return;
	}

	/* Release card resources - assumes all RX buffers consumed in-order
	 * per RX queue
	 */
	efx_unmap_rx_buffer(efx, rx_buf);

	/* Prefetch nice and early so data will (hopefully) be in cache by
	 * the time we look at it.
	 */
	prefetch(rx_buf->data);

	/* Pipeline receives so that we give time for packet headers to be
	 * prefetched into cache.
	 */
	rx_buf->len = len;
	if (rx_queue->channel->rx_pkt)
		__efx_rx_packet(rx_queue->channel,
				rx_queue->channel->rx_pkt,
				rx_queue->channel->rx_pkt_csummed);
	rx_queue->channel->rx_pkt = rx_buf;
	rx_queue->channel->rx_pkt_csummed = checksummed;
}

/* Handle a received packet.  Second half: Touches packet payload. */
void __efx_rx_packet(struct efx_channel *channel,
		     struct efx_rx_buffer *rx_buf, int checksummed)
{
	struct efx_nic *efx = channel->efx;
	enum efx_veto veto;
	struct sk_buff *skb;
#if !defined(EFX_NOT_UPSTREAM) || NETIF_F_LRO != 0
	int lro = efx->net_dev->features & NETIF_F_LRO;
#else
	int lro = efx->lro_enabled;
#endif

	/* If we're in loopback test, then pass the packet directly to the
	 * loopback layer, and free the rx_buf here
	 */
	if (unlikely(efx->loopback_selftest)) {
		efx_loopback_rx_packet(efx, rx_buf->data, rx_buf->len);
		efx_free_rx_buffer(efx, rx_buf);
		goto done;
	}

	if (rx_buf->skb) {
		/* Prefetch more information */
		prefetch(skb_shinfo(rx_buf->skb));

		/* Reserve space for the data */
		skb_put(rx_buf->skb, rx_buf->len);

		/* Move past the ethernet header. rx_buf->data still points
		 * at the ethernet header */
		rx_buf->skb->protocol = eth_type_trans(rx_buf->skb,
						       efx->net_dev);
	}
#ifdef EFX_NOT_UPSTREAM
	channel->rx_packets++;
	channel->rx_bytes += rx_buf->len;
#endif

	/* Both our generic-LRO and SFC-SSR support skb and page based
	 * allocation, but neither support switching from one to the
	 * other on the fly. If we spot that the allocation mode has
	 * changed, then flush the LRO state.
	 */
	if (unlikely(channel->rx_alloc_pop_pages != (rx_buf->page != NULL))) {
		efx_flush_lro(channel);
		channel->rx_alloc_pop_pages = (rx_buf->page != NULL);
	}
#if !defined(EFX_NOT_UPSTREAM) || defined(EFX_USE_GENERIC_LRO)
	if (likely(checksummed && lro)) {
		efx_rx_packet_lro(channel, rx_buf);
		goto done;
	}
#elif defined(EFX_USE_SFC_LRO)
	if (likely(checksummed && lro)) {
		if (efx_ssr(&channel->ssr, rx_buf)) {
			channel->rx_alloc_level += RX_ALLOC_FACTOR_LRO;
			goto done;
		}
	}
#endif

	/* Allow callback to veto the packet */
	veto = EFX_DL_CALLBACK(efx, rx_packet, rx_buf->data, rx_buf->len);
	if (unlikely(veto)) {
		EFX_LOG(efx, "RX vetoed by driverlink %s driver\n",
			efx->dl_cb_dev.rx_packet->driver->name);
		/* Free the buffer now */
		efx_free_rx_buffer(efx, rx_buf);
		goto done;
	}

	/* Form an skb if required */
	if (rx_buf->page) {
		int hdr_len = min(rx_buf->len, EFX_SKB_HEADERS);
		skb = efx_rx_mk_skb(rx_buf, efx, hdr_len);
		if (unlikely(skb == NULL)) {
			efx_free_rx_buffer(efx, rx_buf);
			goto done;
		}
	} else {
		/* We now own the SKB */
		skb = rx_buf->skb;
		rx_buf->skb = NULL;
	}

	EFX_BUG_ON_PARANOID(rx_buf->page);
	EFX_BUG_ON_PARANOID(rx_buf->skb);
	EFX_BUG_ON_PARANOID(!skb);

	/* Set the SKB flags */
	if (unlikely(!checksummed || !efx->rx_checksum_enabled))
		skb->ip_summed = CHECKSUM_NONE;

	/* Pass the packet up */
	netif_receive_skb(skb);

	/* Update allocation strategy method */
	channel->rx_alloc_level += RX_ALLOC_FACTOR_SKB;

	/* fall-thru */
done:
	/* Update statistics */
	efx->net_dev->last_rx = jiffies;
}

void efx_rx_strategy(struct efx_channel *channel)
{
	enum efx_rx_alloc_method method = rx_alloc_method;

#if defined(EFX_NOT_UPSTREAM) && !defined(EFX_USE_COMPOUND_PAGES)
	if (channel->efx->rx_buffer_order > 0) {
		channel->rx_alloc_push_pages = 0;
		return;
	}
#endif

	/* Only makes sense to use page based allocation if LRO is enabled */
#if !defined(EFX_NOT_UPSTREAM) || NETIF_F_LRO != 0
	if (!(channel->efx->net_dev->features & NETIF_F_LRO)) {
#else
	if (!channel->efx->lro_enabled) {
#endif
		method = RX_ALLOC_METHOD_SKB;
	} else if (method == RX_ALLOC_METHOD_AUTO) {
		/* Constrain the rx_alloc_level */
		if (channel->rx_alloc_level < 0)
			channel->rx_alloc_level = 0;
		else if (channel->rx_alloc_level > RX_ALLOC_LEVEL_MAX)
			channel->rx_alloc_level = RX_ALLOC_LEVEL_MAX;

		/* Decide on the allocation method */
		method = ((channel->rx_alloc_level > RX_ALLOC_LEVEL_LRO) ?
			  RX_ALLOC_METHOD_PAGE : RX_ALLOC_METHOD_SKB);
	}

	/* Push the option */
	channel->rx_alloc_push_pages = (method == RX_ALLOC_METHOD_PAGE);
}

int efx_probe_rx_queue(struct efx_rx_queue *rx_queue)
{
	struct efx_nic *efx = rx_queue->efx;
	unsigned int rxq_size;
	int rc;

	EFX_LOG(efx, "creating RX queue %d\n", rx_queue->queue);

	/* Allocate RX buffers */
	rxq_size = (efx->type->rxd_ring_mask + 1) * sizeof(*rx_queue->buffer);
	rx_queue->buffer = kzalloc(rxq_size, GFP_KERNEL);
	if (!rx_queue->buffer) {
		rc = -ENOMEM;
		goto fail1;
	}

	rc = falcon_probe_rx(rx_queue);
	if (rc)
		goto fail2;

	return 0;

 fail2:
	kfree(rx_queue->buffer);
	rx_queue->buffer = NULL;
 fail1:
	/* Mark queue as unused */
	rx_queue->used = 0;

	return rc;
}

int efx_init_rx_queue(struct efx_rx_queue *rx_queue)
{
	struct efx_nic *efx = rx_queue->efx;
	unsigned int max_fill, trigger, limit;

	EFX_LOG(rx_queue->efx, "initialising RX queue %d\n", rx_queue->queue);

	/* Initialise ptr fields */
	rx_queue->added_count = 0;
	rx_queue->notified_count = 0;
	rx_queue->removed_count = 0;
	rx_queue->min_fill = -1U;
	rx_queue->min_overfill = -1U;

	/* Initialise limit fields */
	max_fill = efx->type->rxd_ring_mask + 1 - EFX_RXD_HEAD_ROOM;
	trigger = max_fill * min(rx_refill_threshold, 100U) / 100U;
	limit = max_fill * min(rx_refill_limit, 100U) / 100U;

	rx_queue->max_fill = max_fill;
	rx_queue->fast_fill_trigger = trigger;
	rx_queue->fast_fill_limit = limit;

	/* Set up RX descriptor ring */
	return falcon_init_rx(rx_queue);
}

void efx_fini_rx_queue(struct efx_rx_queue *rx_queue)
{
	int i;
	struct efx_rx_buffer *rx_buf;

	EFX_LOG(rx_queue->efx, "shutting down RX queue %d\n", rx_queue->queue);

	/* Flush RX queue and remove descriptor ring */
	falcon_fini_rx(rx_queue);

	/* Release RX buffers NB start at index 0 not current HW ptr */
	if (rx_queue->buffer) {
		for (i = 0; i <= rx_queue->efx->type->rxd_ring_mask; i++) {
			rx_buf = efx_rx_buffer(rx_queue, i);
			efx_fini_rx_buffer(rx_queue, rx_buf);
		}
	}

	/* For a page that is part-way through splitting into RX buffers */
	if (rx_queue->buf_page != NULL) {
		pci_unmap_page(rx_queue->efx->pci_dev, rx_queue->buf_dma_addr,
			       RX_PAGE_SIZE(rx_queue->efx), PCI_DMA_FROMDEVICE);
		__free_pages(rx_queue->buf_page,
			     rx_queue->efx->rx_buffer_order);
		rx_queue->buf_page = NULL;
	}
}

void efx_remove_rx_queue(struct efx_rx_queue *rx_queue)
{
	EFX_LOG(rx_queue->efx, "destroying RX queue %d\n", rx_queue->queue);

	falcon_remove_rx(rx_queue);

	kfree(rx_queue->buffer);
	rx_queue->buffer = NULL;
	rx_queue->used = 0;
}

/* Flush LRO/SSR state for the given channel */
void efx_flush_lro(struct efx_channel *channel)
{
#if defined(EFX_NOT_UPSTREAM) && defined(EFX_USE_SFC_LRO)
	efx_ssr_end_of_burst(&channel->ssr);
#endif
#if !defined(EFX_NOT_UPSTREAM) || defined(EFX_USE_GENERIC_LRO)
	lro_flush_all(&channel->lro_mgr);
#endif
}


module_param(rx_alloc_method, int, 0644);
MODULE_PARM_DESC(rx_alloc_method, "Allocation method used for RX buffers");

module_param(rx_refill_threshold, uint, 0444);
MODULE_PARM_DESC(rx_refill_threshold,
		 "RX descriptor ring fast/slow fill threshold (%)");

#if defined(EFX_NOT_UPSTREAM) && defined(EFX_USE_SFC_LRO)


/* Size of the LRO hash table.  Must be a power of 2.  A larger table
 * means we can accelerate a larger number of streams.
 */
static unsigned lro_table_size = 128;
module_param(lro_table_size, uint, 0644);
MODULE_PARM_DESC(lro_table_size,
		 "Size of the LRO hash table.  Must be a power of 2");

/* Maximum length of a hash chain.  If chains get too long then the lookup
 * time increases and may exceed the benefit of LRO.
 */
static unsigned lro_chain_max = 20;
module_param(lro_chain_max, uint, 0644);
MODULE_PARM_DESC(lro_chain_max,
		 "Maximum length of chains in the LRO hash table");


/* Maximum time (in jiffies) that a connection can be idle before it's LRO
 * state is discarded.
 */
static unsigned lro_idle_jiffies = HZ / 10 + 1;	/* 100ms */
module_param(lro_idle_jiffies, uint, 0644);
MODULE_PARM_DESC(lro_idle_jiffies, "Time (in jiffies) after which an"
		 " idle connection's LRO state is discarded");


/* Number of packets with payload that must arrive in-order before a
 * connection is eligible for LRO.  The idea is we should avoid coalescing
 * segments when the sender is in slow-start because reducing the ACK rate
 * can damage performance.
 */
static unsigned lro_slow_start_packets = 20;
module_param(lro_slow_start_packets, uint, 0644);
MODULE_PARM_DESC(lro_slow_start_packets, "Number of packets that must "
		 "pass in-order before starting LRO.");


int efx_ssr_init(struct efx_ssr_state *st, struct efx_nic *efx)
{
	unsigned i;
	st->conns_mask = lro_table_size - 1;
	if ((st->conns_mask + 1) & st->conns_mask) {
		EFX_ERR(efx, "lro_table_size(=%u) must be a power of 2\n",
			lro_table_size);
		return -EINVAL;
	}
	st->efx = efx;
	st->conns = kmalloc((st->conns_mask + 1)
			    * sizeof(st->conns[0]), GFP_KERNEL);
	if (st->conns == NULL)
		return -ENOMEM;
	st->conns_n = kmalloc((st->conns_mask + 1)
			      * sizeof(st->conns_n[0]), GFP_KERNEL);
	if (st->conns_n == NULL) {
		kfree(st->conns);
		st->conns = NULL;
		return -ENOMEM;
	}
	for (i = 0; i <= st->conns_mask; ++i) {
		INIT_LIST_HEAD(&st->conns[i]);
		st->conns_n[i] = 0;
	}
	INIT_LIST_HEAD(&st->active_conns);
	INIT_LIST_HEAD(&st->free_conns);
	return 0;
}

/* Drop the given connection, and add it to the free list */
static inline void efx_ssr_drop(struct efx_ssr_state *st,
				struct efx_ssr_conn *c, unsigned conn_hash)
{
	EFX_BUG_ON_PARANOID(c->skb);
	EFX_BUG_ON_PARANOID(st->conns_n[conn_hash] <= 0);
	--st->conns_n[conn_hash];
	list_del(&c->link);
	list_add(&c->link, &st->free_conns);
}

void efx_ssr_fini(struct efx_ssr_state *st)
{
	struct efx_ssr_conn *c;
	unsigned i;

	/* Return cleanly if efx_ssr_init() has not been called. */
	if (st->conns == NULL)
		return;

	EFX_BUG_ON_PARANOID(!list_empty(&st->active_conns));

	for (i = 0; i <= st->conns_mask; ++i)
		while (!list_empty(&st->conns[i])) {
			c = list_entry(st->conns[i].prev,
				       struct efx_ssr_conn, link);
			efx_ssr_drop(st, c, i);
		}

	while (!list_empty(&st->free_conns)) {
		c = list_entry(st->free_conns.prev, struct efx_ssr_conn, link);
		list_del(&c->link);
		EFX_BUG_ON_PARANOID(c->skb);
		kfree(c);
	}

	kfree(st->conns_n);
	kfree(st->conns);
	st->conns = NULL;
}

/* Calc IP checksum and deliver to the OS */
static void efx_ssr_deliver(struct efx_ssr_state *st, struct efx_ssr_conn *c)
{
	struct efx_nic *efx = st->efx;
	int veto, len;

	EFX_BUG_ON_PARANOID(!c->skb);

	++st->n_bursts;

	/* Finish off packet munging and recalculate IP header checksum. */
	c->iph->tot_len = htons(c->iph->tot_len);
	c->iph->check = 0;
	c->iph->check = ip_fast_csum((u8 *) c->iph, c->iph->ihl);

	len = c->skb->len + ((char *)c->iph - (char *)c->eh);
	c->skb->truesize = len + sizeof(struct sk_buff);

	c->th->window = c->th_last->window;
	c->th->ack_seq = c->th_last->ack_seq;
	if (c->th->doff == c->th_last->doff) {
		/* Copy TCP options (take care to avoid going negative). */
		len = ((c->th->doff - 5) & 0xf) << 2u;
		memcpy(c->th + 1, c->th_last + 1, len);
	}

	/* Allow callback to veto the packet. */
	veto = EFX_DL_CALLBACK(efx, rx_packet, (char *)c->eh, len);
	if (unlikely(veto)) {
		EFX_LOG(efx, "RX vetoed by driverlink %s driver\n",
			efx->dl_cb_dev.rx_packet->driver->name);
		dev_kfree_skb_any(c->skb);
	} else {
		netif_receive_skb(c->skb);
	}

	c->skb = NULL;
	list_del_init(&c->active_link);
}

/* Stop tracking connections that have gone idle in order to keep hash
 * chains short.
 */
static void efx_ssr_purge_idle(struct efx_ssr_state *st, unsigned now)
{
	struct efx_ssr_conn *c;
	unsigned i;

	EFX_BUG_ON_PARANOID(!list_empty(&st->active_conns));

	st->last_purge_jiffies = now;
	for (i = 0; i <= st->conns_mask; ++i) {
		if (list_empty(&st->conns[i]))
			continue;

		c = list_entry(st->conns[i].prev, struct efx_ssr_conn, link);
		if (now - c->last_pkt_jiffies > lro_idle_jiffies) {
			++st->n_drop_idle;
			efx_ssr_drop(st, c, i);
		}
	}
}

/* Push held skbs down into network stack.
 * Only called when active list is non-empty.
 */
void __efx_ssr_end_of_burst(struct efx_ssr_state *st)
{
	struct efx_ssr_conn *c;
	unsigned j;

	EFX_BUG_ON_PARANOID(list_empty(&st->active_conns));

	do {
		c = list_entry(st->active_conns.next, struct efx_ssr_conn,
			       active_link);
		EFX_BUG_ON_PARANOID(!c->skb);
		efx_ssr_deliver(st, c);
	} while (!list_empty(&st->active_conns));

	j = jiffies;
	if (unlikely(j != st->last_purge_jiffies))
		efx_ssr_purge_idle(st, j);
}

/* Construct an skb Push held skbs down into network stack.
 * Only called when active list is non-empty.
 */
static inline int
efx_ssr_merge(struct efx_ssr_state *st, struct efx_ssr_conn *c,
	      struct tcphdr *th, int data_length)
{
	/* Increase lengths appropriately */
	c->skb->len += data_length;
	c->skb->data_len += data_length;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
	/*
	 * Keep track of max MSS seen and store in
	 * gso_size for kernel to use
	 */
	if (data_length > skb_shinfo(c->skb)->gso_size)
		skb_shinfo(c->skb)->gso_size = data_length;
#endif

	/* Update the connection state flags */
	c->iph->tot_len += data_length;
	c->th->psh |= th->psh;
	c->th_last = th;
	++st->n_merges;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
	/* This kernel version does not understand LRO, and uses the max
	 * frame received to update rcv_mss.  If we're going above 1/4 of
	 * max window size without scaling pass the packet up.  This is
	 * slightly conservative, but close enough, and avoids rcv_mss
	 * growing too large.  Also stop merging if we got a PSH flag
	 * because if the sender is pushing messages a few times larger
	 * than the real MSS and we let rcv_mss grow larger than that
	 * message size we will end up delaying ACKs that the sender
	 * is waiting for.
	 */
	return (c->skb->len > 16384 || th->psh);
#else
	/* Pass packet up now if another segment could overflow the IP
	 * length.
	 */
	return (c->skb->len > 65536 - 9200);
#endif
}

static inline void
efx_ssr_start(struct efx_ssr_state *st, struct efx_ssr_conn *c,
	      struct tcphdr *th, int data_length)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
	/* Initialise gso_size appropriately */
	skb_shinfo(c->skb)->gso_size = data_length;
#endif

	/* Mangle header fields for later processing */
	c->iph->tot_len = ntohs(c->iph->tot_len);

	/* Move this connection the head of the active list */
	list_del(&c->active_link);
	list_add(&c->active_link, &st->active_conns);
}

static inline int
efx_ssr_conn_page(struct efx_ssr_state *st, struct efx_ssr_conn *c,
		  struct efx_rx_buffer *rx_buf, struct tcphdr *th,
		  int hdr_length, int data_length)
{
	if (likely(c->skb)) {
		struct skb_frag_struct *frag;
		frag = skb_shinfo(c->skb)->frags;
		frag += skb_shinfo(c->skb)->nr_frags;
		frag->page = rx_buf->page;
		frag->page_offset = RX_BUF_OFFSET(rx_buf) + hdr_length;
		frag->size = data_length;
		++skb_shinfo(c->skb)->nr_frags;
		rx_buf->page = NULL;

		if (efx_ssr_merge(st, c, th, data_length) ||
		    (skb_shinfo(c->skb)->nr_frags == MAX_SKB_FRAGS))
			efx_ssr_deliver(st, c);

		return 1;
	} else {
		c->skb = efx_rx_mk_skb(rx_buf, st->efx, hdr_length);
		if (unlikely(c->skb == NULL))
			return 0;

		c->eh = eth_hdr(c->skb);
		c->iph = (struct iphdr *)c->skb->data;
		c->th = (struct tcphdr *)((u8 *) c->iph + c->iph->ihl * 4);
		c->th_last = c->th;

		efx_ssr_start(st, c, th, data_length);

		return 1;
	}
}

static inline void
efx_ssr_conn_skb(struct efx_ssr_state *st, struct efx_ssr_conn *c,
		 struct efx_rx_buffer *rx_buf, struct ethhdr *eh,
		 struct iphdr *iph, struct tcphdr *th, int data_length)
{
	/* Transfer ownership of the rx_buf->skb to the LRO chain */
	struct sk_buff *skb = rx_buf->skb;
	rx_buf->skb = NULL;

	if (likely(c->skb)) {
		/* Remove the headers */
		skb_pull(skb, skb->len - data_length);

		/* Tack the new skb onto the head skb's frag_list. */
		EFX_BUG_ON_PARANOID(skb->next);
		if (!skb_shinfo(c->skb)->frag_list)
			skb_shinfo(c->skb)->frag_list = skb;
		else
			c->skb_tail->next = skb;
		c->skb_tail = skb;

		if (efx_ssr_merge(st, c, th, data_length))
			efx_ssr_deliver(st, c);
	} else {
		c->skb = skb;
		c->eh = eh;
		c->iph = iph;
		c->th = th;
		c->th_last = th;

		efx_ssr_start(st, c, th, data_length);
	}
}

/* Process SKB and decide whether to dispatch it to the stack now or
 * later.
 */
int efx_ssr(struct efx_ssr_state *st, struct efx_rx_buffer *rx_buf)
{

	int eh_proto, data_length, hdr_length, dont_merge;
	struct efx_ssr_conn *c;
	struct ethhdr *eh;
	struct iphdr *iph;
	struct tcphdr *th;
	unsigned th_seq, conn_hash, pkt_length;

	/* This does not handle VLAN code */
	/* Find the IP header. The ethernet header is always at rx_buf->data */
	eh = (struct ethhdr *)rx_buf->data;
	if (rx_buf->page) {
		eh_proto = eh->h_proto;
		iph = (struct iphdr *)(eh + 1);
	} else {
		/* The skb head is at the IP header */
		eh_proto = rx_buf->skb->protocol;
		iph = (struct iphdr *)rx_buf->skb->data;
	}

	/* We're not interested if it isn't TCP over IPv4, or if fragged. */
	if ((eh_proto - htons(ETH_P_IP)) |
	    (iph->protocol - IPPROTO_TCP) |
	    (iph->frag_off & htons(IP_MF | IP_OFFSET)))
		return 0;

	/* Get the TCP protocol */
	th = (struct tcphdr *)((u8 *) iph + iph->ihl * 4);
	hdr_length = (u8 *) th + th->doff * 4 - (u8 *) eh;
	/* Cope with padding after IP header */
	pkt_length = ntohs(iph->tot_len) + (u8 *)iph - (u8 *)eh;
	rx_buf->len = min(pkt_length, rx_buf->len);
	data_length = rx_buf->len - hdr_length;
	th_seq = ntohl(th->seq);
	dont_merge = ((data_length <= 0)
		      | th->urg | th->syn | th->rst | th->fin);

	/* Very cheap and crude hash. */
	conn_hash = (th->source ^ th->dest) & st->conns_mask;

	list_for_each_entry(c, &st->conns[conn_hash], link) {
		if ((c->saddr - iph->saddr) | (c->daddr - iph->daddr) |
		    (c->source - th->source) | (c->dest - th->dest))
			continue;

		/* Re-insert at head of list to reduce lookup time. */
		list_del(&c->link);
		list_add(&c->link, &st->conns[conn_hash]);

		if (unlikely(th_seq - c->next_seq)) {
			/* Out-of-order, so start counting again. */
			if (c->skb)
				efx_ssr_deliver(st, c);
			c->n_in_order_pkts = 0;
			c->next_seq = th_seq + data_length;
			++st->n_misorder;
			return 0;
		}
		c->next_seq = th_seq + data_length;
		c->last_pkt_jiffies = jiffies;

		if (c->n_in_order_pkts < lro_slow_start_packets) {
			/* May be in slow-start, so don't merge. */
			++st->n_slow_start;
			++c->n_in_order_pkts;
			return 0;
		}

		if (unlikely(dont_merge)) {
			if (c->skb)
				efx_ssr_deliver(st, c);
			if (th->fin || th->rst) {
				++st->n_drop_closed;
				efx_ssr_drop(st, c, conn_hash);
			}
			return 0;
		}

		if (rx_buf->page) {
			return efx_ssr_conn_page(st, c, rx_buf, th, hdr_length,
						 data_length);
		} else {
			efx_ssr_conn_skb(st, c, rx_buf, eh, iph, th,
					 data_length);
			return 1;
		}
	}

	/* We're not yet tracking this connection. */
	if (dont_merge)
		return 0;

	if (st->conns_n[conn_hash] >= lro_chain_max) {
		++st->n_too_many;
		return 0;
	}

	if (!list_empty(&st->free_conns)) {
		c = list_entry(st->free_conns.next, struct efx_ssr_conn, link);
		list_del(&c->link);
	} else {
		c = kmalloc(sizeof(*c), GFP_ATOMIC);
		if (c == NULL)
			return 0;
		c->skb = NULL;
		INIT_LIST_HEAD(&c->active_link);
	}

	/* Create the connection tracking data */
	++st->conns_n[conn_hash];
	list_add(&c->link, &st->conns[conn_hash]);
	c->saddr = iph->saddr;
	c->daddr = iph->daddr;
	c->source = th->source;
	c->dest = th->dest;
	c->next_seq = th_seq + data_length;
	c->n_in_order_pkts = 0;
	EFX_BUG_ON_PARANOID(c->skb);
	++st->n_new_stream;
	return 0;
}


#endif /* EFX_USE_SFC_LRO */
