
/*
#    Sfront, a SAOL to C translator    
#    This file: Sfront-specific network functions
#    Copyright (C) 1999  Regents of the University of California
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License (Version 2) as
#    published by the Free Software Foundation.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/


/****************************************************************/
/*           receives RTP, RTCP, and SIP packets                */
/****************************************************************/

int nsys_newdata(void) 

{
  struct sockaddr_in ipaddr;
  unsigned long ssrc;
  struct nsys_source * sptr;
  int len, i, j;
  unsigned char packet[NSYS_UDPMAXSIZE+1];
  unsigned char digest[NSYS_MD5_LENGTH];
  unsigned char * p, * pmax;
  unsigned char ptype, mlen;
  unsigned char cmd = 0;
  unsigned char ndata = 0;
  unsigned char vdata = 0;
  int fromlen = sizeof(struct sockaddr);
  int retry = 0;
  int rtpcode, jcode, fec, prerec;
  int nopacket = 1;
  unsigned short status;
  int fd = nsys_rtp_fd;

  nsys_bufflen = nsys_buffcnt = 0;

  nsys_netout_tstamp += ACYCLE;

  if (nsys_netout_jsend_guard_time && ((--nsys_netout_jsend_guard_time) == 0))
    nsys_netout_guard_tick();
 
  while (1)
    {
      if ((len = recvfrom(fd, packet, NSYS_UDPMAXSIZE, 0, 
			  (struct sockaddr *)&ipaddr, &fromlen)) <= 0)
	{

	  if ((errno == EAGAIN) && (fd == nsys_rtcp_fd))
	    {
	      if (nopacket && (time(NULL) > nsys_nexttime))
		{
		  nsys_netout_rtcptime();		  
		  if (nsys_powerup_mset)
		    nsys_netin_clear_mset(nsys_buff, &nsys_bufflen, 
					  NSYS_BUFFSIZE);
		}
	      return (nsys_bufflen ? NSYS_MIDIEVENTS : NSYS_DONE);
	    }

	  if ((errno == EAGAIN) && (fd == nsys_rtp_fd))
	    {
	      fd = nsys_rtcp_fd;
	      continue;
	    }
	  
	  if (errno == EINTR)
	    {
	      if (++retry > NSYS_MAXRETRY)
		NSYS_ERROR_TERMINATE("Too many I/O retries: nsys_netin_newdata");
	      continue;         
	    }
	  
	  NSYS_ERROR_TERMINATE("Error reading Internet socket");
	}
      else
	{
	  /*****************/
	  /* RTP fast path */
	  /*****************/

	  nopacket = 0;

	  if ((fd == nsys_rtp_fd) && 
	      (packet[NSYS_RTPLOC_BYTE1] == NSYS_RTPVAL_BYTE1))
	    {	      

	      /***************************************/
	      /* parse top of header to extract ssrc */
	      /***************************************/

	      if ((len < NSYS_RTPLEN_HDR) || (len <= NSYS_RTPSIZE_DIGEST))
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, "Truncated RTP packet");
		  continue;
		}

	      ssrc = ntohl(*((unsigned long *)&(packet[NSYS_RTPLOC_SSRC])));
	      ptype = packet[NSYS_RTPLOC_PTYPE] & NSYS_RTPVAL_CLRMARK;
	      fec = (packet[NSYS_RTPLOC_PTYPE] < NSYS_RTPVAL_SETMARK);

	      sptr = nsys_ssrc[ssrc & NSYS_HASHMASK];
	      
	      while (sptr && (sptr->ssrc != ssrc))
		sptr = sptr->xtra;
	      
	      if (!sptr)
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, 
			       "RTP packet from unknown source");
		  continue;
		}

	      if (((ipaddr.sin_addr.s_addr != sptr->rtp_addr->sin_addr.s_addr) ||
		  (ipaddr.sin_port != sptr->rtp_addr->sin_port)) && 
		  (sptr->alt_rtp_addr == NULL))
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, 
			       "RTP packet from unknown IP/port");
		  continue;
		}

	      /*********************/
	      /* do authentication */
	      /*********************/

	      len -= NSYS_RTPSIZE_DIGEST;

	      if (nsys_msession)
		memcpy(&(packet[NSYS_RTPLOC_SSRC]), &nsys_myssrc_net, 
		       sizeof(long));

	      nsys_hmac_md5(packet, len, nsys_keydigest, digest);

	      if (memcmp(&(packet[len]), digest, NSYS_RTPSIZE_DIGEST))
		{		  
		  nsys_warning(NSYS_WARN_UNUSUAL, 
			       "Discarding unauthorized RTP packet");
		  continue;
		}

	      /******************/
	      /* process packet */
	      /******************/

	      if (ptype != sptr->ptype)
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, 
			       "RTP packet with incorrect ptype");
		  continue;
		}

	      len -= NSYS_RTPLEN_HDR;

	      if (sptr->alt_rtp_addr)
		{
		  if ((ipaddr.sin_addr.s_addr != nsys_sip_rtp_inet_addr) ||
		      (ipaddr.sin_port != nsys_sip_rtp_sin_port))
		    {
		      /* normal case */

		      memcpy(sptr->rtp_addr,&ipaddr,sizeof(struct sockaddr_in));
		    }
		  else
		    {
		      /* source-forge trick which may later be implemented */

		      memcpy(sptr->rtp_addr, &(sptr->alt_rtp_addr), 
			     sizeof(struct sockaddr_in));
		    }

		  free(sptr->alt_rtp_addr);
		  sptr->alt_rtp_addr = NULL;
		}

	      rtpcode = nsys_netin_rtpstats(sptr, packet);

	      if (rtpcode == NSYS_RTPCODE_SECURITY)
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, "Possible RTP replay attack");
		  continue;
		}

	      if ((len == 0) || ((rtpcode == NSYS_RTPCODE_DISCARD) && fec))
		{
		  if (len == 0)
		    nsys_warning(NSYS_WARN_UNUSUAL, "RTP payload empty");
		  else
		    nsys_warning(NSYS_WARN_UNUSUAL, "Out of order RTP packet");
		  continue;        
		}

	      p = packet + NSYS_RTPLEN_HDR;

	      if ((rtpcode != NSYS_RTPCODE_NORMAL) && fec)
		{

		  jcode = nsys_netin_journal_recovery(sptr, rtpcode, p, len,
						      nsys_buff, &nsys_bufflen,
						      NSYS_BUFFSIZE);
		  if (jcode == NSYS_JOURNAL_CORRUPTED)
		    {
		      nsys_warning(NSYS_WARN_UNUSUAL, "RTP journal corrupt");
		      continue;
		    }
		  if (jcode == NSYS_JOURNAL_FILLEDBUFF)
		    {
		      nsys_warning(NSYS_WARN_UNUSUAL, "RTP journal too big");
		      return (nsys_bufflen ? NSYS_MIDIEVENTS : NSYS_DONE);
		    }

		  /* NSYS_JOURNAL_RECOVERED falls through */
		}

	      /****************/
	      /* RTP payload  */
	      /****************/

	      if ((mlen = ((*(p++)) & NSYS_SM_MLENMAX)) == 0)
		continue;            

	      if (mlen > (--len))
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, "RTP MIDI section trunction");
		  continue;            
		}

	      pmax = p + mlen;

	      if ((*p) < CSYS_MIDI_NOTEOFF)
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, "RTP running status error");
		  continue;                     
		}

	      while (p < pmax)
		{
		  if ((NSYS_BUFFSIZE - nsys_bufflen) < 4)
		    {
		      nsys_warning(NSYS_WARN_UNUSUAL, 
				   "RTP journal truncated MIDI commands");
		      return (nsys_bufflen ? NSYS_MIDIEVENTS : NSYS_DONE);	
		    }
		  if (((*p) >= CSYS_MIDI_NOTEOFF) && 
		      ((0xF0 & (cmd = (*(p++)))) == CSYS_MIDI_SYSTEM))
		    {
		      nsys_warning(NSYS_WARN_UNUSUAL, 
				   "RTP MIDI System command in stream");
		      break;
		    }
		  if ((sptr->ontime == 0) &&
		      ((0xF0 & cmd) == CSYS_MIDI_NOTEON) && p[1])
		    {
		      p += 2;
		      continue;
		    }
		  nsys_buff[nsys_bufflen++] = cmd;
		  nsys_buff[nsys_bufflen++] = (ndata = *(p++));
		  if ( ((0xF0 & cmd) != CSYS_MIDI_PROGRAM) && 
		       ((0xF0 & cmd) != CSYS_MIDI_CTOUCH))
		    nsys_buff[nsys_bufflen++] = (vdata = *(p++));
		  if (fec)
		    nsys_netin_journal_trackstate(sptr, cmd, ndata, vdata);
		  nsys_buff[nsys_bufflen++] = sptr->mset;
		}

	      /* if no room left for another MIDI command, leave */
	      
	      if ((NSYS_BUFFSIZE - nsys_bufflen) < NSYS_SM_EXPANDMAX)
		return (nsys_bufflen ? NSYS_MIDIEVENTS : NSYS_DONE);

	      continue;
	    }

	  /***********************/
	  /* handle RTCP packets */
	  /***********************/

	  if ((fd == nsys_rtcp_fd) && 
	      (NSYS_RTCPVAL_BYTE1 == (packet[NSYS_RTCPLOC_BYTE1] 
				      & NSYS_RTCPVAL_COOKIEMASK)))
	    {
	      if ((len -= NSYS_RTPSIZE_DIGEST) <= 0)
		{
		  nsys_warning(NSYS_WARN_UNUSUAL, "RTCP packet truncated");
		  continue;
		}
	      
	      if (nsys_msession)
		{
		  memcpy(&ssrc, &(packet[NSYS_RTCPLOC_SSRC]), sizeof(long));
		  memcpy(&(packet[NSYS_RTCPLOC_SSRC]), &nsys_myssrc_net, 
			 sizeof(long));
		}

	      nsys_hmac_md5(packet, len, nsys_keydigest, digest);

	      if (memcmp(&(packet[len]), digest, NSYS_RTPSIZE_DIGEST))
		{		  
		  nsys_warning(NSYS_WARN_UNUSUAL, 
			       "Discarding unauthorized RTCP packet");
		  continue;
		}

	      if (nsys_msession)
		memcpy(&(packet[NSYS_RTCPLOC_SSRC]), &ssrc, sizeof(long));
	      
	      sptr = nsys_netin_rtcp(packet, len, &ipaddr);

	      if (nsys_feclevel)
		nsys_netin_journal_trimstate(sptr);
	
	      if (nsys_powerup_mset)
		{
		  nsys_netin_clear_mset(nsys_buff, &nsys_bufflen,
					NSYS_BUFFSIZE);
		  if ((NSYS_BUFFSIZE - nsys_bufflen) < NSYS_SM_EXPANDMAX)
		    return (nsys_bufflen ? NSYS_MIDIEVENTS : NSYS_DONE);
		}
	      continue;
	    }
	  
	  /***********************/
	  /* handle SIP packets  */
	  /***********************/

	  packet[len] = '\0';  /* null-terminate string */

	  if (sscanf(packet, "SIP/2.0 %hu", &status) == 1)
	    nsys_netin_reply(fd, &ipaddr, packet, status);
	  else
	    if (packet[NSYS_RTPLOC_BYTE1] == 'I')
	      {
		nsys_netin_invite(fd,  &ipaddr, packet);
		if (nsys_powerup_mset)
		  {
		    nsys_netin_clear_mset(nsys_buff, &nsys_bufflen,
					  NSYS_BUFFSIZE);
		    if ((NSYS_BUFFSIZE - nsys_bufflen) < NSYS_SM_EXPANDMAX)
		      return (nsys_bufflen ? NSYS_MIDIEVENTS : NSYS_DONE);
		  }
	      }
	}
    } 
}


/****************************************************************/
/*          returns the next network MIDI command               */
/****************************************************************/

int nsys_midievent(unsigned char * cmd, unsigned char * ndata,
                    unsigned char * vdata, unsigned short * extchan)

{
  *cmd = nsys_buff[nsys_buffcnt];
  *extchan = 0x0F & nsys_buff[nsys_buffcnt++];
  *ndata = nsys_buff[nsys_buffcnt++];

  if ((((*cmd) & 0xF0) != CSYS_MIDI_PROGRAM) && 
      (((*cmd) & 0xF0) != CSYS_MIDI_CTOUCH))
    *vdata = nsys_buff[nsys_buffcnt++];

  *extchan += (NSYS_NETSTART - CSYS_MIDI_NUMCHAN + 
	       (nsys_buff[nsys_buffcnt++] << 4));
  return (nsys_buffcnt == nsys_bufflen) ? NSYS_DONE : NSYS_MIDIEVENTS;
}

/****************************************************************/
/*                    sends RTP packets                         */
/****************************************************************/

void nsys_midisend(unsigned char cmd, unsigned char ndata,
		  unsigned char vdata, unsigned short extchan)

{
  int retry = 0;
  int jsize = 0;
  unsigned char size, cbyte;
  unsigned long tstamp;
  unsigned short seqnum;
  struct nsys_source * sptr;
  struct sockaddr * addr;
  int alt = 0;
  int jstart, dstart;

  /**********************/
  /* MIDI preprocessing */
  /**********************/

  cbyte = cmd & 0xF0;

  size =  (cbyte > CSYS_MIDI_SPECIAL) ? 3 : 0;
  size -= ((cbyte == CSYS_MIDI_PROGRAM) || (cbyte == CSYS_MIDI_CTOUCH));

  cmd = cbyte | (((unsigned char)extchan) & 0x0F);

  if (nsys_msession && 
      ((cbyte == CSYS_MIDI_NOTEON) || (cbyte == CSYS_MIDI_NOTEOFF)) && 
      ((ndata += NSYS_MSESSION_INTERVAL) > 0x7F))
    ndata = 0x7F;

  /*************************/
  /* leave if no receivers */
  /*************************/

  if ((sptr = nsys_srcroot) == NULL)
    { 
      if (nsys_feclevel)
	nsys_netout_journal_addhistory(cmd, ndata, vdata);
      return;
    }

  /**********************/
  /* fill in RTP header */
  /**********************/

  nsys_netout_rtp_packet[NSYS_RTPLOC_PTYPE] = sptr->ptype | nsys_netout_markbit;

  tstamp = htonl((unsigned long)(nsys_netout_tstamp));
  memcpy(&(nsys_netout_rtp_packet[NSYS_RTPLOC_TSTAMP]), &tstamp, sizeof(long));

  nsys_netout_seqnum = ((nsys_netout_seqnum != NSYS_RTPSEQ_HIGHEST) ?
			(nsys_netout_seqnum + 1) : 1); 
  seqnum = htons((unsigned short)(nsys_netout_seqnum & 0x0000FFFF));
  memcpy(&(nsys_netout_rtp_packet[NSYS_RTPLOC_SEQNUM]), &seqnum, 
	 sizeof(short));

  /**************************************/
  /* fill in command section of payload */
  /**************************************/

  nsys_netout_rtp_packet[NSYS_RTPLEN_HDR] = size;

  nsys_netout_rtp_packet[NSYS_RTPLEN_HDR + 1] = cmd;
  nsys_netout_rtp_packet[NSYS_RTPLEN_HDR + 2] = ndata;
  nsys_netout_rtp_packet[NSYS_RTPLEN_HDR + 3] = vdata;

  /**************************************/
  /* fill in journal section of payload */
  /**************************************/

  if (nsys_feclevel)
    {
      jstart = NSYS_RTPLEN_HDR + 1 + size;
      jsize = nsys_netin_journal_create(&(nsys_netout_rtp_packet[jstart]),
					NSYS_UDPMAXSIZE - NSYS_MD5_LENGTH 
					- jstart);
      nsys_netout_journal_addstate(cmd, ndata, vdata);
    }

  /********************************/
  /* add authentication signature */
  /********************************/

  dstart = NSYS_RTPLEN_HDR + 1 + size + jsize;
  nsys_hmac_md5(nsys_netout_rtp_packet, dstart, nsys_keydigest, 
		&(nsys_netout_rtp_packet[dstart]));

  /********************************************/
  /* calculate packet size, update statistics */
  /********************************************/

  nsys_sent_this = 1;
  nsys_sent_packets++;
  nsys_sent_octets += (size + 1 + jsize + NSYS_RTPSIZE_DIGEST);
  size += (NSYS_RTPLEN_HDR + 1 + jsize + NSYS_RTPSIZE_DIGEST);     

  /***************************/
  /* send packet to everyone */
  /***************************/

  do 
    {
      addr = (struct sockaddr *) (alt ? sptr->alt_rtp_addr : sptr->rtp_addr);

      /* when multiple payload types supported, see if *p*[NSYS_RTPLOC_PTYPE] */
      /* differs from (sptr->ptype | nsys_netout_markbit) and reauthenticate  */

      if (sendto(nsys_rtp_fd, nsys_netout_rtp_packet, size, 0, addr, 
		 sizeof(struct sockaddr)) == -1)
	{
	  if (errno == EAGAIN)
	    continue;
	  
	  if ((errno == EINTR) || (errno == ENOBUFS))
	    {
	      if (++retry > NSYS_MAXRETRY)
		NSYS_ERROR_TERMINATE("Too many I/O retries -- nsys_netout_newdata");
	      continue;         
	    }
	  
	  NSYS_ERROR_TERMINATE("Error writing Internet socket");
	}

      if (!(alt = ((!alt) && sptr->alt_rtp_addr)))
	sptr = sptr->next;
    } 
  while (alt || (sptr != nsys_srcroot));

}

/****************************************************************/
/*              called at the end of control processing         */
/****************************************************************/

void nsys_endcycle(void)

{

 if (nsys_netout_jsend_guard_send) 
   nsys_midisend(CSYS_MIDI_NOOP, 0, 0, 0);
 
 if (nsys_graceful_exit && !graceful_exit)
   graceful_exit = 1;

}

/* end  Sfront-specific network functions */

