#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sg_include.h"
#include "sg_err.h"

/* A utility program for the Linux OS SCSI generic ("sg") device driver.
*  Copyright (C) 1999 D. Gilbert and P. Allworth
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   This program is a specialization of the Unix "dd" command in which
   one or both of the given files is a scsi generic device. It assumes
   a 'bs' (block size) of 512 and complains if 'bs' ('ibs' or 'obs') is
   given with some other value.
   If 'if' is not given or 'if=-' then stdin is assumed. If 'of' is
   not given of 'of=-' then stdout assumed. The multipliers "c, b, k, m"
   are recognized on numeric arguments.
   This variant uses command queuing on the 'if' file if it is
   a SCSI generic device.
   As an experiment added an argument "tq" to allow 'tagged queuing' to
   be enabled (1), disabled(0) or left as is (-1) which is the default.
   BEWARE: If the 'of' file is a 'sg' device (eg a disk) then it _will_
   be written to, potentially destroying its previous contents.

   Version 0.93 991127
*/

#define BLOCK_SIZE 512
#define MAX_Q SG_MAX_QUEUE

#define BLOCKS_PER_WBUFF 128    /* this implies 64 KByte working buffer */
/* States for sgq_work structure follow */
#define SGQ_FREE        0
#define SGQ_WRITTEN     1
#define SGQ_READABLE    2
#define SGQ_ERROR       3

/* #define SG_DEBUG */

#define SG_HEAD_SZ sizeof(struct sg_header)
#define SCSI_CMD10_LEN 10
#define READ_CAP_DATA_LEN 8

static unsigned char rdCmdBlk[SCSI_CMD10_LEN] = 
                    {0x28, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char wrCmdBlk[SCSI_CMD10_LEN] = 
                    {0x2a, 0, 0, 0, 0, 0, 0, 0, 0, 0};

typedef struct sgq_work {
    int nxt_read_block; /* next read block address to be issued */
    int blocks_to_read; /* 0 -> no more to read */
    int ind_wr_buff;    /* index of writeable buffer, -1 -> none */
    int nxt_exp_block;  /* next block address waiting to be read */
    unsigned char * buff[MAX_Q];
    int state[MAX_Q];
    int blocks[MAX_Q];
} Sgq_work;


void dtor_sgq_work(Sgq_work * wp)
{
    int k;

    for (k = 0; k < MAX_Q; ++k) {
        if (wp->buff[k])
            free(wp->buff[k]);
    }
    memset(wp, 0, sizeof(Sgq_work)); /* should not hurt */
}

int ctor_sgq_work(int first_block_addr, int blocks_to_read, Sgq_work * wp)
{
    int k;

    memset(wp, 0, sizeof(Sgq_work));
    for (k = 0; k < MAX_Q; ++k) {
        wp->buff[k] = malloc(SG_HEAD_SZ + SCSI_CMD10_LEN +
                             (BLOCK_SIZE * BLOCKS_PER_WBUFF));
        if (! wp->buff[k]) {
            dtor_sgq_work(wp);
            return 2;
        }
        wp->state[k] = SGQ_FREE;
        wp->blocks[k] = 0;
    } 
    wp->nxt_read_block = first_block_addr;
    wp->blocks_to_read = blocks_to_read;
    wp->ind_wr_buff = -1;
    wp->nxt_exp_block = -1;
    return 0;
}

void usage()
{
    printf("Usage: "
           "sgq_dd512 [if=<infile>] [skip=<n>] [of=<ofile>] [seek=<n>]\n"
           "       [count=<n>] [tq=<n>]      {512 byte 'bs' assumed}\n"
           "            either 'if' or 'of' must be a scsi generic device\n"
           " 'tq' is tagged queuing, 1->enable, 0->disable, -1->leave(def)\n");
}

int read_capacity(int sg_fd, int * num_sect, int * sect_sz)
{
    int res, f, err;
    unsigned char rcCmdBlk [10] = {0x25, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    unsigned char rcBuff[SG_HEAD_SZ + sizeof(rcCmdBlk) + 512];
    int rcInLen = SG_HEAD_SZ + sizeof(rcCmdBlk);
    int rcOutLen;
    unsigned char * buffp = rcBuff + SG_HEAD_SZ;
    struct sg_header * rsghp = (struct sg_header *)rcBuff;

    rcOutLen = SG_HEAD_SZ + READ_CAP_DATA_LEN;
    rsghp->pack_len = 0;                /* don't care */
    rsghp->pack_id = 0;
    rsghp->reply_len = rcOutLen;
    rsghp->twelve_byte = 0;
    rsghp->result = 0;
    memcpy(rcBuff + SG_HEAD_SZ, rcCmdBlk, sizeof(rcCmdBlk));

    f = fcntl(sg_fd, F_GETFL);
    if (-1 == f) {
        perror("read_capacity (fcntl(F_GETFL)) error");
        return 1;
    }
    fcntl(sg_fd, F_SETFL, f & (~ O_NONBLOCK)); /* turn on blocking */
    res = write(sg_fd, rcBuff, rcInLen);
    if (res < 0) {
        perror("read_capacity (wr) error");
        fcntl(sg_fd, F_SETFL, f); /* restore flags */
        return 1;
    }
    if (res < rcInLen)
    {
        printf("read_capacity (wr) problems\n");
        fcntl(sg_fd, F_SETFL, f); /* restore flags */
        return 1;
    }
    
    memset(buffp, 0, 8);
    res = read(sg_fd, rcBuff, rcOutLen);
    if (res < 0)
    {
        perror("read_capacity (rd) error");
        fcntl(sg_fd, F_SETFL, f); /* restore flags */
        return 1;
    }
    err = fcntl(sg_fd, F_SETFL, f); /* restore flags */
    if (-1 == err) {
        perror("read_capacity (fcntl(F_SETFL)) error");
        return 1;
    }
    if (res < rcOutLen)
    {
        printf("read_capacity (rd) problems\n");
        return 1;
    }
    if (! sg_chk_n_print("read_capacity", rsghp->target_status, 
                         rsghp->host_status, rsghp->driver_status, 
                         rsghp->sense_buffer, SG_MAX_SENSE))
        return 1;

    *num_sect = 1 + ((buffp[0] << 24) | (buffp[1] << 16) |
                (buffp[2] << 8) | buffp[3]);
    *sect_sz = (buffp[4] << 24) | (buffp[5] << 16) | 
               (buffp[6] << 8) | buffp[7];

/* printf("number of sectors=%d, sector size=%d\n", *num_sect, *sect_sz); */

    return 0;
}

void set_read_outgoing(unsigned char * buff, int from_block, int blocks)
{
    int outLen;
    unsigned char * rdCmd = buff + SG_HEAD_SZ;
    struct sg_header * isghp = (struct sg_header *)buff;

    outLen = SG_HEAD_SZ + (BLOCK_SIZE * blocks);
    isghp->pack_len = 0;                /* don't care */
    isghp->pack_id = from_block;
    isghp->reply_len = outLen;
    isghp->twelve_byte = 0;
    isghp->result = 0;
    memcpy(rdCmd, rdCmdBlk, SCSI_CMD10_LEN);
    rdCmd[2] = (unsigned char)((from_block >> 24) & 0xFF);
    rdCmd[3] = (unsigned char)((from_block >> 16) & 0xFF);
    rdCmd[4] = (unsigned char)((from_block >> 8) & 0xFF);
    rdCmd[5] = (unsigned char)(from_block & 0xFF);
    rdCmd[7] = (unsigned char)((blocks >> 8) & 0xff);
    rdCmd[8] = (unsigned char)(blocks & 0xff);
}

static int num_sg_reads = 0;
static int num_q_fulls = 0;
static int num_reads_waiting = 0;
static int num_reads_zero = 0;
static int num_eagains = 0;

int sg_read(int sg_fd, Sgq_work * sgqp)
{
    int k, blocks, res, find, outLen;
    int inLen = SG_HEAD_SZ + SCSI_CMD10_LEN;
    struct sg_header * osghp;
    int wait_on_read = 0;
    int flags;
    int num_waiting;
    char ebuff[256];
    
    ++num_sg_reads;
    if (ioctl(sg_fd, SG_GET_NUM_WAITING, &num_waiting) < 0) {
        sprintf(ebuff, "sgq_dd512: SG_GET_NUM_WAITING ioctl error");
        perror(ebuff);
    }
    else {
        if (num_waiting)
            num_reads_waiting += num_waiting;
        else
            ++num_reads_zero;
    }
    flags = fcntl(sg_fd, F_GETFL);
    while (1) {
        res = 0;
        if (sgqp->blocks_to_read > 0) {
            for (k = 0; k < MAX_Q; ++k) {
                if (SGQ_FREE == sgqp->state[k])
                    break;
            }
            if (k < MAX_Q) {
                blocks = (sgqp->blocks_to_read > BLOCKS_PER_WBUFF) ? 
                         BLOCKS_PER_WBUFF : sgqp->blocks_to_read;
                set_read_outgoing(sgqp->buff[k], sgqp->nxt_read_block,
                                  blocks);
#ifdef SG_DEBUG
                printf("k=%d, block=%d, num=%d\n", k, sgqp->nxt_read_block,
                        blocks);
#endif
                while (1) {
                    res = write(sg_fd, sgqp->buff[k], inLen);
                    if (res >= 0)
                        break;
                    if ((res < 0) && (EAGAIN != errno)) {
                        sprintf(ebuff, "sg_read (wr) block=%d, error", 
                                sgqp->blocks_to_read);
                        perror(ebuff);
                        return 1;
                    }
                    usleep(10000);      /* sleep 10 milliseconds */
                    ++num_eagains;
                }
                if (res < inLen) {
                    printf("sg_read (wr) block=%d, ask=%d, got=%d\n", 
                           sgqp->blocks_to_read, inLen, res);
                    return 1;
                }
                if (-1 == sgqp->nxt_exp_block)
                    sgqp->nxt_exp_block = sgqp->nxt_read_block;
                sgqp->nxt_read_block += blocks;
                sgqp->blocks_to_read -= blocks;
                sgqp->state[k] = SGQ_WRITTEN;
                sgqp->blocks[k] = blocks;
            }
            else {
                wait_on_read = 1;
                ++num_q_fulls;
            }
        }
        else
            wait_on_read = 1;

        find = sgqp->nxt_exp_block;
        for (k = 0; k < MAX_Q; ++k) {
            if ((SGQ_WRITTEN == sgqp->state[k]) &&
                (find == ((struct sg_header *)sgqp->buff[k])->pack_id))
                break;
        }
        if (k >= MAX_Q) {
            printf("sg_read, nothing readable ??? \n");
            break;
        }
        if (wait_on_read) {
            if (-1 == (fcntl(sg_fd, F_SETFL, flags & (~O_NONBLOCK)))) {
                sprintf(ebuff, "sg_read clear O_NONBLOCK fcntl error");
                perror(ebuff);
                return 1;
            }
        }
        osghp = (struct sg_header *)(sgqp->buff[k] + SCSI_CMD10_LEN);
        outLen = ((struct sg_header *)sgqp->buff[k])->reply_len;
        osghp->pack_id = find;
    
        res = read(sg_fd, osghp, outLen);
        if (res < 0) {
            if (EAGAIN == errno) {
                if (wait_on_read) {
                    printf("sg_read (rd) EAGAIN+waiting?? block=%d\n", find);
                    return 1;
                }
                else
                    ; /* drop through case, continue around the loop */
            }
            else {
                sprintf(ebuff, "sg_read (rd) block=%d, error", find);
                perror(ebuff);
                return 1;
            }
        }
        else { /* got something so clean up */
            if (res < outLen)
            {
                printf("sg_read (rd) block=%d, ask=%d, got=%d\n",
                       find, outLen, res);
                return 1;
            }
            if (! sg_chk_n_print("after read(rd)", osghp->target_status, 
                                 osghp->host_status, osghp->driver_status, 
                                 osghp->sense_buffer, SG_MAX_SENSE))
                return 1;
            if (wait_on_read) {
                if (-1 == (fcntl(sg_fd, F_SETFL, flags | O_NONBLOCK))) { 
                    sprintf(ebuff, 
                            "sg_read setting O_NONBLOCK again fcntl error");
                    perror(ebuff);
                    return 1;
                }
            }
            sgqp->state[k] = SGQ_READABLE;
            sgqp->nxt_exp_block += sgqp->blocks[k];
            sgqp->ind_wr_buff = k;
            return 0;
        }
    }
    return 2;
}

int sg_write(int sg_fd, unsigned char * buff, int blocks, int to_block)
{
    int outLen = SG_HEAD_SZ;
    int inLen, res;
    unsigned char * wrCmd = buff + SG_HEAD_SZ;
    struct sg_header * isghp = (struct sg_header *)buff;
    struct sg_header * osghp = (struct sg_header *)(buff + SCSI_CMD10_LEN);
    
    inLen = SG_HEAD_SZ + SCSI_CMD10_LEN + (BLOCK_SIZE * blocks);
    isghp->pack_len = 0;                /* don't care */
    isghp->pack_id = to_block;
    isghp->reply_len = outLen;
    isghp->twelve_byte = 0;
    isghp->result = 0;
    memcpy(wrCmd, wrCmdBlk, SCSI_CMD10_LEN);
    wrCmd[2] = (unsigned char)((to_block >> 24) & 0xFF);
    wrCmd[3] = (unsigned char)((to_block >> 16) & 0xFF);
    wrCmd[4] = (unsigned char)((to_block >> 8) & 0xFF);
    wrCmd[5] = (unsigned char)(to_block & 0xFF);
    wrCmd[7] = (unsigned char)((blocks >> 8) & 0xff);
    wrCmd[8] = (unsigned char)(blocks & 0xff);

    res = write(sg_fd, buff, inLen);
    if (res < 0) {
        perror("writing (wr) on sg device, error");
        return 1;
    }
    if (res < inLen) {
        printf("writing (wr) on sg device, problems, ask=%d, got=%d\n", 
               inLen, res);
        return 1;
    }

    res = read(sg_fd, buff + SCSI_CMD10_LEN, outLen);
    if (res < 0) {
        perror("writing (rd) on sg device, error");
        return 1;
    }
    if (res < outLen)
    {
        printf("writing (rd) on sg device, problems, ask=%d, got=%d\n",
               outLen, res);
        return 1;
    }
    if (! sg_chk_n_print("after write(rd)", osghp->target_status, 
                         osghp->host_status, osghp->driver_status, 
                         osghp->sense_buffer, SG_MAX_SENSE))
        return 1;
    return 0;
}

int get_num(char * buf)
{
    int res, num;
    char c, cc;
            
    res = sscanf(buf, "%d%c", &num, &c);
    if (0 == res)
        return -1;
    else if (1 == res)
        return num;
    else {
        cc = (char)toupper(c);
        if ('B' == cc)
            return num * 512;
        else if ('C' == cc)
            return num;
        else if ('K' == cc)
            return num * 1024;
        else if ('M' == cc)
            return num * 1024 * 1024;
        else {
            printf("unrecognized multiplier\n");        
            return -1;
        }
    }
}


int main(int argc, char * argv[])
{
    int skip = 0;
    int seek = 0;
    int count = -1;
    int tq = -1;
    char str[512];
    char * key;
    char * buf;
    char inf[512];
    int in_is_sg = 0;
    char outf[512];
    int out_is_sg = 0;
    int bs_bad = 0;
    int res, k, t;
    int infd, outfd, blocks;
    Sgq_work sgq_w;
    int wrkOffset;
    unsigned char * wrkp;
    int in_num_sect, in_sect_sz, out_num_sect, out_sect_sz;
    double anw, pqf, prqe;
    int in_full = 0;
    int in_partial = 0;
    int out_full = 0;
    int out_partial = 0;
    char ebuff[256];

    inf[0] = '\0';
    outf[0] = '\0';
    if (argc < 2) {
        usage();
        return 1;
    }

    for(k = 1; k < argc; k++) {
        if (argv[k])
            strcpy(str, argv[k]);
        else
            continue;
        for(key = str, buf = key; *buf && *buf != '=';)
            buf++;
        if (*buf)
            *buf++ = '\0';
        if (strcmp(key,"if") == 0)
            strcpy(inf, buf);
        else if (strcmp(key,"of") == 0)
            strcpy(outf, buf);
        else if (0 == strcmp(key,"ibs")) {
            if (BLOCK_SIZE != get_num(buf))
                bs_bad = 1;
        }
        else if (0 == strcmp(key,"obs")) {
            if (BLOCK_SIZE != get_num(buf))
                bs_bad = 1;
        }
        else if (0 == strcmp(key,"bs")) {
            if (BLOCK_SIZE != get_num(buf))
                bs_bad = 1;
        }
        else if (0 == strcmp(key,"skip"))
            skip = get_num(buf);
        else if (0 == strcmp(key,"seek"))
            seek = get_num(buf);
        else if (0 == strcmp(key,"count"))
            count = get_num(buf);
        else if (0 == strcmp(key,"tq"))
            tq = get_num(buf);
        else {
            printf("Unrecognized argument '%s'\n", key);
            usage();
            return 1;
        }
    }
    if (bs_bad) {
        printf("If bs/ibs/obs given, must=%d\n", BLOCK_SIZE);
        usage();
        return 1;
    }
    if ((skip < 0) || (seek < 0)) {
        printf("skip and seek cannot be negative\n");
        return 1;
    }
#ifdef SG_DEBUG
    printf("sgq_dd512: if=%s skip=%d of=%s seek=%d count=%d\n",
           inf, skip, outf, seek, count);
#endif
    infd = STDIN_FILENO;
    outfd = STDOUT_FILENO;
    if (inf[0] && ('-' != inf[0])) {
        if ((infd = open(inf, O_RDWR)) >= 0) {
            if (ioctl(infd, SG_GET_PACK_ID, &t) < 0) {
                /* not a scsi generic device so now try and open RDONLY */
                close(infd);
            }
            else {
                in_is_sg = 1;
                res = 0;
                if (0 == tq)
                    res = ioctl(infd, SCSI_IOCTL_TAGGED_DISABLE, &t);
                if (1 == tq)
                    res = ioctl(infd, SCSI_IOCTL_TAGGED_ENABLE, &t);
                if (res < 0)
                    perror("sgq_dd512: SCSI_IOCTL_TAGGED error");
                if ((-1 == (fcntl(infd, F_SETFL, O_NONBLOCK))) ||       
                    (ioctl(infd, SG_SET_FORCE_PACK_ID, &in_is_sg) < 0) ||
                    (ioctl(infd, SG_SET_COMMAND_Q, &in_is_sg) < 0)) {
                    sprintf(ebuff, 
"sgq_dd512: SG_SET_FORCE_PACK_ID/SG_SET_COMMAND_Q ioctl failed on %s ", inf);
                    perror(ebuff);
                    close(infd);
                    return 1;
                }
#ifdef SG_DEF_RESERVED_SIZE
                t = BLOCK_SIZE * BLOCKS_PER_WBUFF;
                res = ioctl(infd, SG_SET_RESERVED_SIZE, &t);
                if (res < 0)
                    perror("sgq_dd512: SG_SET_RESERVED_SIZE error");
#endif
            }
        }
        if (! in_is_sg) {
            if ((infd = open(inf, O_RDONLY)) < 0) {
                sprintf(ebuff, 
        "sgq_dd512: could not open %s for reading", inf);
                perror(ebuff);
                return 1;
            }
            else if (skip > 0) {
                off_t offset = skip;

                offset *= BLOCK_SIZE;       /* could overflow here! */
                if (lseek(infd, offset, SEEK_SET) < 0) {
                    sprintf(ebuff, 
                "sgq_dd512: Couldn't skip to required position on %s", inf);
                    perror(ebuff);
                    return 1;
                }
            }
        }
    }
    if (outf[0] && ('-' != outf[0])) {
        if ((outfd = open(outf, O_RDWR)) >= 0) {
            if (ioctl(outfd, SG_GET_PACK_ID, &t) < 0) {
                /* not a scsi generic device so now try and open RDONLY */
                close(outfd);
            }
            else {
                out_is_sg = 1;
                res = 0;
                if (0 == tq)
                    res = ioctl(outfd, SCSI_IOCTL_TAGGED_DISABLE, &t);
                if (1 == tq)
                    res = ioctl(outfd, SCSI_IOCTL_TAGGED_ENABLE, &t);
                if (res < 0)
                    perror("sgq_dd512: SCSI_IOCTL_TAGGED(o) error");
            }
        }
        if (! out_is_sg) {
            if ((outfd = open(outf, O_WRONLY | O_CREAT, 0666)) < 0) {
                sprintf(ebuff, 
                        "sgq_dd512: could not open %s for writing\n", outf);
                perror(ebuff);
                return 1;
            }
            else if (seek > 0) {
                off_t offset = seek;

                offset *= BLOCK_SIZE;       /* could overflow here! */
                if (lseek(outfd, offset, SEEK_SET) < 0) {
                    sprintf(ebuff,
                "sgq_dd512: Couldn't seek to required position on %s", outf);
                    perror(ebuff);
                    return 1;
                }
            }
        }
    }
    if ((STDIN_FILENO == infd) && (STDOUT_FILENO == outfd)) {
        printf("Can't have both 'if' as stdin _and_ 'of' as stdout\n");
        return 1;
    }
    if (! (in_is_sg || out_is_sg)) {
        printf("Either 'if' or 'of' must be a scsi generic device\n");
        return 1;
    }
    in_num_sect = 0;
    out_num_sect = 0;
    if (0 == count)
        return 0;
    else if (count < 0) {
        if (in_is_sg) {
            if (0 != read_capacity(infd, &in_num_sect, &in_sect_sz)) {
                printf("Unable to read capacity on %s\n", inf);
                in_num_sect = -1;
            }
            else {
                if (in_num_sect > skip)
                    in_num_sect -= skip;
            }
        }
        if (out_is_sg) {
            if (0 != read_capacity(outfd, &out_num_sect, &out_sect_sz)) {
                printf("Unable to read capacity on %s\n", outf);
                out_num_sect = -1;
            }
            else {
                if (out_num_sect > seek)
                    out_num_sect -= seek;
            }
        }
        if (in_num_sect > 0) {
            if (out_num_sect > 0)
                count = (in_num_sect > out_num_sect) ? out_num_sect : 
                                                       in_num_sect;
            else
                count = in_num_sect;
        }
        else
            count = out_num_sect;
    }

    wrkOffset= SG_HEAD_SZ + SCSI_CMD10_LEN;
    if (0 != ctor_sgq_work(skip, count, &sgq_w)) {
        printf("Not enough user memory\n");
        return 1;
    }


/* Finally .... the main loop follows */
    
    while (count) {
        if (in_is_sg) {
            res = sg_read(infd, &sgq_w);
            if (0 != res) {
                printf("sg_read failed, skip=%d\n", skip);
                break;
            }
            else {
                k = sgq_w.ind_wr_buff;
                if (k >= 0)
                    in_full += sgq_w.blocks[k];
            }
        }
        else {
            blocks = (count > BLOCKS_PER_WBUFF) ? BLOCKS_PER_WBUFF : count;
            sgq_w.ind_wr_buff = 0;
            sgq_w.blocks[0] = blocks;
            sgq_w.state[0] = SGQ_READABLE;
            wrkp = sgq_w.buff[0] + wrkOffset;
            res = read(infd, wrkp, blocks * BLOCK_SIZE);
            if (res < 0) {
                sprintf(ebuff, "sgq_dd512: reading, skip=%d ", skip);
                perror(ebuff);
                break;
            }
            else if (res < blocks * BLOCK_SIZE) {
                count = 0;
                blocks = res / BLOCK_SIZE;
                if ((res % BLOCK_SIZE) > 0) {
                    blocks++;
                    in_partial++;
                }
            }
            in_full += blocks;
        }

#if 1
        k = sgq_w.ind_wr_buff;
        if (k >= 0) {
            blocks = sgq_w.blocks[k];
            if (out_is_sg) {
                wrkp = sgq_w.buff[k];
                res = sg_write(outfd, wrkp, blocks, seek);
                if (0 != res) {
                    printf("sg_write failed, seek=%d\n", seek);
                    break;
                }
                else
                    out_full += blocks;
            }
            else {
                wrkp = sgq_w.buff[k] + wrkOffset;
                res = write(outfd, wrkp, blocks * BLOCK_SIZE);
                if (res < 0) {
                    sprintf(ebuff, "sgq_dd512: writing, seek=%d ", seek);
                    perror(ebuff);
                    break;
                }
                else if (res < blocks * BLOCK_SIZE) {
                    printf("output file probably full, seek=%d ", seek);
                    blocks = res / BLOCK_SIZE;
                    out_full += blocks;
                    if ((res % BLOCK_SIZE) > 0)
                        out_partial++;
                    break;
                }
                else
                    out_full += blocks;
            }
            sgq_w.state[k] = SGQ_FREE;
        }
        else
            printf("do nothing on this write loop, count=%d\n", count);
#endif
        if (count > 0)
            count -= blocks;
        skip += blocks;
        seek += blocks;
    }

    dtor_sgq_work(&sgq_w);
    if (STDIN_FILENO != infd)
        close(infd);
    if (STDOUT_FILENO != outfd)
        close(outfd);
    printf("%d+%d records in\n", in_full, in_partial);
    printf("%d+%d records out\n", out_full, out_partial);

    if (num_sg_reads > 0) {
        anw = ((double)num_reads_waiting) / ((double)num_sg_reads);
        pqf = (((double)num_q_fulls) / ((double)num_sg_reads)) * 100.0;
        prqe = (((double)num_reads_zero) / ((double)num_sg_reads)) * 100.0;
        printf("  Avg_num_waits=%5.2f, q fulls=%5.2f%%, "
               "read_q empty=%5.2f%%, eagains=%d\n",
               anw, pqf, prqe, num_eagains);
    }
    if (0 != count) {
        printf("Some error occurred, count=%d\n", count);
        return 1;
    }
    return 0;
}
