/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Written (W) 1999-2010 Soeren Sonnenburg
 * Copyright (C) 1999-2009 Fraunhofer Institute FIRST and Max-Planck-Society
 * Copyright (C) 2010 Berlin Institute of Technology
 */

#include "lib/common.h"
#include "lib/io.h"
#include "features/Features.h"
#include "features/DotFeatures.h"
#include "kernel/LinearKernel.h"

using namespace shogun;

CLinearKernel::CLinearKernel()
: CDotKernel(0), normal(NULL), normal_length(0)
{
	properties |= KP_LINADD;
}

CLinearKernel::CLinearKernel(CDotFeatures* l, CDotFeatures* r)
: CDotKernel(0), normal(NULL), normal_length(0)
{
	properties |= KP_LINADD;
	init(l,r);
}

CLinearKernel::~CLinearKernel()
{
	cleanup();
}

bool CLinearKernel::init(CFeatures* l, CFeatures* r)
{
	CDotKernel::init(l, r);

	return init_normalizer();
}

void CLinearKernel::cleanup()
{
	delete_optimization();

	CKernel::cleanup();
}

void CLinearKernel::clear_normal()
{
	int32_t num = ((CDotFeatures*) lhs)->get_dim_feature_space();
	if (normal==NULL)
	{
		normal = new float64_t[num];
		normal_length=num;
	}

	memset(normal, 0, sizeof(float64_t)*normal_length);

	set_is_initialized(true);
}

void CLinearKernel::add_to_normal(int32_t idx, float64_t weight) 
{
	((CDotFeatures*) lhs)->add_to_dense_vec(
		normalizer->normalize_lhs(weight, idx), idx, normal, normal_length);
	set_is_initialized(true);
}

bool CLinearKernel::init_optimization(
	int32_t num_suppvec, int32_t* sv_idx, float64_t* alphas)
{
	clear_normal();

	for (int32_t i=0; i<num_suppvec; i++)
		add_to_normal(sv_idx[i], alphas[i]);

	set_is_initialized(true);
	return true;
}

bool CLinearKernel::delete_optimization()
{
	delete[] normal;
	normal_length=0;
	normal=NULL;
	set_is_initialized(false);

	return true;
}

float64_t CLinearKernel::compute_optimized(int32_t idx)
{
	ASSERT(get_is_initialized());
	float64_t result = ((CDotFeatures*) rhs)->
		dense_dot(idx, normal, normal_length);
	return normalizer->normalize_rhs(result, idx);
}
