/*
 * Copyright 2008 Klaus Triendl
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free 
 * Software Foundation, 51 Franklin Street, Fifth Floor, 
 * Boston, MA 02110-1301, USA.
*/

#include <sigc++/connection.h>
#include <sigc++/functors/mem_fun.h>
#include <sigc++/adaptors/bind.h>
#include <glib/gatomic.h>
#include "sigx/types.h"
#include "sigx/connection_wrapper.h"
#include "sigx/connection_handler.h"
#include "sigx/signal_wrapper_base.h"
#include "sigx/tunnel_functor.h"
#include "__sigx_pchfence__.h"


namespace sigx
{

connection_wrapper::connection_wrapper(): 
	m_sigc_conn(), 
	m_shared_disp(), 
	m_sigcconn_refcount()
{}


connection_wrapper::connection_wrapper(const shared_dispatchable& _A_disp, const shared_sigc_conn_ptr& _A_conn): 
	m_sigc_conn(_A_conn), 
	m_shared_disp(_A_disp), 
	m_sigcconn_refcount(new int(1))
{}

connection_wrapper::connection_wrapper(const connection_wrapper& other) throw(): 
	m_sigc_conn(other.m_sigc_conn), 
	m_shared_disp(other.m_shared_disp), 
	m_sigcconn_refcount(other.m_sigcconn_refcount)
{
	if (m_sigcconn_refcount)
		g_atomic_int_inc(m_sigcconn_refcount);
}

connection_wrapper::~connection_wrapper() throw()
{
	try
	{
		destroy_self();
	}
	catch (...)
	{}
}

void connection_wrapper::destroy_self()
{
	if (m_sigcconn_refcount)
	{
		// test whether we are the last one pointing to the remote 
		// sigc::connection and whether it's still valid;
		// destroy this connection if yes
		if ((TRUE == g_atomic_int_dec_and_test(m_sigcconn_refcount)	)	&& 
			(g_atomic_pointer_get(&*m_sigc_conn)					)	)
		{
			delete m_sigcconn_refcount;

			try
			{
				open_tunnel_with(
					sigc::compose(
						sigc::ptr_fun(&connection_handler::destroy), 
						sigc::retype_return<const sigc_connection_ptr*>(sigc::mem_fun(&shared_sigc_conn_ptr::get))
					), 
					m_shared_disp
				)
				// read volatile at the other side of the tunnel;
				(m_sigc_conn);
			}
			catch (const bad_dispatcher&)
			{
				// silently ignore;
				// don't mind if the message can't get dispatched, the server 
				// thread has probably already ended and deleted the 
				// sigc::connection
			}
		}

		m_sigcconn_refcount = 0;
		m_sigc_conn.reset();
		// no need to destroy m_shared_disp as it is destroyed by the dtor or assigned
		// anew in operator =
	}
}

connection_wrapper& connection_wrapper::operator =(const connection_wrapper& other)
{
	if (this == &other)
		return *this;

	destroy_self();

	m_sigc_conn = other.m_sigc_conn;
	m_shared_disp = other.m_shared_disp;
	m_sigcconn_refcount = other.m_sigcconn_refcount;

	if (m_sigcconn_refcount)
		g_atomic_int_inc(m_sigcconn_refcount);
	
	return *this;
}

bool connection_wrapper::empty() const
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return false;

	return 
		open_sync_tunnel_with(
			sigc::mem_fun(&sigc::connection::empty), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		)(sigc::ref(*m_sigc_conn));
}

bool connection_wrapper::connected() const
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return false;

	return 
		open_sync_tunnel_with(
			sigc::mem_fun(&sigc::connection::connected), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		)(sigc::ref(*m_sigc_conn));
}

bool connection_wrapper::blocked() const
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return false;

	return 
		open_sync_tunnel_with(
			sigc::mem_fun(&sigc::connection::blocked), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		)(sigc::ref(*m_sigc_conn));
}

bool connection_wrapper::block(bool should_block /*= true*/)
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return false;

	return 
		open_sync_tunnel_with(
			sigc::mem_fun(&sigc::connection::block), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		// the ref() avoids copying the volatile pointer
		)(sigc::ref(*m_sigc_conn), should_block);
}

bool connection_wrapper::unblock()
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return false;

	return 
		open_sync_tunnel_with(
			sigc::mem_fun(&sigc::connection::unblock), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		// the ref() avoids copying the volatile pointer
		)(sigc::ref(*m_sigc_conn));
}

void connection_wrapper::disconnect()
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return;

	return 
		open_tunnel_with(
			sigc::compose(
				sigc::mem_fun(&sigc::connection::disconnect), 
				sigc::mem_fun(&shared_sigc_conn_ptr::operator *)
			), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		// don't ref() because it's an async call
		)(m_sigc_conn);
}

connection_wrapper::operator bool()
{
	if ((!m_sigcconn_refcount					)	|| 
		(!g_atomic_pointer_get(&*m_sigc_conn)	)	)
		return false;

	return 
		open_sync_tunnel_with(
			sigc::mem_fun(&sigc::connection::operator bool), 
			m_shared_disp
		// read volatile at the other side of the tunnel;
		// the ref() avoids copying the volatile pointer
		)(sigc::ref(*m_sigc_conn));
}


} // namespace sigx
