/*
    Silky - A GTK+ client for SILC.
    Copyright (C) 2003,2004 Toni Willberg

    - Buddy list management functions

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    http://silky.sourceforge.net/

*/

#include "includes.h"

SilkyBuddy *buddies = NULL;

extern xmlDocPtr config_buddies;
extern SilkyXMLElement xml_elements[NUM_CONFIG][NUM_CM];
extern gchar *buddiespath;
extern silkyStruct *silky;

SilkyBuddy *buddy_find_by_nick(gchar *nick) {
  SilkyBuddy *b;

  for( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->nickname, nick) ) return b;
  }

  return NULL;
}

SilkyBuddy *buddy_find_by_fp(gchar *fp) {
  SilkyBuddy *b;

  for( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fp) ) return b;
  }

  return NULL;
}


void watch_add(gchar *nick) {

  SilcBuffer idp;

  if (!silky || !silky->conn || !silky->conn->local_id) {
    debug("can't find own id, aborting");
    return;
  }

  debug("adding watch for '%s'", nick);

  idp = silc_id_payload_encode(silky->conn->local_id, SILC_ID_CLIENT);
  if (!idp) {
    debug("unable to encode payload");
    return;
  }

  silc_client_command_send(silky->client, silky->conn,
				 SILC_COMMAND_WATCH, 0, 4,
				 1, idp->data, idp->len,
				 2, nick, strlen(nick),
				 3, NULL, NULL,
				 4, NULL, NULL
				 );

  return;
}


void watch_del(gchar *nick) {
  SilcBuffer idp;

  if (!silky || !silky->conn || !silky->conn->local_id) {
    debug("can't find own id, aborting");
    return;
  }

  debug("deleting watch for '%s'", nick);

  idp = silc_id_payload_encode(silky->conn->local_id, SILC_ID_CLIENT);
  silc_client_command_send(silky->client, silky->conn,
				 SILC_COMMAND_WATCH, 0, 4,
				 1, idp->data, idp->len,
				 2, NULL, NULL,
				 3, nick, strlen(nick),
				 4, NULL, NULL
				 );

  return;
}



gboolean buddy_add(gchar *nick, gchar *fingerprint) {
  SilkyBuddy *b, *newb;


  if( buddy_find_by_fp(fingerprint) ) {
    debug("Buddy with this fingerprint is already in the list, ignoring");
    return FALSE;
  }

  debug("creating struct for new buddy");
  newb = malloc( sizeof( SilkyBuddy ) );
  newb->next = NULL;

  /* let's make sure that no string in here is NULL */
  if( nick ) newb->nickname = g_strdup(nick);
    else newb->nickname = g_strdup("");
  newb->realname = NULL;
  if( fingerprint ) newb->fingerprint = g_strdup(fingerprint);
    else newb->fingerprint = g_strdup("");

  newb->languages = NULL;
  newb->preferred_contact = NULL;
  newb->timezone = NULL;
  newb->geolocation = NULL;

  debug("Adding '%s' (%s)", newb->nickname, newb->fingerprint);

  if( !buddies ) {
    debug("first buddy, creating linked list");
    buddies = newb;
  }  else {
    debug("there are already some buddies, appending");
    for( b = buddies; b->next; b = b->next ) {}
    b->next = newb;
  }

  return TRUE;
}

/**
 * buddy_set_mode_by_fp:
 * @fingerprint: the fingerprint of user to modify
 * @mode: new mode of the user
 *
 * This function changes the user mode of a user stored in Buddy List.
 *
 * Returns: gboolean
 **/
gboolean buddy_set_mode_by_fp(gchar *fingerprint, SilcUInt32 mode) {
 SilkyBuddy *b;

 debug("setting umode %d");

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      b->mode = mode;
      return TRUE;
    }
  }

  return FALSE;
}


/**
 * buddy_init_watchlist:
 *
 * This function sends WATCH command to server for every buddy
 *
 * Returns: nothing
 **/
void buddy_init_watchlist() {
  SilkyBuddy *b;

  debug("sending watch command for all users in the buddy list, if any");
  for ( b = buddies; b; b = b->next ) {
    watch_add(b->nickname);
  }
}

gboolean buddy_set_nick_by_fp(gchar *fingerprint, gchar *nick) {
  SilkyBuddy *b;

  if( !nick ) {
    debug("nick == NULL");
    return FALSE;
  }

  debug("setting nick to '%s'", nick);

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      g_free(b->nickname);
      b->nickname = g_strdup(nick);
      return TRUE;
    }
  }

  return FALSE;
}

gboolean buddy_set_realname_by_fp(gchar *fingerprint, gchar *realname) {
  SilkyBuddy *b;

  if( !realname ) {
    debug("realname == NULL");
    return FALSE;
  }

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      g_free(b->realname);
      b->realname = g_strdup(realname);
      return TRUE;
    }
  }

  return FALSE;
}

gboolean buddy_set_languages_by_fp(gchar *fingerprint, gchar *languages) {
  SilkyBuddy *b;

  if( !languages ) {
    debug("languages == NULL");
    return FALSE;
  }

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      g_free(b->languages);
      b->languages = g_strdup(languages);
      return TRUE;
    }
  }

  return FALSE;
}

gboolean buddy_set_pref_contact_by_fp(gchar *fingerprint, gchar *pc) {
  SilkyBuddy *b;

  if( !pc ) {
    debug("pc == NULL");
    return FALSE;
  }

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      g_free(b->preferred_contact);
      b->preferred_contact = g_strdup(pc);
      return TRUE;
    }
  }

  return FALSE;
}

gboolean buddy_set_timezone_by_fp(gchar *fingerprint, gchar *tz) {
  SilkyBuddy *b;

  if( !tz ) {
    debug("tz == NULL");
    return FALSE;
  }

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      g_free(b->timezone);
      b->timezone = g_strdup(tz);
      return TRUE;
    }
  }

  return FALSE;
}

gboolean buddy_set_geolocation_by_fp(gchar *fingerprint, gchar *gl) {
  SilkyBuddy *b;

  if( !gl ) {
    debug("gl == NULL");
    return FALSE;
  }

  for ( b = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(b->fingerprint, fingerprint) ) {
      g_free(b->geolocation);
      b->geolocation = g_strdup(gl);
      return TRUE;
    }
  }

  return FALSE;
}


/*
  Called as a UI callback when user wants to remove a buddy. 
*/
gboolean ui_buddy_remove (GtkWidget *wid, gpointer *fingerprint) {
  debug("asking to remove a buddy with fingerprint '%s'", *fingerprint);
  buddy_remove_by_fp(*fingerprint);
  return TRUE;
}


gboolean buddy_remove_by_fp(gchar *fingerprint) {
  SilkyBuddy *b, *prevb;
  gint found = 0;

  debug("trying to remove buddy");

  if( !buddies ) {
    debug("No buddies, returning");
    return FALSE;
  }

  for( b = buddies, prevb = buddies; b; b = b->next ) {
    if( !g_ascii_strcasecmp(fingerprint, b->fingerprint) ) {
      debug("found it, nick '%s'", b->nickname);
      found++;
      break;
    }
    prevb = b;
  }

  if( !found ) {
    debug("could not find buddy to remove, returning");
    return FALSE;
  }

  debug("removing buddy from list");
  if( b == buddies ) {
    debug("it is first buddy");
    buddies = buddies->next;
  } else {
    debug("not first buddy");
    prevb->next = b->next;
  }

  debug("freeing buddy from memory");
  g_free(b);

  return TRUE;
}

gboolean init_buddies(gchar *buddiespath) {
  debug("Trying to load buddies from %s", buddiespath);
  if( g_file_test(buddiespath, G_FILE_TEST_EXISTS) ) {
    debug("file found");

    if( !(config_buddies = xml_read_config(CONFIG_BUDDIES, buddiespath)) ) {
      debug("Failed to load buddies from file %s", buddiespath);
      return FALSE;
    }

    if( !parse_buddies() ) {
      debug("Failed to parse buddies, none loaded");
      /* FIXME display warning to user? */
      return TRUE;
    }

    debug("Buddies succesfully loaded, continuing");
    return TRUE;
  }

  return FALSE;
}

gboolean parse_buddies() {
  xmlXPathObjectPtr result;
  xmlXPathContextPtr context;
  xmlNodePtr node, n;
  gint i;

  gchar *nick = NULL;
  gchar *realname = NULL;
  gchar *fp = NULL;

  xml_cleanup_config(CONFIG_BUDDIES);

  debug("Populating SilkyBuddies list...");

  context = xmlXPathNewContext(config_buddies);

  result = xmlXPathEval(g_strdup_printf("/%s/%s", xml_elements[CONFIG_BUDDIES][CB_ROOT].name, xml_elements[CONFIG_BUDDIES][CB_BUDDY].name), context);
  if( !result ) {
    debug("No result found for buddies");
    return FALSE;
  }

  if( xmlXPathNodeSetIsEmpty(result->nodesetval) ) {
    debug("Zaroo buddies found.");
    return TRUE;
  }

  xmlXPathFreeContext(context);
  for( i = 0; i < result->nodesetval->nodeNr; i++ ) {
    node = result->nodesetval->nodeTab[i];

    for( n = node->children; n && g_ascii_strcasecmp(n->name, xml_elements[CONFIG_BUDDIES][CB_BUDDY].name) ; n = n->next ) {
      if( !g_ascii_strcasecmp(n->name, xml_elements[CONFIG_BUDDIES][CB_BUDDY_NICKNAME].name) )
        nick = g_strdup(xmlNodeGetContent(n));

      if( !g_ascii_strcasecmp(n->name, xml_elements[CONFIG_BUDDIES][CB_BUDDY_REALNAME].name) )
        realname = g_strdup(xmlNodeGetContent(n));

      if( !g_ascii_strcasecmp(n->name, xml_elements[CONFIG_BUDDIES][CB_BUDDY_FINGERPRINT].name) )
        fp = g_strdup(xmlNodeGetContent(n));
    }

    if( (!fp || !g_utf8_strlen(fp, -1)) || (!nick || !g_utf8_strlen(nick, -1)) ) {
      debug("Fingerprint empty, ignoring buddy");
    } else {
      debug("Adding buddy '%s', fingerprint '%s'", nick, fp);
      buddy_add(nick, fp);
      buddy_set_realname_by_fp(fp, realname);
    }
  }

  xmlXPathFreeObject(result);
  if( !i ) debug("Zaroo buddies found.");
  return TRUE;
}

gboolean save_buddies() {
  SilkyBuddy *b;
  xmlNodePtr bn, n;

  debug("Starting to save buddies to file '%s'", buddiespath);

  xmlFreeDoc(config_buddies);

  config_buddies = xml_create_config(CONFIG_BUDDIES);

  for( b = buddies; b; b = b->next ) {
    debug("Saving '%s' (%s)", b->nickname, b->fingerprint);
    bn = xmlNewTextChild(xmlDocGetRootElement(config_buddies), NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY].name, NULL);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_NICKNAME].name, b->nickname);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_REALNAME].name, b->realname);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_FINGERPRINT].name, b->fingerprint);

    bn = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_ATTR].name, NULL);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_ATTR_LANGUAGES].name, b->languages);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_ATTR_PREFCONTACT].name, b->preferred_contact);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_ATTR_TIMEZONE].name, b->timezone);
    n = xmlNewTextChild(bn, NULL, xml_elements[CONFIG_BUDDIES][CB_BUDDY_ATTR_GEOLOCATION].name, b->geolocation);
  }

  xml_save_config(CONFIG_BUDDIES, buddiespath);
  return TRUE;
}


/*
  Called from GTK when user selects an user from the buddylist.
*/
gboolean buddy_selected (GtkTreeView *treeview, GdkEventButton *event) {
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;

  gchar *nick;
  gchar *fingerprint;

  GtkMenu *menu;
  GtkWidget *menuitem;
  debug("selected");
  //  debug("button: %d", event->button);

  if (event->button == 3) {
    debug("right click");

    selection = gtk_tree_view_get_selection(treeview);

    /* popup the menu only if something was selected */
    if (gtk_tree_selection_get_selected (selection, &model, &iter))
      {
	gtk_tree_model_get (model, &iter, 2, &nick, -1);
	gtk_tree_model_get (model, &iter, 3, &fingerprint, -1);

	debug ("You opened menu for '%s'", nick);

	/* create the menu */
	menu = GTK_MENU(gtk_menu_new());


	/* ADD MENU items and signal handlers*/

	/* WHOIS */
	/*
	menuitem = gtk_menu_item_new_with_label(_("Who is"));
	g_signal_connect(menuitem, "activate", G_CALLBACK(on_command_whois), clientid);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	gtk_widget_show (menuitem);
	*/

	/* send a message */
	/*
	menuitem = gtk_menu_item_new_with_label(_("Send message"));
	g_signal_connect(menuitem, "activate", G_CALLBACK(on_command_query), clientid);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	gtk_widget_show (menuitem);
	*/


	/* a separator here */
	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	gtk_widget_show(menuitem);

	/* remove from buddylist */
	menuitem = gtk_menu_item_new_with_label(_("Remove Buddy"));
	g_signal_connect(menuitem, "activate", G_CALLBACK(ui_buddy_remove), fingerprint);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
	gtk_widget_show (menuitem);

	gtk_menu_popup (menu, NULL, NULL, NULL, NULL,
			event->button, event->time);
      }
  }
  else {
    debug("left or middle click");
  }

  return FALSE; /* tell GTK to use selector */


}
