/*
    Silky - A GTK+ client for SILC.
    Copyright (C) 2003,2004 Toni Willberg

    - SILC keys management functions

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    http://silky.sourceforge.net/

*/

#include "includes.h"
#include "files.h"

extern silkyStruct *silky;

extern gchar *silkydir;

extern gchar *pubkey; /* own key */
extern gchar *privkey; /* own key */
extern gchar *silkypath; /* from main.c */
extern GladeXML *xmlmain;

/* Utility function to check if received public key is already stored.
 * Returns TRUE if key exists, FALSE if key is yet unknown.
 */
gboolean silky_check_dir_for_key(const char * dir, SilcPublicKey pkey) {
  DIR *pubkeydir;
  struct dirent *entity = NULL;
  SilcPublicKey lkey;
  struct stat sb;
  int ret = 0;
  char fpath[256];
  gboolean try;

  debug("Checking for peer's existing SILC public key...");
	
  /* Open directory for reading */
  pubkeydir = opendir(dir);
  if (pubkeydir == NULL)
    return FALSE;

  /* Init entity so it's not NULL */
  entity = readdir(pubkeydir);

  /* Read whole directory */
  while (entity != NULL) {
    entity = readdir(pubkeydir);
    if (entity != NULL) {
      /* Create path to file */
      memset(fpath, 0, 256);
      g_snprintf(fpath, 256, "%s%c%s", dir, DIR_SEPARATOR, entity->d_name);

      /* Check filetype */
#ifdef SILKY_WIN32
      ret = stat(fpath, &sb);
#else
      ret = lstat(fpath, &sb);
#endif
      if (S_ISREG(sb.st_mode)) {
        /* Regular file, let's try to load a SILC public key */
        try = silc_pkcs_load_public_key(fpath, &lkey, SILC_PKCS_FILE_PEM);
        if (!try)
          try = silc_pkcs_load_public_key(fpath, &lkey, SILC_PKCS_FILE_BIN);

        if (try) {
          /* Okay, it really was a SILC public key */
          if(silc_pkcs_public_key_compare(pkey, lkey)) {
            debug("Key %s matches", fpath);
            closedir(pubkeydir);
            return TRUE;
          }
        }
      }
    }
  }

  closedir(pubkeydir);
  debug("Received unknown key");
  return FALSE;
}


/* this is called when generating keys, as a separate thread */

void silky_keygen (SilkySilcKey *key) {

  gint ret;

  /* create the key */
  debug("silky_keygen [thread]");
  debug("calling silc_create_key_pair()");
  ret = silc_create_key_pair(key->cipher, key->length,
			     pubkey, privkey, /* global variables/pointers */
			     key->identifier, key->passphrase,
			     &silky->client->pkcs,
			     &silky->client->public_key,
			     &silky->client->private_key, FALSE);
  debug("returned from silc_create_key_pair()");

  if (ret) {
    debug("success");
  }
  else {
    debug("failed");
  }

  debug("setting key->ready = TRUE");
  key->ready = TRUE; /* tell the caller we are ready */


  debug("calling g_thread_exit()");

  /* commit suicide */
  g_thread_exit(&ret);
}


/*
  Asks user if he wants to accept public key,
  and saves it
*/

gboolean silc_save_public_key(SilcClient client, SilcClientConnection conn,
                                const char *name, SilcSocketType conn_type,
                                unsigned char *pk, SilcUInt32 pk_len,
                                SilcSKEPKType pk_type,
                                SilcVerifyPublicKey completion, void *context)
{
  int i;
  char file[256], path[256], path2[256], path3[256];
  char *ipf, *hostf = NULL, *fpf = NULL;
  struct stat st;

  GtkLabel *accept_server_key_label;
  GtkWidget *dialog;
  gchar *fingerprint;
  gchar *babbleprint;
  gint result;

  char *entity = "unknown"; /* use to figure out to which dir save the key */

  if  (conn_type == SILC_SOCKET_TYPE_SERVER || conn_type == SILC_SOCKET_TYPE_ROUTER) {
    entity = strdup("server");
  }
  if (conn_type == SILC_SOCKET_TYPE_CLIENT) {
    entity = strdup("client");
  }

  debug("key type: %s (%d)", entity, conn_type);

  SilcPublicKey pubkey;
  bool ret;
  unsigned char *encpk;

  debug("silc_save_public_key()");


  if (pk_type != SILC_SKE_PK_TYPE_SILC) {
    printconsole(_("Received unknown key type."));
    if (completion)
      completion(FALSE, context);
    return FALSE;
  }

  memset(file, 0, sizeof(file));
  memset(path, 0, sizeof(path));
  memset(path2, 0, sizeof(path2));
  memset(path3, 0, sizeof(path2));

  if (conn_type == SILC_SOCKET_TYPE_SERVER ||
      conn_type == SILC_SOCKET_TYPE_ROUTER) {
    if (!name) {
      /* Server key with IP address and port in name */
      g_snprintf(file, sizeof(file) - 1, "%skey_%s_%d.pub", entity,
               conn->sock->ip, conn->sock->port);
      g_snprintf(path, sizeof(path) - 1, "%s%c%skeys%c%s", silkydir, DIR_SEPARATOR, entity, DIR_SEPARATOR, file);

      /* Server key with DNS name and port in name */
      g_snprintf(file, sizeof(file) - 1, "%skey_%s_%d.pub", entity,
               conn->sock->hostname, conn->sock->port);

      g_snprintf(path2, sizeof(path2) - 1, "%s%c%skeys%c%s", silkydir, DIR_SEPARATOR, entity, DIR_SEPARATOR, file);

      /* Additional key, this with only the fingerprint as name */
      fingerprint = silc_hash_fingerprint(NULL, pk, pk_len);
      for (i = 0; i < strlen(fingerprint); i++)
        if (fingerprint[i] == ' ')
          fingerprint[i] = '_';

      g_snprintf(file, sizeof(file) -1, "%skey_%s.pub", entity, fingerprint);
      g_snprintf(path3, sizeof(path) -1, "%s%c%skeys%c%s", silkydir, DIR_SEPARATOR, entity, DIR_SEPARATOR, file);

      silc_free(fingerprint);

      ipf = path;
      hostf = path2;
      fpf = path3;
    } else {
      g_snprintf(file, sizeof(file) - 1, "%skey_%s_%d.pub", entity,
               name, conn->sock->port);
      g_snprintf(path, sizeof(path) - 1, "%s%c%skeys%c%s", silkydir, DIR_SEPARATOR, entity, DIR_SEPARATOR, file);

      ipf = path;
    }
  } else {
    /* Replace all whitespaces with `_'. */
    fingerprint = silc_hash_fingerprint(NULL, pk, pk_len);
    for (i = 0; i < strlen(fingerprint); i++)
      if (fingerprint[i] == ' ')
        fingerprint[i] = '_';

    g_snprintf(file, sizeof(file) - 1, "%skey_%s.pub", entity, fingerprint);
    g_snprintf(path, sizeof(path) - 1, "%s%c%skeys%c%s", silkydir, DIR_SEPARATOR, entity, DIR_SEPARATOR, file);

    silc_free(fingerprint);

    ipf = path;
  }

  /* Check whether this key already exists */
  if (stat(ipf, &st) < 0 && (!hostf || stat(hostf, &st) < 0)
      && (!fpf || stat(fpf, &st))) {

    /* Key does not exist, save it */
    debug("new, not stored key");
    printconsole(_("Received a new, not stored key"));

    /* Decode key and set identifier info */
    ret = silc_pkcs_public_key_decode(pk, pk_len, &pubkey);
    if (!ret) {
      printconsole(_("Could not decode received public key."));
      return FALSE;
    }

    encpk = silc_pkcs_public_key_encode(pubkey, &pk_len);
    if (!encpk) {
      printconsole(_("Could not encode received public key."));
      silc_pkcs_public_key_free(pubkey);
      return FALSE;
    }
    else {


      /* does the user accept the key, before saving? */

      /* create dialog */
      dialog = glade_xml_get_widget (xmlmain, "dialog_accept_new_key");

      /* add text to label */
      accept_server_key_label = GTK_LABEL(glade_xml_get_widget (xmlmain, "accept_new_key_text"));

      fingerprint = silc_hash_fingerprint(NULL, pk, pk_len);
      babbleprint = silc_hash_babbleprint(NULL, pk, pk_len);
      gtk_label_set_text(GTK_LABEL(accept_server_key_label), g_strdup_printf("%s\n%s\n", fingerprint, babbleprint) );
      g_free(fingerprint);
      g_free(babbleprint);

      /* show the dialog, blocking UI */
      debug("showing dialog...");

      gdk_threads_enter();
      result = gtk_dialog_run (GTK_DIALOG(dialog));
      gdk_threads_leave();

      debug("returned from the dialog");

      gtk_widget_hide (GTK_WIDGET(dialog));

      if (result == GTK_RESPONSE_OK) {
	/* user accepted the key */
	debug("user accepted, saving to '%s' '%s', '%s'", path, path2, path3);

	if (path && strlen(path) ) {
	  silc_pkcs_save_public_key(path, pubkey, SILC_PKCS_FILE_PEM);
	}

	if (path2 && strlen(path2)) {
	  silc_pkcs_save_public_key(path2, pubkey, SILC_PKCS_FILE_PEM);
	}

	if(path3 && strlen(path3)) {
	  silc_pkcs_save_public_key(path3, pubkey, SILC_PKCS_FILE_PEM);
	}

	silc_pkcs_public_key_free(pubkey);

	printconsole(_("Key saved.")); /* was it? */

	return TRUE;
      }
      else {
	/* user didn't accept the key */
	debug("user cancelled");
	return FALSE;
      }


    } /* if (!encpk) */


  }
  else {
    debug("known key");
    printconsole(_("This server key is known."));
    return TRUE;
  }
}





/*
  runs after user fills fields to the keygen form,
*/

gint silky_keygen_ui () {

  guint progress_timeout;
  GtkEntry *ekey_name;
  GtkEntry *ekey_email;
  gchar *key_passphrase1;
  GtkEntry *ekey_passphrase1;
  gchar *key_passphrase2;
  GtkEntry *ekey_passphrase2;
  GtkWidget *dialog_keygen_progressbar;
  GtkWidget *keygen_progressbar;
  GThread *thread_id;
  SilkySilcKey key;

  //  gtk_dialog_run(GTK_DIALOG(glade_xml_get_widget (xmlmain, "dialog_firstrun_manual")));
/* fixme; check ret and return -1 if user wants to cancel! */

  /* hide the dialogs */
  //  gtk_widget_hide(glade_xml_get_widget (xmlmain, "dialog_firstrun"));
  // gtk_widget_hide(glade_xml_get_widget (xmlmain, "dialog_firstrun_manual"));


  /* get user's data */
  ekey_name = GTK_ENTRY(glade_xml_get_widget (xmlmain, "fr_key_name"));
  key.name = strdup(gtk_entry_get_text(ekey_name));

  ekey_email = GTK_ENTRY(glade_xml_get_widget (xmlmain, "fr_key_email"));
  key.email = strdup(gtk_entry_get_text(ekey_email));

  ekey_passphrase1 = GTK_ENTRY(glade_xml_get_widget (xmlmain, "fr_key_passphrase1"));
  key_passphrase1 = strdup(gtk_entry_get_text(ekey_passphrase1));

  ekey_passphrase2 = GTK_ENTRY(glade_xml_get_widget (xmlmain, "fr_key_passphrase2"));
  key_passphrase2 = strdup(gtk_entry_get_text(ekey_passphrase2));

  if (!strcmp (key_passphrase1, key_passphrase2) == 0) {
    debug("pass wrong");
    errordialog(_("Passphrases don't match! Try again."));
    memset(key_passphrase1, 0, sizeof(key_passphrase1));
    memset(key_passphrase2, 0, sizeof(key_passphrase2));
    free(key_passphrase1); free(key_passphrase2);

    return 0; /* go back to idle mode, gtk will call us again */

  }
  else {
    debug("pass equals, continuing");
  }

  key.passphrase = g_strdup(key_passphrase1);

  memset(key_passphrase1, 0, sizeof(key_passphrase1));
  memset(key_passphrase2, 0, sizeof(key_passphrase2));
  free(key_passphrase1); free(key_passphrase2);


  /* generate identifier, 'RN=real name, HN=localhost, E=email@address' */
  key.identifier = g_strconcat("RN=", key.name, ", HN=localhost, E=", key.email , NULL);
  debug("Using identifier: '%s'", key.identifier);

  /* verify that the keys are there */
  if (!pubkey || !privkey) {
    g_error("can't find the key file names!");
    /* aborts */
  }


  /* show the progress dialog */
  keygen_progressbar = glade_xml_get_widget (xmlmain, "keygen_progressbar");
  progress_timeout = gtk_timeout_add(50, (GSourceFunc)silky_progressbar_update, keygen_progressbar); /* update the bar every X second */
  dialog_keygen_progressbar = glade_xml_get_widget (xmlmain, "dialog_keygen_progress");


  key.length = (gint)g_strtod(gtk_entry_get_text(GTK_ENTRY(glade_xml_get_widget (xmlmain, "fr_key_length"))), NULL);
  key.cipher = g_strdup("rsa"); /* fixme; put this into the ui */

  debug("calling g_thread_create(silky_keygen)");
  thread_id = g_thread_create( (GThreadFunc)silky_keygen, &key, TRUE, NULL);

  /* this will block until silky_keygen() enables the OK button in the dialog */
  debug("showing progress dialog");
  /* I18N This is a window title */
  gtk_window_set_title(GTK_WINDOW(dialog_keygen_progressbar), g_strdup_printf("Silky v%s : %s", SILKY_VERSION, _("Generating SILC keys...")));
  gtk_widget_show_all (dialog_keygen_progressbar);

  /* we sit here and wait */
  while (key.ready != TRUE) {
    gtk_main_iteration_do(FALSE); /* give GTK some CPU time, non-blocking call */
  }

  debug("key.ready was set TRUE");

  /* stop updating the bar */
  gtk_timeout_remove(progress_timeout);

  if (!GTK_IS_PROGRESS_BAR(keygen_progressbar)) {
    /* this happens if user destroys the progress dialog while we generate the key,
       quitting */
    debug("progress dialog destroyed by user, returning -1");
    return -1;
  }


  /* set text and fraction */
  gtk_progress_bar_set_text( GTK_PROGRESS_BAR(keygen_progressbar), N_("Ready! Click OK to continue."));
  gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR(keygen_progressbar), 1 ); /* set 100% */
  gtk_widget_set_sensitive(glade_xml_get_widget (xmlmain, "button_keygen_ok"), TRUE);

  debug("now waiting for the user");
  while (gtk_events_pending ()) {
    gtk_main_iteration ();
  }

  /* lets wait until user presses OK button */
  gtk_dialog_run (GTK_DIALOG (dialog_keygen_progressbar));

  debug("dialog returned");


  /* hide the dialog window */
  gtk_widget_hide(dialog_keygen_progressbar); /* get rid of the window */


  return 1;
}
